#include "ComponentSpecification.hpp"
#include "PythonException.hpp"
#include "Component.hpp"
#include "regex.hpp"
#include <boost/program_options.hpp>
#include <boost/python/stl_iterator.hpp>

namespace po = boost::program_options;
namespace bp = boost::python;

namespace slmotion {
  namespace python {
      /**
       * The typical constructor: the name of the component, and a
       * key-value dictionary that can be used to construct the actual
       * component.
       */
    ComponentSpecification::ComponentSpecification(const std::string& componentName,
                                                   const boost::python::dict& opts) :
      componentName(componentName) {
      // assess types
      boost::program_options::options_description desc = Component::getConfigurationFileOptionsDescriptionByComponent(componentName);
      const std::vector<boost::shared_ptr<po::option_description>>& optds = desc.options();

      // remove prefixes and create a type map
      std::map<std::string, std::string> optionToType;

      for (auto it = optds.cbegin(); it != optds.cend(); ++it) {
        std::string optionName = (*it)->long_name();
        std::regex r(componentName + "\\.(\\w*)$");
        std::smatch m;
        if (std::regex_match(optionName, m, r))
          optionName = m[1];
        else
          throw PythonException("Malformatted option description name: a name"
                                " prefixed with the option name was expected,"
                                " but got " + optionName);

        boost::any defaultValue;
        boost::shared_ptr<const boost::program_options::value_semantic> semantic = (*it)->semantic();
        if (!semantic->apply_default(defaultValue))
          throw PythonException("Error: the option \"" + optionName + "\" has "
                                "no default value! Cannot determine its type.");
        optionToType[optionName] = string_demangle(defaultValue.type().name());
      }

      boost::python::list keys = opts.keys();
      for (boost::python::stl_input_iterator<std::string> it(keys), end; it != end; ++it) {
        boost::python::object value = opts[boost::python::str(*it)];
        boost::any value_any;
        if (optionToType.count(*it)) {
          try {
            if (optionToType[*it] == 
                string_demangle(typeid(std::string).name())) {
              std::string s = boost::python::extract<std::string>(value);
              value_any = boost::any(s);
            }
            else if (optionToType[*it] == "int") {
              int s = boost::python::extract<int>(value);
              value_any = boost::any(s);
            }
            else if (optionToType[*it] == "double") {
              double s = boost::python::extract<double>(value);
              value_any = boost::any(s);
            }
            else if (optionToType[*it] == "bool") {
              bool s = boost::python::extract<bool>(value);
              value_any = boost::any(s);
            }
            else 
              throw PythonException("Component \"" + componentName + 
                                    "\": Option \"" + *it + 
                                    "\": Invalid type specified: " +
                                    optionToType[*it]);
          }
          catch (boost::python::error_already_set&) {
            std::cerr << "Error: could not convert the argument given as "
              "option \"" + *it + "\" for the component \"" + componentName +
              "\": \"" + optionToType[*it] + "\" was expected." << std::endl;
            throw;
          }
        }
        else if (componentName == "PythonComponent") {
          // python components get special treatment because we do not
          // know their options beforehand
          boost::python::object value = opts[boost::python::str(*it)];
          std::string valueClassName = bp::extract<std::string>(value.attr("__class__").attr("__name__"));
          if (valueClassName == "float") {
            double d = bp::extract<double>(value);
            value_any = boost::any(d);
          }
          else if (valueClassName == "int") {
            int i = bp::extract<int>(value);
            value_any = boost::any(i);
          }
          else if (valueClassName == "bool") {
            bool b = bp::extract<bool>(value);
            value_any = boost::any(b);
          }
          else if (valueClassName == "str") {
            std::string s = bp::extract<std::string>(value);
            value_any = boost::any(s);
          }
          else
            throw PythonException("Don't know how to convert a " + valueClassName + " to a valid option type!");
        }
        else
          throw PythonException("The component " + componentName + " does not"
                                " have an option called \"" + *it + "\"");
        this->opts[*it] = value_any;
      }
    }
  }
}
