#include "CsvImporter.hpp"
#include <boost/algorithm/string.hpp>
#include "regex.hpp"
#include <fstream>

namespace slmotion {
  static CsvImporter DUMMY(true);

  void CsvImporter::process(frame_number_t) {
    assert(false && "This component cannot be called for individual frames.");
  }



  boost::program_options::options_description CsvImporter::getCommandLineOptionsDescription() const {
    boost::program_options::options_description opts("CSV Importer options");
    opts.add_options()("in-csv-file", 
                       boost::program_options::value<std::string>(),
                       "Comma-separated list of files to read in");
    opts.add_options()("csv-fields", 
                       boost::program_options::value<std::string>(),
                       "Semicolon-separated list of field specifications, each "
                       "corresponding to its respective CSV (ie. the number of "
                       "specifications and input CSV files must match). Each "
                       "specification consists of a comma-separated list of "
                       "key-and-list-of-indices pairs; the lists of indices are "
                       "separated from the key by a colon and each index is also"
                       " separated from one another by a colon. The indices are "
                       "0-based column indices of the values on CSV rows. An "
                       "entry may be either a single number or a dash-separated "
                       "range of numbers. As to keys, \"framenr\" is special: it"
                       "is interpreted as to having the 0-based frame number of "
                       "the corresponding row and must always be a single index."
                       " Unless framenr is specified, each row is assumed to "
                       "have originated from a frame number with the same "
                       "0-based row number, unless the whole file contains only "
                       "one row, in which case the values are assumed to be "
                       "global. An example of a valid string of specifications: "
                       "framenr:0,head_pos:1-3,lefthand_pos:4:5:6,"
                       "righthand_pos:7-8:9; global_x:0,global_y:1,global_z:2");
    return opts;
  }



  boost::program_options::options_description CsvImporter::getConfigurationFileOptionsDescription() const {
    boost::program_options::options_description opts;
    opts.add_options()("CsvImporter.filenames", 
                       boost::program_options::value<std::string>()->default_value(""),
                       "Comma-separated list of files to read in");
    opts.add_options()("CsvImporter.fields", 
                       boost::program_options::value<std::string>()->default_value(""),
                       "Semicolon-separated list of field specifications, each "
                       "corresponding to its respective CSV (ie. the number of "
                       "specifications and input CSV files must match). Each "
                       "specification consists of a comma-separated list of "
                       "key-and-list-of-indices pairs; the lists of indices are "
                       "separated from the key by a colon and each index is also"
                       " separated from one another by a colon. The indices are "
                       "0-based column indices of the values on CSV rows. An "
                       "entry may be either a single number or a dash-separated "
                       "range of numbers. As to keys, \"framenr\" is special: it"
                       "is interpreted as to having the 0-based frame number of "
                       "the corresponding row and must always be a single index."
                       " Unless framenr is specified, each row is assumed to "
                       "have originated from a frame number with the same "
                       "0-based row number, unless the whole file contains only "
                       "one row, in which case the values are assumed to be "
                       "global. An example of a valid string of specifications: "
                       "framenr:0,head_pos:1-3,lefthand_pos:4:5:6,"
                       "righthand_pos:7-8:9; global_x:0,global_y:1,global_z:2");
    return opts;
  }



  Component* CsvImporter::createComponentImpl(const boost::program_options::variables_map& opts, 
                                              BlackBoard* blackBoard, FrameSource* frameSource) const {
    CsvImporter c(blackBoard, frameSource);
    std::vector<std::string> filenames, specs;
    if (opts.count("in-csv-file"))
      boost::split(filenames, opts["in-csv-file"].as<std::string>(), 
                   boost::is_any_of(","));
    else if (opts.count("CsvImporter.filenames"))
      boost::split(filenames, opts["CsvImporter.filenames"].as<std::string>(), 
                   boost::is_any_of(","));

    if (opts.count("csv-fields"))
      boost::split(specs, opts["csv-fields"].as<std::string>(), 
                   boost::is_any_of(";"));
    else if (opts.count("CsvImporter.fields"))
      boost::split(specs, opts["CsvImporter.fields"].as<std::string>(), 
                   boost::is_any_of(";"));

    if (filenames.size() != specs.size())
      throw CommandLineException("The number of CSV input files and field"
                                   " specifications must match!");
    c.filenames = std::move(filenames);

    for (const std::string& spec : specs) {
      std::vector< std::pair<std::string,std::vector<size_t> > > parsedSpec;
      std::vector<std::string> fields;
      boost::split(fields, spec, boost::is_any_of(","));
      for (const std::string& field_ref : fields) {
        std::string field = boost::trim_copy(field_ref);
        std::vector<std::string> tokens;
        boost::split(tokens, field, boost::is_any_of(":"));
        if (tokens.size() < 2)
          throw CommandLineException("An invalid field was specified! (No colons?)");

        std::pair<std::string, std::vector<size_t> > parsedField;
        parsedField.first = tokens[0];
        for (auto it = tokens.cbegin() + 1; it != tokens.cend(); ++it) {
          if (it->find('-') == std::string::npos)
            parsedField.second.push_back(boost::lexical_cast<size_t>(*it));
          else {
            std::regex rangeRegex("([0-9]+)-([0-9]+)");
            std::smatch m;
            if (regex_match(*it, m, rangeRegex) && m.size() == 3) {
              size_t first = boost::lexical_cast<size_t>(m[1]);
              size_t last = boost::lexical_cast<size_t>(m[2]);
              for (size_t i = first; i <= last; ++i)
                parsedField.second.push_back(i);
            }
            else
              throw CommandLineException("Invalid CSV index range specified!");
          }
        }
        if (parsedField.first == "framenr" && parsedField.second.size() > 1)
          throw CommandLineException("The framenr must be scalar (only one "
                                     "index allowed");
        parsedSpec.push_back(parsedField);
      }
      c.specs.push_back(parsedSpec);
    }
    return new CsvImporter(c);
  }



  Component::property_set_t CsvImporter::getProvided() const {
    property_set_t props;
    for (const auto& spec : specs)
      for (const auto& prop : spec)
        if (prop.first != "framenr")
          props.insert(prop.first);
    return props;
  }



  /**
   * Reads in a CSV file, assuming each entry is a float, and returns each line 
   * as a separate vector
   */
  static std::vector < std::vector<float> > readCsvFile(const std::string& filename) {
    std::ifstream ifs(filename);
    if (!ifs.good())
      throw CsvException("Could not open " + filename);

    std::vector < std::vector<float> > lines;
    std::string s;
    while(getline(ifs, s)) {
      std::vector<std::string> line;
      boost::split(line, s, boost::is_any_of(","));
      std::vector<float> floatLine(line.size());
      for (size_t i = 0; i < line.size(); ++i)
        floatLine[i] = boost::lexical_cast<float>(boost::trim_copy(line[i]));
      lines.push_back(floatLine);
      if (lines.back().size() != lines.front().size())
        throw CsvException("Malformed CSV file: the line 0 has " + 
                           boost::lexical_cast<std::string>(lines.front().size()) +
                           " entries, but the line " + 
                           boost::lexical_cast<std::string>(lines.size()-1) +
                           " has " + boost::lexical_cast<std::string>(lines.back().size()) +
                           " entries");
    }
    return lines;
  }



  bool CsvImporter::processRangeImplementation(frame_number_t, 
                                               frame_number_t, 
                                               UiCallback*) {
    assert(filenames.size() == specs.size());
    for (size_t i = 0; i < filenames.size(); ++i) {
      std::vector < std::vector<float> > lines = readCsvFile(filenames[i]);
      frame_number_t f = 0;
      size_t frameNumberIndex = SIZE_MAX;
      const auto& spec = specs[i];
      for (const auto& fieldPair : spec) {
        if (fieldPair.first == "framenr") {
          frameNumberIndex = fieldPair.second[0];
          break;
        }
      }

      // the values are considered global if the following conditions are met:
      // no valid framenr is specified and
      // the file contains exactly one line
      bool isGlobal = lines.size() == 1 && frameNumberIndex == SIZE_MAX;

      for (const auto& line : lines) {
        if (frameNumberIndex < SIZE_MAX) {
          if (frameNumberIndex >= line.size())
            throw CsvException("The frame index exceeds the number of fields in"
                               " the file!");
          f = line[frameNumberIndex];
        }

        for (const auto& fieldPair : spec) {
          if (fieldPair.first == "framenr")
            continue;

          std::string propertyName = fieldPair.first;
          std::vector<float> value;
          for (size_t j : fieldPair.second)
            value.push_back(line[j]);

          assert(value.size() == fieldPair.second.size());

          if (isGlobal)
            getBlackBoard().set<std::vector<float> >(propertyName, value);
          else
            getBlackBoard().set<std::vector<float> >(f, propertyName, value);
        }

        ++f;
      }
    }
    return true;
  }
}
