#include "debug.hpp"
#include "XmlDocument.hpp"
#include "SLIO.hpp"
#include "slmotion.hpp"
#include "exceptions.hpp"
#include "CsvToEafConverter.hpp"
#include <fstream>
#include <boost/algorithm/string.hpp>
#include <boost/lexical_cast.hpp>
#include <iostream>
#include <sys/stat.h>

using std::string;
using std::deque;
using std::vector;
using std::map;
using std::cerr;
using std::endl;

// THE FOLLOWING FILE FORMAT IS ASSUMED:
// 
// Comma separated values with the following field names on the first row
// and corresponding data, framewise, on each subsequent row:
/*
  "framenr", // 0
  "timecode", // 1
  "total_num_of_tracked_features", // 2
  "total_total_horizontal_velocity",
  "total_total_vertical_velocity",
  "total_length_sum_vvec",
  "total_length_sum_avec", // 6
  "grosshead_num_of_tracked_features", // 7
  "grosshead_total_horizontal_velocity",
  "grosshead_total_vertical_velocity",
  "grosshead_length_sum_vvec",
  "grosshead_length_sum_avec",
  "grosslefthand_num_of_tracked_features", // 12
  "grosslefthand_total_horizontal_velocity",
  "grosslefthand_total_vertical_velocity",
  "grosslefthand_length_sum_vvec",
  "grosslefthand_length_sum_avec",
  "grossrighthand_num_of_tracked_features", // 17
  "grossrighthand_total_horizontal_velocity",
  "grossrighthand_total_vertical_velocity",
  "grossrighthand_length_sum_vvec",
  "grossrighthand_length_sum_avec",
  "nethead_num_of_tracked_features", // 22
  "nethead_total_horizontal_velocity",
  "nethead_total_vertical_velocity",
  "nethead_length_sum_vvec",
  "nethead_length_sum_avec",
  "netlefthand_num_of_tracked_features", // 27
  "netlefthand_total_horizontal_velocity",
  "netlefthand_total_vertical_velocity",
  "netlefthand_length_sum_vvec",
  "netlefthand_length_sum_avec",
  "netrighthand_num_of_tracked_features", // 32
  "netrighthand_total_horizontal_velocity",
  "netrighthand_total_vertical_velocity",
  "netrighthand_length_sum_vvec",
  "netrighthand_length_sum_avec",
  "overlapping_left_hand_head", // 37
  "overlapping_right_hand_head",
  "overlapping_left_right_hand",
  "head_x_velocity", // 40
  "head_y_velocity",
  "head_velocity_magnitude",
  "lefthand_x_velocity", // 43
  "lefthand_y_velocity",
  "lefthand_velocity_magnitude",
  "righthand_x_velocity", // 46
  "righthand_y_velocity",
  "righthand_velocity_magnitude",
  "orientation_head_theta", // 49
  "orientation_head_omega",
  "orientation_lefthand_theta",
  "orientation_lefthand_omega",
  "orientation_righthand_theta",
  "orientation_righthand_omega" // 54
*/

namespace slmotion {
  void CsvToEafConverter::loadMotionDescriptorsFromFile(const string& filename, deque < vector<double> >& data, map <string, unsigned int>& fieldMapping) const {
    data.clear();
    fieldMapping.clear();

    // read in the input file, line by line
    std::ifstream in(filename);

    if (!in.good()) 
      throw ConvertException("Failed to open the input file: " + filename);
  
    string s;

    // read in the first line, and verify that field names are as expected
    // (this works as an ad-hoc file format specification)
 
    const size_t numFields = 55; // total number of fields
    const string fieldNames[] = {
      "framenr", // 0
      "timecode", // 1
      "total_num_of_tracked_features", // 2
      "total_total_horizontal_velocity",
      "total_total_vertical_velocity",
      "total_length_sum_vvec",
      "total_length_sum_avec", // 6
      "grosshead_num_of_tracked_features", // 7
      "grosshead_total_horizontal_velocity",
      "grosshead_total_vertical_velocity",
      "grosshead_length_sum_vvec",
      "grosshead_length_sum_avec",
      "grosslefthand_num_of_tracked_features", // 12
      "grosslefthand_total_horizontal_velocity",
      "grosslefthand_total_vertical_velocity",
      "grosslefthand_length_sum_vvec",
      "grosslefthand_length_sum_avec",
      "grossrighthand_num_of_tracked_features", // 17
      "grossrighthand_total_horizontal_velocity",
      "grossrighthand_total_vertical_velocity",
      "grossrighthand_length_sum_vvec",
      "grossrighthand_length_sum_avec",
      "nethead_num_of_tracked_features", // 22
      "nethead_total_horizontal_velocity",
      "nethead_total_vertical_velocity",
      "nethead_length_sum_vvec",
      "nethead_length_sum_avec",
      "netlefthand_num_of_tracked_features", // 27
      "netlefthand_total_horizontal_velocity",
      "netlefthand_total_vertical_velocity",
      "netlefthand_length_sum_vvec",
      "netlefthand_length_sum_avec",
      "netrighthand_num_of_tracked_features", // 32
      "netrighthand_total_horizontal_velocity",
      "netrighthand_total_vertical_velocity",
      "netrighthand_length_sum_vvec",
      "netrighthand_length_sum_avec",
      "overlapping_left_hand_head", // 37
      "overlapping_right_hand_head",
      "overlapping_left_right_hand",
      "head_x_velocity", // 40
      "head_y_velocity",
      "head_velocity_magnitude",
      "lefthand_x_velocity", // 43
      "lefthand_y_velocity",
      "lefthand_velocity_magnitude",
      "righthand_x_velocity", // 46
      "righthand_y_velocity",
      "righthand_velocity_magnitude",
      "orientation_head_theta", // 49
      "orientation_head_omega",
      "orientation_lefthand_theta",
      "orientation_lefthand_omega",
      "orientation_righthand_theta",
      "orientation_righthand_omega" // 54
    };

    getline(in, s);
    if (s.length() == 0)
      throw ConvertException("Invalid file format. The first line should begin with a list of fields, but no such line was read.");

    vector<string> temp;
    boost::split(temp, s, boost::is_any_of(","));

    if (temp.size() != numFields) 
      throw ConvertException("Invalid file format: Field number mismatch. "
                             "Read " + 
                             boost::lexical_cast<string>(temp.size()) +
                             " fields, but " +
                             boost::lexical_cast<string>(numFields) +
                             " was expected.");

    for (size_t i = 0; i < numFields; i++) {
      if (temp[i] != fieldNames[i]) 
        throw ConvertException("Invalid file format: Field name mismatch. "
                               "Read " + temp[i] + ", but " + fieldNames[i]
                               + " was expected.");

      fieldMapping.insert(std::pair <string, unsigned int>(fieldNames[i], i));
    }


    // then read in the data
    size_t numLines = 1;

    while (in.good()) {
      getline(in, s);
      numLines++;

      if (s.length() == 0)
        break;

      boost::split(temp, s, boost::is_any_of(","));

      if (temp.size() != numFields) 
        throw ConvertException("Invalid line read (number " +
                               boost::lexical_cast<string>(numLines) +
                               "): Field number mismatch. Read " +
                               boost::lexical_cast<string>(temp.size()) +
                               " fields, but " + 
                               boost::lexical_cast<string>(numFields) +
                               " was expected.");


      // convert from string to double
      data.push_back(vector<double>(numFields, 0));
      size_t i;
      try {
        for (i = 0; i < numFields; i++) {
          data.back()[i] = boost::lexical_cast<double>(temp[i]);
        }
      }
      catch (boost::bad_lexical_cast& e) {
        throw ConvertException("Invalid line read (number " +
                               boost::lexical_cast<string>(numLines) +
                               "): field number " + 
                               boost::lexical_cast<string>(i+1) +
                               " could not be converted to a double "
                               "precision floating point.");
      }

      // The first field should always be an accurate representation of the
      // frame number (that is 1-based line number - 2)
      if (data.back()[0] != (numLines - 2)) 
        throw ConvertException("Invalid line read (number " +
                               boost::lexical_cast<string>(numLines) + 
                               "): Field number 1 should be the 0-based "
                               "frame number, but this does not seem to be "
                               "the case.");
    }
  }



  void CsvToEafConverter::exportToELAN(const string& filename,
                                       const tier_deque_t& tiers,
                                       const string& videoFilename) const {
    if (debug > 1)
      cerr << "Exporting data to ELAN file..." << endl;
  
    struct stat st;
    bool fileExist = stat(filename.c_str(), &st) == 0 ? true : false;

    EafDocument elanDoc = fileExist ? 
      EafDocument(filename) : EafDocument();
    
    if (videoFilename != "")
      elanDoc.addMediaDescriptor(videoFilename);

    // add each tier
    for (auto it = tiers.cbegin(); it != tiers.cend(); ++it) 
      elanDoc.addTier(*it);  
 
    elanDoc.save(filename);
  }



  void CsvToEafConverter::createAnnotationsFromMDs(const deque < vector<double> >& mds, const AnnotationSpecifier& spec,
                                                   deque<Annotation>& anns) {
    // ASSUME the proper ordering of data
    assert(mds.size() > 0);

    int beg = -1; // first frame for the annotation
    int end = -1; // last frame for the annotation

    for (size_t f = 0; f < mds.size(); f++) {
      // assert(mds[f][0] == f);

      // The annotation is considered finished if neither beg nor end is
      // -1.

      bool initialFrame = false; // this is set true, if beg is set during this frame

      // check constraints
      for (deque<AnnotationSpecifier::Constraint>::const_iterator it = spec.constraints.begin();
           it != spec.constraints.end(); it++) {

        if (it->type == AnnotationSpecifier::Constraint::INSIDE ||
            (it->type == AnnotationSpecifier::Constraint::INITINSIDE &&
             (initialFrame || beg == -1))) {
          assert(it->index < mds[f].size());
          if (mds[f][it->index] <= it->max &&
              mds[f][it->index] >= it->min) {
            end = -1;
            if (beg == -1) {
              beg = f;
              initialFrame = true;
            }
          }
          else {
            if (beg != -1) 
              end = f;
            break;
          }
        }
        else if (it->type == AnnotationSpecifier::Constraint::OUTSIDE ||
                 (it->type == AnnotationSpecifier::Constraint::INITOUTSIDE &&
                  (initialFrame || beg == -1))) {
          assert(it->index < mds[f].size());
          if (mds[f][it->index] > it->max ||
              mds[f][it->index] < it->min) {
            end = -1;
            if (beg == -1) {
              beg = f;
              initialFrame = true;
            }
          }
          else {
            if (beg != -1) 
              end = f;
            break;
          }
        }
      }


      // special case: check for the last frame
      if (f + 1 == mds.size())
        end = f + 1;

      if (end != -1) {
        if (beg != -1 && (end - beg >= spec.minLength)) {
          // convert timecodes, add padding
          anns.push_back(Annotation("SLMotion_Automagic_Annotation", ((beg - spec.padding)*1000) / spec.fps, ((end-1 + spec.padding)*1000) / spec.fps));
        }
        beg = -1;
        end = -1;
      }
    }

    // merge annotations, sanitise beginnings  
    for (size_t i = 1; i < anns.size(); ++i) {
      if (anns[i].timerefs.first <= anns[i-1].timerefs.second ||
          anns[i].timerefs.first - anns[i-1].timerefs.second < spec.minGapLength * 1000.0 / spec.fps) {
        anns[i-1].timerefs.second = anns[i].timerefs.second;
        anns.erase(anns.begin() + i);
        --i;
      }
    }

    // make sure that the first annotation begins with a 0 or greater frame number
    if (anns.size() > 0 && anns[0].timerefs.first < 0)
      anns[0].timerefs.first = 0;

    /*
      for (size_t i = 0; i < anns.size(); ++i) {
      std::cout << anns[i].timerefs.first << "->" << 
      anns[i].timerefs.second << std::endl;
      }*/
  }



  void CsvToEafConverter::generateElanAnnotations(const deque<AnnotationSpecifier>& specifiers, const deque< vector<double> >& mds,
                                                  tier_deque_t& tiers) const {
    tiers.clear();

    // ASSUME the proper ordering of data
    for (deque<AnnotationSpecifier>::const_iterator it = specifiers.begin();
         it != specifiers.end(); it++) {
    
      deque<Annotation> anns;
      createAnnotationsFromMDs(mds, *it, anns);

      tiers.push_back(std::pair < string, deque<Annotation> > (it->name, anns));
    }
  }



  void CsvToEafConverter::parseAnnotationSpecifiers(const string& filename, const map <string, unsigned int>& fieldNames, deque<AnnotationSpecifier>& outSpec) const {
    // Validate the annotation specifier file
    xml::XmlValidableDocument annDoc(xml::Validator(getAnnotationSchemaFile()),
                                     filename);
    if (!annDoc.validate())
      throw ConvertException("The annotation file was invalid");


    // Get the root element
    xml::XmlNode annotationNode = annDoc.getRoot();

    // Parse the document

    try {
      // Verify the type of the root
      // if (strcmp(reinterpret_cast<const char*>(annotationNode->name), "AnnotationTemplateDocument") != 0) 
      if (annotationNode.getName() != "AnnotationTemplateDocument")
        throw ConvertException("Invalid root node. AnnotationTemplateDocument was expected but got " + annotationNode.getName());

      // Then, proceed
      outSpec.clear();
      for (auto it = annotationNode.begin(); 
           it != annotationNode.end(); ++it) {
        if (it->getName() != "Annotation") 
          throw ConvertException("Unexpected node '" + it->getName() + "' encountered while parsing the annotation template file. 'Annotation' was expected.");

        // Set fields for the annotation specifier
        string name(it->getProperty("name"));
        int minLength;
        double fps;
        int padding;
        int minGapLength;
        try {
          minLength = boost::lexical_cast<int>(it->getProperty("minLength"));
          fps = boost::lexical_cast<double>(it->getProperty("fps"));
          minGapLength = boost::lexical_cast<int>(it->getProperty("minGapLength"));
          padding = boost::lexical_cast<int>(it->getProperty("padding"));
        }
        catch (boost::bad_lexical_cast& e) {
          throw ConvertException("Bad number encountered on line " + boost::lexical_cast<string>(it->getLineNumber()) + " while constructing an annotation template: " + e.what());
        }

        // Then, fetch constraints
        deque<AnnotationSpecifier::Constraint> consts;

        for (auto constraintNodeIt = it->begin(); constraintNodeIt != it->end();
             ++constraintNodeIt) {
          string initial = constraintNodeIt->getProperty("initial");
          bool init;
          if (initial == "true")
            init = true;
          else if (initial == "false")
            init = false;
          else
            throw ConvertException("Me fail constraint. Should be true or false. This is unpossible! Epic fail with validator encountered.");

          string indexStr = constraintNodeIt->getProperty("field");
          int index;
          double min, max;

          try {
            if (fieldNames.count(indexStr))
              index = fieldNames.find(indexStr)->second;
            else {
              index = boost::lexical_cast<int>(indexStr);
            }
            min = boost::lexical_cast<double>(constraintNodeIt->getProperty("min"));
            max = boost::lexical_cast<double>(constraintNodeIt->getProperty("max"));
          }
          catch (boost::bad_lexical_cast& e) {
            cerr << "l. " << constraintNodeIt->getLineNumber() << endl;
            dumpElement(cerr, annDoc, *constraintNodeIt);
            cerr << endl;            
            throw ConvertException("Parse error while processing a constraint element on line " + boost::lexical_cast<string>(constraintNodeIt->getLineNumber()) + ": Numerical conversion failed.");
          }

          string typeStr = constraintNodeIt->getProperty("type");
          AnnotationSpecifier::Constraint::Type t;
          if (typeStr == "inside") {
            t = init ? AnnotationSpecifier::Constraint::INITINSIDE :
              AnnotationSpecifier::Constraint::INSIDE;
          }
          else if (typeStr == "outside") {
            t = init ? AnnotationSpecifier::Constraint::INITOUTSIDE :
              AnnotationSpecifier::Constraint::OUTSIDE;
          }
          else 
            throw ConvertException("Parse error while processing a constraint element on line " + boost::lexical_cast<string>(constraintNodeIt->getLineNumber()) + ": Invalid type specified.");
	  
          consts.push_back(AnnotationSpecifier::Constraint(t, index, min, max));
        }
        outSpec.push_back(AnnotationSpecifier(name, minLength, fps, padding, minGapLength, consts));

      }  
    }
    catch (ConvertException& e) {
      throw ConvertException(string("Error while parsing annotation templates: ") + e.what());
    }

    if (debug > 1) {
      cerr << "Got following annotation specifiers:\n";
      for (deque<AnnotationSpecifier>::const_iterator it = outSpec.begin();
           it != outSpec.end(); it++) {
        cerr << "name        : " << it->name << "\n"
             << "minlength   : " << it->minLength << "\n"
             << "fps         : " << it->fps << "\n"
             << "padding     : " << it->padding << "\n"
             << "minGapLength: " << it->minGapLength << "\n"
             << "constraints:\n";
        for (deque<AnnotationSpecifier::Constraint>::const_iterator jt =
               it->constraints.begin(); jt != it->constraints.end(); jt++) {
          cerr << "Type : ";
          switch (jt->type) {
          case AnnotationSpecifier::Constraint::INSIDE:
            cerr << "INSIDE";
            break;
          case AnnotationSpecifier::Constraint::INITINSIDE:
            cerr << "INITINSIDE";
            break;
          case AnnotationSpecifier::Constraint::OUTSIDE:
            cerr << "OUTSIDE";
            break;
          case AnnotationSpecifier::Constraint::INITOUTSIDE:
            cerr << "INITOUTSIDE";
            break;
          }
          cerr << "\n"
               << "Index: " << jt->index << "\n"
               << "min  : " << jt->min << "\n"
               << "max  : " << jt->max << "\n";
        }
      }
    }
  }



  void CsvToEafConverter::convert(const string& inputCsv,
                                  const string& inputAnnotationTemplates,
                                  const string& outputEaf,
                                  const string & mediaFilename) const {
    deque < vector<double> > data;
    map <string, unsigned int> fieldMap;
    loadMotionDescriptorsFromFile(inputCsv, data, fieldMap);
    deque<AnnotationSpecifier> specifiers;
    parseAnnotationSpecifiers(inputAnnotationTemplates, fieldMap, specifiers);
    tier_deque_t tiers;
    generateElanAnnotations(specifiers, data, tiers);
    exportToELAN(outputEaf, tiers, mediaFilename);
  }



  void CsvToEafConverter::convertVR(const deque< vector<double> >& mds, const string& annTemplateFilename, const map <string, unsigned int>& fieldNames, const string& outputFilename) const {
    deque<AnnotationSpecifier> specifiers;
    parseAnnotationSpecifiers(annTemplateFilename, fieldNames, specifiers);
    tier_deque_t tiers;
    generateElanAnnotations(specifiers, mds, tiers);
    exportToElanVR(outputFilename, tiers);
  }



  void CsvToEafConverter::exportToElanVR(const std::string& filename, const tier_deque_t& tiers) const {
    if (slmotion::debug > 0)
      slmotion::SLIO::getInfoOut() << "Creating an ELAN output XML file..." << std::endl;  

    for (tier_deque_t::const_iterator it = tiers.begin();
         it != tiers.end(); it++) {
      createTierXmlForElan(filename, *it);
    }
  }




  void slmotion::CsvToEafConverter::createTierXmlForElan(const std::string& filename, const tier_t& tier) const {
    if (slmotion::debug > 0)
      slmotion::SLIO::getInfoOut() << "Creating an ELAN tier XML file..." << std::endl;

    xml::XmlDocument doc;

    xml::XmlNode root("TIER");
    xml::XmlNamespace schemaInstanceNamespace(root.newNamespace("http://www.w3.org/2001/XMLSchema-instance", "xsi"));

    doc.setRootElement(root);

    schemaInstanceNamespace.newProperty("noNamespaceSchemaLocation",
                                        "file:avatech-tier.xsd");

    root.newProperty("columns", tier.first);

    for (std::deque<Annotation>::const_iterator it = tier.second.begin();
         it != tier.second.end(); it++) {
      xml::XmlNode node("span");
      node.newProperty("start", boost::lexical_cast<std::string>(it->timerefs.first));
      node.newProperty("end", boost::lexical_cast<std::string>(it->timerefs.second));
      node.newTextChild("v", "SLMotion_Automagic_Annotation");
      root.addChild(node);
    }

    doc.save(filename);
  }

  CsvToEafConverter::CsvToEafConverter() :
    annValidator([] {
        // load the schema file
        xml::XmlDocument annSchemaDoc(getAnnotationSchemaFile());
        
        return xml::Validator(annSchemaDoc);        
      }()){
  }



  void CsvToEafConverter::convert(const deque< vector<double> >& mds, const string& annTemplateFilename, const map <string, unsigned int>& fieldNames, const string& elanFilename, const string& inputMediaFilename) const {
    deque<AnnotationSpecifier> specifiers;
    parseAnnotationSpecifiers(annTemplateFilename, fieldNames, specifiers);
    tier_deque_t tiers;
    generateElanAnnotations(specifiers, mds, tiers);
    exportToELAN(elanFilename, tiers, inputMediaFilename);
  }
}
