#ifndef SLMOTION_CSVTOEAFCONVERTER
#define SLMOTION_CSVTOEAFCONVERTER

#include "Validator.hpp"
#include "AnnotationSpecifier.hpp"
#include "EafDocument.hpp"
#include <vector>

namespace slmotion {
  /**
   * A class used to convert CSV formatted data into EAF formatted data
   * using XML annotation templates
   */
  class CsvToEafConverter {
  public:
    /**
     * Constructs the converter object, and validates schema files
     */
    CsvToEafConverter();



    /**
     * Converts a CSV file into an EAF annotation file, possibly altering an
     * already existing file, generating annotations according to the annotation
     * template file.
     *
     * @param inputCsv Input CSV filename
     * @param inputAnnotationTemplates Input annotation template filename
     * @param outputEaf Output EAF filename
     * @param mediaFilename Media filename for creating media descriptor entry
     */
    void convert(const std::string& inputCsv,
		 const std::string& inputAnnotationTemplates,
		 const std::string& outputEaf,
		 const std::string& mediaFilename) const;



    /**
     * Generates annotations into the EAF file according to annotation templates.
     * This function is intended to be used directly, by supplying motion
     * descriptor data from internal data structures rather than reading it
     * from a file on the disk.
     *
     * @param mds Motion descriptors, framewise, with each index corresponding
     * to a frame, and the vector being the record of associated motion
     * descriptors (exactly as they would be stored as fields in a file).
     * @param annTemplateFilename Annotation template filename
     * @param fieldNames Field-name-to-index mapping
     * @param elanFilename Output ELAN EAF filename
     * @param inputMediaFilename Filename for the media file used as the media
     * descriptor in the EAF file
     */
    void convert(const std::deque< std::vector<double> >& mds,
		 const std::string& annTemplateFilename,
		 const std::map < std::string, unsigned int >& fieldNames,
		 const std::string& elanFilename,
		 const std::string& inputMediaFilename) const;



    /**
     * Generates annotations into a video recogniser tier XML file according
     * to annotation templates. This function is intended to be used when
     * invoking the program as a video recogniser, adding the new tiers to 
     * ELAN on the fly rather than modifying an existing EAF file or
     * creating a new file altogether.
     *
     * @param mds Motion descriptors, framewise, with each index
     * corresponding to a frame, and the vector being the record of
     * associated motion descriptors (exactly as they would be stored as
     * fields in a file).
     * @param annTemplateFilename Annotation template filename
     * @param fieldNames Field-name-to-index mapping
     * @param outFile Output XML file name
     */
    void convertVR(const std::deque< std::vector<double> >& mds,
		   const std::string& annTemplateFilename,
		   const std::map < std::string, unsigned int >& fieldNames,
		   const std::string& outFile) const;




  private:
    /**
     * Loads motion descriptors from a file, and stores the data in a data
     * structure that corresponds to fields in the analyser class.
     *
     * @param filename The input filename
     * @param data Data structure to store the motion descriptors in.
     * @param fieldMapping Output parameter to store field
     * name-to-CSV-index mapping data
     */
    void loadMotionDescriptorsFromFile(const std::string& filename,
				       std::deque < std::vector<double> >& data,
				       std::map < std::string, unsigned int >& fieldMapping) const;



    /**
     * Creates new annotation filled tiers in the ELAN file
     * 
     * @param filename ELAN output file name. If a file with the given name
     * exists, the new tiers are added to it. Otherwise, a new file is
     * created.
     * @param tiers A list of new tiers. The first element of the pair
     * should be a unique tier identifier. The second element should be a
     * list of annotations.
     * @param videoFilename Filename to be used as media descriptor
     * information when creating a new ELAN file
     */
    void exportToELAN(const std::string& filename, const tier_deque_t& tiers,
		      const std::string& videoFilename) const;



    /**
     * Creates a video recogniser output XML file that ELAN can use to
     * locate the newly created tiers.
     * 
     * @param filename Output XML filename (that is later read by ELAN)
     * @param tiers Tiers of annotations
     */
    void exportToElanVR(const std::string& filename,
			const tier_deque_t& tiers) const;



    /**
     * Creates a tier XML file that can be used to import new tiers into
     * ELAN using the recogniser interface.
     *
     * @param filename Output XML filename (that is later read by ELAN)
     * @param tier One tier
     */
    void createTierXmlForElan(const std::string& filename,
			      const tier_t& tier) const;



  /**
   * Generates annotations and stores them in a deque.
   * @param specifiers A list of annotation specifiers
   * @param mds A reference to framewise motion descriptors
   * @param outTiers Output tiers
   */
    void generateElanAnnotations(const std::deque<AnnotationSpecifier>& specifiers,
				 const std::deque< std::vector<double> >& mds,
				 tier_deque_t& outTiers) const;



    /**
     * Creates annotation specifiers from the given annotation specifier
     * file.
     *
     * @param filename Input filename
     * @param fieldNames Field-name-to-CSV-field-index mapping
     * @param outSpec Output parameter for the list of specifiers
     */
    void parseAnnotationSpecifiers(const std::string& filename,
				   const std::map < std::string, unsigned int >& fieldNames,
				   std::deque<AnnotationSpecifier>& outSpec) const;



    /**
     * Creates list of annotations by going through the deque (the elements of
     * which are assumed to match frames) of motion descriptors, comparing
     * elements in each frame as specified by the annotation specifier, and
     * outputs the resulting annotation sequences into the output deque.
     *
     * @param mds Input framewise motion descriptors
     * @param spec Input motion descriptor
     * @param outAnns Output deque for annotations
     */
    static void createAnnotationsFromMDs(const std::deque < std::vector<double> >& mds, const AnnotationSpecifier& spec,
					 std::deque<Annotation>& outAnns);



    xml::Validator annValidator; ///< Annotation validator object
  };
}
#endif

