#ifndef SLMOTION_EAF_DOCUMENT
#define SLMOTION_EAF_DOCUMENT

#include "XmlValidableDocument.hpp"
#include "Annotation.hpp"
#include <deque>
#include <map>
#include <set>

namespace slmotion {
  /**
   * ELAN EAF schema file
   */
  std::string getEafSchemaFile();


  /**
   * A tier is composed of a name (the first field of the pair), and a deque
   * of annotations
   */
  typedef std::pair < std::string, std::deque<Annotation> > tier_t;

  /**
   * The tier deque type, a deque of tiers
   */
  typedef std::deque < tier_t > tier_deque_t;

  /**
   * A class that represents an ELAN annotation file document in an
   * easier-to-process fashion
   */
  class EafDocument : public xml::XmlValidableDocument {
  public:    
    /**
     * Parses the XML file, and constructs a document accordingly
     *
     * @param xmlDoc An XML EAF document filename
     */
    explicit EafDocument(const std::string& xmlDoc) :
      XmlValidableDocument(xml::Validator(getEafSchemaFile()), xmlDoc),
      timeCodeNameIndexCounter(0),
      annotationIdIndexCounter(0)
    {
      initialise(getRoot());
    }



    /**
     * Constructs an empty document
     */
    EafDocument() :
      XmlValidableDocument(xml::Validator(getEafSchemaFile())),
      timeCodeNameIndexCounter(0),
      annotationIdIndexCounter(0)
    {
      initialise();
    }



    /**
     * Adds a new tier to the document
     *
     * @param tier The new tier
     */
    void addTier(const tier_t& tier);
  
    
    
    tier_deque_t getTiers();



    /**
     * Adds a media descriptor field to the document.
     *
     * @param filename Relative filename
     * @param mime MIME-type (default = video / *)
     *
     * @return True on success, false otherwise. Failure may occur e.g. because
     * the canonical filename could not be found. In the case of failure, the
     * document is not modified.
     */
    bool addMediaDescriptor(const std::string& filename,
			    const std::string& mime = "video/*");



    /**
     * Saves the document in the specified file.
     */
    virtual void save(const std::string& filename);



    int getLastTimecode();


    // move operations
    EafDocument(EafDocument&&);
    EafDocument& operator=(EafDocument&&);

  protected:
    virtual std::ostream& print(std::ostream&) const;

  private:
    std::map < std::string, std::string > timeCodes; ///< Internal representation of timecodes
    std::set < std::string > annotationIds; ///< Internal cache for annotation IDs so that their uniqueness can be guaranteed
    std::set < std::string > tierIds; ///< Internal cache for tier IDs so that their uniqueness can be guaranteed
    size_t timeCodeNameIndexCounter; ///< Time codes will be labelled with an index. This shows the most recent free one, so there should be no need to go through them all
    size_t annotationIdIndexCounter; ///< Annotations will be labelled with an index. This shows the most recent free one, much like the time code index counter above



    /**
     * Prepares an empty document. This function is called by the constructor.
     */
    void initialise();



    /**
     * Prepares a new document from a file.
     * This function is called by the constructor.
     *
     * @param rootNode Root node
     */
    void initialise(xml::XmlNode rootNode);



    /**
     * Adds a new timecode. The identifier is created automagically.
     *
     * @param value Timecode value
     *
     * @return The generated identifier
     */
    std::string addTimeCode(unsigned int value);

    
    
    /**
     * Adds a new timecode. The identifier is created automagically.
     *
     * @param value Timecode value
     *
     * @return The generated identifier
     */
    std::string addTimeCode(const std::string& value);



    /**
     * Returns the header element.
     *
     * @return The HEADER element, throws an exception if it could not be 
     * found
     */
    xml::XmlNode getHeader();



    /**
     * Synchronises the internal data structures with the internal XML
     * representation, i.e. first deletes data from the XML tree that has been
     * stored separately, and then fills the tree with values from the internal
     * separate representation. This function should be called before writing
     * the data on to disk.
     */
    void sync() const;



    /**
     * Adds the specified linguistic type. If a linguistic type by the name
     * specified already exists, nothing is done.
     *
     * @param name Linguistic type name (default = "default-lt")
     */
    void addLingType(const std::string& name = "default-lt") const;



    // no copies
    EafDocument(const EafDocument&) = delete;
    EafDocument& operator=(const EafDocument&) = delete;
  };
}

#endif
