#ifndef SLMOTION_ELM_SKIN_DETECTOR
#define SLMOTION_ELM_SKIN_DETECTOR

#include "SkinDetector.hpp"
#include "FaceDetector.hpp"
#include "GaussianMixture.hpp"
#include "ELMClassifier.hpp"
#include "FeatureExtractor.hpp"


namespace slmotion {
  /**
   * ELM-based skin detector, reworked to work with the new framework
   */
  class ElmSkinDetector : public SkinDetector {
  public:
    explicit ElmSkinDetector(BlackBoard* blackBoard, 
                                  FrameSource* frameSource,
                                  const ColourSpace& c = *ColourSpace::HSV) :
      SkinDetector(blackBoard, frameSource, c),
      cls(NULL),
      trained(false),
      skinthresholdp(0.75),
      blurmagnitude(0)
    { }



    void reset();

    virtual void process(frame_number_t /* frameNumber*/){
      throw std::string("Processing of individual frames not supported"); 
    }

    bool processRangeImplementation(frame_number_t first,
                                    frame_number_t last,
                                    UiCallback* uiCallback);


    /**
     * Trains the model using the given samples.
     *
     * @param samples Skin samples as colour vectors, one per row. Must be
     * of type CV_32FC1.
     * @param responses Response vectors. Ignored in this version.
     * @param update Sets whether the model should be updated or replaced.
     * Ignored in this version.
     */
    void train(const cv::Mat& /*samples*/, const cv::Mat& /*responses*/,
               bool /*update*/ = false){
      // currently not implemented

      throw std::string("ElmSkinDetector can't be trained with external samples yet.");

    }



    /**
     * Reads the image file using the given filename, converts it into skin
     * pixel samples assuming that each non-zero pixel in the image is skin,
     * then trains the model using that particular training data by calling
     * train() as declared above
     *
     * @param filename Image file name
     */
    void train(const std::string& /*filename*/){
      
      throw std::string("ElmSkinDetector can't be trained with external samples yet.");

    }


    /**
     * Reads in frames using the given IO object 
     * and selects suitable training samples on basis of face detections
     * trains the ELM using those

     *
     * @return True if the model was trained. False if it was not (e.g. no
     * faces were found).
     */
    bool trainByFaceDetector();




    virtual std::string getShortDescription() const {
      return "Detects skin coloured pixels using a simple Elm model";
    }

    virtual std::string getLongDescription() const {
      return "Detects skin coloured pixels using a simple Elm model";
    }

    virtual property_set_t getRequirements() const {
      return property_set_t { FACEDETECTOR_BLACKBOARD_ENTRY };
    }

    virtual std::string getShortName() const {
      return "ElmSkinDetector";
    }

    virtual std::string getComponentName() const {
      return "Elm Skin Detector";
    }

    virtual boost::program_options::options_description getConfigurationFileOptionsDescription() const;

 virtual boost::program_options::options_description getCommandLineOptionsDescription() const {
      return boost::program_options::options_description();
    }


    // dummy constructor
    ElmSkinDetector(bool) : SkinDetector(true) {}



    /**
     * Constructs the instance from the given configuration, and passes the
     * configuration also to superclasses
     */
    ElmSkinDetector(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource);


  float getPixelP(const std::vector<float>& x) {

    auto it = pixelPCache.find(x);
    if (it != pixelPCache.end()) 
      return it->second;

        return pixelPCache[x] =  cls->predict_lean(x);

    }



  private:
    virtual Component* createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const;

    void detect(const cv::Mat& inFrame, cv::Mat& outMask);

    void doDetect(const cv::Mat& inFrame, cv::Mat& outMask, cv::Mat *inputmask=NULL);

    ELMClassifier *cls;
    FeatureExtractor fe;


    std::map<std::vector<float>, float> pixelPCache;
    bool trained;
    float skinthresholdp;
    float blurmagnitude;
   
  };

  void findWatershed(cv::Mat &mask,const cv::Mat &frame, cv::Mat &output);


}

#endif
