#ifdef SLMOTION_ENABLE_FFMPEG

extern "C" {
#include <libswscale/swscale.h>
}
#include "FFmpegVideoFileSource.hpp"

using std::shared_ptr;

namespace {
  // this only needs to be done once -- globally
  struct Foo {
    Foo() {
      av_register_all();
      av_log_set_level(0); // to suppress unnecessary libswcale warnings
    }
  } foo;
}

namespace slmotion {
  extern int debug;



  FFmpegVideoFileSource::frame_number_type FFmpegVideoFileSource::size() const {
    return formatContext->streams[videoStream]->duration;
  }



  /**
   * Reads in one frame, converts it to an OpenCV BGR24 
   * Mat object, and returns it, while freeing any used data.
   */
  cv::Mat FFmpegVideoFileSource::readFrame() {
    int frameFinished;
    AVPacket packet;
    cv::Mat openCvMat;
    
    while (av_read_frame(formatContext.get(), &packet) >= 0) {
      // Is this a packet from the video stream?
      if (packet.stream_index==videoStream) {
	// Decode video frame
        avcodec_decode_video2(codecContext.get(), internalFrameYUV.get(), 
                              &frameFinished, &packet);
        
        // Did we get a video frame?
        if (frameFinished) {
          sws_scale(this->swsContext.get(), internalFrameYUV->data,
                    internalFrameYUV->linesize, 0, codecContext->height,
                    internalFrameBGR->data, internalFrameBGR->linesize);
          // convert to OpenCV format
          openCvMat = cv::Mat(codecContext->height, codecContext->width,
                              CV_8UC3);
    
          memcpy(openCvMat.data, internalFrameBGR->data[0], 
                 codecContext->height * codecContext->width * 3);
          break;
        }
      }
      // Free the packet that was allocated by av_read_frame
      av_free_packet(&packet);
    }
    return openCvMat;
  }



  cv::Mat FFmpegVideoFileSource::getFrame() {
#ifdef SLMOTION_THREADING
    std::lock_guard<std::mutex> lock(this->mutex);
#endif
      return readFrame();
  }



  void FFmpegVideoFileSource::tryToOpenFile(const std::string& filename) {
    AVFormatContext* formatContextPtr = NULL;
    // open the file, read the header
    if (av_open_input_file(&formatContextPtr, filename.c_str(), 
                           NULL, 0, NULL) != 0)
      throw SLIOException("Could not open video file.");

    formatContext = std::shared_ptr<AVFormatContext>(formatContextPtr,
                                                     &av_close_input_file);

    // retrieve stream information
    if (av_find_stream_info(formatContextPtr) < 0)
      throw SLIOException("Could not find stream info.");

    // std::cout << formatContext->filename << std::endl;
    // std::cout << formatContext->start_time << std::endl;
    // std::cout << formatContext->duration << std::endl;
    // std::cout << formatContext->file_size << std::endl;
    // std::cout << formatContext->bit_rate << std::endl;
    // std::cout << formatContext->start_time_realtime << std::endl;

    if (slmotion::debug > 1)
      dump_format(formatContextPtr, 0, filename.c_str(), 0);

    // try to locate a video stream
    for(size_t i = 0; i < formatContextPtr->nb_streams; ++i) {
      if (formatContextPtr->streams[i]->codec->codec_type == 
          CODEC_TYPE_VIDEO) {
        codecContext = std::shared_ptr<AVCodecContext>(formatContextPtr->streams[i]->codec,
                                                       &avcodec_close);
        this->videoStream = i;
        break;
      }
    }

    // std::cout << formatContextPtr->streams[videoStream]->duration << std::endl;

    if (codecContext.get() == NULL)
      throw SLIOException("Could not find a codec for the given file.");

    // Find the decoder for the video stream
    codec = avcodec_find_decoder(codecContext->codec_id);

    if(codec == NULL) 
      throw SLIOException("Unsupported codec!");

    // Open codec
    if (avcodec_open(codecContext.get(), codec) < 0)
      throw SLIOException(("Could not open decoder for the given codec (" +
                           std::string(codecContext->codec_name) + ")").c_str());


    // std::cout << codecContext->frame_number << std::endl;
    // std::cout << codecContext->time_base.den << "/" 
    //           << codecContext->time_base.num << std::endl;

    internalFrameYUV = std::shared_ptr<AVFrame>(avcodec_alloc_frame(), 
                                                &av_free);
    internalFrameBGR = std::shared_ptr<AVFrame>(avcodec_alloc_frame(), 
                                                &av_free);

    if (internalFrameYUV.get() == NULL || internalFrameBGR.get() == NULL)
      throw SLIOException("Frame allocation failed.");

    // allocate data buffer
    size_t numBytes = avpicture_get_size(PIX_FMT_BGR24, codecContext->width,
                                         codecContext->height) + 5;
    dataBuffer = shared_ptr<uint8_t>(static_cast<uint8_t*>(av_malloc(numBytes)),
                                     &av_free);
    avpicture_fill(reinterpret_cast<AVPicture*>(internalFrameBGR.get()), 
                   dataBuffer.get(), PIX_FMT_BGR24, codecContext->width, 
                   codecContext->height);

    // allocate frame conversion sws context
    SwsContext* swsContextPtr = sws_getCachedContext(NULL,
                                                     codecContext->width,
                                                     codecContext->height,
                                                     codecContext->pix_fmt,
                                                     codecContext->width,
                                                     codecContext->height,
                                                     PIX_FMT_BGR24,
                                                     SWS_BICUBIC, 0, 0, 0);
    swsContext = std::shared_ptr<SwsContext>(swsContextPtr, &sws_freeContext);
  }
}

#endif // SLMOTION_ENABLE_FFMPEG
