#ifndef SLMOTION_FFMPEG_VIDEOFILESOURCE
#define SLMOTION_FFMPEG_VIDEOFILESOURCE

#ifndef __APPLE__

extern "C" {
#include <libavformat/avformat.h>
}
#include <memory>
#include "VideoFileSource.hpp"
#include "SLIO.hpp"


namespace slmotion {
  /**
   * This class serves as a direct FFMPEG backend for the VideoFileSource 
   * class.
   */
  class FFmpegVideoFileSource : public VideoFileSource {
  public:
    /**
     * The default constructor. Initialises all FFMPEG structs, and sets the
     * frame number initially at zero.
     *
     * @param filename Video filename
     * @param cacheSize Maximum size allowed for the cache.
     */
    explicit FFmpegVideoFileSource(const std::string& filename,
                                   size_t cacheSize = SIZE_MAX) :
      VideoFileSource(filename, cacheSize),
      videoStream(-1), codec(NULL) {
      tryToOpenFile(filename);
    }



    /**
     * The default destructor, implemented because of necessity
     */
    ~FFmpegVideoFileSource() {}



    frame_number_type size() const;



  private:
    /**
     * Attempts to open the video file, or throws on failure
     */
    void tryToOpenFile(const std::string& filename);



    /**
     * Reads in one frame, converts it to an OpenCV BGR24 
     * Mat object, and returns it, while freeing any used data.
     * currentFrameNumber is incremented
     */
    cv::Mat readFrame();



    /**
     * Tries to seek to the given frame
     */
    void seekTo(frame_number_type frnumber);



    cv::Mat getFrame();



    inline void reset(const std::string& filename) {
      tryToOpenFile(filename);
    }



    std::shared_ptr<AVFormatContext> formatContext;
    std::shared_ptr<AVCodecContext> codecContext;
    // preallocated FFmpeg frames
    std::shared_ptr<AVFrame> internalFrameYUV;
    std::shared_ptr<AVFrame> internalFrameBGR;
    std::shared_ptr<uint8_t> dataBuffer;
    int videoStream; ///< Video stream index
    std::shared_ptr<struct SwsContext> swsContext;
    AVCodec* codec;
    AVPacket packet;

#ifdef SLMOTION_THREADING
    mutable std::mutex mutex; ///< If threaded access is used, this object will be used for mutual exclusion
#endif
  };
}
#endif // __APPLE__
#endif
