#include "FaceDetector2.hpp"
#include "util.hpp"

using cv::Rect;
using std::string;

namespace slmotion {
  extern int debug;
  static FaceDetector2 DUMMY(true);



  /**
   * Performs simple linear interpolation for the rects between the first 
   * (inclusive) index and the last (exclusive). First index must not be 0,
   * and first and second must be unequal.
   */
  static void interpolateGap(std::vector<Rect*>& rects, const std::pair<size_t,size_t>& gap) {
    assert(gap.first > 0);
    assert(gap.first < gap.second);
    const Rect& first = *rects[gap.first-1];
    const Rect& last = *rects[gap.second];
    size_t len = gap.second - gap.first;
    for (size_t i = 0; i < len; ++i) {
      double ratio = (i+1) / static_cast<double>(len+1);
      int newX = first.x + (last.x - first.x)*ratio;
      int newY = first.y + (last.y - first.y)*ratio;
      int newW = first.width + (last.width - first.width)*ratio;
      int newH = first.height + (last.height - first.height)*ratio;
    
      *rects[gap.first + i] = Rect(newX, newY, newW, newH);
    }
  }



  /**
   * Locates a gap, beginning at the firstIndex.
   *
   * Return value: first = first invalid detection in a gap,
   * last = first valid
   */
  static std::pair<size_t,size_t> findGap(const std::vector<Rect*>& rects, 
                                          size_t firstIndex) {
    std::pair<size_t,size_t> indices(firstIndex,rects.size());
    while(indices.first < rects.size() &&
          *rects[indices.first] != ViolaJonesFaceDetector::INVALID_FACE)
      ++indices.first;
    
    indices.second = indices.first;
    while(indices.second < rects.size() &&
          *rects[indices.second] == ViolaJonesFaceDetector::INVALID_FACE)
      ++indices.second;
    return indices;
  }



  // static void printRect(Rect* r) {
  //   std::cout << r->tl() << "->" << r->br() << std::endl;
  // }



  void FaceDetector2::process(frame_number_t frnumber) {   
    if (frnumber >= getFrameSource().size())
      throw std::out_of_range("Tried to detect face in a frame whose index exceeds the total number of frames!");

    assert(getFrameSource().size() > 0);
    getBlackBoard().set(frnumber, FACEDETECTOR_BLACKBOARD_ENTRY, getRawFace(frnumber));
  }



  bool FaceDetector2::processRangeImplementation(frame_number_t first, 
                                                frame_number_t last,
                                                UiCallback* uiCallback) {
    for (size_t i = first; i < last; ++i) {
      if (slmotion::debug > 0) {
        if (slmotion::debug == 1)
          std::cerr << '\r';
        std::cerr << i + 1 - first << '/' << last - first;
      }
      process(i);
      if (uiCallback != NULL && !(*uiCallback)(i))
        return false;
    }

    if (slmotion::debug > 0) 
      std::cerr << std::endl;

    std::vector<Rect*> rects = getDetectionsForRange(first, last);

    // Step one: find one detection
    size_t i;
    for(i = 0; i < rects.size() && *rects[i] == ViolaJonesFaceDetector::INVALID_FACE;
        ++i);
    
    // Verify that a detection _exists_, otherwise, do nothing
    if (i >= rects.size())
      return true;

    size_t firstIndex = i; // first detection index

    cv::Point2d meanTl(0,0);
    cv::Point2d meanBr(0,0);
    size_t nValidFaces = 0;
    for(i = 0; i < rects.size(); ++i) {
      if (*rects[i] != ViolaJonesFaceDetector::INVALID_FACE) {
        meanTl.x += rects[i]->tl().x;
        meanTl.y += rects[i]->tl().y;
        meanBr.x += rects[i]->br().x;
        meanBr.y += rects[i]->br().y;
        ++nValidFaces;
      }
    }
    meanTl.x /= nValidFaces;
    meanTl.y /= nValidFaces;
    meanBr.x /= nValidFaces;
    meanBr.y /= nValidFaces;
    cv::Rect meanFace(meanTl, meanBr);

    // Fill any missing detections before the first detection
    // Then, fill any missing detections with the previous detection
    for(i = 0; i < rects.size(); ++i) {
      if (i < firstIndex)
        *rects[i] = *rects[firstIndex];
      else if (*rects[i] == ViolaJonesFaceDetector::INVALID_FACE)
        // *rects[i] = *rects[i-1];
        *rects[i] = meanFace;
    }

    // perform low-pass filtering

    cv::Mat values(4, rects.size(), CV_64FC1);
    for (i = 0; i < rects.size(); ++i) {
      values.at<double>(0,i) = rects[i]->tl().x;
      values.at<double>(1,i) = rects[i]->tl().y;
      values.at<double>(2,i) = rects[i]->br().x;
      values.at<double>(3,i) = rects[i]->br().y;
    }

    cv::Mat valuesDFT(values.size(), CV_64FC1, cv::Scalar::all(0));
    cv::dft(values, valuesDFT, cv::DFT_ROWS);

    valuesDFT.colRange(valuesDFT.cols/8, valuesDFT.cols) = cv::Scalar::all(0);

    cv::dft(valuesDFT, values, cv::DFT_ROWS + cv::DFT_INVERSE + cv::DFT_SCALE);
   
    for(i = 0; i < rects.size(); ++i) 
      *rects[i] = cv::Rect(cv::Point(values.at<double>(0, i), values.at<double>(1, i)),
                           cv::Point(values.at<double>(2, i), values.at<double>(3, i)));
    
    return true;
  }



  /**
   * Interpolate missing detections for the given range
   */
  void FaceDetector2::interpolate(std::vector<Rect*>& rects) {
    if (rects.size() == 0)
      return;

    // Step one: find one detection
    size_t i;
    for(i = 0; i < rects.size() && *rects[i] == ViolaJonesFaceDetector::INVALID_FACE;
        ++i);
    
    // Verify that a detection _exists_, otherwise, do nothing
    if (i >= rects.size())
      return;

    size_t firstIndex = i; // first detection index
    // Fill any missing detections before the first detection
    for(auto it = rects.begin(); it < rects.begin() + firstIndex; ++it)
      **it = *rects[firstIndex];

    if(*rects.front() == INVALID_FACE)
      return; // cannot interpolate if there were NO valid detections

    // interpolate any gaps
    std::pair<size_t,size_t> indices = findGap(rects, firstIndex);
    for (; indices.first < rects.size() && indices.second < rects.size();
         indices = findGap(rects, indices.second))
      interpolateGap(rects, indices);

    // Finally, if required, fill in the last detection
    if (indices.first < rects.size())
      for(auto it = rects.begin() + indices.first; it < rects.end(); ++it)
        **it = *rects[indices.first-1];
  }



  boost::program_options::options_description FaceDetector2::getConfigurationFileOptionsDescription() const {
    boost::program_options::options_description opts;
    opts.add_options()
      ("FaceDetector2.cascade", 
       boost::program_options::value<string>()->default_value(getDefaultHaarCascadeFile()),
       "Sets the cascade filename. A valid cascade file is required for the classifier to work "
       "properly.")
      ("FaceDetector2.scale", boost::program_options::value<string>()->default_value("0.6:1.0"),
       "Input format is <double> | <double>:<double>\n"
       "Scales the face by a factor. If only one factor is specified, both the "
       "width and the height of the face rectangle will be scaled equally. If "
       "two colon-separated factors are specified, the first factor will be used "
       "for scaling the width and the second argument for scaling the height. "
       "E.g. scale=1.5 or scale=0.8:0.7")
      ("FaceDetector2.translate", boost::program_options::value<string>()->default_value("0:0"),
       "Input format is <int>:<int>\n"
       "Translates the face by some amount of pixels. The first parameter "
       "specifies the number of pixels with respect to X axis, and the second "
       "parameter with respect to Y axis.")
      ("FaceDetector2.maxmove", boost::program_options::value<int>()->default_value(25),
       "Sets the maximum number of pixels that the detected face can move with "
       "respect to its position in the previous frame. If the face was not "
       "detected in the previous frame, the distance will be considered with "
       "respect to the last detected position.")
      ("FaceDetector2.maxareachange", boost::program_options::value<double>()->default_value(0.5),
       "Sets the maximum proportional change in the area of the face detected "
       "in respect to the previous frame, or the latest detected face. E.g. if "
       "set to 0.5, the face may change its size up to 50 % between detections.")
      ("FaceDetector2.minneighbours", boost::program_options::value<int>()->default_value(2),
       "Minimum number of overlapping neighbouring detections required for "
       "a detection to be considered valid. Larger number means more accurate "
       "detections (in terms of false positives), but at the cost of possibly "
       "missing some valid detections.")
      ("FaceDetector2.scalefactor", boost::program_options::value<double>()->default_value(1.2),
       "How big a jump there is between different scales; accuracy can be "
       "increased by moving this value closer to 1.0, but this will increase "
       "computation time.");
    return opts;
  }



  Component* FaceDetector2::createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const {
    FaceDetector2 faceDetector(blackBoard, frameSource);

    // if (vm.count("face-cascade"))
    //   config.faceDetectorCascadeFilename = vm["face-cascade"].as<string>();
    // else if (vm.count("facedetection.cascade"))
    //   config.faceDetectorCascadeFilename = vm["facedetection.cascade"].as<string>();

    faceDetector.setCascadeFilename(configuration["FaceDetector2.cascade"].as<std::string>());

    return new FaceDetector2(faceDetector);
  }
}
