#ifdef SLMOTION_WITH_INTRAFACE

#include "FacialStateEstimator.hpp"
#include <boost/filesystem.hpp>
#define DUMP_DATA 1

namespace slmotion {
  extern int debug;
  static FacialStateEstimator DUMMY(true);

  FacialStateEstimator::~FacialStateEstimator() {

  }

  /**
   * Loads the model for the constructor
   * 
  struct svm_model* FacialStateEstimator::getSvmModel(std::string m) {
    struct svm_model *svmModel;
    if ((svmModel = svm_load_model(m.c_str()))==0) {
      std::cerr << "Failed to load model file " << m << std::endl;
      exit(1);
    }
    return svmModel;
  }

  **
   * Translates a vector into a SVM node of sparse representation of the original vector
   *
  static svm_node** matToSvmNode(cv::Mat O) {
    int rowSize = O.rows-1;  // delete to allow expandability in case needed later
    if (rowSize > 0) {
      int colSize = O.cols;
      
      svm_node** node = new svm_node*[rowSize];
      for(int row = 0; row < rowSize; ++row) {
        node[row] = new svm_node[colSize];
      }

      for(int row = 0; row < rowSize; ++row) {
        for (int col = 0; col<colSize; ++col) {
          node[row][col].index = col+1;
          node[row][col].value = O.at<float>(row,col);
        }
        node[row][colSize].index = -1;
      }
      return node;
    } else {
      return NULL;
    }
  }*/

  /**
   * Filter (average) and recover. O - original vector, N - filter intensity
   */
  static cv::Mat recover(cv::Mat O){
    std::vector<float> zerol;
    cv::Mat R(O.rows,1,CV_32FC1);
    R = O;
    
    for (int i=2; i<O.rows-2; i++){
      if (O.at<float>(i,0) == 0)
        zerol.push_back(i);
    }

    for (unsigned int i=0; i<zerol.size(); i++)
      R.at<float>(zerol[i],0) = (O.at<float>(zerol[i]-1,0) + O.at<float>(zerol[i]-2,0))/2;

    return R;
  }

  /**
   * Filter data and reconstruct the lost first and last datapoints.
   */
  static cv::Mat filter2(cv::Mat O, int N){
    cv::Mat A(O.rows,O.cols,CV_32FC1);
    cv::Mat filter2;
    filter2 = cv::Mat::ones(N,1,CV_32FC1) / (float)N;

    cv::filter2D(O,A,CV_32FC1,filter2, cv::Point(-1,-1));

    if (N>3){
      cv::Mat B(O.rows,O.cols,CV_32FC1);
      B = A.rowRange(1,A.rows);
      A = B;
      A.push_back(O.at<float>(O.rows-1,0));
    }

    for (int i=0;i<N-1;i++)
      A.at<float>(i,0) = O.at<float>(i,0);
    
    for (int i=0;i<2;i++){
      A.at<float>(i,0) = A.at<float>(2,0);
      A.at<float>(A.rows-i-1,0) = A.at<float>(A.rows-3,0);
    }
    return A;
  }

  /*
   * Implementation of MATLAB's gradient function. Only valid for column vectors in cv::Mat(N,1) containers
   */
  static cv::Mat matgradient(cv::Mat O){
    cv::Mat S(O.rows,1,CV_32FC1);

    cv::Sobel(O,S,CV_32FC1,0,1,1,0.5,0,cv::BORDER_DEFAULT);
    S.at<float>(0,0) = O.at<float>(1,0) - O.at<float>(0,0);
    S.at<float>(O.rows-1,0) = O.at<float>(O.rows-1,0) - O.at<float>(O.rows-2,0);

    return S; 
  }

  /*
   * Smooth changes above a predefined threshold.
   */
  static cv::Mat removeImpulse(cv::Mat O){
    cv::Mat S(O.rows,1,CV_32FC1);
    cv::Mat R(O.rows,1,CV_32FC1);
    
    S = matgradient(matgradient(O));
    R.at<float>(0,0) = O.at<float>(0,0);
    R.at<float>(1,0) = O.at<float>(1,0);
    R.at<float>(O.rows-1,0) = O.at<float>(O.rows-1,0);

    for (int i=2; i<=O.rows-2; i++){
      if ((S.at<float>(i,0)<(-STEP)) or (S.at<float>(i,0)>STEP))
        R.at<float>(i,0) = (R.at<float>(i-1,0) + R.at<float>(i-2,0))/2;
      else
        R.at<float>(i,0) = O.at<float>(i,0);
    }
    return R;
  }

  /**
   * Parses a comma separated matrix into a cv::Mat struct
   */
  static cv::Mat parseData(std::string f) {
    std::vector<float> y;
    std::ifstream file(f);
    std::string str; 
    int i,j;
    
    i = 0;
    j = 0;
    while (std::getline(file, str)){
      std::istringstream x(str);
      j=0;
      while( x.good() ){
        std::string substr;
        std::getline( x, substr, '\t' );
        y.push_back(atof(substr.c_str()));
        j++;
      }
      i++;
    }

    cv::Mat eig(i,j,CV_32FC1);
    memcpy(eig.data,y.data(),y.size()*sizeof(float));
    return eig;
  }

  /*
   * Naive Bayes
   */
  static int nBayesPredict(cv::Mat test, cv::Mat model){
    cv::Mat mean = cv::Mat(1,model.cols,CV_32FC1,float(0));
    cv::Mat variance = cv::Mat(1,model.cols,CV_32FC1,float(0));
    cv::Mat tmp = cv::Mat(1,model.cols,CV_32FC1,float(0));
    cv::Mat tmplogs = cv::Mat(1,model.cols,CV_32FC1,float(0));
    cv::Mat logPdf = cv::Mat(1,model.rows/2,CV_32FC1,float(0));

    for (int k=0; k < model.rows/2; k++){
      model.row(k*2).copyTo(mean.row(0));
      model.row(k*2+1).copyTo(variance.row(0));

      tmp = ( ( test - mean ) / variance);
      cv::multiply(tmp, tmp, tmp);
      log(variance, variance) ;
      tmplogs = (-0.5 * tmp - variance) -0.5 * log( 2*PI );

      for (int j=0; j<model.cols; j++)
        logPdf.at<float>(0,k) =  logPdf.at<float>(0,k) + tmplogs.at<float>(0,j); 

      tmplogs = cv::Mat(1,model.cols,CV_32FC1,float(0));
    }

    // since normal prior no need to add them to logPdf
    cv::Point p;
    cv::minMaxLoc(logPdf,NULL,NULL,NULL,&p);

    // the classes are ordered for this case, no need to locate class value
    return p.x;
  }

  static float maxi(std::vector<float> O){
    float maxn = O[0];
    for(unsigned int i = 1; i < O.size(); i++)
      if(O[i] > maxn)
        maxn = O[i];

    return maxn;
  }

  static float mini(std::vector<float> O){
    float minn = O[0];
    for(unsigned int i = 1; i < O.size(); i++)
      if(O[i] < minn)
        minn = O[i];

    return minn;
  }

  /**
   *  Set frame color for timeline image
   */
  static void setGreen(cv::Mat& timeline, int position, int start, int xsize, int ysize){
    for (int k = 0; k < xsize; k++)
      for (int j = start; j < ysize; j++){
        timeline.at<cv::Vec3b>(j,k + position*xsize)[0] = 51;
        timeline.at<cv::Vec3b>(j,k + position*xsize)[1] = 255;
        timeline.at<cv::Vec3b>(j,k + position*xsize)[2] = 153;
      }
  }

  /**
   *  Set frame color for timeline image
   */  
  static void setYellow(cv::Mat& timeline, int position, int start, int xsize, int ysize){
    for (int k = 0; k < xsize; k++)
      for (int j = start; j < ysize; j++){
        timeline.at<cv::Vec3b>(j,k + position*xsize)[0] = 51;
        timeline.at<cv::Vec3b>(j,k + position*xsize)[1] = 255;
        timeline.at<cv::Vec3b>(j,k + position*xsize)[2] = 255;
      }
  }

  /**
   *  Set frame color for timeline image
   */
  static void setRed(cv::Mat& timeline, int position, int start, int xsize, int ysize){
    for (int k = 0; k < xsize; k++)
      for (int j = start; j < ysize; j++){
        timeline.at<cv::Vec3b>(j,k + position*xsize)[0] = 51;
        timeline.at<cv::Vec3b>(j,k + position*xsize)[1] = 51;
        timeline.at<cv::Vec3b>(j,k + position*xsize)[2] = 255;
      }
  }

  void FacialStateEstimator::process(frame_number_t frameNumber) {
    BlackBoardPointer<cv::Rect> facePtr = getBlackBoard().get<cv::Rect>(frameNumber, FACEDETECTOR_BLACKBOARD_ENTRY);
    cv::Rect face = *facePtr;
    cv::Mat inFrame = getFrameSource()[frameNumber];

    face.y = round(face.height*0.1 + face.y);
    cv::Mat faceRoi(inFrame,face);
    //cv::blur(faceRoi,faceRoi, cv::Size(3,3), cv::Point(-1,-1));
    
    float er, el, ewr, ewl, mw, m1, m2, b0, bal, bar, b1l, b1r, b2l, b2r, bsl, bsr, roll;
    std::vector<cv::Point2f> landmark;
    cv::Point2f dummy1,bul,bur;
    cv::Mat Xn;
    float score, notFace = 0.5;

    for (int i=0; i<22; i++){
      landmark.push_back(cv::Point2d(0,0));
    }
    face.x = 0;
    face.y = 0;

    if (faceDetector->Detect(faceRoi,face,Xn,score) == INTRAFACE::IF_OK) {
      if (score >= notFace) {
        Xn.row(0) = Xn.row(0)/face.width;
        Xn.row(1) = Xn.row(1)/face.height;
        
        std::vector<float> selected{0,1,2,3,4,5,6,7,8,9,19,21,22,23,25,26,28,30,31,34,37,40};

        for (unsigned int i=0;i<selected.size();i++){       
          landmark[i].x = Xn.at<float>(0,selected[i]);
          landmark[i].y = Xn.at<float>(1,selected[i]);
        }

        dummy1.x = (landmark[18].x+landmark[20].x)/2;
        dummy1.y = (landmark[18].y+landmark[20].y)/2;

        //EYEBROW
        b0 = (float)cv::norm(landmark[5] - landmark[4]);
        b2r = (float)cv::norm(landmark[0] - landmark[10]);
        b1r = (float)cv::norm(landmark[4] - landmark[12]);
        b1l = (float)cv::norm(landmark[5] - landmark[14]);
        b2l = (float)cv::norm(landmark[9] - landmark[16]);
        bal = (float)(landmark[9].x-landmark[5].x) * (maxi(std::vector<float>{landmark[5].y,landmark[6].y,landmark[7].y,landmark[8].y,landmark[9].y}) - mini(std::vector<float>{landmark[5].y,landmark[6].y,landmark[7].y,landmark[8].y,landmark[9].y}));
        bar = (float)(landmark[4].x-landmark[0].x) * (maxi(std::vector<float>{landmark[0].y,landmark[1].y,landmark[2].y,landmark[3].y,landmark[4].y}) - mini(std::vector<float>{landmark[0].y,landmark[1].y,landmark[2].y,landmark[3].y,landmark[4].y}));
        bsr = (float)(landmark[0].y-landmark[4].y)/(landmark[0].x-landmark[4].x);
        bsl = (float)(landmark[9].y-landmark[5].y)/(landmark[9].x-landmark[5].x);
        roll = (float)(landmark[20].y-dummy1.y)/(landmark[20].x-dummy1.x);
        bur.x = (float)(landmark[0].x+landmark[1].x+landmark[2].x+landmark[3].x+landmark[4].x)/5; 
        bur.y = (float)(landmark[0].y+landmark[1].y+landmark[2].y+landmark[3].y+landmark[4].y)/5; 
        bul.x = (float)(landmark[5].x+landmark[6].x+landmark[7].x+landmark[8].x+landmark[9].x)/5; 
        bul.y = (float)(landmark[5].y+landmark[6].y+landmark[7].y+landmark[8].y+landmark[9].y)/5;

        //EYE
        ewr = cv::norm(landmark[10] - landmark[12]);
        er = cv::norm(landmark[11] - landmark[13]);
        ewl = cv::norm(landmark[14] - landmark[16]);
        el = cv::norm(landmark[15] - landmark[17]);
        if (ewr == 0)
          ewr = ewl;
        if (ewl == 0)
          ewl = ewr;
        //MOUTH
        mw = cv::norm(landmark[18] - landmark[20]);
        m1 = cv::norm(landmark[19] - dummy1);
        m2 = cv::norm(landmark[21] - dummy1);
      }
    }

    std::vector<float> v = {bsr,bsl,b1r,b2r,b1l,b2l,b0,bar,bal,er,el,(ewl+ewr)/2,mw,m1,m2, roll,bur.x,bur.y,landmark[13].x,landmark[13].y,
                            bul.x,bul.y,landmark[17].x,landmark[17].y};
    getBlackBoard().set(frameNumber, FACIAL_STATE_BLACKBOARD_ENTRY, v);
  }

  bool FacialStateEstimator::processRangeImplementation(frame_number_t first, frame_number_t last, UiCallback* uiCallback) {
    faceDetector = new INTRAFACE::FaceAlignment(modDetect.c_str(), modTrack.c_str(), &XXDESCRIPTOR);
    if (!faceDetector->Initialized()) {
       cerr << "FaceAlignmentDetect cannot be initialized." << endl;
    }

    for (size_t i = first; i < last; ++i) {
      if (slmotion::debug > 0) {
        if (slmotion::debug > 1)
          std::cout << std::endl;
      }
      this->process(i);
      if (uiCallback != nullptr && !(*uiCallback)((i-first+1)*100./(last-first)))
        return false;
    }
        
    BlackBoardPointer<std::vector<float>> rPtr;
    cv::Mat fv_eyebrow(last,11,CV_32FC1);
    cv::Mat fv_eye(last,3,CV_32FC1);
    cv::Mat fv_mouth(last,5,CV_32FC1);
    cv::Mat fv_vmouth(last,1,CV_32FC1);
    cv::Mat fv_hmouth(last,1,CV_32FC1);
    cv::Mat v_mw(last,3,CV_32FC1);
    cv::Mat fv_temp(last,9,CV_32FC1);

    float mw = 0;
    std::vector<float> bn(11);
    for (unsigned int i=0;i<11;i++)
      bn.push_back(0);

    // recollect frame information and construct vectors
    for (unsigned int i=0; i<last; i++) {
      rPtr = getBlackBoard().get<std::vector<float>>(i,FACIAL_STATE_BLACKBOARD_ENTRY);
      std::vector<float>& r = *rPtr;

      for (unsigned int k=0; k<9; k++)
        fv_eyebrow.at<float>(i,k+2) = r[k];

      for (unsigned int k=0; k<3; k++)
        fv_eye.at<float>(i,k) = r[k+9];
      
      for (unsigned int k=0; k<3; k++)
        v_mw.at<float>(i,k) = r[k+12];

      for (unsigned int k=0; k<9; k++)
        fv_temp.at<float>(i,k) = r[k+15];
      
      fv_temp.at<float>(i,9) = atan(r[15]);

      if (i < (endIx-startIx+1)){
        mw = mw + r[12];
      }
    }
    
    if (slmotion::debug > 0)
      std::cerr << "Postprocessing: ";

    // EYEBROW
    float roll,bux,buy,eycx,eycy = 0;
    cv::Mat tmp;

    // eliminate changes above a treshold that are suspected noisy landmarks
    tmp = matgradient(matgradient((fv_temp.col(0))));
    for (unsigned int i=0; i<last; i++) {
      if (abs(tmp.at<float>(i,0))>0.025)
          fv_temp.at<float>(i,0) = mean(fv_temp.col(0)).val[0];
    }
    filter2(fv_temp.col(0),3).col(0).copyTo(fv_temp.col(0));

    // compute first two unscaled features
    for (unsigned int i=0; i<last; i++) {
      roll = fv_temp.at<float>(i,0);
      bux = fv_temp.at<float>(i,1);
      buy = fv_temp.at<float>(i,2);
      eycx = fv_temp.at<float>(i,3);
      eycy = fv_temp.at<float>(i,4);
      fv_eyebrow.at<float>(i,0) = abs((roll*eycx)-eycy+(buy-roll*bux))/sqrt(1+(roll*roll));
      bux = fv_temp.at<float>(i,5);
      buy = fv_temp.at<float>(i,6);
      eycx = fv_temp.at<float>(i,7);
      eycy = fv_temp.at<float>(i,8);
      fv_eyebrow.at<float>(i,1) = abs((roll*eycx)-eycy+(buy-roll*bux))/sqrt(1+(roll*roll));
    }

    // get seed frames and scale
    for (unsigned int i=startIx; i<= endIx; i++){
      for (unsigned int k=0; k<8; k++)
            bn[k] = bn[k] + fv_eyebrow.at<float>(i,k);
    }
    bn[2] = bn[3] = bn[8] = bn[9] = bn[10] = endIx-startIx+1;
    for (unsigned int i=0; i<11; i++)
      filter2( recover( fv_eyebrow.col(i) ) /( bn[i]/ (endIx-startIx+1) ) , 3).col(0).copyTo(fv_eyebrow.col(i));  

    // EYE
    cv::Mat ew;
    ew = filter2(recover(fv_eye.col(2)),3);
    filter2(recover(fv_eye.col(0))/ew,3).col(0).copyTo(fv_eye.col(0));
    filter2(recover(fv_eye.col(1))/ew,3).col(0).copyTo(fv_eye.col(1));
    fv_eye = fv_eye.colRange(0,fv_eye.cols-1);

    // VMOUTH
    mw = mw/5;
    recover(v_mw.col(0)).col(0).copyTo(v_mw.col(0));
    removeImpulse(recover(v_mw.col(1))).col(0).copyTo(v_mw.col(1));
    removeImpulse(recover(v_mw.col(2))).col(0).copyTo(v_mw.col(2));
    filter2(v_mw.col(0)/mw,3).col(0).copyTo(fv_mouth.col(4));
    filter2(v_mw.col(0)/(v_mw.col(1)+v_mw.col(2)),3).col(0).copyTo(fv_mouth.col(1));
    filter2(v_mw.col(1)/v_mw.col(0),3).col(0).copyTo(fv_mouth.col(3));
    filter2(v_mw.col(1)/v_mw.col(2),3).col(0).copyTo(fv_mouth.col(0));
    filter2(v_mw.col(2)/v_mw.col(0),3).col(0).copyTo(fv_mouth.col(2));

    // HMOUTH
    filter2((v_mw.col(0)/mw)*10,3).col(0).copyTo(fv_hmouth.col(0));

    // ouput the raw features
    if (DUMP_DATA)
      for (unsigned int i=0;i<last;i++){
        std::cout << fv_eyebrow.at<float>(i,0) << "\t" << fv_eyebrow.at<float>(i,1) << "\t" << fv_eyebrow.at<float>(i,2) << "\t" <<fv_eyebrow.at<float>(i,3) << "\t";
        std::cout << fv_eyebrow.at<float>(i,4) << "\t" << fv_eyebrow.at<float>(i,5) << "\t" << fv_eyebrow.at<float>(i,6) << "\t" <<fv_eyebrow.at<float>(i,7) << "\t";
        std::cout << fv_eyebrow.at<float>(i,8) << "\t" << fv_eyebrow.at<float>(i,9) << "\t" << fv_eyebrow.at<float>(i,10) << "\t";
        std::cout << fv_eye.at<float>(i,0) << "\t" << fv_eye.at<float>(i,1) << "\t";
        std::cout << fv_mouth.at<float>(i,0) << "\t" << fv_mouth.at<float>(i,1) << "\t" << fv_mouth.at<float>(i,2) << "\t" << fv_mouth.at<float>(i,3) << "\t";
        std::cout << fv_mouth.at<float>(i,4) << "\n";
      }

    // pca postprocessing
    cv::Mat beig = parseData(eigEyebrow);
    cv::Mat eeig = parseData(eigEye);
    cv::Mat m1eig = parseData(eigVmouth);
    cv::Mat m2eig = parseData(eigHmouth);
    fv_eyebrow = fv_eyebrow * beig;
    fv_eye = fv_eye * eeig;
    fv_vmouth = fv_mouth * m1eig;
    fv_hmouth = fv_hmouth* m2eig;

    if (slmotion::debug > 0)
      std::cerr << "100.00 %" << std::endl;

    if (slmotion::debug > 0)
      std::cerr << "Classification: ";

    // load bayes models and classify each frame
    cv::Mat bmodel = parseData(modEyebrow);
    cv::Mat emodel = parseData(modEye);
    cv::Mat m1model = parseData(modVmouth);
    cv::Mat m2model = parseData(modHmouth);
    cv::Mat result_b(fv_eyebrow.rows,1, CV_32FC1);
    cv::Mat result_e(fv_eye.rows,1, CV_32FC1);
    cv::Mat result_m1(fv_vmouth.rows,1, CV_32FC1);
    cv::Mat result_m2(fv_hmouth.rows,1, CV_32FC1);
      
    for (int i = 0; i < fv_eye.rows; i++){
      result_b.at<float>(i,0) = nBayesPredict(fv_eyebrow.row(i),bmodel);
      result_e.at<float>(i,0) = nBayesPredict(fv_eye.row(i),emodel);
      result_m1.at<float>(i,0) = nBayesPredict(fv_vmouth.row(i),m1model);
      result_m2.at<float>(i,0) = nBayesPredict(fv_hmouth.row(i),m2model);
    }
    
    if (slmotion::debug > 0)
      std::cerr << "100.00 %" << std::endl;
    
    // filter estimations
    cv::Mat result_ef(fv_eye.rows,1, CV_32FC1);
    cv::medianBlur(result_b,result_b,5);
    cv::medianBlur(result_e,result_ef,5);
    cv::medianBlur(result_m1,result_m1,5);
    cv::medianBlur(result_m2,result_m2,5);
    
    // create timeline image and recover squints and close eyes from filter
    cv::Mat timeline;
    if (includeTimeline)
      timeline = cv::Mat(frHeight * 4+annBar, frWidth * result_e.rows, CV_8UC3, cv::Scalar(255,255,255));

    for (int i = 0; i< result_e.rows; i++){
      if (result_ef.at<float>(i,0) == 2)
        result_e.at<float>(i,0) = 2;
      if (result_ef.at<float>(i,0) == 3)
        result_e.at<float>(i,0) = 3;

      if (includeTimeline){
        switch ( (long) (result_b.at<float>(i,0)+0.5)) {
          case 0:
            setRed(timeline, i, 0, frWidth, frHeight);
            break;
          case 2:
            setGreen(timeline, i, 0, frWidth, frHeight);
            break;
        }
        switch ( (long) (result_e.at<float>(i,0)+0.5)) {
          case 0:
            setRed(timeline, i, frHeight, frWidth, frHeight*2);
            break;
          case 1:
            setYellow(timeline, i, frHeight, frWidth, frHeight*2);
            break;
          case 3:
            setGreen(timeline, i, frHeight, frWidth, frHeight*2);
            break;
        }
        switch ( (long) (result_m1.at<float>(i,0)+0.5)) {
          case 1:
            setGreen(timeline, i, frHeight*2, frWidth, frHeight*3);
            break;
          case 2:
            setYellow(timeline, i, frHeight*2, frWidth, frHeight*3);
            break;
        }
        switch ( (long) (result_m2.at<float>(i,0)+0.5)) {
          case 0:
            setYellow(timeline, i, frHeight*3, frWidth, frHeight*4);
            break;
          case 2:
            setGreen(timeline, i, frHeight*3, frWidth, frHeight*4);
            break;
        }

        for (unsigned int j = frHeight*4; j < frHeight*4+annBar; j++){
          timeline.at<cv::Vec3b>(j, i*frWidth-1)[0] = 224;
          timeline.at<cv::Vec3b>(j, i*frWidth-1)[1] = 224;
          timeline.at<cv::Vec3b>(j, i*frWidth-1)[2] = 224;
        }

        if ((i+1)%(annCol) == 0){
          for (unsigned int k = 0; k < frWidth-1; k++)
            for (unsigned int j = frHeight*4; j < frHeight*4+annBar; j++){
              timeline.at<cv::Vec3b>(j, k+i*frWidth)[0] = 224;
              timeline.at<cv::Vec3b>(j, k+i*frWidth)[1] = 224;
              timeline.at<cv::Vec3b>(j, k+i*frWidth)[2] = 224;
            }
        }
      }
    }

    if (slmotion::debug > 0)
      std::cerr << "Results (eyebrow, eye, vmouth, hmouth)" << " (" << last << ")" << std::endl;
    for (unsigned int i=0;i<last;i++)
      std::cout << result_b.at<float>(i,0) << "\t" << result_e.at<float>(i,0) << "\t" << result_m1.at<float>(i,0) << "\t" << result_m2.at<float>(i,0) << "\n";

    if (includeTimeline){
      std::string filename = boost::filesystem::path( getFrameSource().getFileName() ).stem().string();
      cv::imwrite( filename + "timeline.png", timeline);
      std::cerr << filename + "timeline.png" << endl;
    }

    return true;
  }

  Component::property_set_t FacialStateEstimator::getRequirements() const {
    property_set_t props {FACEDETECTOR_BLACKBOARD_ENTRY};
    return props;
  }

  boost::program_options::options_description FacialStateEstimator::getConfigurationFileOptionsDescription() const {
    boost::program_options::options_description opts;
    opts.add_options()
      ("FacialStateEstimator.eigEyebrow", boost::program_options::value<string>()->default_value(DEFAULT_EIG_EYEBROW),
       "Sets the eyebrow position eigenvector file address.")
      ("FacialStateEstimator.eigEye", boost::program_options::value<string>()->default_value(DEFAULT_EIG_EYE),
       "Sets the eye openness eigenvector file address.")
      ("FacialStateEstimator.eigVmouth", boost::program_options::value<string>()->default_value(DEFAULT_EIG_VMOUTH),
       "Sets the vertical mouth state eigenvector file address.")
      ("FacialStateEstimator.eigHmouth", boost::program_options::value<string>()->default_value(DEFAULT_EIG_HMOUTH),
       "Sets the horizontal mouth state eigenvector file address.")
      ("FacialStateEstimator.modEyebrow", boost::program_options::value<string>()->default_value(DEFAULT_MODEL_EYEBROW),
       "Sets the eyebrow position classifier model file address. A Bayes classifier model file is required for the estimations.")
      ("FacialStateEstimator.modEye", boost::program_options::value<string>()->default_value(DEFAULT_MODEL_EYE),
       "Sets the eye openness classifier model file address. A Bayes classifier model file is required for the estimations.")
      ("FacialStateEstimator.modVmouth", boost::program_options::value<string>()->default_value(DEFAULT_MODEL_VMOUTH),
       "Sets the vertical mouth state model file address. A libSVM classifier model file is required for the estimations.")
      ("FacialStateEstimator.modHmouth", boost::program_options::value<string>()->default_value(DEFAULT_MODEL_HMOUTH),
       "Sets the horizontal mouth state model file address. A libSVM classifier model file is required for the estimations.")
      ("FacialStateEstimator.modTrack", boost::program_options::value<string>()->default_value(MODEL_TRACK),
       "Intraface tracking model file location. The file should have a .bin extension.")
      ("FacialStateEstimator.modDetect", boost::program_options::value<string>()->default_value(MODEL_DETECT),
       "Intraface detection model file location. The file should have a .bin extension.")
      ("FacialStateEstimator.startSeed", boost::program_options::value<unsigned int>()->default_value(DEFAULT_START_IX),
       "Start frame number used to initialize the default state for the mouth and eyebrow positions.")
      ("FacialStateEstimator.endSeed", boost::program_options::value<unsigned int>()->default_value(DEFAULT_END_IX),
       "End frame number used to initialize the default state for the mouth and eyebrow positions.")
      ("FacialStateEstimator.frHeight", boost::program_options::value<unsigned int>()->default_value(DEFAULT_FRAME_HEIGHT),
       "Timeline visualization: height of each facial element representation.")
      ("FacialStateEstimator.frWidth", boost::program_options::value<unsigned int>()->default_value(DEFAULT_FRAME_WIDTH),
       "Timeline visualization: frame width of each facial element estimation.")
      ("FacialStateEstimator.annBar", boost::program_options::value<unsigned int>()->default_value(DEFAULT_ANN_HEIGHT),
       "Timeline visualization: height of the annotation bar.")
      ("FacialStateEstimator.annCol", boost::program_options::value<unsigned int>()->default_value(DEFAULT_ANN_MARK),
       "Timeline visualization: each number of marked frames in the annotation bar.")
      ("FacialStateEstimator.includeTimeline", boost::program_options::value<bool>()->default_value(DEFAULT_CREATE_TIMELINE),
       "Timeline visualization: create timeline visualization for the processed video.");
    return opts;
  }

  Component* FacialStateEstimator::createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const { 
    FacialStateEstimator* facialStateEstimator = new FacialStateEstimator(blackBoard, frameSource);
    if (configuration.count("FacialStateEstimator.eigEyebrow")) {
      facialStateEstimator->setEigEyebrow(configuration["FacialStateEstimator.eigEyebrow"].as<string>());
    }
    if (configuration.count("FacialStateEstimator.eigEye")) {
      facialStateEstimator->setEigEye(configuration["FacialStateEstimator.eigEye"].as<string>());
    }
    if (configuration.count("FacialStateEstimator.eigVmouth")) {
      facialStateEstimator->setEigVmouth(configuration["FacialStateEstimator.eigVmouth"].as<string>());
    }
    if (configuration.count("FacialStateEstimator.eigHmouth")) {
      facialStateEstimator->setEigHmouth(configuration["FacialStateEstimator.eigHmouth"].as<string>());
    }
    if (configuration.count("FacialStateEstimator.modEyebrow")) {
      facialStateEstimator->setModEyebrow(configuration["FacialStateEstimator.modEyebrow"].as<string>());
    }
    if (configuration.count("FacialStateEstimator.modEye")) {
      facialStateEstimator->setModEye(configuration["FacialStateEstimator.modEye"].as<string>());
    }
    if (configuration.count("FacialStateEstimator.modVmouth")) {
      facialStateEstimator->setModVmouth(configuration["FacialStateEstimator.modVmouth"].as<string>());
    }
    if (configuration.count("FacialStateEstimator.modHmouth")) {
      facialStateEstimator->setModHmouth(configuration["FacialStateEstimator.modHmouth"].as<string>());
    }
    if (configuration.count("FacialStateEstimator.modTrack")) {
      facialStateEstimator->setModTrack(configuration["FacialStateEstimator.modTrack"].as<string>());
    }
    if (configuration.count("FacialStateEstimator.modDetect")) {
      facialStateEstimator->setModDetect(configuration["FacialStateEstimator.modDetect"].as<string>());
    }
    if (configuration.count("FacialStateEstimator.startSeed")) {
      facialStateEstimator->setStartSeed(configuration["FacialStateEstimator.startSeed"].as<unsigned int>());
    }
    if (configuration.count("FacialStateEstimator.endSeed")) {
      facialStateEstimator->setEndSeed(configuration["FacialStateEstimator.endSeed"].as<unsigned int>());
    }
    if (configuration.count("FacialStateEstimator.frHeight")) {
      facialStateEstimator->setFrHeight(configuration["FacialStateEstimator.frHeight"].as<unsigned int>());
    }
    if (configuration.count("FacialStateEstimator.frWidth")) {
      facialStateEstimator->setFrWidth(configuration["FacialStateEstimator.frWidth"].as<unsigned int>());
    }
    if (configuration.count("FacialStateEstimator.annBar")) {
      facialStateEstimator->setAnnBar(configuration["FacialStateEstimator.annBar"].as<unsigned int>());
    }
    if (configuration.count("FacialStateEstimator.annCol")) {
      facialStateEstimator->setAnnCol(configuration["FacialStateEstimator.annCol"].as<unsigned int>());
    }
    if (configuration.count("FacialStateEstimator.includeTimeline")) {
      facialStateEstimator->setIncludeTimeline(configuration["FacialStateEstimator.includeTimeline"].as<bool>());
    }
    return facialStateEstimator;
  }
}
#endif // SLMOTION_WITH_INTRAFACE
