#include "FeatureCreator.hpp"
#include "slmotion.hpp"
#include "AsmTracker.hpp"
#include "KLTTracker.hpp"
#include "util.hpp"
#include "BlackBoardDumpWriter.hpp"

using std::vector;
using std::deque;
using cv::Vec2d;
using cv::Vec3d;
using cv::Vec4d;
using cv::Vec6d;
using cv::Vec;
using cv::Point2f;
using cv::Point2d;
using std::string;
using std::map;
using std::list;
using std::make_pair;



namespace slmotion {
  const vector<SOM_PAK_Component> SOM_PAK_COMPONENTS {
    {0, "framenr", 0, 999999, SOM_PAK_Component::ValueType::INTEGER, 
     "1"}, // 0
    {1, "timecode", 0, 999999, SOM_PAK_Component::ValueType::INTEGER, 
     "ms"}, // 1
    {2, "total_num_of_tracked_features", 0, 999999, 
     SOM_PAK_Component::ValueType::INTEGER, "1"}, // 2
    {3, "total_total_horizontal_velocity", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {4, "total_total_vertical_velocity", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {5, "total_length_sum_vvec", 0, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {6, "total_length_sum_avec", 0, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"}, // 6
    {7, "grosshead_num_of_tracked_features", 0, 999999, 
     SOM_PAK_Component::ValueType::INTEGER, "1"}, // 7
    {8, "grosshead_total_horizontal_velocity", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {9, "grosshead_total_vertical_velocity", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {10, "grosshead_length_sum_vvec", 0, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {11, "grosshead_length_sum_avec", 0, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {12, "grosslefthand_num_of_tracked_features", 0, 999999, 
     SOM_PAK_Component::ValueType::INTEGER, "1"}, // 12
    {13, "grosslefthand_total_horizontal_velocity", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {14, "grosslefthand_total_vertical_velocity", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {15, "grosslefthand_length_sum_vvec", 0, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {16, "grosslefthand_length_sum_avec", 0, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {17, "grossrighthand_num_of_tracked_features", 0, 999999, 
     SOM_PAK_Component::ValueType::INTEGER, "1"}, // 17
    {18, "grossrighthand_total_horizontal_velocity", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {19, "grossrighthand_total_vertical_velocity", -999999, 999999,
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {20, "grossrighthand_length_sum_vvec", 0, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {21, "grossrighthand_length_sum_avec", 0, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {22, "nethead_num_of_tracked_features", 0, 999999, 
     SOM_PAK_Component::ValueType::INTEGER, "1"}, // 22
    {23, "nethead_total_horizontal_velocity", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {24, "nethead_total_vertical_velocity", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {25, "nethead_length_sum_vvec", 0, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {26, "nethead_length_sum_avec", 0, 999999,
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {27, "netlefthand_num_of_tracked_features", 0, 999999, 
     SOM_PAK_Component::ValueType::INTEGER, "1"}, // 27
    {28, "netlefthand_total_horizontal_velocity", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {29, "netlefthand_total_vertical_velocity", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {30, "netlefthand_length_sum_vvec", 0, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {31, "netlefthand_length_sum_avec", 0, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {32, "netrighthand_num_of_tracked_features", 0, 999999, 
     SOM_PAK_Component::ValueType::INTEGER, "1"}, // 32
    {33, "netrighthand_total_horizontal_velocity", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {34, "netrighthand_total_vertical_velocity", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {35, "netrighthand_length_sum_vvec", 0, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {36, "netrighthand_length_sum_avec", 0, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},

      // "ASM features" begin here 
    {37, "overlapping_left_hand_head", 0, 999999, 
     SOM_PAK_Component::ValueType::INTEGER, "pixel"}, // 37
    {38, "overlapping_right_hand_head", 0, 999999, 
     SOM_PAK_Component::ValueType::INTEGER, "pixel"},
    {39, "overlapping_left_right_hand", 0, 999999, 
     SOM_PAK_Component::ValueType::INTEGER, "pixel"},
    {40, "head_x_velocity", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"}, // 40
    {41, "head_y_velocity", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {42, "head_velocity_magnitude", 0, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {43, "lefthand_x_velocity", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"}, // 43
    {44, "lefthand_y_velocity", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {45, "lefthand_velocity_magnitude", 0, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {46, "righthand_x_velocity", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"}, // 46
    {47, "righthand_y_velocity", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {48, "righthand_velocity_magnitude", 0, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel/frame"},
    {49, "orientation_head_theta", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "radian"}, // 49
    {50, "orientation_head_omega", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "radian/frame"},
    {51, "orientation_lefthand_theta", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "radian"},
    {52, "orientation_lefthand_omega", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "radian/frame"},
    {53, "orientation_righthand_theta", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "radian"},
    {54, "orientation_righthand_omega", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "radian/frame"} // 54
    ,
    {55, "lefthand_x_position", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel"},
    {56, "lefthand_y_position", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel"},
    {57, "righthand_x_position", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel"},
    {58, "righthand_y_position", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel"},
    {59, "nose_x_position", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel"},
    {60, "nose_y_position", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel"},

    {61, "lefthand_x_position_hum", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "huxel"},
    {62, "lefthand_y_position_hum", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "huxel"},
    {63, "righthand_x_position_hum", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "huxel"},
    {64, "righthand_y_position_hum", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "huxel"},
    {65, "nose_x_position_hum", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "huxel"},
    {66, "nose_y_position_hum", -999999, 999999, 
	SOM_PAK_Component::ValueType::REAL, "huxel"},

    {67, "lefthand_x_position_hand", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "huxel"},
    {68, "lefthand_y_position_hand", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "huxel"},
    {69, "righthand_x_position_hand", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "huxel"},
    {70, "righthand_y_position_hand", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "huxel"},

    {71, "handcenter_x_position_hum", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "huxel"},
    {72, "handcenter_y_position_hum", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "huxel"},

    {73, "leftshoulder_x_position", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel"},
    {74, "leftshoulder_y_position", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel"},
    {75, "rightshoulder_x_position", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel"},
    {76, "rightshoulder_y_position", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel"},
    {77, "neck_x_position", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel"},
    {78, "neck_y_position", -999999, 999999, 
     SOM_PAK_Component::ValueType::REAL, "pixel"}
  };

  vector<boost::any> createFrameNumberVector(size_t first, size_t last) {
    assert(last >= first);
    vector<boost::any> v(last-first);
    for (size_t i = first; i < last; ++i)
      v[i - first] = static_cast<int>(i);
    return v;
  }

  vector<boost::any> createTimeCodeVector(size_t first, size_t last, double fps) {
    assert(last >= first);
    vector<boost::any> v(last-first);
    for (size_t i = first; i < last; ++i)
      v[i - first] = static_cast<int>((i*1000)/fps);
    return v;
  }

  /**
   * Removes points that have been tracked for fewer frames than specified
   */
  FeatureCreator::tracked_point_map_vector_t
    FeatureCreator::pruneTracked(const std::vector<std::set<TrackedPoint>>& points,
                                 size_t minimumNumberOfTrackedFrames) {
    // remember past tracked points
    std::set<TrackedPoint::point_id_type_t> pastTracked;
    // these points will be removed
    std::set<TrackedPoint::point_id_type_t> pointsToRemove;

    for (size_t i = 0; i < points.size(); ++i) {
      for (auto it = points[i].cbegin(); it != points[i].cend(); ++it) {
        if (!pastTracked.count(it->getId()) && 
            !pointsToRemove.count(it->getId())) {
          size_t k = 1;
          for (size_t j = i + 1; j < i + minimumNumberOfTrackedFrames &&
                 j < points.size(); ++j)
            if (points[j].count(*it))
              ++k;
            else
              break;
          if (k >= minimumNumberOfTrackedFrames) 
            pastTracked.insert(it->getId());
          else
            pointsToRemove.insert(it->getId());
        }
      }
    }

    // second pass:
    // create maps
    tracked_point_map_vector_t maps(points.size());
    for (size_t i = 0; i < points.size(); ++i) 
      for (auto it = points[i].cbegin(); it != points[i].cend(); ++it) 
        if (pastTracked.count(it->getId()))
          maps[i].insert(std::make_pair(it->getId(),*it));

    return maps;
  }



  FeatureCreator::velocity_vector_map_vector_t FeatureCreator::createVelocityVectors(const tracked_point_map_vector_t& points) {
    velocity_vector_map_vector_t velocityVectors(points.size());

    for (size_t i = 0; i < points.size(); ++i) {
      for (auto it = points[i].begin(); it != points[i].end(); ++it) {
        velocity_vector_t vvec(0,0);
        if (i > 0 && i + 1 < points.size() && points[i-1].count(it->first)
            && points[i+1].count(it->first)) 
          vvec = 0.5 * (points[i+1].find(it->first)->second -
                        points[i-1].find(it->first)->second);
        
        velocityVectors[i].insert(std::make_pair(it->first, vvec));
      }
    }

    return velocityVectors;
  }



  FeatureCreator::acceleration_vector_map_vector_t FeatureCreator::createAccelerationVectors(const tracked_point_map_vector_t& points) {
    acceleration_vector_map_vector_t accelerationVectors(points.size());

    for (size_t i = 0; i < points.size(); ++i) {
      for (auto it = points[i].begin(); it != points[i].end(); ++it) {
        acceleration_vector_t avec(0,0);
        if (i > 0 && i + 1 < points.size() && points[i-1].count(it->first)
            && points[i+1].count(it->first)) 
          avec = points[i+1].find(it->first)->second +
            points[i-1].find(it->first)->second -
            2.0 * static_cast<TrackedPoint::point_t>(it->second);
        
        accelerationVectors[i].insert(std::make_pair(it->first, avec));
      }
    }

    return accelerationVectors;
  }

  /**
   * Creates the features (5 in total) assuming that the features match the
   * body parts
   * If strict is enabled, only the desired body parts are counted. 
   * Otherwise, anything that "isA" bodypart will do
   */
  void FeatureCreator::computeKLTFeatures(vector<vector<boost::any>>::iterator begin,
                                          vector<vector<boost::any>>::iterator end,
                                          const FeatureCreator::velocity_vector_map_vector_t& velocityVectors,
                                          const FeatureCreator::acceleration_vector_map_vector_t& accelerationVectors,
                                          const FeatureCreator::point_id_to_part_name_map_vector_t& bodyPartAssociation,
                                          BodyPart::PartName allowedBodyPart,
                                          bool strict) {
    assert(end - begin == 5);
    assert(velocityVectors.size() == accelerationVectors.size());

    for (auto it = begin; it != end; ++it)
      *it = vector<boost::any>(velocityVectors.size());

    for (size_t i = 0; i < velocityVectors.size(); ++i) {
      assert(velocityVectors[i].size() == accelerationVectors[i].size());
      int numOfTrackedFeatures = 0;
      Point2f sumVelocity(0,0);
      Point2f sumAcceleration(0,0);

      for (auto it = velocityVectors[i].begin(); 
           it != velocityVectors[i].end(); ++it) {
        assert(accelerationVectors[i].count(it->first));
        assert(bodyPartAssociation[i].count(it->first));

        BodyPart::PartName bpn = bodyPartAssociation[i].find(it->first)->second;
        if ((strict && allowedBodyPart == bpn) ||
            (!strict && BodyPart::isA(bpn, allowedBodyPart))) {
          ++numOfTrackedFeatures;
          sumVelocity += it->second;
          sumAcceleration += accelerationVectors[i].find(it->first)->second;
        }
      }

      auto it = begin;
      (*it++)[i] = boost::any(numOfTrackedFeatures);
      (*it++)[i] = boost::any(static_cast<double>(sumVelocity.x));
      (*it++)[i] = boost::any(static_cast<double>(sumVelocity.y));
      (*it++)[i] = boost::any(cv::norm(sumVelocity));
      (*it++)[i] = boost::any(cv::norm(sumAcceleration));
    }
  }

  static std::vector < std::map<TrackedPoint::point_id_type_t, BodyPart::PartName> > getBodyPartMap(const FeatureCreator::tracked_point_map_vector_t& points) {
    std::vector < std::map<TrackedPoint::point_id_type_t, BodyPart::PartName> > bodyParts(points.size());
    for (size_t i = 0; i < points.size(); ++i)
      for (auto it = points[i].begin(); it != points[i].end(); ++it)
        bodyParts[i].insert(std::make_pair(it->second.getId(),
                                           it->second.getPartName()));
    return bodyParts;
  }



  static void storeKLTPointsAndVectorsToBlackboard(BlackBoard& bb, 
                                                   const FeatureCreator::tracked_point_map_vector_t& prunedPoints, 
                                                   const FeatureCreator::velocity_vector_map_vector_t& velocityVectors,
                                                   const FeatureCreator::acceleration_vector_map_vector_t& accelerationVectors,
                                                   const FeatureCreator::point_id_to_part_name_map_vector_t& partIds,
                                                   size_t firstFrame) {
    assert(prunedPoints.size() == velocityVectors.size());
    assert(velocityVectors.size() == accelerationVectors.size());

    for (size_t i = 0; i < prunedPoints.size(); ++i) {
      bb.set(i + firstFrame, 
             FEATURECREATOR_KLT_PRUNED_POINTS_BLACKBOARD_ENTRY,
             prunedPoints[i]);
      bb.set(i + firstFrame,
             FEATURECREATOR_KLT_VELOCITY_VECTORS_BLACKBOARD_ENTRY,
             velocityVectors[i]);
      bb.set(i + firstFrame,
             FEATURECREATOR_KLT_ACCELERATION_VECTORS_BLACKBOARD_ENTRY,
             accelerationVectors[i]);

      bb.set(i + firstFrame,
             FEATURECREATOR_KLT_IDENTITIES, partIds[i]);
    }
  }



  void FeatureCreator::createKLTFeatures(const AnalysisResult& data, 
                                         vector < vector<boost::any> >& features,
                                         BlackBoard& blackBoard) {
    tracked_point_map_vector_t prunedPoints = 
      FeatureCreator::pruneTracked(data.KLTPoints, 3);

    velocity_vector_map_vector_t velocityVectors = createVelocityVectors(prunedPoints);
    acceleration_vector_map_vector_t accelerationVectors = createAccelerationVectors(prunedPoints);
    std::vector < std::map<TrackedPoint::point_id_type_t, BodyPart::PartName> > bodyParts = getBodyPartMap(prunedPoints);

    // 5 features times (1 total + 3 strict (net) + 3 relaxed (gross)) = 35
    const size_t FEATURE_SET_SIZE = 5;
    const size_t NUMBER_OF_FEATURE_SETS = 7;
    const size_t KLT_FEATURE_COUNT = FEATURE_SET_SIZE * NUMBER_OF_FEATURE_SETS;
    features.insert(features.end(), KLT_FEATURE_COUNT, vector<boost::any>());
    auto kltBegin = features.end() - KLT_FEATURE_COUNT;
    // helper function for the loop
    // in the pair, first = strict? second = allowed body part
    auto kltInserter = [&](const std::pair<bool, BodyPart::PartName>& sbp) {
      computeKLTFeatures(kltBegin, kltBegin + FEATURE_SET_SIZE, 
                         velocityVectors, accelerationVectors, bodyParts, 
                         sbp.second, sbp.first);
      kltBegin += FEATURE_SET_SIZE;
    };
    // helper vector, strict/bodypart values
    vector<std::pair<bool, BodyPart::PartName>> sbps {
      std::make_pair(false, BodyPart::UNKNOWN), 
        std::make_pair(false, BodyPart::HEAD), 
        std::make_pair(false, BodyPart::LEFT_HAND),
        std::make_pair(false, BodyPart::RIGHT_HAND),
        std::make_pair(true, BodyPart::HEAD), 
        std::make_pair(true, BodyPart::LEFT_HAND),
        std::make_pair(true, BodyPart::RIGHT_HAND)
    };
    std::for_each(sbps.begin(), sbps.end(), kltInserter);

    storeKLTPointsAndVectorsToBlackboard(blackBoard, prunedPoints,
                                         velocityVectors, 
                                         accelerationVectors,
                                         bodyParts, data.firstFrame);
  }



  FeatureCreator::point_id_to_part_name_map_vector_t FeatureCreator::createBodyPartAssociationMaps(const tracked_point_map_vector_t& points) {
    point_id_to_part_name_map_vector_t associations(points.size());

    for (size_t i = 0; i < points.size(); ++i) 
      for (auto it = points[i].cbegin(); it != points[i].cend(); ++it)
        associations[i][it->first] = it->second.getPartName();

    return associations;
  }



  bool FeatureCreator::processRangeImplementation(frame_number_t first, 
                                                  frame_number_t last,
                                                  UiCallback*) {
    if (debug > 1)
      std::cerr << "slmotion: FeatureCreator: extractRawResults" << std::endl;
    auto data = extractRawResults(first, last, getBlackBoard());

    if (debug > 1)
      std::cerr << "slmotion: FeatureCreator: Copy Components" << std::endl;
    vector<SOM_PAK_Component> components(SOM_PAK_COMPONENTS.begin(),
                                         SOM_PAK_COMPONENTS.end());
    if (debug > 1)
      std::cerr << "slmotion: FeatureCreator: Construct Any-Vector \"features\"" << std::endl;
    vector<vector<boost::any>> features;

    // frame number and timecode
    if (debug > 1)
      std::cerr << "slmotion: FeatureCreator: Push timecode and frame number" << std::endl;
    features.push_back(createFrameNumberVector(data.firstFrame, 
                                               data.lastFrame));
    features.push_back(createTimeCodeVector(data.firstFrame, data.lastFrame,
                                            getFrameSource().getFps()));

    if (debug > 1)
      std::cerr << "slmotion: FeatureCreator: Create KLT features" << std::endl;
    createKLTFeatures(data, features, getBlackBoard());
    if (debug > 1)
      std::cerr << "slmotion: FeatureCreator: Create ASM features" << std::endl;
    createASMFeatures(data, features);

    if (debug > 1)
      std::cerr << "slmotion: FeatureCreator: Set on black board" << std::endl;
    getBlackBoard().set(FEATURECREATOR_FEATUREVECTOR_BLACKBOARD_ENTRY,
                        FeatureVector(components, features, 
                                      data.firstFrame, data.lastFrame));

    if (debug > 1)
      std::cerr << "slmotion: FeatureCreator: Return." << std::endl;
    return true;
  }



  //  typedef std::pair<PDM,cv::Point> ppp_t; // PDM-Point-pair
  // typedef std::tuple<ppp_t,ppp_t,ppp_t> pp3t_t; // PDM-Point-3-tuple
  struct AsmTriple {
    Asm::Instance head;
    Asm::Instance leftHand;
    Asm::Instance rightHand;
  };


  /**
   * Computes the overlap features, and stores the tracks at the given
   * iterators. Tuple order is assumed to be head-lefthand-righthand
   */
  static void computeAsmOverlap(vector<std::vector<boost::any>>::iterator begin,
                                vector<std::vector<boost::any>>::iterator end,
                                const AsmTriple& asms) {
    assert(end - begin == 3);
    int headLeftHandOverlap = intersectionArea(asms.leftHand, asms.head);
      // PDM::intersectionArea(std::get<0>(asms).first,
      //                       std::get<1>(asms).first,
      //                       std::get<0>(asms).second,
      //                       std::get<1>(asms).second);

    int headRightHandOverlap = intersectionArea(asms.rightHand, asms.head);
      // PDM::intersectionArea(std::get<0>(asms).first,
      //                       std::get<2>(asms).first,
      //                       std::get<0>(asms).second,
      //                       std::get<2>(asms).second);

    int leftRightHandOverlap = intersectionArea(asms.leftHand, 
                                                asms.rightHand);
      // PDM::intersectionArea(std::get<1>(asms).first,
      //                       std::get<2>(asms).first,
      //                       std::get<1>(asms).second,
      //                       std::get<2>(asms).second);

    begin++->push_back(boost::any(headLeftHandOverlap));
    begin++->push_back(boost::any(headRightHandOverlap));
    begin++->push_back(boost::any(leftRightHandOverlap));
  }



  /**
   * Computes the centroid-related features (mainly velocities), orientation
   * angles, and angular velocities, and stores the tracks at the given 
   * iterators. Tuple order is assumed to be head-lefthand-righthand
   *
   * @param futureAsms PDM/Anchor-point pairs for the frame following the 
   * current one
   * @param currentAsms As above but fot the current frame
   * @param pastAsms as above but for the previous frame
   */
  static void computeAsmFeatures(vector<std::vector<boost::any>>::iterator begin,
				 vector<std::vector<boost::any>>::iterator end,
				 const AsmTriple& futureAsms,
				 const AsmTriple& currentAsms,
				 const AsmTriple& pastAsms) {
    assert(end - begin == 39);

    // this function does the magic
    auto computeXYVelAndMag = [&begin](const Asm::Instance& futureAsm,
                                       const Asm::Instance& pastAsm) {
      Point2d p = futureAsm.computeCentroid();
      p -= pastAsm.computeCentroid();
      p *= 0.5;
      // now p contains the difference
      begin++->push_back(boost::any(p.x));
      begin++->push_back(boost::any(p.y));
      begin++->push_back(boost::any(cv::norm(p)));
    };

    computeXYVelAndMag(futureAsms.head, pastAsms.head);
    computeXYVelAndMag(futureAsms.leftHand, pastAsms.leftHand);
    computeXYVelAndMag(futureAsms.rightHand, pastAsms.rightHand);

    // reapply the same principle
    auto computeThetaOmega = [&begin](const Asm::Instance& futureAsm,
                                      const Asm::Instance& currentAsm,
                                      const Asm::Instance& pastAsm) {
      double theta = computeOrientation(currentAsm);
      double omega = (computeOrientation(futureAsm) -
                      computeOrientation(pastAsm)) * 0.5;
      // now p contains the difference
      begin++->push_back(boost::any(theta));
      begin++->push_back(boost::any(omega));
    };

    computeThetaOmega(futureAsms.head, currentAsms.head,
                      pastAsms.head);
    computeThetaOmega(futureAsms.leftHand, currentAsms.leftHand,
                      pastAsms.leftHand);
    computeThetaOmega(futureAsms.rightHand, currentAsms.rightHand,
                      pastAsms.rightHand);

    // 109500:40 :
    Point2d l_shldr(238, 275);
    Point2d r_shldr(506, 274);
    Point2d neck(367, 231);

    PixelToBodyCoordinateTransformer tr(r_shldr.x-l_shldr.x, neck);

    Point2d p = currentAsms.leftHand.computeCentroid();
    begin++->push_back(boost::any(p.x));
    begin++->push_back(boost::any(p.y));
    Point2d lh = tr(p);

    p = currentAsms.rightHand.computeCentroid();
    begin++->push_back(boost::any(p.x));
    begin++->push_back(boost::any(p.y));
    Point2d rh = tr(p);
    
    Point2d nose(-1, -1);
    p = nose;
    begin++->push_back(boost::any(p.x));
    begin++->push_back(boost::any(p.y));
    Point2d ns = tr(p);

    begin++->push_back(boost::any(lh.x));
    begin++->push_back(boost::any(lh.y));
    begin++->push_back(boost::any(rh.x));
    begin++->push_back(boost::any(rh.y));
    begin++->push_back(boost::any(ns.x));
    begin++->push_back(boost::any(ns.y));

    Point2d hcenter = (lh+rh)*0.5;
    
    p = lh-hcenter;
    begin++->push_back(boost::any(p.x));
    begin++->push_back(boost::any(p.y));
    p = rh-hcenter;
    begin++->push_back(boost::any(p.x));
    begin++->push_back(boost::any(p.y));

    p = hcenter;
    begin++->push_back(boost::any(p.x));
    begin++->push_back(boost::any(p.y));

    begin++->push_back(boost::any(l_shldr.x));
    begin++->push_back(boost::any(l_shldr.y));
    begin++->push_back(boost::any(r_shldr.x));
    begin++->push_back(boost::any(r_shldr.y));
    begin++->push_back(boost::any(neck.x));
    begin++->push_back(boost::any(neck.y));
  }

  void FeatureCreator::createASMFeatures(const AnalysisResult& data,
                                         std::vector<std::vector<boost::any>>& features) {
    // 3 × overlap + 3×3×velocity + 3×(1 theta + 1 omega) + 2x3x2 x pixel&hum position 
    //  + 2x2 hand position + 4x2 coord systems = 42
    const size_t ASM_FEATURE_COUNT = 42; // total 37+42=79
    features.insert(features.end(), ASM_FEATURE_COUNT, vector<boost::any>());
    auto asmBegin = features.end() - ASM_FEATURE_COUNT;

    assert(data.headASMs.size() == data.leftHandASMs.size() || data.headASMs.size() == 0 ||
           data.leftHandASMs.size() == 0);
    assert(data.leftHandASMs.size() == data.rightHandASMs.size() || 
           data.leftHandASMs.size() == 0 || data.rightHandASMs.size() == 0);
    // if (data.headASMs.size() == 0)
    //   return;
    
    // gets the tuple for the given index
    auto getAsmTuple = [&data](size_t i) {
      return AsmTriple { data.headASMs.size() > 0 ? data.headASMs[i] : Asm::EMPTY, 
                         data.leftHandASMs.size() > 0 ? data.leftHandASMs[i] : Asm::EMPTY, 
			 data.rightHandASMs.size() > 0 ? data.rightHandASMs[i] : Asm::EMPTY
      };
    };

    computeAsmOverlap(asmBegin, asmBegin + 3, getAsmTuple(0));
    if (data.headASMs.size() > 1) 
      computeAsmFeatures(asmBegin + 3, asmBegin + ASM_FEATURE_COUNT, 
			 getAsmTuple(1), getAsmTuple(0), 
			 getAsmTuple(0));
    else {
      computeAsmFeatures(asmBegin + 3, asmBegin + ASM_FEATURE_COUNT, 
			 getAsmTuple(0), getAsmTuple(0), 
			 getAsmTuple(0));
      return;
    }

    size_t i;
    for (i = 1; i < data.headASMs.size()-1; ++i) {
      computeAsmOverlap(asmBegin, asmBegin + 3, getAsmTuple(i));
      computeAsmFeatures(asmBegin + 3, asmBegin + ASM_FEATURE_COUNT, 
			 getAsmTuple(i+1), getAsmTuple(i), 
			 getAsmTuple(i-1));
    }
    computeAsmOverlap(asmBegin, asmBegin + 3, 
                      getAsmTuple(i));      
    computeAsmFeatures(asmBegin + 3, asmBegin + ASM_FEATURE_COUNT, 
		       getAsmTuple(i), getAsmTuple(i), 
		       getAsmTuple(i-1));
  }



  Component::property_set_t FeatureCreator::getRequirements() const {
    return property_set_t {
        // ASM_TRACKER_LEFT_HAND_ANCHOR_GUESS_BLACKBOARD_ENTRY,
        // ASM_TRACKER_RIGHT_HAND_ANCHOR_GUESS_BLACKBOARD_ENTRY,
        // ASM_TRACKER_HEAD_ANCHOR_GUESS_BLACKBOARD_ENTRY,
        // ASM_TRACKER_LEFT_HAND_ANCHOR_BLACKBOARD_ENTRY,
        // ASM_TRACKER_RIGHT_HAND_ANCHOR_BLACKBOARD_ENTRY,
        // ASM_TRACKER_HEAD_ANCHOR_BLACKBOARD_ENTRY,  
        // ASM_TRACKER_MEAN_HEAD_PDM,
        // ASM_TRACKER_MEAN_LEFT_HAND_PDM,
        // ASM_TRACKER_MEAN_RIGHT_HAND_PDM,
        // ASM_TRACKER_RIGHT_HAND_ASM,
        // ASM_TRACKER_LEFT_HAND_ASM,
        // ASM_TRACKER_HEAD_ASM,
      ASM_TRACKER_HEAD_ASM_INSTANCE,
        ASM_TRACKER_LEFT_HAND_ASM_INSTANCE,
        ASM_TRACKER_RIGHT_HAND_ASM_INSTANCE,
        KLTTRACKER_BLACKBOARD_TRACKED_POINTS_ENTRY
      };
  }



  AnalysisResult FeatureCreator::extractRawResults(size_t firstFrame, size_t lastFrame, const BlackBoard& blackBoard) {
    AnalysisResult result;
    assert(lastFrame > firstFrame);
    result.firstFrame = firstFrame;
    result.lastFrame = lastFrame;
    result.KLTPoints.clear();
    result.headASMs.clear();
    result.leftHandASMs.clear();
    result.rightHandASMs.clear();

    if (blackBoard.has(firstFrame, KLTTRACKER_BLACKBOARD_TRACKED_POINTS_ENTRY))
      for (size_t i = firstFrame; i < lastFrame; ++i)
        result.KLTPoints.push_back(*blackBoard.get<std::set<TrackedPoint>>(i, KLTTRACKER_BLACKBOARD_TRACKED_POINTS_ENTRY));

    // if (blackBoard.has(firstFrame, ASM_TRACKER_HEAD_ASM)) 
    //   for (size_t i = firstFrame; i < lastFrame; ++i)
    //     result.headASMs.push_back(make_pair(blackBoard.get<PDM>(i, ASM_TRACKER_HEAD_ASM), blackBoard.get<cv::Point>(i, ASM_TRACKER_HEAD_ANCHOR_BLACKBOARD_ENTRY)));
    if (blackBoard.has(firstFrame, ASM_TRACKER_HEAD_ASM_INSTANCE)) 
      for (size_t i = firstFrame; i < lastFrame; ++i)
        result.headASMs.push_back(*blackBoard.get<Asm::Instance>(i, ASM_TRACKER_HEAD_ASM_INSTANCE));
    

    // if (blackBoard.has(firstFrame, ASM_TRACKER_LEFT_HAND_ASM)) 
    //   for (size_t i = firstFrame; i < lastFrame; ++i)
    //     result.leftHandASMs.push_back(make_pair(blackBoard.get<PDM>(i, ASM_TRACKER_LEFT_HAND_ASM), blackBoard.get<cv::Point>(i, ASM_TRACKER_LEFT_HAND_ANCHOR_BLACKBOARD_ENTRY)));
    
    // if (blackBoard.has(firstFrame, ASM_TRACKER_RIGHT_HAND_ASM)) 
    //   for (size_t i = firstFrame; i < lastFrame; ++i)
    //     result.rightHandASMs.push_back(make_pair(blackBoard.get<PDM>(i, ASM_TRACKER_RIGHT_HAND_ASM), blackBoard.get<cv::Point>(i, ASM_TRACKER_RIGHT_HAND_ANCHOR_BLACKBOARD_ENTRY)));

    if (blackBoard.has(firstFrame, ASM_TRACKER_LEFT_HAND_ASM_INSTANCE)) 
      for (size_t i = firstFrame; i < lastFrame; ++i)
        result.leftHandASMs.push_back(*blackBoard.get<Asm::Instance>(i, ASM_TRACKER_LEFT_HAND_ASM_INSTANCE));
    
    if (blackBoard.has(firstFrame, ASM_TRACKER_RIGHT_HAND_ASM_INSTANCE)) 
      for (size_t i = firstFrame; i < lastFrame; ++i)
        result.rightHandASMs.push_back(*blackBoard.get<Asm::Instance>(i, ASM_TRACKER_RIGHT_HAND_ASM_INSTANCE));


    return result;
  }



  FeatureCreator::FeatureCreator(bool) :
    Component(true) {
    BlackBoardDumpWriter::registerAnyWriter<velocity_vector_map_t>();
    BlackBoardDumpWriter::registerAnyWriter<point_id_to_part_name_map_t>();
    BlackBoardDumpWriter::registerAnyWriter<tracked_point_map_t>();
    BlackBoardDumpWriter::registerAnyWriter<FeatureVector>();
    BlackBoardDumpWriter::registerDumbWriter<FeatureVector>([](BlackBoardDumpWriter* const w, std::ostream& ofs, const FeatureVector& data) {
        /**
         * Format:
         * [vector<SOM_PAK_Component>][uint64_t][vector<double/uchar/int>]{n}[uint64_t][uint64_t]
         *             |                   |                  |                 |         |
         *             |                   |                  |                 |         +---last frame
         *             |                   |                  |                 +---first frame
         *             |                   |                  +---data vector (depending on component type), times n
         *             |                   +---number of components (and associated data vectors)
         *             +---component descriptions
         */
        auto comps = data.getComponents();
        w->dumbWrite(ofs, comps);
        uint64_t nComps = comps.size();
        w->dumbWrite(ofs, nComps);

        const std::vector<std::vector<boost::any>>& rawFeatures = data.getRawFeatures();
        assert(nComps == rawFeatures.size());
        if (nComps > 0) {
          size_t nFrames = rawFeatures.front().size();
          vector<int> intv(nFrames);
          vector<uchar> boolv(nFrames);
          vector<double> doublev(nFrames);
          for (size_t i = 0; i < nComps; ++i) {
            assert(rawFeatures[i].size() == nFrames);
            SOM_PAK_Component::ValueType type = comps[i].getValueType();
            for (size_t j = 0; j < nFrames; ++j) {
              switch(type) {
              case SOM_PAK_Component::ValueType::INTEGER:
                intv[j] = boost::any_cast<int>(rawFeatures[i][j]);
                break;

              case SOM_PAK_Component::ValueType::REAL:
                doublev[j] = boost::any_cast<double>(rawFeatures[i][j]);
                break;

              case SOM_PAK_Component::ValueType::BOOLEAN:
                boolv[j] = boost::any_cast<bool>(rawFeatures[i][j]);
                break;
              }
            }

            switch(type) {
            case SOM_PAK_Component::ValueType::INTEGER:
              w->dumbWrite(ofs, intv);
              break;

            case SOM_PAK_Component::ValueType::REAL:
              w->dumbWrite(ofs, doublev);
              break;

            case SOM_PAK_Component::ValueType::BOOLEAN:
              w->dumbWrite(ofs, boolv);
              break;
            }
          }
        }
        uint64_t v = data.getFirstFrame();
        w->dumbWrite(ofs, v);
        v = data.getLastFrame();
        w->dumbWrite(ofs, v);
      });

    BlackBoardDumpWriter::registerSizeComputer<FeatureVector>([](const BlackBoardDumpWriter* const w, const FeatureVector& data) {
        /**
         * Format:
         * [vector<SOM_PAK_Component>][uint64_t][vector<double/uchar/int>]{n}[uint64_t][uint64_t]
         *             |                   |                  |                 |         |
         *             |                   |                  |                 |         +---last frame
         *             |                   |                  |                 +---first frame
         *             |                   |                  +---data vector (depending on component type), times n
         *             |                   +---number of components (and associated data vectors)
         *             +---component descriptions
         */
        size_t size = 3*sizeof(uint64_t);
        auto comps = data.getComponents();
        size += w->getSize(comps);
        size_t nComps = comps.size();
        const std::vector<std::vector<boost::any>>& rawFeatures = data.getRawFeatures();
        assert(nComps == rawFeatures.size());
        if (nComps > 0) {
          size_t nFrames = rawFeatures.front().size();
          vector<int> intv(nFrames, 0);
          vector<uchar> boolv(nFrames, 0);
          vector<double> doublev(nFrames, 0);
          for (size_t i = 0; i < nComps; ++i) {
            assert(rawFeatures[i].size() == nFrames);
            switch(comps[i].getValueType()) {
            case SOM_PAK_Component::ValueType::INTEGER:
              size += w->getSize(intv);
              break;

            case SOM_PAK_Component::ValueType::REAL:
              size += w->getSize(doublev);
              break;

            case SOM_PAK_Component::ValueType::BOOLEAN:
              size += w->getSize(boolv);
              break;
            }
          }
        }
        return size;
      });

    BlackBoardDumpWriter::registerSizeComputer<SOM_PAK_Component>([](const BlackBoardDumpWriter* const w, const SOM_PAK_Component& data) {
        /**
         * Format:
         * [uint64_t][string][uint8_t[8]][uint8_t[8]][uint8_t][string]
         *     |        |        |           |         |         |
         *     |        |        |           |         |         +--- real-life unit name
         *     |        |        |           |         +--- value type enum
         *     |        |        |           +--- max value (union type, 64 bits)
         *     |        |        +--- min value (union type, 64 bits)
         *     |        +--- component name                        
         *     +--- component index                               
         */
        static_assert(sizeof(uint64_t) == sizeof(double),
                      "assumptions are made about type sizes that seem not to hold; is char an 8-bit byte?");
        static_assert(sizeof(uint8_t[8]) == sizeof(double),
                      "assumptions are made about type sizes that seem not to hold; is char an 8-bit byte?");
        return sizeof(uint64_t) + w->getSize(data.getName()) + 2*sizeof(uint8_t[8]) 
          + sizeof(uint8_t) + w->getSize(data.getUnit());
      });

    BlackBoardDumpWriter::registerDumbWriter<SOM_PAK_Component>([](BlackBoardDumpWriter* const w, std::ostream& ofs, const SOM_PAK_Component& data) {
        /**
         * Format:
         * [uint64_t][string][uint8_t[8]][uint8_t[8]][uint8_t][string]
         *     |        |        |           |         |         |
         *     |        |        |           |         |         +--- real-life unit name
         *     |        |        |           |         +--- value type enum
         *     |        |        |           +--- max value (union type, 64 bits)
         *     |        |        +--- min value (union type, 64 bits)
         *     |        +--- component name                        
         *     +--- component index    
         *
         * min/max is as a double   if valueType == REAL
         *                 int64_t  if valueType == INTEGER
         *                 uint64_t if valueType == BOOLEAN                          
         */
        uint64_t index = data.getIndex();
        w->dumbWrite(ofs, index);
        w->dumbWrite(ofs, data.getName());
        uint8_t min[8], max[8];
        memset(min, 0, sizeof(min));
        memset(max, 0, sizeof(max));
        SOM_PAK_Component::ValueType valueType = data.getValueType();
        switch(valueType) {
        case SOM_PAK_Component::ValueType::INTEGER:
          *reinterpret_cast<int64_t*>(min) = data.getMinValue<int>();
          *reinterpret_cast<int64_t*>(max) = data.getMaxValue<int>();
          break;
        case SOM_PAK_Component::ValueType::REAL:
          *reinterpret_cast<double*>(min) = data.getMinValue<double>();
          *reinterpret_cast<double*>(max) = data.getMaxValue<double>();
          break;
        case SOM_PAK_Component::ValueType::BOOLEAN:
          *reinterpret_cast<uint64_t*>(min) = data.getMinValue<bool>();
          *reinterpret_cast<uint64_t*>(max) = data.getMaxValue<bool>();
          break;
        }
        w->dumbWrite(ofs, min);
        w->dumbWrite(ofs, max);

        uint8_t valueTypeInt = static_cast<uint8_t>(valueType);
        w->dumbWrite(ofs, valueTypeInt);
        w->dumbWrite(ofs, data.getUnit());
      });

    BlackBoardDumpReader::registerUnDumper<SOM_PAK_Component>([](BlackBoardDumpReader* const w, std::istream& ifs) {
        /**
         * Format:
         * [uint64_t][string][uint8_t[8]][uint8_t[8]][uint8_t][string]
         *     |        |        |           |         |         |
         *     |        |        |           |         |         +--- real-life unit name
         *     |        |        |           |         +--- value type enum
         *     |        |        |           +--- max value (union type, 64 bits)
         *     |        |        +--- min value (union type, 64 bits)
         *     |        +--- component name                        
         *     +--- component index    
         *
         * min/max is as a double   if valueType == REAL
         *                 int64_t  if valueType == INTEGER
         *                 uint64_t if valueType == BOOLEAN                          
         */
        uint64_t index;
        w->dumbRead(ifs, index);
        string name;
        w->dumbRead(ifs, name);
        uint8_t min[8], max[8];
        memset(min, 0, sizeof(min));
        memset(max, 0, sizeof(max));
        w->dumbRead(ifs, min);
        w->dumbRead(ifs, max);
        uint8_t valType;
        w->dumbRead(ifs, valType);
        std::string unitName;
        w->dumbRead(ifs, unitName);

        SOM_PAK_Component::ValueType valueType = static_cast<SOM_PAK_Component::ValueType>(valType);
        int imin, imax;
        double dmin, dmax;
        bool bmin, bmax;

        if (valueType == SOM_PAK_Component::ValueType::INTEGER) {
          imin = *reinterpret_cast<int64_t*>(min);
          imax = *reinterpret_cast<int64_t*>(max);
          return SOM_PAK_Component(index, name, imin, imax, valueType, 
                                   unitName);
        }
        else if (valueType == SOM_PAK_Component::ValueType::REAL) {
          dmin = *reinterpret_cast<double*>(min);
          dmax = *reinterpret_cast<double*>(max);
          return SOM_PAK_Component(index, name, dmin, dmax, valueType, 
                                   unitName);
        }
        else { // SOM_PAK_Component::ValueType::BOOLEAN:
          assert(valueType == SOM_PAK_Component::ValueType::BOOLEAN);
          bmin = *reinterpret_cast<uint64_t*>(min);
          bmax = *reinterpret_cast<uint64_t*>(max);
          return SOM_PAK_Component(index, name, bmin, bmax, valueType, 
                                   unitName);
        }
      });

    BlackBoardDumpReader::registerUnDumper<FeatureVector>([](BlackBoardDumpReader* const w, std::istream& ifs) {
        /**
         * Format:
         * [vector<SOM_PAK_Component>][uint64_t][vector<double/bool/int>]{n}[uint64_t][uint64_t]
         *             |                   |                  |                 |         |
         *             |                   |                  |                 |         +---last frame
         *             |                   |                  |                 +---first frame
         *             |                   |                  +---data vector (depending on component type), times n
         *             |                   +---number of components (and associated data vectors)
         *             +---component descriptions
         */
        vector<SOM_PAK_Component> comps;
        w->dumbRead(ifs, comps);
        uint64_t nComps;
        w->dumbRead(ifs, nComps);
        if (nComps != comps.size())
          throw IOException("The number of component descriptions does not matched the stored number of components!");

        std::vector<std::vector<boost::any> > rawFeatures(nComps);
        vector<int> intv;
        vector<bool> boolv;
        vector<double> doublev;
        size_t nFrames = 0;
        for (size_t i = 0; i < nComps; ++i) {
          SOM_PAK_Component::ValueType type = comps[i].getValueType();
          switch(type) {
          case SOM_PAK_Component::ValueType::INTEGER:
            w->dumbRead(ifs, intv);
            break;

          case SOM_PAK_Component::ValueType::REAL:
            w->dumbRead(ifs, doublev);
            break;

          case SOM_PAK_Component::ValueType::BOOLEAN:
            w->dumbRead(ifs, boolv);
            break;
          }

          if (nFrames == 0)
            nFrames = std::max(std::max(intv.size(), boolv.size()), doublev.size());

          vector<boost::any>& anyVector = rawFeatures[i];

          switch(type) {
          case SOM_PAK_Component::ValueType::INTEGER:
            insert_transform(intv, anyVector, [](int i) {
                return boost::any(i);
              });
            break;

          case SOM_PAK_Component::ValueType::REAL:
            insert_transform(doublev, anyVector, [](double d) {
                return boost::any(d);
              });
            break;
            
          case SOM_PAK_Component::ValueType::BOOLEAN:
            insert_transform(boolv, anyVector, [](bool b) {
                return boost::any(b);
              });
            break;
          }

          if (anyVector.size() != nFrames)
            throw IOException("Vector size mismatch: expected " + 
                              boost::lexical_cast<string>(nFrames) + " elements"
                              " but got " + 
                              boost::lexical_cast<string>(anyVector.size()) + 
                              " instead!");
        }

        uint64_t first, last;
        w->dumbRead(ifs, first);
        w->dumbRead(ifs, last);

        return FeatureVector(comps, rawFeatures, first, last);
      });
  }

  static FeatureCreator DUMMY(true);

  Component* FeatureCreator::createComponentImpl(const boost::program_options::variables_map&, 
                                                 BlackBoard* blackBoard, FrameSource* frameSource) const {
    FeatureCreator fc(blackBoard, frameSource);
    return new FeatureCreator(fc);
  }

}

