#ifndef SLMOTION_FEATURE_CREATOR
#define SLMOTION_FEATURE_CREATOR

#include "debug.hpp"
#include "FeatureVectors.hpp"
#include "SLIO.hpp"
#include "AnalysisResult.hpp"
#include "FeatureVector.hpp"
#include "Component.hpp"

#include <list>

namespace slmotion {
  const std::string FEATURECREATOR_FEATUREVECTOR_BLACKBOARD_ENTRY =
    "featurevector";

  const std::string FEATURECREATOR_KLT_PRUNED_POINTS_BLACKBOARD_ENTRY = "persistentklttrackedpoints";
  const std::string FEATURECREATOR_KLT_VELOCITY_VECTORS_BLACKBOARD_ENTRY = "kltvelocityvectors";
  const std::string FEATURECREATOR_KLT_ACCELERATION_VECTORS_BLACKBOARD_ENTRY = "kltaccelerationvectors";
  const std::string FEATURECREATOR_KLT_IDENTITIES = "kltidentities";

  extern const std::vector<SOM_PAK_Component> SOM_PAK_COMPONENTS;

  /**
   * Eats measurements (created by the analyser), outputs features.
   */
  class FeatureCreator : public Component {
  public:
    virtual std::string getShortName() const {
      return "FeatureCreator";
    }



    /**
     * Constructs the object
     *
     * @param slio A pointer to an I/O object that will be used to write the
     * results. This may or may not be the same object as the one used by 
     * the Analyser.
     */
    explicit FeatureCreator(BlackBoard* blackBoard, 
                            FrameSource* frameSource) : 
      Component(blackBoard, frameSource) {}



    void process(frame_number_t) {
      assert(false && "FeatureCreator cannot be called on a single frame.");
    }



    property_set_t getRequirements() const;



    inline property_set_t getProvided() const {
      return property_set_t { FEATURECREATOR_FEATUREVECTOR_BLACKBOARD_ENTRY,
          FEATURECREATOR_KLT_ACCELERATION_VECTORS_BLACKBOARD_ENTRY,
          FEATURECREATOR_KLT_VELOCITY_VECTORS_BLACKBOARD_ENTRY,
          FEATURECREATOR_KLT_PRUNED_POINTS_BLACKBOARD_ENTRY };
    }



    inline virtual std::string getComponentName() const {
      return "Feature Creator";
    }



    virtual void reset() { }



    FeatureCreator(bool);



    /**
     * Extracts features, and attempts to write them to the file as set in 
     * the internal I/O object.
     *
     * @param data Data from Analyser objects
     *
     * @return The resulting features, regardless of whether they were 
     * written or not
     */
    // std::list<FeatureVector> outputFeatures(const std::list<AnalysisResult>& data,
    //                                                double fps,
    //                                                const std::string& elanFilename,
    //                                                const std::string& annTemplateFilename, const std::string& inputFilename,
    //                                                const std::string& outTierXmlFile = "");



    typedef std::map<TrackedPoint::point_id_type_t,TrackedPoint> tracked_point_map_t;
    typedef std::vector<tracked_point_map_t> tracked_point_map_vector_t;

    typedef TrackedPoint::point_t velocity_vector_t;
    typedef velocity_vector_t acceleration_vector_t;
    typedef std::map<TrackedPoint::point_id_type_t, velocity_vector_t> velocity_vector_map_t;
    typedef velocity_vector_map_t acceleration_vector_map_t;
    typedef std::vector<velocity_vector_map_t> velocity_vector_map_vector_t;
    typedef velocity_vector_map_vector_t acceleration_vector_map_vector_t;
    typedef std::map<TrackedPoint::point_id_type_t, BodyPart::PartName> point_id_to_part_name_map_t;
    typedef std::vector<point_id_to_part_name_map_t> point_id_to_part_name_map_vector_t;



    /**
     * Extracts raw results associated with the given frame. This 
     * function should be called after all components have been processed.
     *
     * @param firstFrame (inclusive)
     * @param lastFrame (exclusive)
     * @param blackBoard Black Board to read the data from
     */
    static AnalysisResult extractRawResults(size_t firstFrame, 
                                            size_t lastFrame,
                                            const BlackBoard& blackBoard);



    virtual std::string getShortDescription() const {
      return "This component goes through entries on the black board, and computes some features.";
    }



    virtual std::string getLongDescription() const {
      return "This component goes through entries on the black board, and computes some features.";
    }



    virtual boost::program_options::options_description getConfigurationFileOptionsDescription() const {
      return boost::program_options::options_description();
    }

    virtual boost::program_options::options_description getCommandLineOptionsDescription() const {
      return boost::program_options::options_description();
    }


  private:
    virtual Component* createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const;



    /**
     * Loads features from previous components, then 
     * computes whatever is expected, and stores the results on the 
     * blackboard.
     */
    bool processRangeImplementation(frame_number_t first, 
                                    frame_number_t last,
                                    UiCallback* uiCallback = NULL);



    void createFeatures(const std::string& outTierXmlFile = "");


    
    void writeFeatures(const std::list<FeatureVector>& results);



    /**
     * Processes one data entry, and creates corresponding features.
     */
    FeatureVector createFeaturesFromData(const AnalysisResult& data,
                                         double fps);



    /**
     * Computes the following KLT features:
     *  (i)   Total number of tracked points
     *  (ii)  Sum of horizontal velocity components
     *  (iii) Sum of vertical velocity components
     *  (iv)  L2 norm of the sum of all velocity vectors
     *  (v)   L2 norm of the sum of all acceleration vectors
     *
     * Then stores the values at the given tracks between the given random
     * access iterators
     *
     * @param begin A random access iterator pointing towards at the first 
     * of the desired output tracks
     * @param end A random access iterator pointing at an element just past
     * the last desired target track
     * @param velocityVectors Tracked-point-id-to-velocity-vector maps for
     * each frame
     * @param accelerationVectors As above, but for acceleration vectors
     * @param bodyPartAssociation Association between point ids and body 
     * part names. There should be an entry for each body part for each 
     * frame.
     * @param allowedBodyPart The desired body part name
     * @param strict If enabled, only those points are considered whose body
     * part name is strictly equal to the one given as the previous 
     * parameter. Otherwise, any part will do, as long as it "isA" 
     * allowedBodyPart.
     */
    static void computeKLTFeatures(std::vector<std::vector<boost::any>>::iterator begin,
                                   std::vector<std::vector<boost::any>>::iterator end,
                                 const FeatureCreator::velocity_vector_map_vector_t& velocityVectors,
                                 const FeatureCreator::acceleration_vector_map_vector_t& accelerationVectors,
                                 const point_id_to_part_name_map_vector_t& bodyPartAssociation,
                                 BodyPart::PartName allowedBodyPart,
                                 bool strict);



    /**
     * Creates a vector of maps containing body part associations for each 
     * feature in each frame (vector element), keyed with the point id.
     */
    static point_id_to_part_name_map_vector_t createBodyPartAssociationMaps(const tracked_point_map_vector_t& points);



    /**
     * Creates 'full motion descriptors' by reading in analysis results and
     * interpretation results, preforming some calculations and merging the 
     * result
     *
     * @param frnumber Current frame number
     * @param fps Frames per second, used to convert between time codes and 
     * frame numbers
     * @param ar Input analysis result
     * @param ir Input interpretation result
     * @return The full motion descriptors. In a typical usage scenario,
     * these values would be assigned to the input struct as members.
     */
    static std::deque < std::vector<double> > computeFeatureData(double fps,
                                                                 const AnalysisResult& ar, 
                                                                 const FeatureVectors& fv);



    /**
     * Removes points that have been tracked for fewer frames than specified
     *
     * Returns a vector of map of points where each key corresponds to the 
     * point ID.
     */
    static tracked_point_map_vector_t pruneTracked(const std::vector<std::set<TrackedPoint>>& points,
                                                   size_t minimumNumberOfTrackedFrames);

    /**
     * Eats points, outputs velocity vectors
     *
     * Return values are defined as follows:
     *  for each frame f (vector element) in (0,numberOfFrames)
     *   for each point n with location vector (n,f)
     *    v(n,f) = (x(n,f+1)-x(n,f-1))/2
     * given that location vectors x(n,f-1) and x(n,f+1) exist
     * otherwise v(n,f) = 0
     */
    static velocity_vector_map_vector_t createVelocityVectors(const tracked_point_map_vector_t& points);

    /**
     * Eats points, outputs acceleration vectors
     *
     * Return values are defined as follows:
     *  for each frame f (vector element) in (0,numberOfFrames)
     *   for each point n with location vector (n,f)
     *    a(n,f) = x(n,f+1) - 2x(n,f) + x(n,f-1)
     * given that location vectors x(n,f-1) and x(n,f+1) exist
     * otherwise a(n,f) = 0
     */
    static acceleration_vector_map_vector_t createAccelerationVectors(const tracked_point_map_vector_t& points);



    /**
     * Append new feature tracks to the existing vector
     */
    static void createKLTFeatures(const AnalysisResult& data, 
                                  std::vector<std::vector<boost::any>>& features,
                                  BlackBoard& blackBoard);



    /**
     * Creates features from the ASMs and appends new tracks at the end of 
     * the existing feature track vector
     */
    static void createASMFeatures(const AnalysisResult& data,
                                  std::vector<std::vector<boost::any>>& features);
  };



  std::vector<boost::any> createFrameNumberVector(size_t first, size_t last);
  std::vector<boost::any> createTimeCodeVector(size_t first, size_t last, double fps);
}
#endif

