#include "FeatureVector.hpp"
#include <stdexcept>
#include <string>
#include <boost/lexical_cast.hpp>


namespace slmotion {
  void FeatureVector::verify() {
    if (lastFrame < firstFrame)
      throw std::invalid_argument("First frame should go before the last frame.");

    if (components.size() != features.size()) {
      std::stringstream ss;
      ss << components.size() << " vs " << features.size();
      throw std::invalid_argument("FeatureVector construction failed because"
				  " of mismatch between the number of components, "
				  "and the number of feature tracks: "+ss.str());
    }

    size_t s = features.size() > 0 ? features.front().size() : 0;
    if (s != lastFrame - firstFrame)
      throw std::invalid_argument("There should be exactly one element per each "
				  "frame on each track");

    if (std::count_if(features.begin(), features.end(), 
                      [s](const std::vector<boost::any>& v) {
                        return v.size() == s;
                      }) != static_cast<int>(features.size()))
      throw std::invalid_argument("All feature tracks must be of equal size.");

    std::string errorMessage = "FeatureVector construction failed while verifying component ";
    for (size_t i = 0; i < components.size(); ++i) {
      const SOM_PAK_Component& comp = components[i];
      if (i != comp.getIndex())
        throw std::invalid_argument(errorMessage + '"' + comp.getName() + 
                                    "\": index mismatch (got " + 
                                    boost::lexical_cast<std::string>(i) + 
                                    " but " + boost::lexical_cast<std::string>(comp.getIndex()) +
                                    " was expected)");
      double d;
      int k;
      for (size_t j = 0; j < features[i].size(); ++j) {
        if (features[i][j].type() != typeid(std::vector<float>))
          switch(comp.getValueType()) {
            typedef SOM_PAK_Component::ValueType ValueType;
          case ValueType::INTEGER:
            if (features[i][j].type() != typeid(int))
              throw std::invalid_argument(errorMessage + '"' + comp.getName() + 
                                          "\": a non-integer element "
                                          "encountered.");

            k = boost::any_cast<int>(features[i][j]);
            if (k > comp.getMaxValue<int>() || k < comp.getMinValue<int>())
              throw std::invalid_argument(errorMessage + '"' + comp.getName() + 
                                          "\": element out of range");
            break;

          case ValueType::REAL:
            if (features[i][j].type() != typeid(double))
              throw std::invalid_argument(errorMessage + '"' + comp.getName() + 
                                          "\": a non-double element "
                                          "encountered.");

            d = boost::any_cast<double>(features[i][j]);
            if (d > comp.getMaxValue<double>() || 
                d < comp.getMinValue<double>())
              throw std::invalid_argument(errorMessage + '"' + comp.getName() + 
                                          "\": element out of range");
            break;

          case ValueType::BOOLEAN:
            if (features[i][j].type() != typeid(bool))
              throw std::invalid_argument(errorMessage + '"' + comp.getName() + 
                                          "\": a non-boolean element "
                                          "encountered.");
            break;
          }
      }
    }
  }



  static double toDouble(const boost::any& value, SOM_PAK_Component::ValueType type) {
    switch(type) {
    case SOM_PAK_Component::ValueType::INTEGER:
      return boost::any_cast<int>(value);
      break;

    case SOM_PAK_Component::ValueType::REAL:
      return boost::any_cast<double>(value);
      break;

    case SOM_PAK_Component::ValueType::BOOLEAN:
      return boost::any_cast<bool>(value);
      break;
    }

    return INFINITY;
  }

  std::deque<std::vector<double>> FeatureVector::toDoubleVectorDeque() const {
    std::deque<std::vector<double>> deck(lastFrame - firstFrame,
                                         std::vector<double>(components.size()));
    for (size_t i = 0; i < lastFrame - firstFrame; ++i) 
      for (size_t j = 0; j < features.size(); ++j)
        deck[i][j] = toDouble(features[j][i], components[j].getValueType());
    return deck;
  }

  std::map<std::string, unsigned int> FeatureVector::getFieldNameMap() const {
    std::map<std::string, unsigned int> map;
    for (const auto& c : components)
      map[c.getName()] = c.getIndex();
    return map;
  }
}
