#ifndef SLMOTION_FEATURE_VECTOR
#define SLMOTION_FEATURE_VECTOR

#include "SOM_PAK_Component.hpp"
#include <map>
#include <cassert>
#include <deque>
#include <stdexcept>
#include <boost/any.hpp>
#include <iostream>
#include <vector>

namespace slmotion {
  /**
   * Output feature vector type:
   * several 'tracks' of arbitrary components.
   */
  class FeatureVector {
  public:
    /**
     * Constructs the feature vector. Each feature 'track' should be of 
     * equal length and the number of tracks should match that of 
     * components. Each feature element should meet requirements specified
     * by the associated component object. If one or more of these 
     * requirements is not fulfilled, an exception is thrown.
     */
    FeatureVector(const std::vector<SOM_PAK_Component>& components,
                  const std::vector<std::vector<boost::any>> features,
                  size_t firstFrame, size_t lastFrame) :
      components(components), features(features), firstFrame(firstFrame),
      lastFrame(lastFrame) {
      verify();
    }



    FeatureVector() : firstFrame(0), lastFrame(0) {}



    std::deque<std::vector<double>> toDoubleVectorDeque() const;


    inline const std::vector<SOM_PAK_Component>& getComponents() const {
      return components;
    }



    /**
     * Returns the value of the feature in the given frame 
     */
    template<typename T>
    T get(size_t feature, size_t frame) {
      assert(components.size() == features.size());
      if (feature >= features.size())
        throw std::out_of_range("Tried to access a non-existent feature track");

      if (frame >= features[feature].size())
        throw std::out_of_range("Tried to access a feature for a non-existent frame");

      switch(components[feature].getValueType()) {
      case SOM_PAK_Component::ValueType::INTEGER:
        if (typeid(T) != typeid(typename SOM_PAK_Component::CppType<SOM_PAK_Component::ValueType::INTEGER>::type))
          throw std::invalid_argument("Invalid type for the given component! Integer type expected.");
        break;

      case SOM_PAK_Component::ValueType::REAL:
        if (typeid(T) != typeid(typename SOM_PAK_Component::CppType<SOM_PAK_Component::ValueType::REAL>::type))
          throw std::invalid_argument("Invalid type for the given component! Real type expected.");
        break;

      case SOM_PAK_Component::ValueType::BOOLEAN:
        if (typeid(T) != typeid(typename SOM_PAK_Component::CppType<SOM_PAK_Component::ValueType::BOOLEAN>::type))
          throw std::invalid_argument("Invalid type for the given component! Boolean type expected.");
        break;
      }

      return boost::any_cast<T>(features[feature][frame]);
    }



    /**
     * Returns the value of the feature in the given frame by name
     */
    template<typename T>
    T get(const std::string& featureName , size_t frame) {
      size_t index = SIZE_MAX;
      for (const auto& c : components) {
        if (c.getName() == featureName) {
          index = c.getIndex();
          break;
        }
      }

      if (index == SIZE_MAX)
        throw std::out_of_range("Tried to access a non-existent feature track");

      return get<T>(index, frame);
    }



    /**
     * Returns a map from field names to field indices
     */
    std::map<std::string, unsigned int> getFieldNameMap() const;

    inline size_t getFirstFrame() const {
      return firstFrame;
    }

    inline size_t getLastFrame() const {
      return lastFrame;
    }

    const std::vector<std::vector<boost::any>>& getRawFeatures() const {
      return features;
    }

  private:
    void verify();

    std::vector<SOM_PAK_Component> components; ///< Component description describing the type, names and maximum and minimum values of each valid component
    std::vector<std::vector<boost::any>> features; ///< "tracks" of components, the outer vector ought to be equal in size to the components vector, and each inner vectour ought to be of equal size
    size_t firstFrame; // inclusive
    size_t lastFrame; // exclusive
  };
}

#endif
