#ifndef SLMOTION_FILTER
#define SLMOTION_FILTER

#include "opencv.hpp"

#include "Clonable.hpp"

namespace slmotion {
  /**
   * A simple class for applying various filters
   */
  class PostProcessFilter : public Clonable {
  public:
    /**
     * Applies the filter to the target binary mask.
     *
     * @param original Original 3-channel colour image
     * @param targetMask Input/output parameter; the binary mask that is to 
     * be filtered
     */
    virtual void apply(const cv::Mat& original, 
                       cv::Mat& targetMask) const = 0;
  
  
  
    virtual ~PostProcessFilter() = 0;



    virtual bool operator==(const PostProcessFilter&) = 0;
  };



  /**
   * Crops the mask.
   */
  class PostProcessCropFilter : public PostProcessFilter {
  public:
    /**
     * @param r Rectangle that represents the area that should be left intact;
     * everything else is zeroed out.
     */
    explicit PostProcessCropFilter(const cv::Rect& r) : 
      cropArea(r) { }



    /**
     * Crops the mask.
     */
    void apply(const cv::Mat&, cv::Mat& target) const;



    inline PostProcessCropFilter* clone() const {
      return new PostProcessCropFilter(*this);
    }



    bool operator==(const PostProcessFilter& other) {
      auto otherPtr = dynamic_cast<const PostProcessCropFilter*>(&other);
      if (otherPtr)
        return otherPtr->cropArea == this->cropArea;
      return false;
    }



  private:
    cv::Rect cropArea; ///< Area used for cropping. If (-1,-1)->(-1,-1), no cropping is performed.
  };



  class PostProcessMorphologicalTransformationFilter : public PostProcessFilter {
  public:
    enum class Type {
      OPEN, CLOSE, ERODE, DILATE
        };



    /**
     * The constructor
     *
     * @param type Type of transformation
     * @param iterations Number of iterations
     */
    PostProcessMorphologicalTransformationFilter(Type type, int iterations) :
      type(type), iterations(iterations) {}



    void apply(const cv::Mat&, cv::Mat& target) const;



    inline PostProcessMorphologicalTransformationFilter* clone() const {
      return new PostProcessMorphologicalTransformationFilter(*this);
    }



    bool operator==(const PostProcessFilter& other) {
      auto otherPtr = dynamic_cast<const PostProcessMorphologicalTransformationFilter*>(&other);
      if (otherPtr)
        return otherPtr->type == this->type && otherPtr->iterations == this->iterations;
      return false;
    }



  private:
    Type type;
    int iterations;
  };


  /**
   * This filter fills holes in the mask.
   */
  class PostProcessHoleFillerFilter : public PostProcessFilter {
  public:
    void apply(const cv::Mat&, cv::Mat& target) const;



    inline PostProcessHoleFillerFilter* clone() const {
      return new PostProcessHoleFillerFilter(*this);
    }



    bool operator==(const PostProcessFilter& other) {
      return dynamic_cast<const PostProcessHoleFillerFilter*>(&other) != NULL;
    }
  };



  /**
   * Performs watershed segmentation on the mask
   */
  class PostProcessWatershedFilter : public PostProcessFilter {
  public:
    /**
     * @param inColour Input colour image
     * @param target Input/output mask
     */
    void apply(const cv::Mat& inColour, cv::Mat& target) const;

    explicit PostProcessWatershedFilter(size_t n) : dilateAmount(n) {}



    inline PostProcessWatershedFilter* clone() const {
      return new PostProcessWatershedFilter(*this);
    }



    bool operator==(const PostProcessFilter& other) {
      auto otherPtr = dynamic_cast<const PostProcessWatershedFilter*>(&other);
      if (otherPtr)
        return this->dilateAmount == otherPtr->dilateAmount;
      return false;
    }



  private:
    size_t dilateAmount; ///< How much to dilate
  };


  /**
   * Removes small components  the mask
   */
  class SmallComponentRemoverFilter : public PostProcessFilter {
  public:
    /**
     * @param inColour Input colour image
     * @param target Input/output mask
     */
    void apply(const cv::Mat& inColour, cv::Mat& target) const;

    explicit SmallComponentRemoverFilter(size_t n) : sizeThreshold(n) {}



    inline SmallComponentRemoverFilter* clone() const {
      return new SmallComponentRemoverFilter(*this);
    }



    bool operator==(const PostProcessFilter& other) {
      auto otherPtr = dynamic_cast<const SmallComponentRemoverFilter*>(&other);
      if (otherPtr)
        return this->sizeThreshold == otherPtr->sizeThreshold;
      return false;
    }



  private:
    size_t sizeThreshold; ///< determines how large a blob needs to be for it to be retained
  };
}



#endif
