#include <opencv2/opencv.hpp>
#include "FrameSource.hpp"
#include "debug.hpp"

namespace slmotion {
  cv::Mat FrameSource::scale(const cv::Mat& src) {
    if (aspect < 0 && (targetScale.width < 0 || targetScale.height < 0))
      return src;

    // perform scaling and aspect ratio correction as necessary     
    cv::Size newSize = targetScale.width > 0 && targetScale.height > 0 ? targetScale : src.size();
    if (aspect > 0)
      newSize.width = newSize.height * aspect;

    if (debug > 1)
      std::cerr << "Converting video to " << newSize.width 
                << "x" << newSize.height << "pixels" << std::endl;
      
    cv::Mat temp;
    resize(src, temp, newSize);
    return temp;
  }



  FrameSource::~FrameSource() {}



  void FrameSource::cacheFrame(frame_number_type frameNumber, const cv::Mat& frame) {
#ifdef SLMOTION_THREADING
    std::lock_guard<std::mutex> lk(this->mutex);
#endif
    while (frameCache.size() > 0 && frameCache.size() >= maxCacheSize) {
      frameCache.erase(fifoIndices.front());
      fifoIndices.pop_front();
    }
    frameCache[frameNumber] = 
      std::shared_ptr<cv::Mat>(new cv::Mat(scale(frame)));
    fifoIndices.push_back(frameNumber);
  }


  FrameSource::Iterator FrameSource::Iterator::operator++(int) {
    Iterator copy(*this);
    ++currentFrame;
    return copy;
  }



  FrameSource::Iterator FrameSource::Iterator::operator--(int) {
    Iterator copy(*this);
    --currentFrame;
    return copy;
  }



  FrameSource::Iterator operator+(FrameSource::difference_type amount, 
                                            const FrameSource::Iterator& rhs) {
    FrameSource::Iterator copy = rhs;
    return copy += amount;
  }



  FrameSource::Iterator operator-(FrameSource::difference_type amount, 
                                  const FrameSource::Iterator& rhs) {
    FrameSource::Iterator copy = rhs;
    return copy -= amount;
  }



  FrameSource::Iterator FrameSource::Iterator::operator+(FrameSource::difference_type amount) const {
    Iterator copy = *this;
    return copy += amount;    
  }



  FrameSource::Iterator FrameSource::Iterator::operator-(FrameSource::difference_type amount) const {
    Iterator copy = *this;
    return copy -= amount;    
  }

  FrameSource::difference_type FrameSource::Iterator::operator-(const FrameSource::Iterator& rhs) const {
    return this->currentFrame - rhs.currentFrame;
  }
}
