#include "Gaussian.hpp"
#include "math.hpp"

namespace slmotion {
  using math::PI;

  Gaussian::Gaussian(const cv::Mat& samples) : nVars(samples.cols), 
                                               covDetSqrt(0) {
    cv::Mat coVar;
    cv::calcCovarMatrix(samples, coVar, means, CV_COVAR_NORMAL | 
                        CV_COVAR_SCALE | CV_COVAR_ROWS,
                        CV_32FC1);
    covDetSqrt = sqrt(determinant(coVar));

    if (covDetSqrt == 0)
      throw std::invalid_argument("Covariance matrix is singular! Distribution is ill-defined.");

    inverseCovariance = coVar.inv(cv::DECOMP_SVD);
  }



  double Gaussian::computeLogProbability(const std::vector<float>& x) const 
  {
    cv::Mat xMinusM;
    means.copyTo(xMinusM);
    assert(xMinusM.cols == static_cast<int>(nVars));
    for (size_t i = 0; i < nVars; ++i) 
      xMinusM.at<float>(0,i) = x[i] - xMinusM.at<float>(0,i);
        
    cv::Mat result = xMinusM * inverseCovariance * xMinusM.t();
    return -0.5 * result.at<float>(0,0) - log(pow(2.0*PI,static_cast<double>(nVars)/2.0)*covDetSqrt);
  }



#if 0
  Gaussian& Gaussian::operator=(Gaussian&& other) {
    if (this != &other) {
      std::swap(this->nVars, other.nVars);
      std::swap(this->inverseCovariance, other.inverseCovariance);
      std::swap(this->means, means);
      std::swap(this->covDetSqrt, other.covDetSqrt);
    }
    return *this;
  }
#endif



  double Gaussian::computePdf(const std::vector<float>& x) const {
    cv::Mat xMinusM;
    means.copyTo(xMinusM);
    assert(xMinusM.cols == static_cast<int>(nVars));
    for (size_t i = 0; i < nVars; ++i) 
      xMinusM.at<float>(0,i) = x[i] - xMinusM.at<float>(0,i);
        
    cv::Mat result = xMinusM * inverseCovariance * xMinusM.t();

    return pow(2.0*PI, -static_cast<double>(nVars)/2.0) / covDetSqrt *
      exp(-0.5 * result.at<float>(0,0));
  }
}
