#ifndef SLMOTION_GAUSSIAN
#define SLMOTION_GAUSSIAN

#include <opencv2/opencv.hpp>
#include "util.hpp"

namespace slmotion {
  /**
   * This class represents a single multivariate Gaussian distribution
   */
  class Gaussian {
  public:
    /**
     * Constructs the Gaussian distribution.
     *
     * @param samples Sample vectors from which covariance and means are 
     * computed, one sample per row. Type assumed to be CV_32FC1.
     */
    Gaussian(const cv::Mat& samples);



    /**
     * Your standard copy constructor. Member matrices will be cloned for 
     * your convenience.
     */
    Gaussian(const Gaussian& other) : 
      nVars(other.nVars), 
      inverseCovariance(other.inverseCovariance.clone()),
      means(other.means.clone()), covDetSqrt(other.covDetSqrt) {
    }



    /**
     * The default move constructor
     */
    // Gaussian(Gaussian&&) = default;



    /**
     * Ordinary copy-assignment. As with the copy constructor, member 
     * matrices shall be cloned.
     */
    Gaussian& operator=(const Gaussian& other) {
      if (this != &other) {
        cv::Mat inverseCovarianceTemp = other.inverseCovariance.clone();
        cv::Mat meansTemp = other.means.clone();
        this->nVars = other.nVars;
        this->covDetSqrt = other.nVars;
        std::swap(this->inverseCovariance, inverseCovarianceTemp);
        std::swap(this->means, meansTemp);
      }
      return *this;
    }



    /**
     * Default move-assignment. Implemented manually because defaulting is 
     * not possible as of GCC 4.5.1
     */
    // Gaussian& operator=(Gaussian&&);



    /**
     * Computes logarithm of the pdf, i.e.
     *  -0.5 * (x-m)^T * S^-1 * (x-m) - log((2pi)^(n/2)) * |S|^(1/2)
     *
     * where
     *  m  is the mean
     *  S  is the covariance matrix
     *  x  is the input sample
     *  n  is the number of components
     * |S| is the determinant of the covariance matrix
     */
    double computeLogProbability(const std::vector<float>& x) const;



    /**
     * Computes the value of the probability density function given the 
     * vector, i.e. returns
     *
     * (2pi)^(-k/2) |Sigma|^(-1/2) exp { -(1/2)(x-mu)^T Sigma^(-1) (x-mu) }
     */
    double computePdf(const std::vector<float>& x) const;



    /**
     * Returns true if the two Gaussians are equal memberwise
     */
    inline bool operator==(const Gaussian& other) {
      return nVars == other.nVars && covDetSqrt == other.covDetSqrt &&
        equal(inverseCovariance, other.inverseCovariance) &&
        equal(means, other.means);
    }



  private:
    size_t nVars; ///< Number of variables
    cv::Mat inverseCovariance; ///< Inverse covariance matrix
    cv::Mat means; ///< Means for each variable
    double covDetSqrt; ///< Square root of the determinant of covariance matrix
  };
}

#endif
