#ifndef SLMOTION_GAUSSIAN_MIXTURE
#define SLMOTION_GAUSSIAN_MIXTURE

#include <stdexcept>
#include "Gaussian.hpp"

namespace slmotion {
  /**
   * This class represents a weighted mixture of multivariate Gaussians
   */
  class GaussianMixture {
  public:
    /**
     * One way to construct; give the distributions and weights explicitly
     * No checks are made about whether the weights sum up to 1.
     */
    GaussianMixture(const std::vector<Gaussian>& gaussians,
                    const std::vector<double>& weights) :
      gaussians(gaussians), weights(weights) { 
      if (gaussians.size() != weights.size())
        throw std::invalid_argument("There must be an equal amount of "
                                    "Gaussians and weights!");
    }



    /**
     * Takes in the clusters of samples, and constructs the Gaussians 
     * accordingly. For each Gaussian i with k_i samples, the weights are 
     * set at k_i / (sum_j k_j)
     *
     * @param samples Samples for each cluster. One Gaussian will be 
     * generated per cluster. Samples are assumed to be row vectors. Each 
     * cluster must have an equal number of columns (variables).
     */
    explicit GaussianMixture(const std::vector<cv::Mat>& samples);
    


    GaussianMixture() = default;



    /**
     * Given Gaussian distributions p_i with weights w_i, computes 
     * log sum_i(p_i(x) w_i)
     */
    inline double computeLogMixPdf(const std::vector<float>& x) {
      return log(computeMixPdf(x));
    }



    /**
     * Given Gaussian distributions p_i with weights w_i, computes 
     * sum_i(p_i(x) w_i)
     */
    double computeMixPdf(const std::vector<float>& x) {
      double val = 0;
      for (size_t k = 0; k < gaussians.size(); ++k) 
        val += gaussians[k].computePdf(x) * weights[k];
      return val;
    }



    /**
     * Given Gaussian distributions p_i with weights w_i, computes 
     * max_i log(p_i(x) w_i)
     */
    double computeLogMaxPdf(const std::vector<float>& x) {
      std::vector<double> vals(x.size());

      for (size_t k = 0; k < gaussians.size(); ++k) 
        vals[k] = gaussians[k].computeLogProbability(x);

      double maxval = -DBL_MAX;
      for (size_t k = 0; k < gaussians.size(); ++k)
        maxval = std::max(maxval, vals[k]);
      return maxval;
    }



    /**
     * Given Gaussian distributions p_i with weights w_i, computes 
     * max_i p_i(x) w_i
     */
    double computeMaxPdf(const std::vector<float>& x) {
      std::vector<double> vals(x.size());

      for (size_t k = 0; k < gaussians.size(); ++k) 
        vals[k] = gaussians[k].computePdf(x);

      double maxval = 0;
      for (size_t k = 0; k < gaussians.size(); ++k)
        maxval = std::max(maxval, vals[k]);
      return maxval;
    }



    inline size_t size() const {
      assert(gaussians.size() == weights.size());
      return gaussians.size();
    }



  private:
    std::vector<Gaussian> gaussians;
    std::vector<double> weights;
  };
}
#endif
