#include <boost/lexical_cast.hpp>
#include "GaussianSkinDetector.hpp"
#include "exceptions.hpp"
#include "BlackBoardExplorer.hpp"

using cv::Mat;
using cv::TermCriteria;
using cv::Rect;
using boost::lexical_cast;
using boost::program_options::value;
using std::string;
using std::vector;



namespace slmotion {
  static GaussianSkinDetector DUMMY(true);

  void GaussianSkinDetector::setNumberOfDistributions(size_t k, size_t r) {
    if (!(0 < r && r <= k) && k != 0)
      throw std::invalid_argument("Tried to set skin detection distribution number values k = " + lexical_cast<string>(k) + " and r = " + lexical_cast<string>(r) + ", which is not allowed because constraint 0 < r <= k was not satisfied.");
    totalColourClasses = k;
    relevantColourClasses = r;
  }



  void GaussianSkinDetector::train(const cv::Mat& samples, const cv::Mat&,
                                   bool) {
    assert(samples.type() == CV_32FC1);
    this->reset();

    vector<Mat> clusters;
    Mat clusterMeans;
    getBiggestClusters(samples, clusters, clusterMeans, 
                       relevantColourClasses, totalColourClasses);

    gaussMix = GaussianMixture(clusters);
  }



  void GaussianSkinDetector::detect(const cv::Mat& inFrame,
                                    cv::Mat& outMask) {
    if (gaussMix.size() == 0) {
      // attempt to train

      if (trainFilename.length() == 0)
        trainByFaceDetector();
      else
        train(trainFilename);

      if (gaussMix.size() == 0)
        throw SLMotionException("No components defined for the Gaussian Skin Detector! The detector may not have been trained. Cannot detect anything.");
    }

    assert(inFrame.size() == outMask.size());
    assert(inFrame.depth() == CV_8U || inFrame.depth() == CV_32F);
    // assert(inFrame.channels() == static_cast<int>(componentCount));
    assert(outMask.type()  == CV_8UC1);

    if (inFrame.depth() == CV_8U)
      detect<uchar>(inFrame, outMask);
    else if (inFrame.depth() == CV_32F)
      detect<float>(inFrame, outMask);
  }



  bool GaussianSkinDetector::trainByFaceDetector() {
    size_t framenr = 0;
    Rect faceLocation;
    try {       
      if (!getBlackBoard().has(framenr, FACEDETECTOR_BLACKBOARD_ENTRY)) 
        throw SLMotionException("No face data on the black board!");
      for (FrameSource::const_iterator it = getFrameSource().cbegin();
           it < getFrameSource().cend(); ++framenr, ++it) {
        // read frames until successful detection
        faceLocation = *getBlackBoard().get<Rect>(framenr, FACEDETECTOR_BLACKBOARD_ENTRY);
        if (faceLocation != FaceDetector::INVALID_FACE) {         
          // create an elliptical mask within face
          // train the model using colour vectors from here
          Mat temp;
          getColourSpace().convertColour(*it, temp);

          Mat samples = nonZeroToSamples(temp, getEllipseMask(it->size(), faceLocation));

          train(samples, Mat(), false);
          return true;
        }
      }
    }
    catch(std::exception& e) {
      throw SLMotionException((string("An exception was thrown while training the skin detector using face detection results: ") + e.what()).c_str());
    }
    return false;
  }



  void GaussianSkinDetector::train(const std::string& filename) {
    cv::Mat trainImage = cv::imread(filename);
    if (trainImage.empty())
      throw std::invalid_argument("Tried to train the model using an empty or invalid image file");
    if (trainImage.type() != CV_8UC3)
      throw std::invalid_argument("Tried to train the model using an image of wrong type. Expected a 3-channel 8-bit image, got something completely different.");

    getColourSpace().convertColour(trainImage, trainImage);
    Mat samples;
    if (trainImage.depth() == CV_8U)
      samples = nonZeroToSamples<uchar>(trainImage, Mat(trainImage.size(), CV_8UC1, cv::Scalar::all(255)));
    else if (trainImage.depth() == CV_32F)
      samples = nonZeroToSamples<float>(trainImage, Mat(trainImage.size(), CV_8UC1, cv::Scalar::all(255)));
   
    train(samples, Mat(), false);

    // for (size_t i = 0; i < relevantColourClasses; ++i) {
    //   std::cout << "Mean vector " << i << ": " << std::endl << means[i];
    //   std::cout << "Cov det sqrt i: " << covDetSqrt[i] << std::endl;
    //   std::cout << "InvCov i: " << std::endl;
    //   std::cout << inverseCovariances[i] << std::endl;
    // }
  }



  void GaussianSkinDetector::reset() {
    SkinDetector::reset();
    gaussMix = GaussianMixture();
    pixelValueCache.clear();
  }



  boost::program_options::options_description GaussianSkinDetector::getConfigurationFileOptionsDescription() const {
    boost::program_options::options_description opts;
    opts.add(SkinDetector::getConfigurationFileOptionsDescription());
    opts.add_options()
      ("GaussianSkinDetector.logdensitythreshold", 
       value<double>()->default_value(-14),
       "Sets the threshold for value of logarithmic "
       "probability density function that a pixel should have"
       " to be interpreted as a skin pixel. Pixels whose "
       "value exceeds this threshold will be classified as "
       "skin pixels.")
      ("GaussianSkinDetector.kcolours", value<int>()->default_value(2),
       "Sets the number K of clusters that should be considered when "
       "performing K-means. Please see the description above.")
      ("GaussianSkinDetector.rcolours", value<int>()->default_value(1),
       "Sets the number R <= K colours that should be considered by the "
       "filter. I.e. the R most common clusters will be used for computing "
       "R probability distributions. Please see the description above.")
      ("GaussianSkinDetector.trainimage", value<string>()->default_value(""),
       "Train the detector using an image file rather than the face "
       "detector. Non-zero pixels are chosen.")
      ("GaussianSkinDetector.weights", value<bool>()->default_value(true),
       "If true, a weighted linear combination of distribution values is "
       "used. Otherwise, only maximum value is used.");
    return opts;
  }



  Component* GaussianSkinDetector::createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const {
    return new GaussianSkinDetector(configuration, blackBoard, frameSource);
  }



  GaussianSkinDetector::GaussianSkinDetector(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) :
    SkinDetector(configuration, blackBoard, frameSource),
    classificationThreshold(configuration.count("GaussianSkinDetector.logdensitythreshold") ?
                            configuration["GaussianSkinDetector.logdensitythreshold"].as<double>() :
                            -14),
    weights(configuration.count("GaussianSkinDetector.weights") ?
            configuration["GaussianSkinDetector.weights"].as<bool>() : true),
    trainFilename(configuration.count("GaussianSkinDetector.trainimage") ?
                  configuration["GaussianSkinDetector.trainimage"].as<string>() : "") {
    setNumberOfDistributions(configuration.count("GaussianSkinDetector.kcolours") ? 
                             configuration["GaussianSkinDetector.kcolours"].as<int>() : 2,
                             configuration.count("GaussianSkinDetector.rcolours") ?
                             configuration["GaussianSkinDetector.rcolours"].as<int>() : 1);
            
            // if (config.skinDetectorTrainingImageFilename.length() > 0)
            //   gauss->train(config.skinDetectorTrainingImageFilename);
            // else {
            //   faceDetector = findComponentNoThrow<FaceDetector>(comps);
            //   if (faceDetector) 
            //     gauss->train(*faceDetector);              
            //   else
            //     throw CommandLineException("No training image file presented for the Gaussian skin detector, and face detection is disabled! The detector cannot be trained.");
            // }          
  }

}

