#ifdef SLMOTION_WITH_GTK

#include "GraphicalBlackBoardEditor.hpp"
#include "util.hpp"
#include <iostream>
#include <boost/lexical_cast.hpp>



namespace slmotion {
  namespace gui {
    gboolean GraphicalBlackBoardEditor::selectCallback(GtkTreeSelection* selection,
                                                       GtkTreeModel* model,
                                                       GtkTreePath* path,
                                                       gboolean,
                                                       gpointer data) {
      // short description: check if a *new* selection is being made
      // then get the value from the black board and display it (somehow)
      GtkTreeIter iter;
      SelectionParams* sp = reinterpret_cast<SelectionParams*>(data);
      assert(gtk_tree_model_get_iter(model, &iter, path));
      if ((iter.stamp != sp->currentRow.stamp ||
           iter.user_data != sp->currentRow.user_data ||
           iter.user_data2 != sp->currentRow.user_data2 ||
           iter.user_data3 != sp->currentRow.user_data3) &&
          !gtk_tree_selection_iter_is_selected(selection, &iter)) {
        sp->currentRow = iter;
        gchar *prop, *frnumber, *type;
        gtk_tree_model_get(model, &iter, sp->propertyColumn, &prop, 
                           sp->frameNumberColumn, &frnumber, sp->typeColumn, 
                           &type, -1);

        // now we have everything we need to get the value from the black board

        BlackBoardPointer<boost::any> value;
        if (strcmp(frnumber, "n/a") == 0) 
          value = sp->editor->blackboard->get(prop);
        else
          value = sp->editor->blackboard->get(boost::lexical_cast<size_t>(frnumber),
                                              prop);

        // matrices are handled as a special case because an image
        // representation is created for them
        if (sp->editor->pixBuf != nullptr)
          g_object_unref(sp->editor->pixBuf);
        if (value->type() == typeid(cv::Mat)) {
          const cv::Mat& m = boost::any_cast<cv::Mat>(*value);
          sp->editor->pixBuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, false, 8,
                                              m.cols, m.rows);
          guchar* pixelData = gdk_pixbuf_get_pixels(sp->editor->pixBuf);
          int rowstride = gdk_pixbuf_get_rowstride(sp->editor->pixBuf);
          int width = gdk_pixbuf_get_width(sp->editor->pixBuf);
          int height = gdk_pixbuf_get_height(sp->editor->pixBuf);
          int n_channels = gdk_pixbuf_get_n_channels(sp->editor->pixBuf);
          
          size_t pixelDataSize = rowstride * std::max(height-1,0) + width * 
            ((n_channels * 8 + 7) / 8);
          
          if (m.type() == CV_8UC3) {
            // OpenCV handles images as BGR whereas GTK assumes RGB ordering
            for (size_t i = 0; i < pixelDataSize; i += n_channels) {
              (pixelData + i)[0] = (m.data + i)[2];
              (pixelData + i)[1] = (m.data + i)[1];
              (pixelData + i)[2] = (m.data + i)[0];
            }
          }
          else if (m.type() == CV_8UC1) {
            for (size_t i = 0; i < pixelDataSize; i += n_channels) 
              (pixelData + i)[2] = (pixelData + i)[1] = (pixelData + i)[0] 
                = m.data[i/n_channels];
          }
          else {
            memset(pixelData, 0, pixelDataSize);
            std::cerr << "Warning: Cannot handle the given matrix type" << std::endl;
          }
        }
        else 
          sp->editor->pixBuf = nullptr;
        
        gtk_image_set_from_pixbuf(sp->editor->imageBox,
                                  sp->editor->pixBuf);

        gtk_text_buffer_set_text(sp->editor->textBuffer, 
                                 anyToString(*value).c_str(), -1);
        
        g_free(prop);
        g_free(frnumber);
        g_free(type);
      }
      return true;
    }

    GraphicalBlackBoardEditor::GraphicalBlackBoardEditor(SingletonWindow** pointer,
                                                         std::shared_ptr<BlackBoard> bb) :
      SingletonWindow(pointer),
      blackboard(bb),
      textBuffer(nullptr),
      textView(nullptr),
      imageBox(nullptr),
      pixBuf(nullptr)
    {
      init();
    }

    void GraphicalBlackBoardEditor::init() {
      GtkWindow* editorWindow = getWindow();
      gtk_window_set_title(editorWindow, "BlackBoard Editor");
      sp.propertyColumn = gss.addColumn("Property");
      sp.frameNumberColumn = gss.addColumn("Frame number");
      sp.typeColumn = gss.addColumn("Type");
      gss.createModel();

      const BlackBoard::frame_board_t& frameBoard = blackboard->getFrameBoard();
      for (auto it = frameBoard.cbegin(); it != frameBoard.cend(); ++it) {
        std::vector<std::string> v { "", "", "" };
        v[sp.propertyColumn] = it->first.second;
        v[sp.frameNumberColumn] = boost::lexical_cast<std::string>(it->first.first);
        v[sp.typeColumn] = string_demangle(blackboard->get(it->first.first, 
                                                           it->first.second)
                                           ->type().name());
        gss.addRow(v);
      }

      const BlackBoard::global_board_t& globalBoard = blackboard->getGlobalBoard();
      for (auto it = globalBoard.cbegin(); it != globalBoard.cend(); ++it) {
        std::vector<std::string> v { "", "", "" };
        v[sp.propertyColumn] = it->first;
        v[sp.frameNumberColumn] = "n/a";
        v[sp.typeColumn] = string_demangle(blackboard->get(it->first)->type()
                                           .name());
        gss.addRow(v);
      }

      gtk_widget_set_size_request(gss.toGtkWidget(), 640, 480);
      GtkWidget* hbox = gtk_hbox_new(false, 2);
      gtk_container_add(GTK_CONTAINER(hbox), gss.toGtkWidget());

      GtkWidget* vbox = gtk_vbox_new(false, 2);
      imageBox = GTK_IMAGE(gtk_image_new());
      // gtk_widget_set_size_request(imageBox, 720, 576);
      gtk_container_add(GTK_CONTAINER(vbox), GTK_WIDGET(imageBox));
      gtk_container_add(GTK_CONTAINER(hbox), vbox);

      textView = GTK_TEXT_VIEW(gtk_text_view_new());
      textBuffer = gtk_text_view_get_buffer(textView);
      gtk_text_view_set_editable(textView, false);
      gtk_container_add(GTK_CONTAINER(vbox), GTK_WIDGET(textView));
      gtk_widget_set_size_request(GTK_WIDGET(imageBox), 720, 576);

      sp.currentRow.stamp = 0;
      sp.editor = this;

      GtkTreeSelection* selection = gtk_tree_view_get_selection(gss.toGtkTreeView());
      gtk_tree_selection_set_select_function(selection, &selectCallback, &sp, nullptr);

      gtk_container_add(GTK_CONTAINER(editorWindow), hbox);

      gtk_widget_show_all(GTK_WIDGET(editorWindow));

      pixBuf = nullptr;
    }

    GraphicalBlackBoardEditor::~GraphicalBlackBoardEditor() {
      if (pixBuf != nullptr)
        g_object_unref(pixBuf);
    }
  }
}

#endif // SLMOTION_WITH_GTK
