#include "HandConfiguration.hpp"
#include "AdvancedString.hpp"
#include "SLMotionException.hpp"
#include <boost/math/constants/constants.hpp>
#include <fstream>

namespace slmotion {
  HandConfiguration::FingerBase& HandConfiguration::getLj(unsigned int id) {
    switch (id) {
    case 0:
      return thumb;
      break;

    case 1:
      return index;
      break;

    case 2:
      return middle;
      break;

    case 3:
      return ring;
      break;

    case 4:
      return little;
      break;

    default:
      throw std::out_of_range("The finger number must be between 0 and 4!");      
    }

    // bogus
    return thumb;
  }


  cv::Mat HandConfiguration::toMat() const {
    cv::Mat v(25, 1, CV_64FC1);
    int i = 0;
    for (const Finger* f : {&little, &ring, &middle, &index}) {
      v.at<double>(i++) = f->getMcp().first;
      v.at<double>(i++) = f->getMcp().second;
      v.at<double>(i++) = f->getPip();
      v.at<double>(i++) = f->getDip();
    }
    v.at<double>(i++) = thumb.getCm().first;
    v.at<double>(i++) = thumb.getCm().second;
    v.at<double>(i++) = thumb.getMcp().first;
    v.at<double>(i++) = thumb.getDip();
    v.at<double>(i++) = rc.first;
    v.at<double>(i++) = rc.second;
    v.at<double>(i++) = baseMetaJoint.first;
    v.at<double>(i++) = baseMetaJoint.second;
    v.at<double>(i++) = dru;
    assert(i == v.rows);
    return v;
  }

  HandConfiguration::HandConfiguration(const cv::Mat& v) {
    if (v.type() != CV_64FC1 || v.cols != 1 || v.rows != 25)
      throw std::invalid_argument("The input matrix is expected to be a "
                                  "25-column vector of doubles!");
    int i = 0;
    for (Finger* f : {&little, &ring, &middle, &index}) {
      f->getMcp().first = v.at<double>(i++);
      f->getMcp().second = v.at<double>(i++);
      f->getPip() = v.at<double>(i++);
      f->getDip() = v.at<double>(i++);
    }
    thumb.getCm().first = v.at<double>(i++);
    thumb.getCm().second = v.at<double>(i++);
    thumb.getMcp().first = v.at<double>(i++);
    thumb.getDip() = v.at<double>(i++);
    rc.first = v.at<double>(i++);
    rc.second = v.at<double>(i++);
    baseMetaJoint.first = v.at<double>(i++);
    baseMetaJoint.second = v.at<double>(i++);
    dru = v.at<double>(i++);
    assert(i == v.rows);
  }

#ifdef SLMOTION_ENABLE_LIBHAND
  void HandConfiguration::apply(libhand::FullHandPose& pose) const {
    static const int LITTLE_FINGER_MCP_INDEX = 0;
    static const int LITTLE_FINGER_PIP_INDEX = 1;
    static const int LITTLE_FINGER_DIP_INDEX = 2;

    static const int RING_FINGER_MCP_INDEX = 3;
    static const int RING_FINGER_PIP_INDEX = 4;
    static const int RING_FINGER_DIP_INDEX = 5;

    static const int MIDDLE_FINGER_MCP_INDEX = 6;
    static const int MIDDLE_FINGER_PIP_INDEX = 7;
    static const int MIDDLE_FINGER_DIP_INDEX = 8;

    static const int INDEX_FINGER_MCP_INDEX = 9;
    static const int INDEX_FINGER_PIP_INDEX = 10;
    static const int INDEX_FINGER_DIP_INDEX = 11;

    static const int THUMB_CM_INDEX = 12;
    static const int THUMB_MCP_INDEX = 13;
    static const int THUMB_DIP_INDEX = 14;

    static const int RADIOCARPAL_INDEX = 15;

    static const int BASE_METAJOINT_INDEX = 16;

    static const int DISTAL_RADIOULNAR_INDEX = 17;

    // n joints = 18??

    double PI = boost::math::constants::pi<double>();
    
    pose.bend(LITTLE_FINGER_MCP_INDEX) = little.getMcp().first / 180. * PI;
    pose.side(LITTLE_FINGER_MCP_INDEX) = little.getMcp().second / 180. * PI;

    pose.bend(LITTLE_FINGER_PIP_INDEX) = little.getPip() / 180. * PI;
    pose.bend(LITTLE_FINGER_DIP_INDEX) = little.getDip() / 180. * PI;

    pose.bend(RING_FINGER_MCP_INDEX) = ring.getMcp().first / 180. * PI;
    pose.side(RING_FINGER_MCP_INDEX) = ring.getMcp().second / 180. * PI;

    pose.bend(RING_FINGER_PIP_INDEX) = ring.getPip() / 180. * PI;
    pose.bend(RING_FINGER_DIP_INDEX) = ring.getDip() / 180. * PI;

    pose.bend(MIDDLE_FINGER_MCP_INDEX) = middle.getMcp().first / 180. * PI;
    pose.side(MIDDLE_FINGER_MCP_INDEX) = middle.getMcp().second / 180. * PI;

    pose.bend(MIDDLE_FINGER_PIP_INDEX) = middle.getPip() / 180. * PI;
    pose.bend(MIDDLE_FINGER_DIP_INDEX) = middle.getDip() / 180. * PI;

    pose.bend(INDEX_FINGER_MCP_INDEX) = index.getMcp().first / 180. * PI;
    pose.side(INDEX_FINGER_MCP_INDEX) = index.getMcp().second / 180. * PI;

    pose.bend(INDEX_FINGER_PIP_INDEX) = index.getPip() / 180. * PI;
    pose.bend(INDEX_FINGER_DIP_INDEX) = index.getDip() / 180. * PI;

    pose.bend(THUMB_CM_INDEX) = thumb.getCm().first / 180. * PI;
    pose.side(THUMB_CM_INDEX) = thumb.getCm().second / 180. * PI;

    // the MCP of the thumb is saddle
    pose.bend(THUMB_MCP_INDEX) = thumb.getMcp().first / 180. * PI;
    pose.bend(THUMB_DIP_INDEX) = thumb.getDip() / 180. * PI;

    pose.bend(RADIOCARPAL_INDEX) = rc.first / 180. * PI;
    pose.side(RADIOCARPAL_INDEX) = rc.second / 180. * PI;

    pose.bend(BASE_METAJOINT_INDEX) = baseMetaJoint.first / 180. * PI;
    pose.side(BASE_METAJOINT_INDEX) = baseMetaJoint.second / 180. * PI;

    pose.bend(DISTAL_RADIOULNAR_INDEX) = dru / 180. * PI;
  }



  std::string HandConfiguration::jointName(int i) {
    if (i == 0)
      return "little finger mcp bend";
    else if (i == 1)
      return "little finger mcp side";
    else if (i == 2)
      return "little finger pip bend";
    else if (i == 3)
      return "little finger f dip bend";
    else if (i == 4)
      return "ring finger mcp bend";
    else if (i == 5)
      return "ring finger mcp side";
    else if (i == 6)
      return "ring finger pip bend";
    else if (i == 7)
      return "ring finger dip bend";
    else if (i == 8)
      return "middle finger mcp bend";
    else if (i == 9)
      return "middle finger mcp side";
    else if (i == 10)
      return "middle finger pip bend";
    else if (i == 11)
      return "middle finger dip bend";
    else if (i == 12)
      return "index finger mcp bend";
    else if (i == 13)
      return "index finger mcp side";
    else if (i == 14)
      return "index finger pip bend";
    else if (i == 15)
      return "index finger dip bend";
    else if (i == 16)
      return "thumb cm bend";
    else if (i == 17)
      return "thumb cm side";
    else if (i == 18)
      return "thumb mcp bend";
    else if (i == 19)
      return "thumb dip bend";
    else if (i == 20)
      return "rc bend";
    else if (i == 21)
      return "rc side";
    else if (i == 22)
      return "base metajoint bend";
    else if (i == 23)
      return "base metajoint side";
    else if (i == 24)
      return "dru bend";
    else
      return "N/A";
  }



  HandConfiguration HandConfiguration::fromFile(const std::string& filename) {
    std::ifstream ifs(filename);
    cv::Mat configuration(25, 1, CV_64FC1);
    configuration = 0.0;
    std::vector<slmotion::AdvancedString> lines;
    if (!ifs.good()) 
      throw SLMotionException(("Could not read \"" + filename + "\"").c_str());
    std::string s;
    while (std::getline(ifs, s))
      lines.push_back(s);
    if (lines.size() != 25) 
      throw SLMotionException(("ERROR: \"" + filename + "\" contains an "
                               "invalid hand configuration. Parameter "
                               "count mismatch: Expected 25 parameters but "
                               "got " + AdvancedString(lines.size()) + "!").c_str());    
    for (int i = 0; i < 25; ++i)
      configuration.at<double>(i,0) = static_cast<float>(lines[i]);

    return HandConfiguration(configuration);
  }

#endif // SLMOTION_ENABLE_LIBHAND
}
