#ifndef SLMOTION_HAND_CONFIGURATION
#define SLMOTION_HAND_CONFIGURATION

#ifdef SLMOTION_ENABLE_LIBHAND
#include <hand_pose.h>
#endif

#include <opencv2/opencv.hpp>

#include <stdexcept>

namespace slmotion {
  /**
   * This class represents an abstraction of the hand configuration
   */
  class HandConfiguration {
  public:
    class FingerBase {
    public:
      FingerBase() : mcp(std::pair<double,double>(0,0)), pip(0), dip(0), 
                     cm(std::pair<double,double>(0,0)) {}
      virtual ~FingerBase() {}

      /**
       * Returns a reference to the metacarpophalangeal joint
       */
      std::pair<double, double>& getMcp() { return mcp; }

      /**
       * Returns a reference to the metacarpophalangeal joint
       */
      const std::pair<double, double>& getMcp() const { return mcp; }

      /**
       * Returns a reference to the distal interphalangeal joint
       */
      double& getDip() { return dip; }

      /**
       * Returns a reference to the distal interphalangeal joint
       */
      const double& getDip() const { return dip; }

      /**
       * Returns a reference to the proximal interphalangeal joint
       */
      virtual double& getPip() { return pip; }

      /**
       * Returns a reference to the proximal interphalangeal joint
       */
      virtual const double& getPip() const { return pip; }

      /**
       * Returns the carpo-metacarpal joint (notice the 2-dimensionality)
       */
      virtual std::pair<double,double>& getCm() { 
        return cm;
      }

      /**
       * Returns the carpo-metacarpal joint (notice the 2-dimensionality)
       */
      virtual const std::pair<double,double>& getCm() const { 
        return cm;
      }


    private:
      std::pair<double, double> mcp; ///< metacarpophalangeal joint (degrees)
      double pip; ///< proximal interphalangeal joint (degrees)
      double dip; ///< dip interphalangeal joint (degrees)
      std::pair<double,double> cm; ///< carpometacarpal joint (degrees)
    };

    class Finger : public FingerBase {
    public:
      Finger() {}

      virtual std::pair<double,double>& getCm() { 
        throw std::out_of_range("Only the thumb has a modifiable carpo-metacarpal joint!");
        return FingerBase::getCm();
      }      

      virtual const std::pair<double,double>& getCm() const { 
        throw std::out_of_range("Only the thumb has a modifiable carpo-metacarpal joint!");
        return FingerBase::getCm();
      }      
    };

    class Thumb : public FingerBase {
    public:
      Thumb() {}

      virtual double& getPip() { 
        throw std::out_of_range("The thumb does not have a proximal interphalangeal joint!");
        return FingerBase::getPip(); 
      }

      virtual const double& getPip() const { 
        throw std::out_of_range("The thumb does not have a proximal interphalangeal joint!");
        return FingerBase::getPip(); 
      }

    private:
    };

    Finger& getIndexFinger() {
      return index;
    }

    Finger& getMiddleFinger() {
      return middle;
    }

    Finger& getRingFinger() {
      return ring;
    }

    Finger& getLittleFinger() {
      return little;
    }

    Thumb& getThumb() {
      return thumb;
    }

    /**
     * Returns the radiocarpal joint
     */
    std::pair<double,double>& getRc() {
      return rc;
    }

    /**
     * Returns the "base metajoint", ie. an imaginary joint at the
     * base where the hand is cut
     */
    std::pair<double,double>& getBaseMetaJoint() {
      return baseMetaJoint;
    }

    /**
     * Returns the distal radioulnar joint. The value here
     * corresponds to a twisting articulation.
     */
    double& getDru() {
      return dru;
    }

#ifdef SLMOTION_ENABLE_LIBHAND
    /**
     * Applies the hand configuration to a LibHand hand pose
     */
    void apply(libhand::FullHandPose& pose) const;
#endif

    /**
     * Returns a reference to a finger according to the Liddell-Johnson numbering, ie.
     *
     * 0 = thumb
     * 1 = index finger
     * 2 = middle finger
     * 3 = little finger
     * 4 = pinky 
     */
    FingerBase& getLj(unsigned int id);

    HandConfiguration() : rc(0,0), baseMetaJoint(0,0), dru(0) {}

    /**
     * Returns a 4 * 4 + 4 + 2 + 2 + 1 = 25-vector of doubles (CV_64FC1)
     * as follows (all values are in DEGREES): 
     * [ little finger mcp bend ("toward or outwards of                    ]
     * [ the palm", corresponding to extension in Liddell and Johnson"),   ]
     * [ little finger mcp side (" toward or outwards of other fingers",   ]
     * [ corresponding to adduction and abduction in Liddell and Johnson), ]
     * [ lf pip bend,                                                      ]
     * [ lf dip bend,                                                      ]
     * [ ring finger mcp bend,                                             ]
     * [ ring finger mcp side,                                             ]
     * [ ring finger pip bend,                                             ]
     * [ ring finger dip bend,                                             ]
     * [ middle finger mcp bend,                                           ]
     * [ middle finger mcp side,                                           ]
     * [ middle finger pip bend,                                           ]
     * [ middle finger dip bend,                                           ]
     * [ index finger mcp bend,                                            ]
     * [ index finger mcp side,                                            ]
     * [ index finger pip bend,                                            ]
     * [ index finger dip bend,                                            ]
     * [ thumb cm bend,                                                    ]
     * [ thumb cm side,                                                    ]
     * [ thumb mcp bend,                                                   ]
     * [ thumb dip bend,                                                   ]
     * [ rc bend (wrist up and down),                                      ]
     * [ rc side (wrist left and right),                                   ]
     * [ base metajoint bend (up and down from where the hand is cut),     ]
     * [ base metajoint side (left and right),                             ]
     * [ dru bend (twisting of the hand)                                   ]
     */
    cv::Mat toMat() const;


    
    /**
     * Given an index, returns the corresponding joint name (as above), or N/A 
     * if i is invalid.
     */
    static std::string jointName(int i);



    /**
     * Alternate form of the toMat() above.
     */
    inline explicit operator cv::Mat() const {
      return toMat();
    }



    /**
     * Constructs a hand configuration object where the joint angles
     * are specified below (the format is the same as the one returned
     * by jointsToVector):
     *
     * A 4 * 4 + 4 + 2 + 1 = 23-vector of doubles (CV_64FC1)
     * as follows (all values are in DEGREES): 
     * [ little finger mcp bend ("toward or outwards of                    ]
     * [ the palm", corresponding to extension in Liddell and Johnson"),   ]
     * [ little finger mcp side (" toward or outwards of other fingers",   ]
     * [ corresponding to adduction and abduction in Liddell and Johnson), ]
     * [ lf pip bend,                                                      ]
     * [ lf dip bend,                                                      ]
     * [ ring finger mcp bend,                                             ]
     * [ ring finger mcp side,                                             ]
     * [ ring finger pip bend,                                             ]
     * [ ring finger dip bend,                                             ]
     * [ middle finger mcp bend,                                           ]
     * [ middle finger mcp side,                                           ]
     * [ middle finger pip bend,                                           ]
     * [ middle finger dip bend,                                           ]
     * [ index finger mcp bend,                                            ]
     * [ index finger mcp side,                                            ]
     * [ index finger pip bend,                                            ]
     * [ index finger dip bend,                                            ]
     * [ thumb cm bend,                                                    ]
     * [ thumb cm side,                                                    ]
     * [ thumb mcp bend,                                                   ]
     * [ thumb dip bend,                                                   ]
     * [ rc bend (wrist up and down),                                      ]
     * [ rc side (wrist left and right),                                   ]
     * [ base metajoint bend (up and down from where the hand is cut),     ]
     * [ base metajoint side (left and right),                             ]
     * [ dru bend (twisting of the hand)                                   ]
     */
    HandConfiguration(const cv::Mat& vector);



    /**
     * Reads the configuration from a file
     */
    static HandConfiguration fromFile(const std::string& filename);



  private:
    Finger index, middle, ring, little;
    Thumb thumb;
    std::pair<double,double> rc;
    std::pair<double,double> baseMetaJoint;
    double dru;
  };
}

#endif
