#ifdef SLMOTION_ENABLE_LIBHAND
#include "HandEvaluator.hpp"
#include "math.hpp"

namespace slmotion {
  static HandEvaluator DUMMY(true);
  void HandEvaluator::process(frame_number_t fn) {  
    auto handBox = getBlackBoard().get<cv::Mat>(fn, HAND_FITTER_HAND_BOX_IMG_ENTRY);
    auto handBoxBinary = getBlackBoard().get<cv::Mat>(fn, HAND_FITTER_HAND_BOX_BINARY_IMG_ENTRY);
    auto renderedImg = getBlackBoard().get<cv::Mat>(fn, HAND_FITTER_RENDERED_IMG_ENTRY);

    HandFitter::Instance instance;
    instance.handBox = &(*handBox);
    instance.handBoxBinary = &(*handBoxBinary);
    instance.trimmedThreshold = trimmedThreshold;
    instance.trimmedMaxSize = trimmedMaxSize;
    instance.phogLevels = phogLevels;
    instance.phogBins = phogBins;
    instance.cannyThreshold = cannyThreshold;
    instance.renderedImg = *renderedImg;

    std::set<std::string> blackboardKeys = std::set<std::string>();

    // compute features first
    std::set<HandFitter::Feature> allFeatures;
    for (auto f : HandFitter::VALID_FEATURE_METRIC_COMBINATIONS)
      allFeatures.insert(f.first);



    for (auto f : allFeatures) {
      HandFitter::computeFeatureForHand(f, instance);
      HandFitter::computeFeatureForRendered(f, instance);
    }



    for (auto f : HandFitter::VALID_FEATURE_METRIC_COMBINATIONS) {
      timerOn();
      double d = HandFitter::computeDistance(f.first, f.second, instance);
      assert (d >= 0);
      time_t t = timerOff();
      std::string comboName = HandFitter::featureToString(f.first) + "+" +
        HandFitter::distanceFunctionToString(f.second);
      getBlackBoard().set(fn, "handevaluator " + comboName + " distance", d);
      blackboardKeys.insert("handevaluator " + comboName + " distance");
      getBlackBoard().set(fn, "handevaluator " + comboName + " time", t);
      blackboardKeys.insert("handevaluator " + comboName + " time");
      std::cerr << comboName << ": " << d << " (" << t << " ms)" << std::endl;
    }

    blackboardKeys.insert("handevaluator hog size");
    getBlackBoard().set(fn, "handevaluator hog size", instance.handHog->getDescriptorSize());
    blackboardKeys.insert("handevaluator phog size");
    getBlackBoard().set(fn, "handevaluator phog size", instance.handPhog->getRawValues().size());
    blackboardKeys.insert("handevaluator trimmed hog hand size");
    getBlackBoard().set(fn, "handevaluator trimmed hog hand size",
                        instance.handTrimmedHog.rows);
    blackboardKeys.insert("handevaluator trimmed hog rendered size");
    getBlackBoard().set(fn, "handevaluator trimmed hog rendered size",
                        instance.renderedTrimmedHog.rows);
    blackboardKeys.insert("handevaluator hand sift count");
    getBlackBoard().set(fn, "handevaluator hand sift count",
                        instance.handSiftKeypoints.size());
    blackboardKeys.insert("handevaluator rendered sift count");
    getBlackBoard().set(fn, "handevaluator rendered sift count",
                        instance.renderedSiftKeypoints.size());
    blackboardKeys.insert("handevaluator hand surf count");
    getBlackBoard().set(fn, "handevaluator hand surf count",
                        instance.handSurfKeypoints.size());
    blackboardKeys.insert("handevaluator rendered surf count");
    getBlackBoard().set(fn, "handevaluator rendered surf count",
                        instance.renderedSurfKeypoints.size());
    blackboardKeys.insert("handevaluator hand canny count");
    getBlackBoard().set(fn, "handevaluator hand canny count",
                        instance.handCannyEdgesSize);
    blackboardKeys.insert("handevaluator rendered canny count");
    getBlackBoard().set(fn, "handevaluator rendered canny count",
                        instance.renderedCannyEdgesSize);

    assert((int)instance.handSiftKeypoints.size() == instance.handSiftDescriptors.rows);
    assert((int)instance.handSiftKeypoints.size() == instance.handSiftWeights.rows);
    assert(instance.handSiftWeights.cols == 1);
    assert(instance.handSiftDescriptors.cols == 128);

    assert((int)instance.handSurfKeypoints.size() == instance.handSurfDescriptors.rows);
    assert((int)instance.handSurfKeypoints.size() == instance.handSurfWeights.rows);
    assert(instance.handSurfWeights.cols == 1);
    assert(instance.handSurfDescriptors.cols == 64);

    assert((int)instance.renderedSiftKeypoints.size() == instance.renderedSiftDescriptors.rows);
    assert((int)instance.renderedSiftKeypoints.size() == instance.renderedSiftWeights.rows);
    assert(instance.renderedSiftWeights.cols == 1);
    assert(instance.renderedSiftDescriptors.cols == 128);

    assert((int)instance.renderedSurfKeypoints.size() == instance.renderedSurfDescriptors.rows);
    assert((int)instance.renderedSurfKeypoints.size() == instance.renderedSurfWeights.rows);
    assert(instance.renderedSurfWeights.cols == 1);
    assert(instance.renderedSurfDescriptors.cols == 64);

    assert(instance.handHog->getDescriptorSize() == (handBox->rows / 8 - 1) *
           (handBox->cols / 8 - 1) * 4 * 9);
    assert(instance.handHog->getDescriptorSize() == instance.renderedHog->getDescriptorSize());

    assert(instance.handToRenderedHogDm.rows == instance.handHog->getDescriptorSize());
    assert(instance.handToRenderedHogDm.cols == instance.handHog->getDescriptorSize());

    assert(instance.handHog.get() != nullptr);
    assert(instance.renderedHog.get() != nullptr);

    assert(instance.handPhogValues.rows == (int)instance.handPhog->getRawValues().size());
    assert(instance.renderedPhogValues.rows == (int)instance.renderedPhog->getRawValues().size());
    for (int i = 0; i < instance.renderedPhogValues.rows; ++i) {
      assert(instance.handPhogValues.at<float>(i) == instance.handPhog->getRawValues()[i]);
      assert(instance.renderedPhogValues.at<float>(i) == instance.renderedPhog->getRawValues()[i]);
    }

    assert(instance.handHogValues.rows == instance.handHog->getDescriptorSize());
    for (int i = 0; i < instance.handHogValues.rows; ++i)
      assert(instance.handHogValues.at<float>(i) == instance.handHog->getRawValues()[i]);
    assert(instance.handHogValuesNormalized.rows == instance.handHogValues.rows);
    assert(std::abs(math::sum<float>(instance.handHogValuesNormalized) - 1.0) < 0.001);

    assert(instance.renderedHogValues.rows == instance.renderedHog->getDescriptorSize());
    for (int i = 0; i < instance.renderedHogValues.rows; ++i)
      assert(instance.renderedHogValues.at<float>(i) == instance.renderedHog->getRawValues()[i]);
    assert(instance.renderedHogValuesNormalized.rows == instance.renderedHogValues.rows);
    assert(std::abs(math::sum<float>(instance.renderedHogValuesNormalized) - 1.0) < 0.001);

    assert(instance.handContour.type() == CV_32FC1);
    assert(math::max(instance.handContour) == 1);
    assert(math::min(instance.handContour) == 0);
    assert(instance.handContourSize == cv::countNonZero(instance.handContour));
    assert(instance.handContourDistanceTransform.type() == CV_32FC1);
    assert(instance.handContourDistanceTransform.size() == instance.handContour.size());

    assert(instance.renderedContour.type() == CV_32FC1);
    assert(math::max(instance.renderedContour) == 1);
    assert(math::min(instance.renderedContour) == 0);
    assert(instance.renderedContourSize == cv::countNonZero(instance.renderedContour));
    assert(instance.renderedContourDistanceTransform.type() == CV_32FC1);
    assert(instance.renderedContourDistanceTransform.size() == instance.renderedContour.size());

    assert(instance.handCannyEdges.type() == CV_32FC1);
    assert(math::max(instance.handCannyEdges) == 1);
    assert(math::min(instance.handCannyEdges) == 0);
    assert(instance.handCannyEdgesSize == cv::countNonZero(instance.handCannyEdges));
    assert(instance.handCannyEdgesDistanceTransform.type() == CV_32FC1);
    assert(instance.handCannyEdgesDistanceTransform.size() == instance.handCannyEdges.size());

    assert(instance.renderedCannyEdges.type() == CV_32FC1);
    assert(math::max(instance.renderedCannyEdges) == 1);
    assert(math::min(instance.renderedCannyEdges) == 0);
    assert(instance.renderedCannyEdgesSize == cv::countNonZero(instance.renderedCannyEdges));
    assert(instance.renderedCannyEdgesDistanceTransform.type() == CV_32FC1);
    assert(instance.renderedCannyEdgesDistanceTransform.size() == instance.renderedCannyEdges.size());

    std::cerr << "blackboard keys:" << std::endl;
    for (auto s : blackboardKeys)
      std::cerr << s << std::endl;
  }



  Component* HandEvaluator::createComponentImpl(const boost::program_options::variables_map& vm, 
                                                BlackBoard* blackBoard, FrameSource* frameSource) const {
    HandEvaluator hf(blackBoard, frameSource);
    
    if (vm.count("HandFitter.trimmedthreshold"))
      hf.trimmedThreshold = vm["HandFitter.trimmedthreshold"].as<double>();

    if (vm.count("HandFitter.trimmedmaxsize"))
      hf.trimmedMaxSize = vm["HandFitter.trimmedmaxsize"].as<int>();

    if (vm.count("HandFitter.phoglevels"))
      hf.phogLevels = vm["HandFitter.phoglevels"].as<int>();

    if (vm.count("HandFitter.phogbins"))
      hf.phogBins = vm["HandFitter.phogbins"].as<int>();

    if (vm.count("HandFitter.cannythreshold"))
      hf.cannyThreshold = vm["HandFitter.cannythreshold"].as<double>();

    return new HandEvaluator(hf);
  }



  boost::program_options::options_description HandEvaluator::getConfigurationFileOptionsDescription() const {
    boost::program_options::options_description opts;

    return opts;
  }
}
#endif // SLMOTION_ENABLE_LIBHAND
