#ifdef SLMOTION_ENABLE_LIBHAND
#include "HandFitter.hpp"
#include "AdvancedString.hpp"
#include "HandVisualiser.hpp"
#include "math.hpp"
#include "util.hpp"
#ifdef SLMOTION_WITH_LIBLBFGS
#include "lbfgs.h"


namespace slmotion {
  static HandFitter DUMMY(true);

  /*
  static double degreeToUnit(double d) {
    while (d > 180.0)
      d -= 360;
    while (d < -180.0)
        d += 360;
    
    return d/180.0;
  };

  static double radianToUnit(double d) {
    while (d > math::PI)
      d -= math::PI;
    while (d < -math::PI)
      d += math::PI;

    return d/math::PI;
  };

  static double unitToRadian(double d) {
    return d*math::PI;
  };

  static double unitToDegree(double d) {
    return d*180.0;
  };

  static double radianToDegree(double d) {
    return d/math::PI*180.0;
  };

  static double degreeToRadian(double d) {
    return d/180.0*math::PI;
    };*/



  static std::string lbfgsRetValToString(int r) {
    if (r == LBFGS_SUCCESS)
      return "LBFGS_SUCCESS";
    else if (r == LBFGS_CONVERGENCE)
      return "LBFGS_CONVERGENCE";
    else if (r == LBFGS_STOP)
      return "LBFGS_STOP";
    else if (r == LBFGS_ALREADY_MINIMIZED)
      return "LBFGS_ALREADY_MINIMIZED";
    else if (r == LBFGSERR_UNKNOWNERROR)
      return "LBFGSERR_UNKNOWNERROR";
    else if (r == LBFGSERR_LOGICERROR)
      return "LBFGSERR_LOGICERROR";
    else if (r == LBFGSERR_OUTOFMEMORY)
      return "LBFGSERR_OUTOFMEMORY";
    else if (r == LBFGSERR_CANCELED)
      return "LBFGSERR_CANCELED";

    else if (r == LBFGSERR_INVALID_N)
      return "LBFGSERR_INVALID_N";
    else if (r == LBFGSERR_INVALID_N_SSE)
      return "LBFGSERR_INVALID_N_SSE";
    else if (r == LBFGSERR_INVALID_X_SSE)
      return "LBFGSERR_INVALID_X_SSE";
    else if (r == LBFGSERR_INVALID_EPSILON)
      return "LBFGSERR_INVALID_EPSILON";

    else if (r == LBFGSERR_INVALID_TESTPERIOD)
      return "LBFGSERR_INVALID_TESTPERIOD";
    else if (r == LBFGSERR_INVALID_DELTA)
      return "LBFGSERR_INVALID_DELTA";
    else if (r == LBFGSERR_INVALID_LINESEARCH)
      return "LBFGSERR_INVALID_LINESEARCH";
    else if (r == LBFGSERR_INVALID_MINSTEP)
      return "LBFGSERR_INVALID_MINSTEP";

    else if (r == LBFGSERR_INVALID_MAXSTEP)
      return "LBFGSERR_INVALID_MAXSTEP";
    else if (r == LBFGSERR_INVALID_FTOL)
      return "LBFGSERR_INVALID_FTOL";
    else if (r == LBFGSERR_INVALID_WOLFE)
      return "LBFGSERR_INVALID_WOLFE";
    else if (r == LBFGSERR_INVALID_GTOL)
      return "LBFGSERR_INVALID_GTOL";

    else if (r == LBFGSERR_INVALID_XTOL)
      return "LBFGSERR_INVALID_XTOL";
    else if (r == LBFGSERR_INVALID_MAXLINESEARCH)
      return "LBFGSERR_INVALID_MAXLINESEARCH";
    else if (r == LBFGSERR_INVALID_ORTHANTWISE)
      return "LBFGSERR_INVALID_ORTHANTWISE";
    else if (r == LBFGSERR_INVALID_ORTHANTWISE_START)
      return "LBFGSERR_INVALID_ORTHANTWISE_START";

    else if (r == LBFGSERR_INVALID_ORTHANTWISE_END)
      return "LBFGSERR_INVALID_ORTHANTWISE_END";
    else if (r == LBFGSERR_OUTOFINTERVAL)
      return "LBFGSERR_OUTOFINTERVAL";
    else if (r == LBFGSERR_INCORRECT_TMINMAX)
      return "LBFGSERR_INCORRECT_TMINMAX";
    else if (r == LBFGSERR_ROUNDING_ERROR)
      return "LBFGSERR_ROUNDING_ERROR";

    else if (r == LBFGSERR_MINIMUMSTEP)
      return "LBFGSERR_MINIMUMSTEP";
    else if (r == LBFGSERR_MAXIMUMSTEP)
      return "LBFGSERR_MAXIMUMSTEP";
    else if (r == LBFGSERR_MAXIMUMLINESEARCH)
      return "LBFGSERR_MAXIMUMLINESEARCH";
    else if (r == LBFGSERR_MAXIMUMITERATION)
      return "LBFGSERR_MAXIMUMITERATION";

    else if (r == LBFGSERR_WIDTHTOOSMALL)
      return "LBFGSERR_WIDTHTOOSMALL";
    else if (r == LBFGSERR_INVALIDPARAMETERS)
      return "LBFGSERR_INVALIDPARAMETERS";
    else if (r == LBFGSERR_INCREASEGRADIENT)
      return "LBFGSERR_INCREASEGRADIENT";
    else
      return "UNKNOWN ERROR";
  }


  // normalises by forcing angles to a sensible range and prevents r
  // from going too small or large
#if 0
  static cv::Mat normaliseConfig(const cv::Mat& config, const cv::Size& frameSize) {
    assert(config.cols == 1 && config.rows == 6 && config.type() == CV_64FC1);
    cv::Mat normalised(config.rows, config.cols, config.type());
    // assume r in [1,10]
    normalised.at<double>(0) = std::max(std::min(config.at<double>(0),10.0),1.0);
    for (int i = 1; i < 4; ++i) {
      double& a = normalised.at<double>(i);
      a = math::toRange(config.at<double>(i), -math::PI, math::PI);
      assert(math::inRange(a, -math::PI, math::PI));
    }
    normalised.at<double>(4) = std::max(std::min(config.at<double>(4),
                                                 (double)frameSize.width), 
                                        (double)-frameSize.width);
    normalised.at<double>(5) = std::max(std::min(config.at<double>(5),
                                                 (double)frameSize.height), 
                                        (double)-frameSize.height);
    return normalised;
  }

  // normalises variables to range [-1,1]
  static cv::Mat normaliseConfig(const cv::Mat& config, const cv::Size& frameSize) {
    assert(config.cols == 1 && config.rows == 6 && config.type() == CV_64FC1);
    cv::Mat normalised(config.rows, config.cols, config.type());
    // assume r in [1,10]
    normalised.at<double>(0) = (std::max(std::min(config.at<double>(0),10.0),1.0) - 5.0)/5.0;
    for (int i = 1; i < 4; ++i)
      normalised.at<double>(i) = radianToUnit(config.at<double>(i));
    normalised.at<double>(4) = config.at<double>(4)/frameSize.width;
    normalised.at<double>(5) = config.at<double>(5)/frameSize.height;
    return normalised;
  }

  // given a normalised configuration, forces values into proper range
  static cv::Mat reNormaliseConfig(const cv::Mat& config) {
    assert(config.cols == 1 && config.rows == 6 && config.type() == CV_64FC1);
    cv::Mat normalised(config.rows, config.cols, config.type());
    // assume r in [1,10]
    normalised.at<double>(0) = std::max(std::min(config.at<double>(0),1.0),-1.0);
    for (int i = 1; i < 4; ++i)
      normalised.at<double>(i) = math::toRange(config.at<double>(i), -1.0, 1.0);
    normalised.at<double>(4) = std::max(std::min(config.at<double>(4),1.0),-1.0);
    normalised.at<double>(5) = std::max(std::min(config.at<double>(5),1.0),-1.0);
    return normalised;
  }

  static cv::Mat deNormaliseConfig(const cv::Mat& config, const cv::Size& frameSize) {
    assert(config.cols == 1 && config.rows == 6 && config.type() == CV_64FC1);
    cv::Mat deNormalised(config.rows, config.cols, config.type());
    deNormalised.at<double>(0) = config.at<double>(0)*5.0 + 5.0;
    for (int i = 1; i < 4; ++i)
      deNormalised.at<double>(i) = unitToRadian(config.at<double>(i));
    deNormalised.at<double>(4) = config.at<double>(4)*frameSize.width;
    deNormalised.at<double>(5) = config.at<double>(5)*frameSize.height;
    return deNormalised;
  }
#endif
                                      


  template<typename T>
  class HandGradient {
  public:
    HandGradient(T f) : f(f) {}

    cv::Mat operator()(const cv::Mat& config) {
      assert(config.rows == 6 && config.cols == 1 && config.type() == CV_64FC1);
      cv::Mat gradient(config.rows, config.cols, config.type(), 
                       cv::Scalar::all(0));
      double angularH = 0.1;
      std::vector<double> hs {0.1, angularH, angularH, angularH, 1.0, 1.0};
      for (int i = 0; i < config.rows; ++i) {
        cv::Mat h(config.rows, config.cols, CV_64FC1, cv::Scalar::all(0));
        h.at<double>(i) = hs[i];
        gradient.at<double>(i) = (f(config+h)-f(config-h))/(2.0*hs[i]);
      }
      return gradient;
    };

  private:
    T f;
    int N;
  };

  template<typename T>
  HandGradient<T> handGradient(T f) {
    return HandGradient<T>(f);
  }


  
  /**
   * Determines the initial distance R as the value that, given other
   * parameters, maximizes the ratio between intersection and union of
   * non-zero pixels.
   */
  static double determineInitialR(const cv::Mat& handBoxGreyscale,
                                  HandVisualiser& hv,
                                  const HandConfiguration& hc,
                                  double theta,
                                  double phi,
                                  double tilt) {
    assert(handBoxGreyscale.type() == CV_8UC1);
    double bestR = 1.0;
    double bestRatio = 0.0;
    for (double R = 1.0; R < 20.0; R += 1.0) {
      cv::Mat renderedImg = hv.render(hc, R, theta, phi, tilt);
      cv::Mat renderedImgGreyscale;
      cv::cvtColor(renderedImg, renderedImgGreyscale, CV_BGR2GRAY);
      double unionCount = cv::countNonZero(cv::max(renderedImgGreyscale, handBoxGreyscale));
      double intersectionCount = cv::countNonZero(cv::min(renderedImgGreyscale, handBoxGreyscale));
      double ratio = intersectionCount / unionCount;
      if (ratio > bestRatio) {
        bestRatio = ratio;
        bestR = R;
      }
    }
    return bestR;
  }



  static void extractContour(const cv::Mat& binaryImage,
                             cv::Mat& outContour,
                             cv::Mat& outDistanceTransform,
                             int& outContourSize) {
    outContour = extractOuterContour(binaryImage);
    // cv::Mat outContourFloat;
    // outContour.convertTo(outContourFloat, CV_32FC1);
    // outContourFloat = cv::Scalar::all(1.0) - outContourFloat / 255.0;
    cv::distanceTransform(cv::Scalar::all(255) - outContour, outDistanceTransform, CV_DIST_L2, 5);
    outContourSize = cv::countNonZero(outContour);
    cv::Mat temp;
    outContour.convertTo(temp, CV_32FC1);
    outContour = temp / 255.0;
  }

 

  static void extractHog(const cv::Mat& img,
                         std::unique_ptr<HOGDescriptor>& outHogPtr,
                         cv::Mat& outHogValuesNormalized,
                         cv::Mat& outHogValues) {
    outHogPtr.reset(new HOGDescriptor(img));
    outHogValues = outHogPtr->getHogVector();
    outHogValuesNormalized = outHogValues / math::sum<float>(outHogValues);
  }



  static void extractPhog(const cv::Mat& img,
                          std::unique_ptr<PHOG>& outPhog,
                          cv::Mat& phogValues,
                          int nBins,
                          int nLevels) {
    outPhog.reset(new PHOG(img, nBins, nLevels));
    phogValues = outPhog->getPhogVector();
  }



  static void extractSurf(const cv::SURF& surf,
                          const cv::Mat& img, 
                          std::vector<cv::KeyPoint>& keypoints, 
                          cv::Mat& descriptors,
                          cv::Mat& weights) {
    surf(img, cv::noArray(), keypoints, descriptors);
    weights = cv::Mat(keypoints.size(), 1, CV_32FC1);
    for (size_t i = 0; i < keypoints.size(); ++i)
      weights.at<float>(i) = keypoints[i].response;
  }



  static void extractSift(const cv::SIFT& sift,
                          const cv::Mat& img, 
                          std::vector<cv::KeyPoint>& keypoints, 
                          cv::Mat& descriptors,
                          cv::Mat& weights) {
    sift(img, cv::noArray(), keypoints, descriptors);
    weights = cv::Mat(keypoints.size(), 1, CV_32FC1);
    for (size_t i = 0; i < keypoints.size(); ++i)
      weights.at<float>(i) = keypoints[i].response;
  }



  void extractCanny(const cv::Mat& img,
                    cv::Mat& outEdges,
                    double threshold,
                    int& outCannyEdgesSize,
                    cv::Mat& outDistanceTransform) {
    const double RATIO = 3;
    cv::Canny(img, outEdges, threshold, RATIO * threshold);
    cv::distanceTransform(cv::Scalar::all(255) - outEdges, outDistanceTransform, CV_DIST_L2, 5);
    outCannyEdgesSize = cv::countNonZero(outEdges);
    cv::Mat temp;
    outEdges.convertTo(temp, CV_32FC1);
    outEdges = temp / 255.0;
  }



  std::string HandFitter::featureToString(HandFitter::Feature f) {
    switch(f) {
    case HandFitter::Feature::HOG:
      return "hog";
    case HandFitter::Feature::PHOG:
      return "phog";
    case HandFitter::Feature::CONTOUR:
      return "contour";
    case HandFitter::Feature::CANNY:
      return "canny";
    case HandFitter::Feature::SURF:
      return "surf";
    case HandFitter::Feature::SIFT:
      return "sift";
    case HandFitter::Feature::TRIMMED_HOG:
      return "trimmedhog";
    }

    assert(false && "Should never reach here");
    return "";
  }



  /**
   * Computes the given feature and stores it in the instance
   */
  void HandFitter::computeFeatureForHand(HandFitter::Feature feature,
                                         HandFitter::Instance& instance) {
    switch(feature) {
    case Feature::CONTOUR:
      extractContour(*instance.handBoxBinary,
                     instance.handContour,
                     instance.handContourDistanceTransform,
                     instance.handContourSize);
      break;

    case Feature::HOG:
    case Feature::TRIMMED_HOG:
      extractHog(*instance.handBox, instance.handHog, instance.handHogValuesNormalized, 
                 instance.handHogValues);
      break;

    case Feature::PHOG:
      extractPhog(*instance.handBox, instance.handPhog, instance.handPhogValues,
                  instance.phogBins, instance.phogLevels);
      break;

    case Feature::SIFT:
      extractSift(instance.sift, *instance.handBox, 
                  instance.handSiftKeypoints,
                  instance.handSiftDescriptors,
                  instance.handSiftWeights);
      break;

    case Feature::SURF:
      extractSurf(instance.surf, *instance.handBox, 
                  instance.handSurfKeypoints, 
                  instance.handSurfDescriptors,
                  instance.handSurfWeights);
      break;

    case Feature::CANNY:
      extractCanny(*instance.handBox,
                   instance.handCannyEdges,
                   instance.cannyThreshold,
                   instance.handCannyEdgesSize,
                   instance.handCannyEdgesDistanceTransform);
      break;

    default:
      std::cerr << featureToString(feature) << std::endl;
      assert(false && "NOT IMPLEMENTED YET");
    }
  }



  const std::set< std::pair<HandFitter::Feature, 
                                   HandFitter::DistanceFunction > > 
  HandFitter::VALID_FEATURE_METRIC_COMBINATIONS {
    std::make_pair(HandFitter::Feature::HOG, HandFitter::DistanceFunction::CHI_SQUARED),
    std::make_pair(HandFitter::Feature::HOG, HandFitter::DistanceFunction::EMD),
    std::make_pair(HandFitter::Feature::HOG, HandFitter::DistanceFunction::EMD_HAT),
    std::make_pair(HandFitter::Feature::HOG, HandFitter::DistanceFunction::EUCLIDEAN),

    std::make_pair(HandFitter::Feature::PHOG, HandFitter::DistanceFunction::CHI_SQUARED),
    std::make_pair(HandFitter::Feature::PHOG, HandFitter::DistanceFunction::EUCLIDEAN),

    std::make_pair(HandFitter::Feature::TRIMMED_HOG, HandFitter::DistanceFunction::EMD),
    std::make_pair(HandFitter::Feature::TRIMMED_HOG, HandFitter::DistanceFunction::EMD_HAT),

    std::make_pair(HandFitter::Feature::CONTOUR, HandFitter::DistanceFunction::CHAMFER),
    std::make_pair(HandFitter::Feature::CANNY, HandFitter::DistanceFunction::CHAMFER),

    std::make_pair(HandFitter::Feature::SIFT, HandFitter::DistanceFunction::EMD_EUCLIDEAN_BASE),
    std::make_pair(HandFitter::Feature::SIFT, HandFitter::DistanceFunction::EMD_HAT_EUCLIDEAN_BASE),
    std::make_pair(HandFitter::Feature::SURF, HandFitter::DistanceFunction::EMD_EUCLIDEAN_BASE),
    std::make_pair(HandFitter::Feature::SURF, HandFitter::DistanceFunction::EMD_HAT_EUCLIDEAN_BASE)
  };



  std::string HandFitter::distanceFunctionToString(HandFitter::DistanceFunction f) {
    switch (f) {
    case HandFitter::DistanceFunction::CHAMFER:
      return "chamfer";
    case HandFitter::DistanceFunction::CHI_SQUARED:
      return "chisquared";
    case HandFitter::DistanceFunction::EMD:
      return "emd";
    case HandFitter::DistanceFunction::EMD_HAT:
      return "emdhat";
    case HandFitter::DistanceFunction::EUCLIDEAN:
      return "euclidean";
    case HandFitter::DistanceFunction::EMD_EUCLIDEAN_BASE:
      return "emdeuclideanbase";
    case HandFitter::DistanceFunction::EMD_HAT_EUCLIDEAN_BASE:
      return "emdhateuclideanbase";
    }

    assert(false && "Should never reach here");
    return "";
  }



  double HandFitter::computeDistance(HandFitter::Feature feature,
                                     HandFitter::DistanceFunction metric,
                                     HandFitter::Instance& instance) {
    if (!VALID_FEATURE_METRIC_COMBINATIONS.count(std::make_pair(feature, metric)))
      throw std::invalid_argument(featureToString(feature) + "+" + 
                                  distanceFunctionToString(metric) + " is not"
                                  " a valid feature/metric combination!");

    if (feature == HandFitter::Feature::TRIMMED_HOG &&
        (instance.handTrimmedHog.empty() ||
         instance.handTrimmedHogNormalized.empty() ||
         instance.renderedTrimmedHog.empty() ||
         instance.renderedTrimmedHogNormalized.empty() ||
         instance.handToRenderedTrimmedHogDm.empty())) {
      instance.handHog->generateTrimmedDistanceMatrix(instance.handHog->getCellSize().width,
                                                      instance.trimmedThreshold,
                                                      instance.trimmedMaxSize,
                                                      *instance.renderedHog,
                                                      instance.handToRenderedTrimmedHogDm,
                                                      instance.handTrimmedHog,
                                                      instance.renderedTrimmedHog);
        
      instance.handTrimmedHogNormalized = 
        instance.handTrimmedHog / math::sum<float>(instance.handTrimmedHog);
      instance.renderedTrimmedHogNormalized = 
        instance.renderedTrimmedHog / math::sum<float>(instance.renderedTrimmedHog);
    }


    if (feature == HandFitter::Feature::HOG && 
        (metric == HandFitter::DistanceFunction::EMD ||
         metric == HandFitter::DistanceFunction::EMD_HAT)) {
        if (instance.handToRenderedHogDm.empty())
          instance.handToRenderedHogDm = 
            instance.handHog->generateDistanceMatrix(instance.handHog->getCellSize().width,
                                                     *instance.renderedHog);
    }

    if (feature == HandFitter::Feature::SURF &&
        metric == HandFitter::DistanceFunction::EMD_EUCLIDEAN_BASE) {
      if (instance.handToRenderedSurfL2Dm.empty())
        instance.handToRenderedSurfL2Dm = 
          math::generateDescriptorDmL2(instance.handSurfDescriptors,
                                       instance.renderedSurfDescriptors);
      return cv::EMD(instance.handSurfWeights / math::sum<float>(instance.handSurfWeights),
                     instance.renderedSurfWeights / math::sum<float>(instance.renderedSurfWeights),
                     CV_DIST_USER,
                     instance.handToRenderedSurfL2Dm);
    }     

    if (feature == HandFitter::Feature::SURF &&
        metric == HandFitter::DistanceFunction::EMD_HAT_EUCLIDEAN_BASE) {
      if (instance.handToRenderedSurfL2Dm.empty())
        instance.handToRenderedSurfL2Dm = 
          math::generateDescriptorDmL2(instance.handSurfDescriptors,
                                       instance.renderedSurfDescriptors);
      return math::emdHat<float>(instance.handSurfWeights, 
                                 instance.renderedSurfWeights,
                                 instance.handToRenderedSurfL2Dm);
    }     

    if (feature == HandFitter::Feature::SIFT &&
        metric == HandFitter::DistanceFunction::EMD_EUCLIDEAN_BASE) {
      if (instance.handToRenderedSurfL2Dm.empty())
        instance.handToRenderedSurfL2Dm = 
          math::generateDescriptorDmL2(instance.handSurfDescriptors,
                                       instance.renderedSurfDescriptors);
      return cv::EMD(instance.handSurfWeights / math::sum<float>(instance.handSurfWeights),
                     instance.renderedSurfWeights / math::sum<float>(instance.renderedSurfWeights),
                     CV_DIST_USER,
                     instance.handToRenderedSurfL2Dm);
    }

    if (feature == HandFitter::Feature::SIFT &&
        metric == HandFitter::DistanceFunction::EMD_HAT_EUCLIDEAN_BASE) {
      if (instance.handToRenderedSiftL2Dm.empty())
        instance.handToRenderedSiftL2Dm = 
          math::generateDescriptorDmL2(instance.handSiftDescriptors,
                                       instance.renderedSiftDescriptors);
      return math::emdHat<float>(instance.handSiftWeights, 
                                 instance.renderedSiftWeights,
                                 instance.handToRenderedSiftL2Dm);
    }     

    cv::Mat* P = nullptr, *Q = nullptr, *D = nullptr;

    if (feature == HandFitter::Feature::HOG && metric == HandFitter::DistanceFunction::EMD) {
      P = &instance.handHogValuesNormalized;
      Q = &instance.renderedHogValuesNormalized;
      D = &instance.handToRenderedHogDm;
    }
    else if (feature == HandFitter::Feature::HOG) {
      P = &instance.handHogValues;
      Q = &instance.renderedHogValues;
      D = &instance.handToRenderedHogDm;
    }
    else if (feature == HandFitter::Feature::TRIMMED_HOG && metric == HandFitter::DistanceFunction::EMD) {
      P = &instance.handTrimmedHogNormalized;
      Q = &instance.renderedTrimmedHogNormalized;
      D = &instance.handToRenderedTrimmedHogDm;
    }
    else if (feature == HandFitter::Feature::TRIMMED_HOG) {
      P = &instance.handTrimmedHog;
      Q = &instance.renderedTrimmedHog;
      D = &instance.handToRenderedTrimmedHogDm;
    }
    else if (feature == HandFitter::Feature::PHOG) {
      P = &instance.handPhogValues;
      Q = &instance.renderedPhogValues;    
    }
    else if (feature == HandFitter::Feature::CONTOUR && 
             metric == HandFitter::DistanceFunction::CHAMFER) {
      return math::chamferDistance(instance.handContour,
                                   instance.handContourSize,
                                   instance.handContourDistanceTransform,
                                   instance.renderedContour,
                                   instance.renderedContourSize,
                                   instance.renderedContourDistanceTransform);
    }
    else if (feature == HandFitter::Feature::CANNY && 
             metric == HandFitter::DistanceFunction::CHAMFER) {
      return math::chamferDistance(instance.handCannyEdges,
                                   instance.handCannyEdgesSize,
                                   instance.handCannyEdgesDistanceTransform,
                                   instance.renderedCannyEdges,
                                   instance.renderedCannyEdgesSize,
                                   instance.renderedCannyEdgesDistanceTransform);
    }


    if (!P || !Q) {
      std::cerr << featureToString(feature) << "+" << distanceFunctionToString(metric) << std::endl;
      assert(false && "Not implemented yet!");
    }

    if (metric == HandFitter::DistanceFunction::CHI_SQUARED) {
      return math::chiSqDist(*P, *Q);
    }
    else if (metric == HandFitter::DistanceFunction::EMD) {
      return cv::EMD(*P, *Q, CV_DIST_USER, *D);
    }
    else if (metric == HandFitter::DistanceFunction::EMD_HAT) {
      return math::emdHat<float>(*P, *Q, *D);
    }
    else if (metric == HandFitter::DistanceFunction::EUCLIDEAN) {
      return cv::norm(*P - *Q);
    }

    std::cerr << featureToString(feature) << "+" << distanceFunctionToString(metric) << std::endl;
    assert(false && "Not implemented yet!");
    return 0;
  }

  void HandFitter::computeFeatureForRendered(HandFitter::Feature feature,
                                             HandFitter::Instance& instance) {
    switch (feature) {

    case Feature::CONTOUR:
      extractContour(instance.renderedImg,
                     instance.renderedContour,
                     instance.renderedContourDistanceTransform,
                     instance.renderedContourSize);
      break;

    case Feature::HOG:
      extractHog(instance.renderedImg,
                 instance.renderedHog,
                 instance.renderedHogValuesNormalized,
                 instance.renderedHogValues);
      instance.handToRenderedHogDm = cv::Mat();
      break;

    case Feature::TRIMMED_HOG:
      extractHog(instance.renderedImg,
                 instance.renderedHog,
                 instance.renderedHogValuesNormalized,
                 instance.renderedHogValues);
      instance.handToRenderedTrimmedHogDm = cv::Mat();
      instance.handTrimmedHog = cv::Mat();
      instance.renderedTrimmedHog = cv::Mat();
      instance.handTrimmedHogNormalized = cv::Mat();
      instance.renderedTrimmedHogNormalized = cv::Mat();
      break;

    case Feature::PHOG:
      extractPhog(instance.renderedImg, instance.renderedPhog, 
                  instance.renderedPhogValues, instance.phogBins, 
                  instance.phogLevels);
      break;

    case Feature::SIFT:
      extractSift(instance.sift, instance.renderedImg, 
                  instance.renderedSiftKeypoints,
                  instance.renderedSiftDescriptors,
                  instance.renderedSiftWeights);
      break;

    case Feature::SURF:
      extractSurf(instance.surf, instance.renderedImg, 
                  instance.renderedSurfKeypoints, 
                  instance.renderedSurfDescriptors,
                  instance.renderedSurfWeights);
      break;

    case Feature::CANNY:
      extractCanny(instance.renderedImg,
                   instance.renderedCannyEdges,
                   instance.cannyThreshold,
                   instance.renderedCannyEdgesSize,
                   instance.renderedCannyEdgesDistanceTransform);
      break;

    default:
      assert(false && "NOT IMPLEMENTED YET");
    }
  }



  void HandFitter::updateBestPose(double initialTheta,
                                  double initialPhi,
                                  double initialTilt,
                                  double initialR,
                                  HandFitter::Instance& instance,
                                  HandFitter::Feature scanFeature,
                                  HandFitter::DistanceFunction scanMetric,
                                  HandFitter::Optimization optimization,
                                  const HandConfiguration& hc,
                                  HandVisualiser& hv) {
    double cost = DBL_MAX;
    double theta = initialTheta;
    double phi = initialPhi;
    double tilt = initialTilt;
    double r = initialR;
    double tx = 0;
    double ty = 0;
    instance.renderedImg = hv.render(hc, r, theta, phi, tilt);
    computeFeatureForRendered(scanFeature, instance);

    cv::Mat pose(6, 1, CV_64FC1, cv::Scalar::all(0));
    pose.at<double>(0) = r;
    pose.at<double>(1) = theta;
    pose.at<double>(2) = phi;
    pose.at<double>(3) = tilt;

    struct CostFunction {
      double operator()(const cv::Mat& x) {
        instancePtr->renderedImg = hvPtr->render(*hcPtr, x.at<double>(0), x.at<double>(1),
                                                 x.at<double>(2), x.at<double>(3),
                                                 x.at<double>(4), x.at<double>(5));
        computeFeatureForRendered(scanFeature, *instancePtr);
        return computeDistance(scanFeature, scanMetric, *instancePtr);
      }

      HandFitter::Instance* instancePtr;
      HandFitter::Feature scanFeature;
      HandFitter::DistanceFunction scanMetric;
      HandVisualiser* hvPtr;
      const HandConfiguration* hcPtr;
    } costFunction;
    costFunction.instancePtr = &instance;
    costFunction.scanFeature = scanFeature;
    costFunction.scanMetric = scanMetric;
    costFunction.hvPtr = &hv;
    costFunction.hcPtr = &hc;

    auto grad = math::gradient(costFunction, 6);

    struct LbfgsInstance {
      decltype(&grad) gradPtr;
      CostFunction* costFunctionPtr;
      cv::Mat* posePtr;
    } lbfgsInstance;
    lbfgsInstance.gradPtr = &grad;
    lbfgsInstance.costFunctionPtr = &costFunction;
    lbfgsInstance.posePtr = &pose; 

    auto lbfgsEvaluate = [](void *instance,
                            const lbfgsfloatval_t *x,
                            lbfgsfloatval_t *g,
                            const int n,
                            const lbfgsfloatval_t /* step */)->lbfgsfloatval_t {
      assert(n == 6);
      LbfgsInstance* lbfgsInstance = reinterpret_cast<LbfgsInstance*>(instance);
      for (int i = 0; i < n; ++i)
        lbfgsInstance->posePtr->at<double>(i) = x[i];
      cv::Mat G = lbfgsInstance->gradPtr->eval(*lbfgsInstance->posePtr);
      for (int i = 0; i < n; ++i)
        g[i] = G.at<double>(i);
      return (*lbfgsInstance->costFunctionPtr)(*lbfgsInstance->posePtr);
    };

    cost = costFunction(pose);

    if (optimization == HandFitter::Optimization::NONE) {
      // do nothing
    }
    else if (optimization == HandFitter::Optimization::GRADIENT_DESCENT) {
      double gamma = 1.0;

      while (gamma > 0.001) {
        gamma = 1.0;
        cv::Mat G = grad.eval(pose);

        cv::Mat newPose = pose - gamma * G;
        double newCost = costFunction(newPose);
        while (newCost > cost) {
          gamma *= 0.5;
          newPose = pose - gamma * G;
          newCost = costFunction(newPose);
        }
        
        pose = newPose;
        cost = newCost;
        gamma = std::min(gamma, cv::norm(G));
      }
    }
    else if (optimization == HandFitter::Optimization::LBFGS) {
      static const int N = 6;
      lbfgsfloatval_t *x = lbfgs_malloc(N);
      for (int i = 0; i < pose.rows; ++i)
        x[i] = pose.at<double>(i);
      lbfgsfloatval_t fx = FLT_MAX;
      lbfgs_parameter_t param;
      lbfgs_parameter_init(&param);
      param.linesearch = LBFGS_LINESEARCH_BACKTRACKING_STRONG_WOLFE;
      param.epsilon = 1.0;

      lbfgs(N, x, &fx, lbfgsEvaluate, nullptr/*progress*/, &lbfgsInstance, &param);
      // int ret = lbfgs(N, x, &fx, lbfgsEvaluate, nullptr/*progress*/, &lbfgsInstance, &param);
      // std::cerr << "lbfgs return value: " << lbfgsRetValToString(ret) << std::endl;
      cost = fx;
    }
    else
      assert (false && "NOT IMPLEMENTED YET");

    r = pose.at<double>(0);
    theta = pose.at<double>(1);
    phi = pose.at<double>(2);
    tilt = pose.at<double>(3);
    tx = pose.at<double>(4);
    ty = pose.at<double>(5);

    if (cost < instance.bestCost) {
      instance.bestCost = cost;
      instance.bestTheta = theta;
      instance.bestPhi = phi;
      instance.bestTilt = tilt;
      instance.bestR = r;
      instance.bestTx = tx;
      instance.bestTy = ty;
    } 
  }


  void HandFitter::process(frame_number_t fn) {
    auto frame = getFrameSource()[fn];
    auto skinmask = getBlackBoard().get<cv::Mat>(fn, SKINDETECTOR_BLACKBOARD_MASK_ENTRY);

    auto bb = getBlackBoard().get<cv::Rect>(fn, HANDLOCATOR_BLACKBOARD_ENTRY);
    cv::Rect paddedBb = padBy50PercentAndAlign16(*bb);
    // prevent overflow
    if (paddedBb.x < 0) {
      paddedBb.x = 0;
    }
    if (paddedBb.y < 0) {
      paddedBb.y = 0;
    }
    
    cv::Mat handBox = frame(paddedBb).clone();
    cv::Mat handBoxBinary = (*skinmask)(paddedBb).clone();
    for (int i = 0; i < handBoxBinary.rows; ++i)
      for (int j = 0; j < handBoxBinary.cols; ++j)
        if (!bb->contains(cv::Point(j + paddedBb.x,
                                    i + paddedBb.y)))
          handBoxBinary.at<uchar>(i,j) = 0;

    HandVisualiser hv(handBox.size());
    HandConfiguration hc(initialConfiguration);

    Instance instance;
    instance.handBox = &handBox;
    instance.handBoxBinary = &handBoxBinary;
    instance.trimmedThreshold = trimmedThreshold;
    instance.trimmedMaxSize = trimmedMaxSize;
    instance.phogLevels = phogLevels;
    instance.phogBins = phogBins;
    instance.cannyThreshold = cannyThreshold;

    // std::set<Feature> allFeatures = finalFeatures;
    // allFeatures.insert(scanFeature);
    // for (Feature f : allFeatures) {
    //   computeFeatureForHand(f, instance);
    // }
    computeFeatureForHand(scanFeature, instance);

    double step = math::PI/4.0;
    double minTheta = math::PI;
    double maxTheta = 2.0*math::PI + step;
    double minPhi = -math::PI;
    double maxPhi = math::PI;
    double minTilt = -math::PI;
    double maxTilt = math::PI;


    std::set<std::string> blackboardKeys = std::set<std::string>();

    timerOn();
    for (double theta = minTheta; theta < maxTheta; theta += step) {
      for (double phi = minPhi; phi < maxPhi; phi += step) {
        for (double tilt = minTilt; tilt < maxTilt; tilt += step) {
          double r = determineInitialR(handBoxBinary, hv, hc, theta, phi, tilt);
          updateBestPose(theta, phi, tilt, r, instance, scanFeature, scanMetric,
                         optimization, hc, hv);

          /*
          cv::imshow("", sideBySide(*instance.handBox,
                                    hv.render(hc, instance.bestR, instance.bestTheta,
                                              instance.bestPhi, instance.bestTilt)));
          cv::waitKey(20);
          */
        }
      }
    }
    blackboardKeys.insert("handfitter optimization time");
    getBlackBoard().set("handfitter optimization time", timerOff());
    std::cerr << "Optimization took " << timerOff() << " ms" << std::endl;

    instance.renderedImg = hv.render(hc, instance.bestR, instance.bestTheta,
                                     instance.bestPhi, instance.bestTilt);
    blackboardKeys.insert(HAND_FITTER_RENDERED_IMG_ENTRY);
    getBlackBoard().set(fn, HAND_FITTER_RENDERED_IMG_ENTRY, instance.renderedImg.clone());
    blackboardKeys.insert(HAND_FITTER_HAND_BOX_IMG_ENTRY);
    getBlackBoard().set(fn, HAND_FITTER_HAND_BOX_IMG_ENTRY, instance.handBox->clone());
    blackboardKeys.insert(HAND_FITTER_HAND_BOX_BINARY_IMG_ENTRY);
    getBlackBoard().set(fn, HAND_FITTER_HAND_BOX_BINARY_IMG_ENTRY, instance.handBoxBinary->clone());

    blackboardKeys.insert("handfitter bestr");
    getBlackBoard().set(fn, "handfitter bestr", instance.bestR);
    blackboardKeys.insert("handfitter besttheta");
    getBlackBoard().set(fn, "handfitter besttheta", instance.bestTheta);
    blackboardKeys.insert("handfitter bestphi");
    getBlackBoard().set(fn, "handfitter bestphi", instance.bestPhi);
    blackboardKeys.insert("handfitter besttilt");
    getBlackBoard().set(fn, "handfitter besttilt", instance.bestTilt);
    blackboardKeys.insert("handfitter besttx");
    getBlackBoard().set(fn, "handfitter besttx", instance.bestTx);
    blackboardKeys.insert("handfitter bestty");
    getBlackBoard().set(fn, "handfitter bestty", instance.bestTy);

    return;
#if 0
    cv::Mat skinmaskBox = (*skinmask)(*bb);

    cv::HOGDescriptor hog;
    std::vector<float> inputHogs;
    hog.compute(handBox, inputHogs);
    cv::Mat dblHogsInput(inputHogs.size(),1,CV_64FC1);
    for (int i = 0; i < dblHogsInput.rows; ++i)
        dblHogsInput.at<double>(i) = inputHogs[i];

    cv::Mat renderedImg;
    std::vector<float> renderedHogs;
    cv::Mat dblHogsRendered(inputHogs.size(),1,CV_64FC1);
    
    //const int N = 29;
    const int N = 6;

    cv::SURF surf;
    std::vector<cv::KeyPoint> inputSurfKeypoints;
    surf(handBox, skinmaskBox, inputSurfKeypoints);
    // cv::Mat temp = handBox.clone();
    // float rMin = FLT_MAX, rMax = 0;
    // for (cv::KeyPoint& k : inputSurfKeypoints) {
    //   if (k.response > rMax)
    //     rMax = k.response;
    //   if (k.response < rMin)
    //     rMin = k.response;
    // }
    // float rRange = rMax - rMin;
    // for (cv::KeyPoint& k : inputSurfKeypoints) {
    //   int radius = 1 + (k.response - rMin)/rRange*10.0 + 0.5;
    //   cv::circle(temp, k.pt, radius, cv::Scalar::all(255));
    // }
    // cv::imshow("", temp);
    // cv::waitKey(0);

    std::vector<double> inputPointCloud(inputSurfKeypoints.size());
    for (size_t i = 0; i < inputSurfKeypoints.size(); ++i)
      inputPointCloud[i] = inputSurfKeypoints[i].response;

    auto surfEmdDistance = [&](const cv::Mat& m) {
      renderedImg = hv.render(hc, m.at<double>(0), m.at<double>(1), 
                              m.at<double>(2), m.at<double>(3),
                              m.at<double>(3), m.at<double>(4));
      std::vector<cv::KeyPoint> renderedSurfKeypoints;
      surf(renderedImg, cv::Mat(), renderedSurfKeypoints);
      
      std::vector<double> renderedPointCloud(renderedSurfKeypoints.size());
      for (size_t i = 0; i < renderedSurfKeypoints.size(); ++i)
        renderedPointCloud[i] = renderedSurfKeypoints[i].response;
      
      cv::Mat surfDistanceMatrix(renderedPointCloud.size(), inputPointCloud.size(), CV_64FC1);
      for (int i = 0; i < surfDistanceMatrix.rows; ++i)
        for (int j = 0; j < surfDistanceMatrix.cols; ++j)
          surfDistanceMatrix.at<double>(i,j) = 
            cv::norm(renderedSurfKeypoints[i].pt - inputSurfKeypoints[i].pt);

      assert (false && "Does not work");
      // return math::emdGreedy(renderedPointCloud, inputPointCloud, surfDistanceMatrix);
      return 0.0;
    };

    auto hogDistance = [&](const cv::Mat& m) {
      assert(m.type() == CV_64FC1);
      assert(m.cols == 1);
      // hc = m.rowRange(0,25);
      // renderedImg = hv.render(hc, 
      //                         m.at<double>(25),
      //                         m.at<double>(26),
      //                         m.at<double>(27),
      //                         m.at<double>(28));
      assert(m.rows == 1 || m.rows == 4 || m.rows == 29);
      if (m.rows == 1)
        renderedImg = hv.render(hc, 
                                m.at<double>(0),
                                0, 0, 0);
      else if (m.rows == 4) {
        renderedImg = hv.render(hc, 
                                m.at<double>(0),
                                m.at<double>(1),
                                m.at<double>(2),
                                m.at<double>(3));
      }
      else if (m.rows == 29) {
        hc = m.rowRange(0,25);
        renderedImg = hv.render(hc, 
                                m.at<double>(25),
                                m.at<double>(26),
                                m.at<double>(27),
                                m.at<double>(28));

      }
      
      hog.compute(renderedImg, renderedHogs);
      assert(static_cast<int>(renderedHogs.size()) == dblHogsRendered.rows);
      for (int i = 0; i < dblHogsRendered.rows; ++i)
        dblHogsRendered.at<double>(i) = renderedHogs[i];
      cv::Mat diffM = dblHogsRendered - dblHogsInput;
      cv::Mat val = diffM.t()*diffM;
      assert(val.cols == 1 && val.rows == 1);
      return val.at<double>(0,0);
    };
#endif
#if 0
    cv::Mat inputDistanceTransform;
    cv::Mat skinMaskOuterContour = extractOuterContour(skinmaskBox);
    cv::Mat skinMaskOuterContourFloat;
    skinMaskOuterContour.convertTo(skinMaskOuterContourFloat, CV_32FC1);
    skinMaskOuterContourFloat /= 255.0;

    cv::distanceTransform(cv::Scalar::all(1.0) - skinMaskOuterContour, 
                          inputDistanceTransform, CV_DIST_L2, 5);

    int inputContourSize = countNonZero(skinMaskOuterContour);

    // double maxDistance;
    // cv::minMaxLoc(inputDistanceTransform, nullptr, &maxDistance);
    auto chamferDistance = [&](const cv::Mat& x) {
      assert(x.rows == 6 && x.cols == 1 && x.type() == CV_64FC1);
      // don't accept NaNs
      if (cv::countNonZero(cv::Mat(x != x)))
        return std::numeric_limits<double>::infinity();
      // cv::Mat x = deNormaliseConfig(X, handBox.size());
      // no hand configuration at this point
      renderedImg = hv.render(hc, x.at<double>(0),
                              x.at<double>(1), x.at<double>(2), 
                              x.at<double>(3), x.at<double>(4), 
                              x.at<double>(5));
      cv::Mat renderedDistanceTransform;
      cv::Mat renderedOuterContour = extractOuterContour(renderedImg);
      cv::Mat renderedOuterContourFloat;
      renderedOuterContour.convertTo(renderedOuterContourFloat, CV_32FC1);
      renderedOuterContourFloat /= 255.0;
      cv::distanceTransform(cv::Scalar::all(1.0) - renderedOuterContour, 
                            renderedDistanceTransform, CV_DIST_L2, 5);

      int renderedContourSize = countNonZero(renderedOuterContour);
      
      double inputToRenderedDcd = 
      directedChamferDistance(skinMaskOuterContourFloat, 
                              inputContourSize,
                              renderedDistanceTransform);
      double renderedToInputDcd = 
      directedChamferDistance(renderedOuterContourFloat, 
                              renderedContourSize,
                              inputDistanceTransform);
      return inputToRenderedDcd + renderedToInputDcd;
    };

    cv::Mat config(N, 1, CV_64FC1, cv::Scalar::all(0));
    config.at<double>(0) = 5;
    config.at<double>(2) = math::PI;
    cv::imshow("", hv.render(hc, config.at<double>(0),
                             config.at<double>(1), config.at<double>(2), 
                             config.at<double>(3), config.at<double>(4), 
                             config.at<double>(5)));
    // config = normaliseConfig(config, handBox.size());

    // auto G = math::gradient(chamferDistance, N, 0.1);
    auto Gchamfer = handGradient(chamferDistance);
    auto GsurfEmd = handGradient(surfEmdDistance);

    cv::Mat bestConfiguration;
    double bestObjectiveFunctionValue = DBL_MAX;
    for (double theta = -3.1415; theta < 3.1415+3.1415/4.0; theta += 3.1415/4.0) {// 45) {
      for (double phi = -3.1415; phi < 3.1415; phi += 3.1415/4.0) {
        for (double tilt = -3.1415; tilt < 3.1415; tilt += 3.1415/4.0) {
          double initialR = 1.0;
          bool borderPixels = true;
          while (borderPixels) {
            borderPixels = false;
            renderedImg = hv.render(hc, 
                                    initialR,
                                    theta,
                                    phi,
                                    tilt);
            for (int i : { 0, renderedImg.rows-1}) {
              for (int j = 0; j < renderedImg.cols; ++j) {
                const cv::Vec3b& v = renderedImg.at<cv::Vec3b>(i,j);
                if (v[0] != 0 || v[1] != 0 || v[2] != 0)
                  borderPixels = true;
              }
            }
            for (int j : { 0, renderedImg.cols-1}) {
              for (int i = 0; i < renderedImg.rows; ++i) {
                const cv::Vec3b& v = renderedImg.at<cv::Vec3b>(i,j);
                if (v[0] != 0 || v[1] != 0 || v[2] != 0)
                  borderPixels = true;
              }
            }
            if (borderPixels)
              ++initialR;
          }

          std::cerr << "initialR: " << initialR << std::endl;
          // cv::Mat config(N, 1, CV_64FC1, cv::Scalar::all(0));
          //config.at<double>(25,0) = 5;
          //config.at<double>(26,0) = theta; //-90;
          //config.at<double>(27,0) = phi; // 0;
          //config.at<double>(28,0) = tilt; //-90;
          config.at<double>(0) = initialR;
          config.at<double>(1) = theta;
          config.at<double>(2) = phi;
          config.at<double>(3) = tilt;
          config.at<double>(4) = 0;
          config.at<double>(5) = 0;

          // auto G = math::gradient(hogDistance, 29);
          // auto G = math::gradient(hogDistance, N);
          auto& G = GsurfEmd;
          // auto G = math::gradient(colorDiffDistance, N);
          // auto H = math::hessian(hogDistance, N, true);
          auto& distance = surfEmdDistance;
          std::cerr << "Initial theta: " << theta << std::endl;
          std::cerr << "Initial phi: " << phi << std::endl;
          std::cerr << "Initial tilt: " << tilt << std::endl;
          std::cerr << "Initial config:" << std::endl
                    << config << std::endl;
          double oldDistance = distance(config);
          while (true) {
            // auto Heval = H.eval(config);
            // std::cerr << "Hessian:" << std::endl
            //           << Heval << std::endl;
            std::cerr << "Evaluating gradient at " << config << std::endl;
            auto Geval = G(config);
            std::cerr << "Gradient evaluated: " << Geval << std::endl;
            double gamma = 1.0 / cv::norm(Geval);
            cv::Mat newConfig = config - gamma*Geval;
            std::cerr << "Estimating gamma. Initial newConfig: " 
                      << newConfig << std::endl;
            double newDistance = distance(newConfig);
            while(newDistance > oldDistance && gamma > 0) {
              gamma /= 10.0;
              if (gamma > 0) {
                newConfig = config - gamma*Geval;
                newDistance = distance(newConfig);
                assert(newConfig.at<double>(0) > 0);
              }
            }
            std::cerr << "Estimated gamma = " << gamma << std::endl;
            config = newConfig;
            oldDistance = newDistance;
      
            std::cerr << "Gradient:" << std::endl
                      << Geval << std::endl;
            std::cerr << "Config:" << std::endl
                      << config << std::endl;
            std::cerr << "Objective function value: " <<
              oldDistance << " (best: " << bestObjectiveFunctionValue << ")" 
                      <<  std::endl;
            std::cerr << "gamma = " << gamma << std::endl;
            if (oldDistance < bestObjectiveFunctionValue) {
              bestConfiguration = config.clone();
              bestObjectiveFunctionValue = oldDistance;

              cv::Mat temp = handBox.clone();
              for (int i = 0; i < temp.rows; ++i) {
                for (int j = 0; j < temp.cols; ++j) {
                  const cv::Vec3b& v = renderedImg.at<cv::Vec3b>(i,j);
                  if (v[0] != 0 || v[1] != 0 || v[2] != 0)
                    temp.at<cv::Vec3b>(i,j) = v;
                }
              }
              cv::imshow("", temp);
              cv::waitKey(20);
            }

            if (gamma < 1e-13)
              break;
          }
        }
      }
    }
    return;
#endif

#if 0    
    typedef decltype(GsurfEmd) surfEmdGradType;
    typedef decltype(Gchamfer) chamferGradType;
    // typedef decltype(hogDistance) hogDistanceType;
    typedef decltype(chamferDistance) chamferDistanceType;
    typedef decltype(surfEmdDistance) surfEmdDistanceType;


    instance.config = &config;
    instance.renderedImg = &renderedImg;
    instance.Gchamfer = &Gchamfer;
    instance.chamferDistance = &chamferDistance;
    instance.handBox = &handBox;
    instance.GsurfEmd = &GsurfEmd;
    instance.surfEmdDistance = &surfEmdDistance;

    auto evaluateChamfer = [](void *instance,
                              const lbfgsfloatval_t *x,
                              lbfgsfloatval_t *g,
                              const int n,
                              const lbfgsfloatval_t /* step */)->lbfgsfloatval_t {
      //std::cerr << "evaluate function called" << std::endl;
      Instance* inst = reinterpret_cast<Instance*>(instance);
      assert(n == inst->config->rows);
      assert(inst->config->cols == 1);
      assert(n > 0);
      assert(inst->config->type() == CV_64FC1);
      for (int i = 0; i < n; ++i)
        inst->config->at<double>(i) = x[i];
      std::cerr << "config:" << std::endl << *inst->config << std::endl;
      // cv::Mat gradientValue = inst->G->eval(*inst->config);
      cv::Mat gradient = (*inst->Gchamfer)(*inst->config);
      assert(gradient.size() == inst->config->size() && 
             gradient.type() == inst->config->type());
      std::cerr << "gradient:" << std::endl << gradient << std::endl;
      for (int i = 0; i < n; ++i)
        g[i] = gradient.at<double>(i);
      double val = (*inst->chamferDistance)(*inst->config);
      std::cerr << "val = " << val << std::endl;
      return val;
    };

    auto evaluateSurfEmd = [](void *instance,
                              const lbfgsfloatval_t *x,
                              lbfgsfloatval_t *g,
                              const int n,
                              const lbfgsfloatval_t /*step*/)->lbfgsfloatval_t {
      Instance* inst = reinterpret_cast<Instance*>(instance);
      assert(n == inst->config->rows);
      assert(inst->config->cols == 1);
      assert(n > 0);
      assert(inst->config->type() == CV_64FC1);
      for (int i = 0; i < n; ++i)
        inst->config->at<double>(i) = x[i];
      std::cerr << "Evaluate called" << std::endl;
      std::cerr << "config:" << std::endl << *inst->config << std::endl;
      cv::Mat gradient = (*inst->GsurfEmd)(*inst->config);
      assert(gradient.size() == inst->config->size() && 
             gradient.type() == inst->config->type());
      std::cerr << "gradient:" << std::endl << gradient << std::endl;
      for (int i = 0; i < n; ++i)
        g[i] = gradient.at<double>(i);
      // double val = (*inst->chamferDistance)(*inst->config);
      double val = (*inst->surfEmdDistance)(*inst->config);
      cv::imshow("", *inst->renderedImg);
      cv::waitKey(10);
      std::cerr << "val = " << val << std::endl;
      return val;
    };

    auto progress = [](void *instance,
                       const lbfgsfloatval_t *x,
                       const lbfgsfloatval_t */*g*/,
                       const lbfgsfloatval_t fx,
                       const lbfgsfloatval_t xnorm,
                       const lbfgsfloatval_t gnorm,
                       const lbfgsfloatval_t step,
                       int n,
                       int k,
                       int /*ls*/)->int
    {
      Instance* inst = reinterpret_cast<Instance*>(instance);
      std::cerr << "Iteration " << k << ":" << std::endl;
      std::cerr << "x = [" << x[0];;
      for (int i = 1; i < n; ++i) {
        std::cerr << ", " << x[i];
      }
      std::cerr << "]" << std::endl;
      std::cerr << "fx = " << fx << std::endl;
      std::cerr << "xnorm = " << xnorm << std::endl;
      std::cerr << "gnorm = " << gnorm << std::endl;
      std::cerr << "step = " << step << std::endl;
      cv::Mat temp(inst->handBox->rows, inst->handBox->cols*2, CV_8UC3);
      cv::Mat left = temp(cv::Rect(0, 0, inst->handBox->cols, inst->handBox->rows));
      inst->handBox->copyTo(left);
      cv::Mat right = temp(cv::Rect(inst->handBox->cols, 0, inst->handBox->cols, inst->handBox->rows));
      inst->renderedImg->copyTo(right);
      cv::imshow("", temp);
      cv::waitKey(0);
      return 0;
    };

    lbfgsfloatval_t *x = lbfgs_malloc(N);
    for (int i = 0; i < N; ++i)
      x[i] = config.at<double>(i);
    lbfgsfloatval_t fx;
    lbfgs_parameter_t param;
    lbfgs_parameter_init(&param);
    // param.xtol = 1.0;
    // param.min_step = 1e-10;
    // param.linesearch = LBFGS_LINESEARCH_BACKTRACKING_ARMIJO;
    // param.ftol = 1e-3;
    // param.gtol = 0.1;
    param.linesearch = LBFGS_LINESEARCH_BACKTRACKING_STRONG_WOLFE;
    param.epsilon = 1.0;
    param.max_linesearch = 1000;


    auto lbfgsRetValToString = [](int r)->std::string {
      if (r == LBFGS_SUCCESS)
        return "LBFGS_SUCCESS";
      else if (r == LBFGS_CONVERGENCE)
        return "LBFGS_CONVERGENCE";
      else if (r == LBFGS_STOP)
        return "LBFGS_STOP";
      else if (r == LBFGS_ALREADY_MINIMIZED)
        return "LBFGS_ALREADY_MINIMIZED";
      else if (r == LBFGSERR_UNKNOWNERROR)
        return "LBFGSERR_UNKNOWNERROR";
      else if (r == LBFGSERR_LOGICERROR)
        return "LBFGSERR_LOGICERROR";
      else if (r == LBFGSERR_OUTOFMEMORY)
        return "LBFGSERR_OUTOFMEMORY";
      else if (r == LBFGSERR_CANCELED)
        return "LBFGSERR_CANCELED";

      else if (r == LBFGSERR_INVALID_N)
        return "LBFGSERR_INVALID_N";
      else if (r == LBFGSERR_INVALID_N_SSE)
        return "LBFGSERR_INVALID_N_SSE";
      else if (r == LBFGSERR_INVALID_X_SSE)
        return "LBFGSERR_INVALID_X_SSE";
      else if (r == LBFGSERR_INVALID_EPSILON)
        return "LBFGSERR_INVALID_EPSILON";

      else if (r == LBFGSERR_INVALID_TESTPERIOD)
        return "LBFGSERR_INVALID_TESTPERIOD";
      else if (r == LBFGSERR_INVALID_DELTA)
        return "LBFGSERR_INVALID_DELTA";
      else if (r == LBFGSERR_INVALID_LINESEARCH)
        return "LBFGSERR_INVALID_LINESEARCH";
      else if (r == LBFGSERR_INVALID_MINSTEP)
        return "LBFGSERR_INVALID_MINSTEP";

      else if (r == LBFGSERR_INVALID_MAXSTEP)
        return "LBFGSERR_INVALID_MAXSTEP";
      else if (r == LBFGSERR_INVALID_FTOL)
        return "LBFGSERR_INVALID_FTOL";
      else if (r == LBFGSERR_INVALID_WOLFE)
        return "LBFGSERR_INVALID_WOLFE";
      else if (r == LBFGSERR_INVALID_GTOL)
        return "LBFGSERR_INVALID_GTOL";

      else if (r == LBFGSERR_INVALID_XTOL)
        return "LBFGSERR_INVALID_XTOL";
      else if (r == LBFGSERR_INVALID_MAXLINESEARCH)
        return "LBFGSERR_INVALID_MAXLINESEARCH";
      else if (r == LBFGSERR_INVALID_ORTHANTWISE)
        return "LBFGSERR_INVALID_ORTHANTWISE";
      else if (r == LBFGSERR_INVALID_ORTHANTWISE_START)
        return "LBFGSERR_INVALID_ORTHANTWISE_START";

      else if (r == LBFGSERR_INVALID_ORTHANTWISE_END)
        return "LBFGSERR_INVALID_ORTHANTWISE_END";
      else if (r == LBFGSERR_OUTOFINTERVAL)
        return "LBFGSERR_OUTOFINTERVAL";
      else if (r == LBFGSERR_INCORRECT_TMINMAX)
        return "LBFGSERR_INCORRECT_TMINMAX";
      else if (r == LBFGSERR_ROUNDING_ERROR)
        return "LBFGSERR_ROUNDING_ERROR";

      else if (r == LBFGSERR_MINIMUMSTEP)
        return "LBFGSERR_MINIMUMSTEP";
      else if (r == LBFGSERR_MAXIMUMSTEP)
        return "LBFGSERR_MAXIMUMSTEP";
      else if (r == LBFGSERR_MAXIMUMLINESEARCH)
        return "LBFGSERR_MAXIMUMLINESEARCH";
      else if (r == LBFGSERR_MAXIMUMITERATION)
        return "LBFGSERR_MAXIMUMITERATION";

      else if (r == LBFGSERR_WIDTHTOOSMALL)
        return "LBFGSERR_WIDTHTOOSMALL";
      else if (r == LBFGSERR_INVALIDPARAMETERS)
        return "LBFGSERR_INVALIDPARAMETERS";
      else if (r == LBFGSERR_INCREASEGRADIENT)
        return "LBFGSERR_INCREASEGRADIENT";
      else
        return "UNKNOWN ERROR";
    };

    cv::Mat bestConfiguration(N, 1, CV_64FC1);
    double bestObjectiveFunctionValue = DBL_MAX;
    for (double theta = -3.1415; theta < 3.1415+3.1415/4.0; theta += 3.1415/4.0) {
      for (double phi = -3.1415; phi < 3.1415; phi += 3.1415/4.0) {
        for (double tilt = -3.1415; tilt < 3.1415; tilt += 3.1415/4.0) {
          config.at<double>(0) = 5.0;
          config.at<double>(1) = theta;
          config.at<double>(2) = phi;
          config.at<double>(3) = tilt;
          config.at<double>(4) = 0.0;
          config.at<double>(5) = 0.0;
          for (int i = 0; i < N; ++i)
            x[i] = config.at<double>(i);

          int ret = lbfgs(N, x, &fx, evaluateChamfer, nullptr/*progress*/, &instance, &param);
          std::cerr << "lbfgs return value: " << lbfgsRetValToString(ret) << std::endl;
          if (fx < bestObjectiveFunctionValue) {
            for (int i = 0; i < N; ++i)
              bestConfiguration.at<double>(i) = x[i];
            bestObjectiveFunctionValue = fx;
            cv::imshow("", hv.render(hc, bestConfiguration.at<double>(0),
                                     bestConfiguration.at<double>(1),
                                     bestConfiguration.at<double>(2),
                                     bestConfiguration.at<double>(3),
                                     bestConfiguration.at<double>(4),
                                     bestConfiguration.at<double>(5)));
            cv::waitKey(20);
          }
          std::cerr << "fx = " << fx << " (best: " << bestObjectiveFunctionValue << ")" << std::endl;
        }
      }
    }
    cv::waitKey(0);
    lbfgs_free(x);
#endif
  }




  static HandFitter::Feature stringToFeature(const std::string& s) {
      if (s == "hog")
        return HandFitter::Feature::HOG;
      else if (s == "phog")
        return HandFitter::Feature::PHOG;
      else if (s == "contour")
        return HandFitter::Feature::CONTOUR;
      else if (s == "canny")
        return HandFitter::Feature::CANNY;
      else if (s == "surf")
        return HandFitter::Feature::SURF;
      else if (s == "sift")
        return HandFitter::Feature::SIFT;
      else if (s == "trimmedhog")
        return HandFitter::Feature::TRIMMED_HOG;
      else
        throw std::invalid_argument(s + " is not a valid feature!");
      return HandFitter::Feature::CONTOUR;
  }



  static HandFitter::DistanceFunction stringToDistanceFunction(const std::string& s) {
    if (s == "chamfer")
      return HandFitter::DistanceFunction::CHAMFER;
    else if (s == "chisquared")
      return HandFitter::DistanceFunction::CHI_SQUARED;
    else if (s == "emd")
      return HandFitter::DistanceFunction::EMD;
    else if (s == "emdhat")
      return HandFitter::DistanceFunction::EMD_HAT;
    else if (s == "euclidean")
      return HandFitter::DistanceFunction::EUCLIDEAN;
    else
      throw std::invalid_argument(s + " is not a valid metric!");
    return HandFitter::DistanceFunction::CHAMFER;
  }



  static HandFitter::Optimization stringToOptimization(const std::string& s) {
    if (s == "none")
      return HandFitter::Optimization::NONE;
    else if (s == "gradientdescent")
      return HandFitter::Optimization::GRADIENT_DESCENT;
    else if (s == "lbfgs")
      return HandFitter::Optimization::LBFGS;
    else
      throw std::invalid_argument(s + " is not a valid optimization scheme!");
    return HandFitter::Optimization::NONE;
  }



  Component* HandFitter::createComponentImpl(const boost::program_options::variables_map& vm, 
                                             BlackBoard* blackBoard, FrameSource* frameSource) const {
    HandFitter hf(blackBoard, frameSource);
    
    if (vm.count("HandFitter.config"))
      hf.initialConfiguration = HandConfiguration::fromFile(vm["HandFitter.config"].as<std::string>());
    else
      std::cerr << "Warning: No hand configuration set. Using default configuration." << std::endl;

    if (vm.count("HandFitter.scanfeature")) 
      hf.scanFeature = stringToFeature(vm["HandFitter.scanfeature"].as<std::string>());

    if (vm.count("HandFitter.scanmetric")) 
      hf.scanMetric = stringToDistanceFunction(vm["HandFitter.scanmetric"].as<std::string>());

    if (vm.count("HandFitter.optimization")) 
      hf.optimization = stringToOptimization(vm["HandFitter.optimization"].as<std::string>());

    if (vm.count("HandFitter.trimmedthreshold"))
      hf.trimmedThreshold = vm["HandFitter.trimmedthreshold"].as<double>();

    if (vm.count("HandFitter.trimmedmaxsize"))
      hf.trimmedMaxSize = vm["HandFitter.trimmedmaxsize"].as<int>();

    if (vm.count("HandFitter.phoglevels"))
      hf.phogLevels = vm["HandFitter.phoglevels"].as<int>();

    if (vm.count("HandFitter.phogbins"))
      hf.phogBins = vm["HandFitter.phogbins"].as<int>();

    if (vm.count("HandFitter.cannythreshold"))
      hf.cannyThreshold = vm["HandFitter.cannythreshold"].as<double>();

    return new HandFitter(hf);
  }



  boost::program_options::options_description HandFitter::getConfigurationFileOptionsDescription() const {
    boost::program_options::options_description opts;
    opts.add_options()
      ("HandFitter.config",
       boost::program_options::value<std::string>()->default_value(""),
       "Filename of the configuration to load on start-up")
      ("HandFitter.scanfeature",
       boost::program_options::value<std::string>()->default_value(""),
       "Feature used for scanning the best configuration")
      ("HandFitter.scanmetric",
       boost::program_options::value<std::string>()->default_value(""),
       "Metric used for scanning the best configuration")
      ("HandFitter.optimization",
       boost::program_options::value<std::string>()->default_value(""),
       "Metric used for scanning the best configuration")
      ("HandFitter.trimmedthreshold",
       boost::program_options::value<double>()->default_value(1.0),
       "Threshold of how much the remaining values need to explain about the "
       "mass of the histogram when computing trimmed hog")
      ("HandFitter.trimmedmaxsize",
       boost::program_options::value<int>()->default_value(INT_MAX),
       "Absolute maximum size of trimmed histograms")
      ("HandFitter.phoglevels",
       boost::program_options::value<int>()->default_value(3),
       "Number of levels in the pyramidal HOG")
      ("HandFitter.phogbins",
       boost::program_options::value<int>()->default_value(20),
       "Number of bins in the pyramidal HOG")
      ("HandFitter.cannythreshold",
       boost::program_options::value<double>()->default_value(100),
       "Canny edge detector lower threshold");
    return opts;
  }
}
#endif // SLMOTION_WITH_LIBLBFGS
#endif // SLMOTION_ENABLE_LIBHAND
