#ifndef SLMOTION_HAND_FITTER
#define SLMOTION_HAND_FITTER

#include "HandLocator.hpp"
#include "HandConfiguration.hpp"
#include "HOGDescriptor.hpp"
#include "PHOG.hpp"
#include "HandVisualiser.hpp"

#include <opencv2/nonfree/nonfree.hpp>

namespace slmotion {
  const std::string HAND_FITTER_RENDERED_IMG_ENTRY = "handfitter best rendered";
  const std::string HAND_FITTER_HAND_BOX_IMG_ENTRY = "handfitter handbox";
  const std::string HAND_FITTER_HAND_BOX_BINARY_IMG_ENTRY = "handfitter handbox binary";



  /**
   * Attempts to locate the bounding box of the hand on the basis of
   * skin blobs detected
   */
  class HandFitter : public Component {
  public:
    HandFitter(bool) : Component(true) { }

    HandFitter(BlackBoard* blackBoard, FrameSource* frameSource) :
      Component(blackBoard, frameSource),
      scanFeature(Feature::CONTOUR),
      scanMetric(DistanceFunction::CHAMFER),
      optimization(Optimization::NONE),
      trimmedThreshold(1.0),
      trimmedMaxSize(INT_MAX),
      phogLevels(3),
      phogBins(20),
      cannyThreshold(100)
    { }

    virtual void process(frame_number_t frameNumber);

    virtual std::string getShortDescription() const {
      return "Hand Fitter";
    }

    virtual std::string getLongDescription() const {
      return "Tries to fit the hand to the image and evaluates if there is "
        "some level of similarity between the input image and the given "
        "abstract hand configuration.";
    }

    virtual property_set_t getRequirements() const {
      return property_set_t { HANDLOCATOR_BLACKBOARD_ENTRY };
    }

    virtual property_set_t getProvided() const {
      return property_set_t {
        HAND_FITTER_RENDERED_IMG_ENTRY,
          HAND_FITTER_HAND_BOX_IMG_ENTRY,
          HAND_FITTER_HAND_BOX_BINARY_IMG_ENTRY
          };
    }

    virtual std::string getShortName() const {
      return "HandFitter";
    }

    virtual std::string getComponentName() const {
      return "Hand Fitter";
    }

    virtual void reset() {
    }

    virtual boost::program_options::options_description getConfigurationFileOptionsDescription() const;

    virtual boost::program_options::options_description getCommandLineOptionsDescription() const {
      return boost::program_options::options_description();
    }

    enum class DistanceFunction : uint64_t {
      CHAMFER = 1,
      CHI_SQUARED = 2,
      EMD = 4,
      EMD_HAT = 8,
      EUCLIDEAN = 16,
      EMD_EUCLIDEAN_BASE = 32,
      EMD_HAT_EUCLIDEAN_BASE = 64
    };

    enum class Feature : uint64_t {
      HOG = 1,
      PHOG = 2,
      CONTOUR = 4,
      CANNY = 8,
      SURF = 16,
      SIFT = 32,
      TRIMMED_HOG = 64
    };

    enum class Optimization : uint64_t {
      NONE = 1,
      LBFGS = 2,
      GRADIENT_DESCENT = 4
    };



  static const std::set< std::pair<Feature, 
                                   DistanceFunction > > 
  VALID_FEATURE_METRIC_COMBINATIONS;



    static std::string featureToString(HandFitter::Feature f);

    static std::string distanceFunctionToString(HandFitter::DistanceFunction f);

    struct Instance {
      Instance() : 
        handBox(nullptr),
        handBoxBinary(nullptr),
        handContourSize(0),
        handCannyEdgesSize(0),
        renderedContourSize(0),
        renderedCannyEdgesSize(0),
        bestR(10),
        bestTheta(0),
        bestPhi(0),
        bestTilt(0),
        bestTx(0),
        bestTy(0),
        bestCost(DBL_MAX),
        trimmedThreshold(1.0),
        trimmedMaxSize(INT_MAX),
        phogLevels(3),
        phogBins(20),
        cannyThreshold(0)
      {}

      cv::Mat* handBox;
      cv::Mat* handBoxBinary;
      cv::Mat handContour;
      int handContourSize;
      cv::Mat handContourDistanceTransform;
      std::unique_ptr<HOGDescriptor> handHog;
      cv::Mat handHogValues;
      cv::Mat handHogValuesNormalized;
      cv::Mat handTrimmedHog;
      cv::Mat handTrimmedHogNormalized;
      std::unique_ptr<PHOG> handPhog;
      cv::Mat handPhogValues;
      std::vector<cv::KeyPoint> handSiftKeypoints;
      cv::Mat handSiftWeights;
      cv::Mat handSiftDescriptors;
      std::vector<cv::KeyPoint> handSurfKeypoints;
      cv::Mat handSurfDescriptors;
      cv::Mat handSurfWeights;
      cv::Mat handCannyEdges;
      int handCannyEdgesSize;
      cv::Mat handCannyEdgesDistanceTransform;

      cv::Mat renderedImg;
      cv::Mat renderedContour;
      int renderedContourSize;
      cv::Mat renderedContourDistanceTransform;
      std::unique_ptr<HOGDescriptor> renderedHog;
      cv::Mat renderedHogValues;
      cv::Mat renderedHogValuesNormalized;
      cv::Mat renderedTrimmedHog;
      cv::Mat renderedTrimmedHogNormalized;
      std::unique_ptr<PHOG> renderedPhog;
      cv::Mat renderedPhogValues;
      std::vector<cv::KeyPoint> renderedSiftKeypoints;
      cv::Mat renderedSiftDescriptors;
      cv::Mat renderedSiftWeights;
      std::vector<cv::KeyPoint> renderedSurfKeypoints;
      cv::Mat renderedSurfDescriptors;
      cv::Mat renderedSurfWeights;
      cv::Mat renderedCannyEdges;
      int renderedCannyEdgesSize;
      cv::Mat renderedCannyEdgesDistanceTransform;

      cv::Mat handToRenderedHogDm;
      cv::Mat handToRenderedTrimmedHogDm;
      cv::Mat handToRenderedSiftL2Dm;
      cv::Mat handToRenderedSurfL2Dm;

      cv::SIFT sift;
      cv::SURF surf;

      double bestR;
      double bestTheta;
      double bestPhi;
      double bestTilt;
      double bestTx;
      double bestTy;
      double bestCost;
      
      /// OPTIONS
      double trimmedThreshold; // threshold for trimmed hog
      size_t trimmedMaxSize;
      size_t phogLevels;
      size_t phogBins;
      double cannyThreshold;
    };



    static void computeFeatureForHand(HandFitter::Feature feature,
                                      HandFitter::Instance& instance);



    static void computeFeatureForRendered(HandFitter::Feature feature,
                                          HandFitter::Instance& instance);



    static double computeDistance(HandFitter::Feature feature,
                                  HandFitter::DistanceFunction metric,
                                  HandFitter::Instance& instance);



  private:
virtual Component* createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const;



    static void updateBestPose(double initialTheta,
                               double initialPhi,
                               double initialTilt,
                               double initialR,
                               HandFitter::Instance& instance,
                               HandFitter::Feature scanFeature,
                               HandFitter::DistanceFunction scanMetric,
                               HandFitter::Optimization optimization,
                               const HandConfiguration& hc,
                               HandVisualiser& hv);



    HandConfiguration initialConfiguration;
    Feature scanFeature;
    DistanceFunction scanMetric;
    Optimization optimization;
    double trimmedThreshold; // threshold for trimmed hog
    size_t trimmedMaxSize;
    size_t phogLevels;
    size_t phogBins;
    double cannyThreshold;
  };
}

#endif

