#include "HandLocator.hpp"
#include "Blob.hpp"

namespace slmotion {
  static HandLocator DUMMY(true);

  void HandLocator::process(frame_number_t fn) {
    cv::Rect face = *getBlackBoard().get<cv::Rect>(fn, FACEDETECTOR_BLACKBOARD_ENTRY);
    cv::Point faceCentroid = 0.5 * (face.tl() + face.br());

    auto blobs = getBlackBoard().get<std::vector<Blob> >(fn, BLACKBOARD_BLOBS_ENTRY);
    const Blob* currentBlob = nullptr;
    for (const Blob& blob : *blobs) {
      cv::Rect boundingBox = blob.getBoundingBox();

      // ignore the face
      if (boundingBox.contains(faceCentroid))
        continue;

      // select the "leftmost" blob (corresponding to the right hand
      // unless the hands are crossed)
      if (currentBlob == nullptr || 
          (!mirror && blob.getCentroid().x < currentBlob->getCentroid().x) ||
          (mirror && blob.getCentroid().x > currentBlob->getCentroid().x))
        currentBlob = &blob;
    }

    cv::Rect bb;
    if (currentBlob) {
      bb = currentBlob->getBoundingBox();
      if (padding > 0) {
        bb.x -= padding;
        bb.y -= padding;
        bb.width += padding*2;
        bb.height += padding*2;
      }
    }
    else
      bb = cv::Rect();
    getBlackBoard().set(fn, HANDLOCATOR_BLACKBOARD_ENTRY, bb);
  }



  Component* HandLocator::createComponentImpl(const boost::program_options::variables_map& vm, BlackBoard* blackBoard, FrameSource* frameSource) const {
    int padding = 0;
    int mirror = false;
    if (vm.count("HandLocator.padding"))
      padding = vm["HandLocator.padding"].as<int>();

    if (vm.count("HandLocator.mirror"))
      mirror = vm["HandLocator.mirror"].as<bool>();

    return new HandLocator(blackBoard, frameSource, padding, mirror);
  }



  boost::program_options::options_description HandLocator::getConfigurationFileOptionsDescription() const {
    boost::program_options::options_description opts;
    opts.add_options()
      ("HandLocator.padding",
       boost::program_options::value<int>()->default_value(0),
       "Add n pixels of padding to the resulting rectangle")
      ("HandLocator.mirror",
       boost::program_options::value<bool>()->default_value(false),
       "If true, mirrors the handedness assumption");
    return opts;
  }
}
