#ifdef SLMOTION_WITH_LIBSVM
#include "HeadPoseEstimator.hpp"
#include "SVM.h"

#define DUMP_DATA 1

namespace slmotion {
  extern int debug;
  static HeadPoseEstimator DUMMY(true);

  HeadPoseEstimator::~HeadPoseEstimator() {
    if (svmYawModel)
      svm_free_and_destroy_model(&svmYawModel);
    if (svmPitModel)
      svm_free_and_destroy_model(&svmPitModel);
  }

  /**
   * Loads the model for the constructor
   */  
  struct svm_model* getSvmModel(std::string m) {
    struct svm_model *svmModel;
    if ((svmModel = svm_load_model(m.c_str()))==0) {
      std::cerr << "Failed to load model file " << m << std::endl;
      exit(1);
    }
    return svmModel;
  }

  /**
   * Translates a vector into a SVM node of sparse representation of the original vector
   */
  static svm_node** vectorToSvmNode(std::vector<double> v) {
    //  std::vector<std::vector<double>> range
    int rowSize = 1;  // delete to allow expandability in case needed later
    if (rowSize > 0) {
      int colSize = v.size(); // v[0] for multirow
      // if ((unsigned) colSize != range[0].size()) {
      //   std::cout << "Ranges are not aligned with data" << std::endl;
      //   return NULL;
      // }
      svm_node** node = new svm_node*[rowSize];
      for(int row = 0; row < rowSize; ++row) {
        node[row] = new svm_node[colSize + 1];
      }

      for(int row = 0; row < rowSize; ++row) {
        for (int col = 0; col<colSize; ++col) {
          node[row][col].index = col+1;
          // node[row][col].value = (((v[col]-range[0][col])/(range[1][col]-range[0][col]))-0.5)*2; //v[row][col]
          node[row][col].value = v[col];
        }
        node[row][colSize].index = -1;
      }

      return node;
    } else {
      return NULL;
    }
  }

  /**
   * Parses a 1D comma separated filter into a cv::Mat struct
   */
  static std::vector<double> parseFilter(std:: string f) {
    std::vector<double> y;
    std::stringstream x(f);
   
    while( x.good() ) {
      std::string substr;
      getline( x, substr, ',' );
      y.push_back(atof(substr.c_str()));
    }

    return y;
  }

  /**
   * Filters data in vector X with filter in vector B and A. 1D version of the "Direct Form II Transposed"
   * port from http://mechatronics.ece.usu.edu/yqchen/filter.c/
   */
  static std::vector<double> filter1D(std::vector<double> B, std::vector<double> A, std::vector<double> X) {
    int order = std::max(A.size(),B.size());
    int N = X.size();

    if (order > N) {
      std::cerr << "Filter can not be larger than data vector" << std::endl;
      return X;
    }

    std::vector<double> y(N);
    
    y.at(0) = B.at(0)*X.at(0);
    for (int i=1; i<order; i++){
      y.at(i) = 0;
      for (int j=0; j<i+1; j++)
        y.at(i) = y.at(i) + B.at(j)*X.at(i-j);
      for (int j=0; j<i; j++)
        y.at(i) = y.at(i) - A.at(j+1)*y.at(i-j-1);
    }
    for (int i=order; i<N; i++){
      y.at(i) =0;
      for (int j=0; j<order; j++)
        y.at(i) = y.at(i) + B.at(j)*X.at(i-j);
      for (int j=0; j<order-1; j++)
        y.at(i) = y.at(i) - A.at(j+1)*y.at(i-j-1);
    }
    return y;
  }

  void HeadPoseEstimator::process(frame_number_t frameNumber) {
    BlackBoardPointer<cv::Rect> facePtr = getBlackBoard().get<cv::Rect>(frameNumber, FACEDETECTOR_BLACKBOARD_ENTRY);
    BlackBoardPointer<std::vector<cv::Point2f>> pointsPtr = getBlackBoard().get<std::vector<cv::Point2f>>(frameNumber, FACIAL_LANDMARK_BLACKBOARD_ENTRY);
    BlackBoardPointer<cv::Mat> maskPtr { getBlackBoard().get<cv::Mat>(frameNumber, SKINDETECTOR_BLACKBOARD_MASK_ENTRY) };
    cv::Rect face = *facePtr;
    std::vector<cv::Point2f>& points = *pointsPtr;

    if (!nativePrecision) {
      for (int i=0; i<8; i++) {
        double tmpx = ceil((points[i].x * pow(10,2) ) -0.499999999999999999999999) / pow(10,2);
        double tmpy = ceil((points[i].y * pow(10,2) ) -0.499999999999999999999999) / pow(10,2);
        points.push_back(cv::Point2f(tmpx, tmpy));
      }  
      points.erase(points.begin(),points.begin()+8);
    }
    
    cv::Mat maskFull = *maskPtr;

    double dim, mL, mR, mT, mB = 0;
    dim = face.width;
    cv::Mat mask;
    mask = maskFull(face);
    cv::Point center;

    center.x = ceil(points[0].x - face.x);
    center.y = ceil(points[0].y - face.y);

    mL = cv::countNonZero(255-mask(cv::Range(0,dim), cv::Range(0,center.x)))/dim;
    mR = cv::countNonZero(255-mask(cv::Range(0,dim), cv::Range(center.x,dim)))/dim;
    mT = cv::countNonZero(255-mask(cv::Range(0,center.y), cv::Range(0,dim)))/dim;
    mB = cv::countNonZero(255-mask(cv::Range(center.y,dim), cv::Range(0,dim)))/dim;

    std::vector<double> yawRawVec, pitRawVec;
    double rolRawVec;
    for (int i=0; i<7; i++) {
      yawRawVec.push_back((points[i+1].x-face.x-mL+mR)/dim);
      yawRawVec.push_back((points[i+1].y-face.y-mT+mB)/dim);
    }
    for (int i=0; i<8; i++) {
      pitRawVec.push_back((points[i].x-face.x)/dim);
      pitRawVec.push_back((points[i].y-face.y)/dim);
    }
    pitRawVec.push_back(mL/dim);
    pitRawVec.push_back(mR/dim);
    pitRawVec.push_back(mT/dim);
    pitRawVec.push_back(mB/dim);

    cv::Point2f emR, emL;
    emR.x = (yawRawVec[0]+yawRawVec[8])/2;
    emR.y = (yawRawVec[1]+yawRawVec[9])/2;
    emL.x = (yawRawVec[2]+yawRawVec[10])/2;
    emL.y = (yawRawVec[3]+yawRawVec[11])/2;
    rolRawVec = atan( ((emR.y-emL.y)/(emR.x-emL.x)) )*(180/M_PI);

    svm_node** yawFeatureSpa = vectorToSvmNode(yawRawVec);
    svm_node** pitFeatureSpa = vectorToSvmNode(pitRawVec);


    double labelYaw = svm_predict(svmYawModel, yawFeatureSpa[0]);
    double labelPit = svm_predict(svmPitModel, pitFeatureSpa[0]);

    std::vector<double> v = {labelYaw*15,labelPit*15,-rolRawVec};
    getBlackBoard().set(frameNumber, HEAD_POSE_BLACKBOARD_ENTRY, v );
  }

  bool HeadPoseEstimator::processRangeImplementation(frame_number_t first, frame_number_t last, UiCallback* uiCallback) {
    if (!(svm_get_svm_type(svmYawModel) == 3) && !(svm_get_svm_type(svmPitModel) == 3)) {
      std::cerr << "Model check failed" << std::endl;
      return false;
    }

    for (size_t i = first; i < last; ++i) {
      if (slmotion::debug > 0) {
        if (slmotion::debug > 1)
          std::cout << std::endl;
      }
      this->process(i);
      if (uiCallback != nullptr && !(*uiCallback)((i-first+1)*100./(last-first)))
        return false;
    }

    if (slmotion::debug > 0)
      std::cout << std::endl << "Filtering (finite impulse response)..." << std::endl;

    // read back blackboard entry for post processing
    BlackBoardPointer<std::vector<double>> rPtr;
    std::vector<double> yawVec, pitVec, rolVec;
    for (unsigned int i=0; i<last; i++) {
      rPtr = getBlackBoard().get<std::vector<double>>(i,HEAD_POSE_BLACKBOARD_ENTRY);
      std::vector<double>& r = *rPtr;
      yawVec.push_back(r[0]);
      pitVec.push_back(r[1]);
      rolVec.push_back(r[2]);
    }

    // read filter vector and build new filtered values
    std::vector<double> yawF, pitF, rolF;
    yawF = parseFilter(yawFilter);
    pitF = parseFilter(pitFilter);
    rolF = parseFilter(rolFilter);
        
    std::vector<double> yawFinal, pitFinal, rolFinal;
    std::vector<double> yA(yawF.size()), pA(pitF.size()), rA(rolF.size());
    yA[0] = pA[0] = rA[0] = 1;

    yawFinal = filter1D(yawF, yA, yawVec);
    pitFinal = filter1D(pitF, pA, pitVec);
    rolFinal = filter1D(rolF, rA, rolVec);
    
    for (unsigned int i=0; i<last; i++) {
      std::vector<double> v = {yawFinal[i], pitFinal[i], rolFinal[i]};
      getBlackBoard().set(i, HEAD_POSE_BLACKBOARD_ENTRY, v);
      if (DUMP_DATA)
        printf("%2d\t %2.5f\t %2.5f\t %2.5f\n",i, v[0], v[1], v[2]);
    }

  	return true;
  }

  Component::property_set_t HeadPoseEstimator::getRequirements() const {
    property_set_t props {FACEDETECTOR_BLACKBOARD_ENTRY};
    props.insert(FACIAL_LANDMARK_BLACKBOARD_ENTRY);
    props.insert(SKINDETECTOR_BLACKBOARD_MASK_ENTRY);
    return props;
  }

  boost::program_options::options_description HeadPoseEstimator::getConfigurationFileOptionsDescription() const {
    boost::program_options::options_description opts;
    opts.add_options()
      ("HeadPoseEstimator.nativePrecision", boost::program_options::value<bool>()->default_value(DEFAULT_NATIVE_PRECISION),
       "Sets the precision type for facial landmarks. true: float coordinates (more than 4 decimals), false: coordinates rounded at 2 decimals.")
      ("HeadPoseEstimator.yawModel", boost::program_options::value<string>()->default_value(DEFAULT_YAW_MODEL),
       "Sets the libSVM yaw model file address. A valid model file is required for the classifier to work properly.")
      ("HeadPoseEstimator.pitModel", boost::program_options::value<string>()->default_value(DEFAULT_PIT_MODEL),
       "Sets the libSVM pitch model file address. A valid model file is required for the classifier to work properly.")
      ("HeadPoseEstimator.yawFilter", boost::program_options::value<string>()->default_value(DEFAULT_YAW_FILTER),
       "Sets the postprocessing windowed FIR filter for the yaw estimations.")
      ("HeadPoseEstimator.pitFilter", boost::program_options::value<string>()->default_value(DEFAULT_PIT_FILTER),
       "Sets the postprocessing windowed FIR filter for the pitch estimations.")
      ("HeadPoseEstimator.rolFilter", boost::program_options::value<string>()->default_value(DEFAULT_ROL_FILTER),
       "Sets the postprocessing windowed FIR filter for the roll estimations.");
    return opts;
  }

  Component* HeadPoseEstimator::createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const { 
    HeadPoseEstimator* headPoseEstimator = new HeadPoseEstimator(blackBoard, frameSource);

    if (configuration.count("HeadPoseEstimator.nativePrecision")) {
      headPoseEstimator->setNativePrecision(configuration["HeadPoseEstimator.nativePrecision"].as<bool>());
    }
    if (configuration.count("HeadPoseEstimator.yawModel")) {
      headPoseEstimator->setYawModel(configuration["HeadPoseEstimator.yawModel"].as<string>());
    }
    if (configuration.count("HeadPoseEstimator.pitModel")) {
      headPoseEstimator->setPitModel(configuration["HeadPoseEstimator.pitModel"].as<string>());
    }
    if (configuration.count("HeadPoseEstimator.yawFilter")) {
      headPoseEstimator->setYawFilter(configuration["HeadPoseEstimator.yawFilter"].as<string>());
    }
    if (configuration.count("HeadPoseEstimator.pitFilter")) {
      headPoseEstimator->setPitFilter(configuration["HeadPoseEstimator.pitFilter"].as<string>());
    }
    if (configuration.count("HeadPoseEstimator.rolFilter")) {
      headPoseEstimator->setRolFilter(configuration["HeadPoseEstimator.rolFilter"].as<string>());
    }
    return headPoseEstimator;
  }
}
#endif // SLMOTION_WITH_LIBSVM
