#ifndef SLMOTION_FEATURETRACKER
#define SLMOTION_FEATURETRACKER

#include <set>
#include "Component.hpp"
#include "exceptions.hpp"
#include "TrackedPoint.hpp"
#include "SkinDetector.hpp"
#include "ColourSpaceConverter.hpp"
#include "BodyPartCollector.hpp"



namespace slmotion {
  const std::string KLTTRACKER_BLACKBOARD_TRACKED_POINTS_ENTRY = "klttrackedpoints";
  // std::set<TrackedPoint>



  /**
   * A pyramidial Lucas-Kanade sparse optical flow tracker, tracking 
   * features selected using the Shi-Tomasi method.
   *
   * The state is preserved by default over three consecutive frames.
   */
  class KLTTracker : public Component {
  public:
    virtual std::string getShortDescription() const {
      return "A Kanade-Lukas-Tomasi interest point tracker";
    }

    virtual std::string getLongDescription() const {
      return "A Kanade-Lukas-Tomasi interest point tracker. Detects "
        "Shi-Tomasi corner features, and tracks them using a pyramidal "
        "Lucas-Kanade tracker.";
    }

    /**
     * The default constructor. Constructs the tracker with default values.
     */
    KLTTracker(BlackBoard* blackBoard, FrameSource* frameSource) :
      Component(blackBoard, frameSource),
      maxFrameError(1000),
      removeNonMaskedFeatures(true),
      qualityLevel(0.01),
      minDistance(3),
      maxMove(30),
      maxPoints(1000)
    {};



    explicit KLTTracker(bool);



    virtual std::string getComponentName() const {
      return "Kanade-Lucas-Tomasi Tracker";
    }

    virtual std::string getShortName() const {
      return "KLTTracker";
    }

    /**
     * If the tracker has not been initialised, initialise the tracker
     * Otherwise, track points.
     *
     * Greyscale copies of frames must exist on the black board.
     */
    virtual void process(frame_number_t frameNumber);


  
    /**
     * Setter for the max points variable
     *
     * @param newMax New max points
     */
    inline void setMaxPoints(int newMax) {
      maxPoints = std::max(newMax,0);
    }



    /// Trivial setter
    inline void setRemoveNonMaskedFeatures(bool b) {
      removeNonMaskedFeatures = b;
    }



    /// Trivial setter
    inline void setMaxFrameError(double d) {
      maxFrameError = d;
    }



    /// Trivial setter
    inline void setQualityLevel(double d) {
      qualityLevel = d;
    }



    /// Trivial setter
    inline void setMinDistance(double d) {
      minDistance = d;
    }



    /// Trivial setter
    inline void setMaxMove(int i) {
      maxMove = i;
    }



    inline virtual property_set_t getProvided() const {
      return property_set_t { KLTTRACKER_BLACKBOARD_TRACKED_POINTS_ENTRY };
    }



    inline virtual property_set_t getRequirements() const {
      return property_set_t { SKINDETECTOR_BLACKBOARD_MASK_ENTRY,
          COLOURSPACECONVERTER_BLACKBOARD_GSIMAGE_ENTRY,
          BODYPARTCOLLECTOR_BLACKBOARD_ENTRY };
    }

    virtual void reset() {}

    virtual boost::program_options::options_description getConfigurationFileOptionsDescription() const;

    virtual boost::program_options::options_description getCommandLineOptionsDescription() const {
      return boost::program_options::options_description();
    }

  private:
    Component* createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const;



    /**
     * A simple implementation that forces sequentiality (and forcefully 
     * disables the multithreaded default implementation)
     */
    virtual bool processRangeImplementation(frame_number_t first, 
                                            frame_number_t last, 
                                            UiCallback* uiCallback);



    /**
     * Sets body part identities for the given frame point set, returns 
     * an updated copy
     */
    std::set<TrackedPoint> updateBodyPartIdentities(const std::set<TrackedPoint>& trackedPoints, frame_number_t frnumber);



    /**
     * Initialises the tracker by performing Shi-Tomasi feature detection
     *
     * @param gsframe Input greyscale frame
     * @param mask Input mask (skin detection result)
     *
     * @return Initial point set
     */
    std::set<TrackedPoint> initialise(const cv::Mat& gsframe, const cv::Mat& mask);



    /**
     * Assuming the detector has been initialised properly, use the 
     * pyramidial LK algorithm to track features
     *
     * @param frame Input frame (greyscale)
     * @param previousFrame Previous input frame (greyscale)
     * @param oldPoints Points from the previous frame
     * @param mask Input binary mask
     */
    std::set<TrackedPoint> track(const cv::Mat& frame,
                                 const cv::Mat& previousFrame,
                                 const std::set<TrackedPoint>& oldPoints,
                                 const cv::Mat& mask);



    /**
     * Attempts to locate corners in the greyscale image within the area of
     * the mask if the mask is set (i.e. it is not empty)
     *
     * @param gsimg Greyscale input image
     * @param mask Binary mask, or an empty matrix
     *
     * @return a vector of detected corners
     */
    std::vector<cv::Point2f> findCorners(const cv::Mat& gsImg, 
                                         const cv::Mat& mask);



    /**
     * Attempts to rediscover lost points, and removes points that have 
     * become invalid in some respect
     *
     * @param img should be a (greyscale) frame from a later point in time.
     *
     * @param points should be a current array of points that are being
     * tracked.
     *
     * @param featuresFound should be an array created by
     * calcOpticalFlowPyrLK where indices match those of points and each 
     * lost point is marked by a 0 and others non-zero.
     *
     * @param errors Should be be a vector of floats as created by 
     * calcOpticalFlowPyrLK that represents tracking error in the given 
     * frame
     *
     * @param mask Should be a binary mask (possibly created by the skin
     * detector) that is used to mask away undesired pixels when selecting
     * good features to track.
     */
    void purgeAndReplacePoints(const cv::Mat& img, 
                               std::vector<cv::Point2f>& points, 
                               std::vector<uchar>& featuresFound,
                               std::vector<float>& errors,
                               const cv::Mat& mask = cv::Mat());



    /**
     * Removes points that are too near one another, marking them zero
     * in featuresFound
     *
     * @param corners A vector of corners
     * @param featuresFound For each corresponding corner, mark 1 if the
     * corner should be preserved, or 0 if it should be deemed lost
     * @param errors Error terms as produced by calcOpticalFlowPyrLK
     * @param mask An optional binary mask; if the mask is zero for the 
     * given point, it is removed.
     */
    void removeInvalidPoints(std::vector<cv::Point2f>& corners,
                             std::vector<uchar>& featuresFound,
                             std::vector<float>& errors,
                             const cv::Mat& = cv::Mat());



    double maxFrameError; ///< Reject any tracked points whose track_error (as calculated by cvCalcOpticalFlowPyrLK) is greater than this threshold

    bool removeNonMaskedFeatures; ///< If enabled, features that have travelled outside the boundaries of the skin mask will be removed

    double qualityLevel; ///< quality level for finding good features
    double minDistance;  ///< minimum distance between good features

    int maxMove; ///< how many pixels the feature may move between successive frames
    int maxPoints; ///< This variable is here because it has constraints (ie. it must be positive and less than or equal to ST_MAX_POINTS_UPPERBOUND)
  };



  /**
   * A very simple class to signal exceptional events during tracking
   */
  class KLTTrackerException : public SLMotionException {
  public:
    /**
     * A very simple constructor
     *
     * @param msg An explanatory message
     */
    explicit KLTTrackerException(const char* msg) throw() :
      SLMotionException(msg)
    {}

    virtual ~KLTTrackerException() throw() {}
  };
}
#endif
