#include "KinectExampleComponent.hpp"
#include "configuration.hpp"

using cv::Mat;

namespace slmotion {
  /**
   * This will cause the component to be registered.
   */
  static KinectExampleComponent DUMMY(true);



  boost::program_options::options_description KinectExampleComponent::getConfigurationFileOptionsDescription() const {
    // The ConfigurationFileOptionsDescription is a simple wrapper around
    // the boost::program_options::options_description class. Basically, 
    // just define the options in the usual manner, then add a section 
    // description giving some information about the component.

    boost::program_options::options_description opts;
    opts.add_options()
      ("KinectExample.constant", 
       boost::program_options::value<double>()->default_value(1.0),
       "This is a constant by which each pixel is multiplied");
    // notice the syntax: section.key
    // in the configuration file, the option will be set like this:
    // [KinectExample]
    // constant=1.0
    
    return opts;
  }



  void KinectExampleComponent::process(frame_number_t frameNumber) {
    // operator[] is used for accessing frames
    Mat depthImg = getFrameSource().getTrack(1)[frameNumber];

    // Check whether the depth data is from ONI or ordinary video file
    // (elements differ)
    assert(depthImg.type() == CV_8UC3 || depthImg.type() == CV_16UC1);

    if (depthImg.type() == CV_8UC3) {   
      // convert to greyscale
      Mat temp;
      cv::cvtColor(depthImg, temp, CV_BGR2GRAY);
      temp.convertTo(depthImg, CV_64FC1);
      assert(depthImg.type() == CV_64FC1);
    }
    else if (depthImg.type() == CV_16UC1) {
      Mat temp;
      depthImg.convertTo(temp, CV_64FC1);
      depthImg = temp;
    }

    // normalise the data
    double minVal, maxVal;
    cv::minMaxLoc(depthImg, &minVal, &maxVal);
    double diff = maxVal - minVal;
    
    // manipulate data
    assert(getFrameSource().getDefaultTrackNumber() == 0);
    // the operator[] on a multi-track frame source should always return
    // the frame from the default track
    Mat frame = getFrameSource()[frameNumber];

    assert(frame.type() == CV_8UC3);
    for (int i = 0; i < frame.rows; ++i) {
      for (int j = 0; j < frame.cols; ++j) { 
        for (int k = 0; k < 3; ++k) {
          uchar b = frame.at<cv::Vec3b>(i, j)[k];
          b *= (depthImg.at<double>(i, j) - minVal) * (constant / diff);
          frame.at<cv::Vec3b>(i, j)[k] = b;
        }
      }
    }

    // finally, store the result on the black board
    getBlackBoard().set(frameNumber, "KinectEntry", frame);
  }



  Component* KinectExampleComponent::createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const {
    KinectExampleComponent newComponent(blackBoard, frameSource);
    if (configuration.count("KinectExample.constant"))
      newComponent.setConstant(configuration["KinectExample.constant"].as<double>());
    return new KinectExampleComponent(newComponent);
  }
}

