#include "MainWindow.hpp"
#include "AdvancedString.hpp"

#include "SingletonWindow.hpp"
#include "GraphicalBlackBoardEditor.hpp"
#include "slmotion.hpp"
#include "DummyFrameSource.hpp"
#include "BlackBoardDumpReader.hpp"
#include "BlackBoardDumpWriter.hpp"
#include "configuration.hpp"
#include "ConfigurationWindow.hpp"

#include <string>

#include <sys/types.h>
#include <pwd.h>
#include <sys/stat.h>
#include <unistd.h>

#if defined(SLMOTION_WITH_GTK)
#include <gtk/gtk.h>
#elif defined(SLMOTION_WITH_QT)
#include <QApplication>
#include <QGridLayout>
#include <QStringListModel>
#include <QCheckBox>
#include <QMenuBar>
#include <QStatusBar>
#include <QMessageBox>
#include <QFileDialog>
#endif 

#include <numeric>

using std::vector;
using std::string;

namespace slmotion {
  namespace gui {

    static const std::string HOME_DIR = getpwuid(getuid())->pw_dir;
    static const std::string CONFIGURATION_SUBDIRNAME = ".slmotion";
    static const std::string DEFAULT_CONF_FILE = "default.conf";
    static const std::string DEFAULT_CONF_FILE_FULL_PATH = HOME_DIR + "/" + 
      CONFIGURATION_SUBDIRNAME + "/" + DEFAULT_CONF_FILE;



#if defined(SLMOTION_WITH_GTK) || defined(SLMOTION_WITH_QT)
    static std::vector<char> stringToCharArray(const std::string& s) {
      std::vector<char> c(s.length()+1); 
      strcpy(&c[0], s.c_str());
      return c;
    }



    static std::vector<std::vector<char> > stringArrayToCharArrayArray(const std::vector<std::string>& strings) {
      std::vector<std::vector<char> > charArrays(strings.size());
      for (size_t i = 0; i < strings.size(); ++i)
        charArrays[i] = stringToCharArray(strings[i]);
      return charArrays;
    }
#endif



#if defined(SLMOTION_WITH_GTK)
    using std::vector;
    using std::string;

    /**
     * parameters to pass to the "main" function
     */
    struct Params {
      GtkTreeView* inputFilename;
      GtkFileChooser* scriptFilename;
      GtkEntry* outVideoFile;
      GtkEntry* outCsvFile;
      GtkFileChooser* annotationTemplateFile;
      GtkEntry* outElanFile;
      GtkStatusbar* statusBar;
      char* argv0;
      GtkCheckButton* blackBoardClearCheckButton;
      GtkWindow* mainWindow;
      std::string configFile;
      std::shared_ptr<slmotion::BlackBoard> bb;
    };



    static vector<string> generateCmdOpts(struct Params* p) {
      vector<string> argvStrings;

      GtkTreeSelection* selection = gtk_tree_view_get_selection(p->inputFilename);
      gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);
      gtk_tree_selection_select_all(selection);
      if (gtk_tree_selection_count_selected_rows(selection) > 0) {
        gtk_tree_selection_selected_foreach(selection,
                                            [](GtkTreeModel* model,
                                               GtkTreePath*,
                                               GtkTreeIter* iter,
                                               gpointer data) {
                                              gchar *value;
                                              gtk_tree_model_get (model, iter, 0, &value, -1);
                                              reinterpret_cast<std::vector<std::string>*>(data)->push_back(value);
                                              g_free(value);
                                            },
                                            &argvStrings);
      }

      if (GFile* f = gtk_file_chooser_get_file(p->scriptFilename)) {
        char* path = g_file_get_path(f);
        argvStrings.push_back("--script");
        argvStrings.push_back(path);
        g_free(path);
        g_object_unref(f);
      }

      const gchar* s = gtk_entry_get_text(p->outVideoFile);
      if (strlen(s)) {
        argvStrings.push_back("-o");
        argvStrings.push_back(s);
      }

      s = gtk_entry_get_text(p->outCsvFile);
      if (strlen(s)) {
        argvStrings.push_back("--feature-out");
        argvStrings.push_back(s);
      }

      if (GFile* f = gtk_file_chooser_get_file(p->annotationTemplateFile)) {
        char* path = g_file_get_path(f);
        argvStrings.push_back("--ann-template-file");
        argvStrings.push_back(path);
        g_free(path);
        g_object_unref(f);
      }

      s = gtk_entry_get_text(p->outElanFile);
      if (strlen(s)) {
        argvStrings.push_back("--elan-out");
        argvStrings.push_back(s);
      }

      if (p->configFile.length() > 0) {
        argvStrings.push_back("--config");
        argvStrings.push_back(p->configFile);
      }

      return argvStrings;
    }



    static void getCmdOpts(GtkWidget*, gpointer data) {
      vector<string> opts = generateCmdOpts(reinterpret_cast<struct Params*>(data));
      // add quotation marks around options that contain spaces (e.g. filenames)
      for (size_t i = 0; i < opts.size(); ++i)
        if (opts[i].find(' ') != string::npos)
          opts[i] = '"' + opts[i] + '"';

      GtkWidget* dialog = gtk_message_dialog_new(nullptr, GtkDialogFlags(0),
                                                 GTK_MESSAGE_INFO,
                                                 GTK_BUTTONS_OK,
                                                 "Command line options");
      gtk_window_set_title(GTK_WINDOW(dialog), "Command line options");
      gtk_message_dialog_format_secondary_text(GTK_MESSAGE_DIALOG(dialog),
                                               "These CLI options correspond to "
                                               "the current selections:\n%s",
                                               AdvancedString::join(" ", opts)
                                               .c_str());
      gtk_widget_show(dialog);
      gtk_dialog_run(GTK_DIALOG(dialog));
      gtk_widget_destroy(dialog);
    }



    static void quit(GtkWidget*, gpointer shouldQuit) {
      *reinterpret_cast<bool*>(shouldQuit) = true;
      gtk_main_quit();
    }


    
    static void editBlackBoard(GtkWidget*, gpointer ptr) {
      static slmotion::gui::SingletonWindow* editor = nullptr;
      if (editor == nullptr)
        editor = new slmotion::gui::GraphicalBlackBoardEditor(&editor, 
                                                              *reinterpret_cast<std::shared_ptr<slmotion::BlackBoard>*>(ptr));
    }



    static void about(GtkWidget*, gpointer) {
      GtkWidget* dialog = gtk_message_dialog_new(nullptr, GtkDialogFlags(0),
                                                 GTK_MESSAGE_INFO,
                                                 GTK_BUTTONS_OK,
                                                 "About slmotion");
      gtk_window_set_title(GTK_WINDOW(dialog), "About slmotion");
      gtk_message_dialog_format_secondary_text(GTK_MESSAGE_DIALOG(dialog),
                                               "%s\nCopyright © Aalto University "
                                               "2013\n"
                                               "http://users.ics.aalto.fi/jmkarppa/"
                                               "slmotion/\nFor enquiries, please "
                                               "contact Matti Karppa at "
                                               "matti.karppa@aalto.fi",
                                               slmotion::getVersions().c_str());
      gtk_widget_show(dialog);
      gtk_dialog_run(GTK_DIALOG(dialog));
      gtk_widget_destroy(dialog);
    }



    static GtkFileFilter* getDumpFilter() {
      GtkFileFilter* dumpFilter = gtk_file_filter_new();
      gtk_file_filter_add_pattern(dumpFilter, "*.dump");
      gtk_file_filter_set_name(dumpFilter, "Dump files");
      return dumpFilter;
    }

    static GtkFileFilter* getGzipFilter() {
      GtkFileFilter* dumpFilter = gtk_file_filter_new();
      gtk_file_filter_add_pattern(dumpFilter, "*.gz");
      gtk_file_filter_set_name(dumpFilter, "Gzipped files");
      return dumpFilter;
    }

    static GtkFileFilter* getAnyFilter() {
      GtkFileFilter* dumpFilter = gtk_file_filter_new();
      gtk_file_filter_add_pattern(dumpFilter, "*.*");
      gtk_file_filter_set_name(dumpFilter, "Any files");
      return dumpFilter;
    }

    static GtkFileFilter* getStillImageFilter() {
      GtkFileFilter* stillImageFilter = gtk_file_filter_new();
      gtk_file_filter_add_pattern(stillImageFilter, "*.jpg");
      gtk_file_filter_add_pattern(stillImageFilter, "*.png");
      gtk_file_filter_add_pattern(stillImageFilter, "*.bmp");
      gtk_file_filter_set_name(stillImageFilter, "Still image files");
      return stillImageFilter;
    }



    static void makeErrorDialog(const char* what) {
      GtkWidget* dialog = gtk_message_dialog_new(nullptr, GtkDialogFlags(0),
                                                 GTK_MESSAGE_ERROR,
                                                 GTK_BUTTONS_OK,
                                                 "Error");
      gtk_window_set_title(GTK_WINDOW(dialog), "Error");
      gtk_message_dialog_format_secondary_text(GTK_MESSAGE_DIALOG(dialog),
                                               "Operation failed:\n%s",
                                               what);
      gtk_widget_show(dialog);
      gtk_dialog_run(GTK_DIALOG(dialog));
      gtk_widget_destroy(dialog);
    }



    /**
     * Shows a dialog and loads the contents of the dump file to the
     * Python black board.
     */
    static void loadBlackBoardDump(GtkWidget*, gpointer ptr) {
      std::pair<GtkStatusbar*,std::shared_ptr<slmotion::BlackBoard>*>* pair =
        reinterpret_cast<std::pair<GtkStatusbar*,std::shared_ptr<slmotion::BlackBoard>*>* >(ptr);
      auto statusbar = pair->first;
      GtkWidget* dialog = gtk_file_chooser_dialog_new("Select a file", nullptr,
                                                      GTK_FILE_CHOOSER_ACTION_OPEN,
                                                      GTK_STOCK_CANCEL,
                                                      GTK_RESPONSE_CANCEL,
                                                      GTK_STOCK_OPEN,
                                                      GTK_RESPONSE_ACCEPT,
                                                      nullptr);
      gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialog), getDumpFilter());
      gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialog), getGzipFilter());
      gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialog), getAnyFilter());
      if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) {
        gchar* filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
        gtk_widget_destroy(dialog);
        slmotion::DummyFrameSource dummySource;
        slmotion::BlackBoardDumpReader bbdr(pair->second->get(),
                                            &dummySource);

        class LoadDumpCallback : public slmotion::UiCallback {
        public:
          explicit LoadDumpCallback(GtkStatusbar* statusBar) : 
            pushed(false),
            statusBar(statusBar) {}

          bool operator()(double percentComplete) {
            if (pushed)
              gtk_statusbar_pop(statusBar, 0);

            char percent[10];
            sprintf(percent, "%0.2f", percentComplete);
            gtk_statusbar_push(statusBar, 0, (std::string("Loading ") + percent +
                                              " %...").c_str());
            pushed = true;

            while (gtk_events_pending())
              gtk_main_iteration ();

            return true;
          }

          bool pushed;
          GtkStatusbar* statusBar;
        };

        bbdr.setFilenames(std::vector<std::string>{ filename });
        LoadDumpCallback callback(GTK_STATUSBAR(statusbar));
        try {
          bbdr.processRange(0, SIZE_MAX, &callback);
        }
        catch (slmotion::IOException& e) {
          makeErrorDialog(e.what());
        }
        if (callback.pushed) 
          gtk_statusbar_pop(GTK_STATUSBAR(statusbar), 0);
        gtk_statusbar_push(GTK_STATUSBAR(statusbar), 0, "Blackboard dump loaded!");

        g_free(filename);
      }
      else
        gtk_widget_destroy(dialog);
    }



    // returns false on failure; this should be called from the main
    // window initialisation function
    static bool checkAndCreateDefaultConfiguration(GtkWindow* mainWindow) {
      std::string dir = HOME_DIR + "/" + CONFIGURATION_SUBDIRNAME;

      struct stat s;
      int err = stat(dir.c_str(), &s);
      if (err == -1 && errno == ENOENT) {
        // no such directory, try to create
        if (mkdir(dir.c_str(), 0777)) {
          GtkWidget* dialog = gtk_message_dialog_new(mainWindow, GTK_DIALOG_MODAL,
                                                     GTK_MESSAGE_WARNING,
                                                     GTK_BUTTONS_OK,
                                                     "Warning");
          gtk_window_set_title(GTK_WINDOW(dialog), "Warning");
          gtk_message_dialog_format_secondary_text(GTK_MESSAGE_DIALOG(dialog),
                                                   "%s",
                                                   ("Could not create the default "
                                                    "configuration directory \"" + 
                                                    (HOME_DIR + "/" + 
                                                     CONFIGURATION_SUBDIRNAME) + "\":"
                                                    " " + strerror(errno)).c_str());
          gtk_widget_show(dialog);
          gtk_dialog_run(GTK_DIALOG(dialog));
          gtk_widget_destroy(dialog);

          return false;
        }
      }
      else if (!S_ISDIR(s.st_mode)) {
        GtkWidget* dialog = gtk_message_dialog_new(mainWindow, GTK_DIALOG_MODAL,
                                                   GTK_MESSAGE_WARNING,
                                                   GTK_BUTTONS_OK,
                                                   "Warning");
        gtk_window_set_title(GTK_WINDOW(dialog), "Warning");
        gtk_message_dialog_format_secondary_text(GTK_MESSAGE_DIALOG(dialog),
                                                 "%s",
                                                 ("Could not create configuration "
                                                  "directory \"" + dir + "\": "
                                                  "The file already exists but is "
                                                  "not a directory.").c_str());
        gtk_widget_show(dialog);
        gtk_dialog_run(GTK_DIALOG(dialog));
        gtk_widget_destroy(dialog);
        return false;
      }

      bool fileExists = std::ifstream(DEFAULT_CONF_FILE_FULL_PATH);

      if (!fileExists) {
        std::ofstream ofs(DEFAULT_CONF_FILE_FULL_PATH);
        if (!ofs) {
          GtkWidget* dialog = gtk_message_dialog_new(mainWindow, GTK_DIALOG_MODAL,
                                                     GTK_MESSAGE_WARNING,
                                                     GTK_BUTTONS_OK,
                                                     "Warning");
          gtk_window_set_title(GTK_WINDOW(dialog), "Warning");
          gtk_message_dialog_format_secondary_text(GTK_MESSAGE_DIALOG(dialog),
                                                   "%s",
                                                   ("Could not create default "
                                                    "configuration file \"" +
                                                    DEFAULT_CONF_FILE_FULL_PATH +
                                                    "\": " + strerror(errno))
                                                   .c_str());
          gtk_widget_show(dialog);
          gtk_dialog_run(GTK_DIALOG(dialog));
          gtk_widget_destroy(dialog);
          return false;
        }
        slmotion::configuration::generateDefaultConfig(ofs, 80);
      }

      return true;
    }



    class SlmotionGtkCallback : public slmotion::UiCallback {
    public:
      explicit SlmotionGtkCallback(GtkStatusbar* statusBar,
                                   MainWindow* mainWindow) : pushed(false), 
                                                   statusBar(statusBar),
                                                   mainWindow (mainWindow) {}

      ~SlmotionGtkCallback() {}

      bool operator()(double percentComplete) {
        if (pushed)
          gtk_statusbar_pop(statusBar, 0);

        char percent1[10];
        sprintf(percent1, "%0.2f", percentComplete);
        char percent2[10];
        sprintf(percent2, "%0.2f", (double(getCurrentComponentIndex())/
                                    double(getNComponents()) + percentComplete)/
                getNComponents());
        gtk_statusbar_push(statusBar, 0, ("Processing " + getCurrentComponentName()
                                          + ": " + percent1 + " % / total: " +
                                          percent2 + " %").c_str());
        pushed = true;

        while (gtk_events_pending())
          gtk_main_iteration ();

        return !mainWindow->getShouldQuit();
      }

      bool getPushed() const {
        return pushed;
      }
  
    private:
      bool pushed;
      GtkStatusbar* statusBar;
      MainWindow* mainWindow;
    };


    // this is the signal handler id of the process/cancel button
    static gulong processCancelButtonHandlerId;

    static void process(GtkWidget* button, gpointer data);

    static void cancel(GtkWidget* button, gpointer data) {
      reinterpret_cast<MainWindow*>(data)->setShouldQuit(true);

      g_signal_handler_disconnect(button,
                                  processCancelButtonHandlerId);
      gtk_button_set_label(GTK_BUTTON(button), "Process");
      processCancelButtonHandlerId = g_signal_connect(button, "clicked",
                                                      G_CALLBACK(&process), data);
    }



    static void process(GtkWidget* button, gpointer data) {
      reinterpret_cast<MainWindow*>(data)->setShouldQuit(false);

      // change the button to a cancel button
      g_signal_handler_disconnect(button,
                                  processCancelButtonHandlerId);
      gtk_button_set_label(GTK_BUTTON(button), "Cancel");
      processCancelButtonHandlerId = g_signal_connect(button, "clicked",
                                                      G_CALLBACK(&cancel), data);
  
      // auto p = reinterpret_cast<struct Params*>(data);
      auto p = reinterpret_cast<MainWindow*>(data)->getParamsPtr();
      gtk_statusbar_push(p->statusBar, 0, "Processing...");

      // construct parameters
      std::vector<std::string> argvStrings = generateCmdOpts(p); 
      argvStrings.insert(argvStrings.begin(), p->argv0);

      std::vector<std::vector<char> > argvArrays = stringArrayToCharArrayArray(argvStrings);

      std::vector<char*> argv(argvArrays.size());
      for (size_t i = 0; i < argvArrays.size(); ++i)
        argv[i] = &argvArrays[i][0];

      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(p->blackBoardClearCheckButton)) ||
          p->bb.get() == nullptr) 
        p->bb = std::shared_ptr<slmotion::BlackBoard>(new slmotion::BlackBoard());
  
      SlmotionGtkCallback callback(p->statusBar, reinterpret_cast<MainWindow*>(data));
      int returnValue = slmotion::main(argv.size(), &argv[0], p->bb, &callback);
      if (callback.getPushed())
        gtk_statusbar_pop(p->statusBar, 0);
  
      if (returnValue == 0)
        gtk_statusbar_push(p->statusBar, 0, "Done!");
      else {
        gtk_statusbar_push(p->statusBar, 0, "An error occurred.");
        GtkWidget* dialog = gtk_message_dialog_new(p->mainWindow, GTK_DIALOG_MODAL,
                                                   GTK_MESSAGE_ERROR,
                                                   GTK_BUTTONS_OK,
                                                   "Error");
        gtk_window_set_title(GTK_WINDOW(dialog), "Error");
        const char* c = getenv("SLMOTION_ERROR_LOG_FILE");
        std::string errMess = "An error occurred while processing files. Please see ";
        errMess += c ? c : "stderr";
        errMess += " for more information.";
        gtk_message_dialog_format_secondary_text(GTK_MESSAGE_DIALOG(dialog),
                                                 "%s",
                                                 errMess.c_str());
        gtk_widget_show(dialog);
        gtk_dialog_run(GTK_DIALOG(dialog));
        gtk_widget_destroy(dialog);
      }
      // for (auto it = argv.begin(); it != argv.end(); ++it)
      //   delete[] *it;

      // change the button to a process button
      g_signal_handler_disconnect(button,
                                  processCancelButtonHandlerId);
      gtk_button_set_label(GTK_BUTTON(button), "Process");
      processCancelButtonHandlerId = g_signal_connect(button, "clicked",
                                                      G_CALLBACK(&process), data);
    }




    static GtkFileFilter* getVideoFileFilter() {
      GtkFileFilter* videoFilter = gtk_file_filter_new();
      gtk_file_filter_add_pattern(videoFilter, "*.avi");
      gtk_file_filter_add_pattern(videoFilter, "*.mkv");
      gtk_file_filter_add_pattern(videoFilter, "*.mpg");
      gtk_file_filter_add_pattern(videoFilter, "*.mov");
      gtk_file_filter_set_name(videoFilter, "Video files");
      return videoFilter;
    }

 


    static void configure(GtkWidget*, gpointer conffilePtr) {
      static slmotion::gui::SingletonWindow* configWindow = nullptr;
      if (configWindow == nullptr)
        configWindow = new slmotion::gui::ConfigurationWindow(&configWindow, 
                                                              *reinterpret_cast<std::string*>(conffilePtr));
    }





    /**
     * Shows a dialog and saves the contents of the Python black board to
     * a dump file
     */
    static void saveBlackBoardDump(GtkWidget*, gpointer ptr) {
      std::pair<GtkStatusbar*,std::shared_ptr<slmotion::BlackBoard>*>* pair =
        reinterpret_cast<std::pair<GtkStatusbar*,std::shared_ptr<slmotion::BlackBoard>*>* >(ptr);
      auto statusbar = pair->first;
      GtkWidget* dialog = gtk_file_chooser_dialog_new("Select a file", nullptr,
                                                      GTK_FILE_CHOOSER_ACTION_SAVE,
                                                      GTK_STOCK_CANCEL,
                                                      GTK_RESPONSE_CANCEL,
                                                      GTK_STOCK_SAVE,
                                                      GTK_RESPONSE_ACCEPT,
                                                      nullptr);
      gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialog), getDumpFilter());
      gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialog), getGzipFilter());
      gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialog), getAnyFilter());
      gtk_file_chooser_set_do_overwrite_confirmation(GTK_FILE_CHOOSER(dialog), true);
      if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) {
        gchar* filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
        gtk_widget_destroy(dialog);

        slmotion::DummyFrameSource dummySource;
        slmotion::BlackBoardDumpWriter bbdw(pair->second->get(), &dummySource);

        class SaveDumpCallback : public slmotion::UiCallback {
        public:
          explicit SaveDumpCallback(GtkStatusbar* statusBar) : 
            pushed(false),
            statusBar(statusBar) {}

          bool operator()(double percentComplete) {
            if (pushed)
              gtk_statusbar_pop(statusBar, 0);

            char percent[10];
            sprintf(percent, "%0.2f", percentComplete);
            gtk_statusbar_push(statusBar, 0, (std::string("Saving ") + percent +
                                              " %...").c_str());
            pushed = true;

            while (gtk_events_pending())
              gtk_main_iteration ();

            return true;
          }

          bool pushed;
          GtkStatusbar* statusBar;
        };

        bbdw.setFilename(filename);
        SaveDumpCallback callback(GTK_STATUSBAR(statusbar));
        try {
          bbdw.processRange(0, SIZE_MAX, &callback);
        }
        catch (slmotion::IOException& e) {
          makeErrorDialog(e.what());
        }

        if (callback.pushed) 
          gtk_statusbar_pop(GTK_STATUSBAR(statusbar), 0);
        gtk_statusbar_push(GTK_STATUSBAR(statusbar), 0, "Blackboard dump saved!");

        g_free(filename);
      }
      else
        gtk_widget_destroy(dialog);
    }



    int MainWindow::exec(int argc, char* argv[]) {
        gtk_init(&argc, &argv);

        // GtkBuilder* builder = gtk_builder_new();
        // gtk_builder_add_from_file(builder, UI_XML_FILENAME, NULL);
        
        GtkWidget* window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
        gtk_widget_set_size_request(window, 640, -1);
        gtk_container_set_border_width(GTK_CONTAINER(window), 10);
        gtk_window_set_title(GTK_WINDOW(window), "slmotion");
        g_signal_connect(window, "destroy", G_CALLBACK(&quit), &shouldQuit);
        
        GtkWidget* menuBar = gtk_menu_bar_new();
        GtkWidget* fileMenuItem = gtk_menu_item_new_with_label("_File");
        gtk_menu_item_set_use_underline(GTK_MENU_ITEM(fileMenuItem), true);
        GtkWidget* fileMenu = gtk_menu_new();

        GtkWidget* loadBbdMenuItem = gtk_menu_item_new_with_label("_Load blackboard dump");
        gtk_menu_item_set_use_underline(GTK_MENU_ITEM(loadBbdMenuItem), true);
        gtk_menu_shell_append(GTK_MENU_SHELL(fileMenu), loadBbdMenuItem);

        GtkWidget* saveBbdMenuItem = gtk_menu_item_new_with_label("_Save blackboard dump");
        gtk_menu_item_set_use_underline(GTK_MENU_ITEM(saveBbdMenuItem), true);
        gtk_menu_shell_append(GTK_MENU_SHELL(fileMenu), saveBbdMenuItem);

        GtkWidget* quitMenuItem = gtk_menu_item_new_with_label("_Quit");
        gtk_menu_item_set_use_underline(GTK_MENU_ITEM(quitMenuItem), true);
        g_signal_connect(quitMenuItem, "activate", G_CALLBACK(&quit), NULL);
        gtk_menu_shell_append(GTK_MENU_SHELL(fileMenu), quitMenuItem);
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(fileMenuItem), fileMenu);
        gtk_menu_shell_append(GTK_MENU_SHELL(menuBar), fileMenuItem);

        Params& p = getParams();
        p.bb = std::shared_ptr<slmotion::BlackBoard>(new slmotion::BlackBoard());

        GtkWidget* editMenuItem = gtk_menu_item_new_with_label("_Edit");
        gtk_menu_item_set_use_underline(GTK_MENU_ITEM(editMenuItem), true);
        GtkWidget* editMenu = gtk_menu_new();
        GtkWidget* editBbMenuItem = gtk_menu_item_new_with_label("E_dit blackboard");
        g_signal_connect(editBbMenuItem, "activate", G_CALLBACK(&editBlackBoard),
                         &p.bb);
        gtk_menu_item_set_use_underline(GTK_MENU_ITEM(editBbMenuItem), true);
        gtk_menu_shell_append(GTK_MENU_SHELL(editMenu), editBbMenuItem);

        GtkWidget* configureMenuItem = gtk_menu_item_new_with_label("_Configure");
        gtk_menu_item_set_use_underline(GTK_MENU_ITEM(configureMenuItem), true);
        gtk_menu_shell_append(GTK_MENU_SHELL(editMenu), configureMenuItem);

        GtkWidget* getCmdOptsMenuItem = gtk_menu_item_new_with_label("_Get command line options");
        gtk_menu_item_set_use_underline(GTK_MENU_ITEM(getCmdOptsMenuItem), true);
        gtk_menu_shell_append(GTK_MENU_SHELL(editMenu), getCmdOptsMenuItem);
        g_signal_connect(getCmdOptsMenuItem, "activate", G_CALLBACK(&getCmdOpts),
                         &p);

        gtk_menu_item_set_submenu(GTK_MENU_ITEM(editMenuItem), editMenu);
        gtk_menu_shell_append(GTK_MENU_SHELL(menuBar), editMenuItem);

        GtkWidget* helpMenuItem = gtk_menu_item_new_with_label("_Help");
        gtk_menu_item_set_use_underline(GTK_MENU_ITEM(helpMenuItem), true);
        GtkWidget* helpMenu = gtk_menu_new();
        GtkWidget* aboutMenuItem = gtk_menu_item_new_with_label("_About");
        gtk_menu_item_set_use_underline(GTK_MENU_ITEM(aboutMenuItem), true);
        g_signal_connect(aboutMenuItem, "activate", G_CALLBACK(&about), NULL);
        gtk_menu_shell_append(GTK_MENU_SHELL(helpMenu), aboutMenuItem);
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(helpMenuItem), helpMenu);
        gtk_menu_shell_append(GTK_MENU_SHELL(menuBar), helpMenuItem);

        GtkWidget* outerVBox = gtk_vbox_new(false, 0);
        GtkWidget* vbox = gtk_vbox_new(false, 2);
        gtk_container_add(GTK_CONTAINER(outerVBox), menuBar);
        gtk_container_add(GTK_CONTAINER(outerVBox), vbox);
        gtk_container_add(GTK_CONTAINER(window), outerVBox);

        GtkStatusbar* statusBar = reinterpret_cast<GtkStatusbar*>(gtk_statusbar_new());
        gtk_statusbar_push(statusBar, 0, "Welcome to slmotion!");

        auto pair = std::make_pair(statusBar, &p.bb);
        g_signal_connect(loadBbdMenuItem, "activate", 
                         G_CALLBACK(&loadBlackBoardDump), &pair);
        g_signal_connect(saveBbdMenuItem, "activate", 
                         G_CALLBACK(&saveBlackBoardDump), statusBar);

        GtkTable* grid = GTK_TABLE(gtk_table_new(7,3,false));
        gtk_container_add(GTK_CONTAINER(vbox), GTK_WIDGET(grid));
        p.mainWindow = GTK_WINDOW(window);

        p.inputFilename = GTK_TREE_VIEW(gtk_tree_view_new());
        GtkCellRenderer *renderer = gtk_cell_renderer_text_new();
        GtkTreeViewColumn* column = gtk_tree_view_column_new_with_attributes("Filenames",
                                                                             renderer, "text", 0, NULL);
        gtk_tree_view_append_column(GTK_TREE_VIEW(p.inputFilename), column);
        GtkListStore* store = gtk_list_store_new(1, G_TYPE_STRING);
        gtk_tree_view_set_model(GTK_TREE_VIEW(p.inputFilename), 
                                GTK_TREE_MODEL(store));
        GtkTreeSelection* selection = gtk_tree_view_get_selection(p.inputFilename);
        gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);

        p.scriptFilename = GTK_FILE_CHOOSER(gtk_file_chooser_button_new("Select a file", GTK_FILE_CHOOSER_ACTION_OPEN));
        p.outVideoFile = GTK_ENTRY(gtk_entry_new());
        p.outCsvFile = GTK_ENTRY(gtk_entry_new());
        p.annotationTemplateFile = GTK_FILE_CHOOSER(gtk_file_chooser_button_new("Select a file", GTK_FILE_CHOOSER_ACTION_OPEN));
        p.outElanFile = GTK_ENTRY(gtk_entry_new());
        p.statusBar = statusBar;
        p.argv0 = argv[0];

        std::vector<GtkWidget*> align(6);
        for (int i = 0; i < 6; ++i)
          align[i] = gtk_alignment_new(1,1,0,1);

        gtk_container_add(GTK_CONTAINER(align[0]), gtk_label_new("Input files: "));
        gtk_table_attach(grid, align[0], 0, 1, 0, 2, GTK_FILL, GTK_FILL, 0, 0);
        gtk_widget_set_size_request(GTK_WIDGET(p.inputFilename), -1, 100);
        // gtk_table_attach(grid, GTK_WIDGET(p.inputFilename), 1, 2, 0, 2, 
        //                  GtkAttachOptions(GTK_EXPAND|GTK_FILL), GTK_FILL, 0, 0);
        // add scrollbars
        GtkWidget* scrolledWindow = gtk_scrolled_window_new(nullptr, nullptr);
        gtk_container_add(GTK_CONTAINER(scrolledWindow), 
                          GTK_WIDGET(p.inputFilename));
        gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledWindow),
                                       GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
        gtk_table_attach(grid, scrolledWindow, 1, 2, 0, 2, 
                         GtkAttachOptions(GTK_EXPAND|GTK_FILL), GTK_FILL, 0, 0);
        GtkWidget* inputFileAddButton = gtk_button_new_with_label("Add");
        gtk_table_attach(grid, GTK_WIDGET(inputFileAddButton), 2, 3, 0, 1, 
                         GTK_FILL, GTK_FILL, 0, 0);

        GtkWidget* inputFileRemoveButton = gtk_button_new_with_label("Remove");
        gtk_table_attach(grid, GTK_WIDGET(inputFileRemoveButton), 2, 3, 1, 2, 
                         GTK_FILL, GTK_FILL, 0, 0);

        struct InputFileCallback {
          static void addCallback(GtkButton*, gpointer d) {
            GtkTreeView* data = GTK_TREE_VIEW(d);
            GtkWidget* dialog = gtk_file_chooser_dialog_new("Select a file",
                                                            nullptr,
                                                            GTK_FILE_CHOOSER_ACTION_OPEN,
                                                            GTK_STOCK_CANCEL,
                                                            GTK_RESPONSE_CANCEL,
                                                            GTK_STOCK_OPEN,
                                                            GTK_RESPONSE_ACCEPT,
                                                            nullptr);
            gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialog), 
                                        getVideoFileFilter());
            gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialog),
                                        getStillImageFilter());
            gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(dialog), true);
          
            if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) {
              GSList* filenames = gtk_file_chooser_get_filenames(GTK_FILE_CHOOSER(dialog));
              g_slist_foreach(filenames, [](gpointer data, gpointer inputfiles) {
                  GtkTreeViewColumn* column = gtk_tree_view_get_column(GTK_TREE_VIEW(inputfiles), 0);
                  assert(column != nullptr);
              
                  GtkListStore* store = GTK_LIST_STORE(gtk_tree_view_get_model
                                                       (GTK_TREE_VIEW(inputfiles)));
                  assert(store != nullptr);

                  GtkTreeIter iter;
                  gtk_list_store_append(store, &iter);
                  gtk_list_store_set(store, &iter, 0, reinterpret_cast<gchar*>(data), -1);
                }, data);
              g_slist_free(filenames);
            }
            gtk_widget_destroy(dialog);
          }

          static void removeCallback(GtkButton*, gpointer d) {
            GtkTreeView* fileTreeView = GTK_TREE_VIEW(d);
            GtkTreeSelection* selection = gtk_tree_view_get_selection(fileTreeView);
            GtkTreeModel* model = gtk_tree_view_get_model(fileTreeView);
            GtkTreeIter iter;
            if (gboolean b = gtk_tree_model_get_iter_first(model, &iter)) {
              do {
                b = gtk_tree_selection_iter_is_selected(selection, &iter);
                if (b)
                  b = gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
                else
                  b = gtk_tree_model_iter_next(model, &iter);
              } while(b);
            }
          }
        };

        g_signal_connect(inputFileAddButton, "clicked", 
                         G_CALLBACK(&InputFileCallback::addCallback), 
                         p.inputFilename);

        g_signal_connect(inputFileRemoveButton, "clicked", 
                         G_CALLBACK(&InputFileCallback::removeCallback), 
                         p.inputFilename);

        gtk_container_add(GTK_CONTAINER(align[1]), gtk_label_new("Script file: "));
        gtk_table_attach(grid, align[1], 0, 1, 2, 3, GTK_FILL,
                         GTK_FILL, 0, 0);
        gtk_table_attach(grid, GTK_WIDGET(p.scriptFilename), 1, 3, 2, 3, 
                         GtkAttachOptions(GTK_EXPAND|GTK_FILL), GTK_FILL, 0, 0);

        gtk_container_add(GTK_CONTAINER(align[4]), gtk_label_new("Annotation template file: "));
        gtk_table_attach(grid, align[4], 0, 1, 3, 4, GTK_FILL,
                         GTK_FILL, 0, 0);
        gtk_table_attach(grid, GTK_WIDGET(p.annotationTemplateFile), 1, 3, 3, 4, 
                         GtkAttachOptions(GTK_EXPAND|GTK_FILL), GTK_FILL, 0, 0);

        typedef GtkFileFilter*(*filter_creator_t)();
       
        auto connectSelectButtonToFileChooserDialog = [](GtkButton* selectButton,
                                                         std::pair<GtkEntry*,filter_creator_t>* data) {
          struct s {
            static void callback(GtkButton*, gpointer data) {
              GtkWidget* dialog = gtk_file_chooser_dialog_new("Select a file",
                                                              nullptr,
                                                              GTK_FILE_CHOOSER_ACTION_SAVE,
                                                              GTK_STOCK_CANCEL,
                                                              GTK_RESPONSE_CANCEL,
                                                              GTK_STOCK_SAVE,
                                                              GTK_RESPONSE_ACCEPT,
                                                              nullptr);
              gtk_file_chooser_set_do_overwrite_confirmation(GTK_FILE_CHOOSER(dialog), true);
              auto dataPair = reinterpret_cast<std::pair<GtkEntry*,filter_creator_t>*>(data);
              gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialog), dataPair->second());
              if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT)
                gtk_entry_set_text(dataPair->first, 
                                   gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog)));
              gtk_widget_destroy(GTK_WIDGET(dialog));
            }
          };
          g_signal_connect(selectButton, "clicked", G_CALLBACK(&s::callback), 
                           data);
        };

        GtkWidget* outVideoFileEntry = GTK_WIDGET(p.outVideoFile);
        GtkWidget* outVideoFileSelectButton = gtk_button_new_with_label("Select");
        gtk_container_add(GTK_CONTAINER(align[2]), gtk_label_new("Out video file: "));
        gtk_table_attach(grid, align[2], 0, 1, 4, 5, GTK_FILL,
                         GTK_FILL, 0, 0);
        gtk_table_attach(grid, outVideoFileEntry, 1, 2, 4, 5,
                         GtkAttachOptions(GTK_EXPAND|GTK_FILL), GTK_FILL, 0, 0);
        gtk_table_attach(grid, outVideoFileSelectButton, 2, 3, 4, 5,
                         GTK_FILL, GTK_FILL, 0, 0);
        gtk_editable_set_editable(GTK_EDITABLE(outVideoFileEntry), false);

        GtkWidget* outCsvFileEntry = GTK_WIDGET(p.outCsvFile);
        GtkWidget* outCsvFileSelectButton = gtk_button_new_with_label("Select");
        gtk_editable_set_editable(GTK_EDITABLE(outCsvFileEntry), false);
        gtk_container_add(GTK_CONTAINER(align[3]), gtk_label_new("Out CSV file: "));
        gtk_table_attach(grid, align[3], 0, 1, 5, 6, GTK_FILL,
                         GTK_FILL, 0, 0);
        gtk_table_attach(grid, outCsvFileEntry, 1, 2, 5, 6,
                         GtkAttachOptions(GTK_EXPAND|GTK_FILL), GTK_FILL, 0, 0);
        gtk_table_attach(grid, outCsvFileSelectButton, 2, 3, 5, 6,
                         GTK_FILL, GTK_FILL, 0, 0);


        GtkWidget* outElanEntry = GTK_WIDGET(p.outElanFile);
        GtkWidget* elanSelectButton = gtk_button_new_with_label("Select");
        gtk_editable_set_editable(GTK_EDITABLE(outElanEntry), false);
        gtk_container_add(GTK_CONTAINER(align[5]), gtk_label_new("Out ELAN file: "));
        gtk_table_attach(grid, align[5], 0, 1, 6, 7, GTK_FILL,
                         GTK_FILL, 0, 0);
        gtk_table_attach(grid, outElanEntry, 1, 2, 6, 7,
                         GtkAttachOptions(GTK_EXPAND|GTK_FILL), GTK_FILL, 0, 0);
        gtk_table_attach(grid, elanSelectButton, 2, 3, 6, 7,
                         GTK_FILL, GTK_FILL, 0, 0);

        auto csvFilterCreator = [] {
          GtkFileFilter* csvFilter = gtk_file_filter_new();
          gtk_file_filter_add_pattern(csvFilter, "*.csv");
          gtk_file_filter_set_name(csvFilter, "CSV files");
          return csvFilter;
        };

        auto eafFilterCreator = [] {
          GtkFileFilter* eafFilter = gtk_file_filter_new();
          gtk_file_filter_add_pattern(eafFilter, "*.eaf");
          gtk_file_filter_set_name(eafFilter, "EAF files");
          return eafFilter;
        };

        std::pair<GtkEntry*, filter_creator_t> entriesAndFilterCreators[3] { 
          std::pair<GtkEntry*, filter_creator_t>(GTK_ENTRY(outVideoFileEntry), 
                                                 &getVideoFileFilter),
            std::pair<GtkEntry*, filter_creator_t>(GTK_ENTRY(outCsvFileEntry),
                                                   csvFilterCreator),
            std::pair<GtkEntry*, filter_creator_t>(GTK_ENTRY(outElanEntry),
                                                   eafFilterCreator)
            };


        connectSelectButtonToFileChooserDialog(GTK_BUTTON(outVideoFileSelectButton),
                                               &entriesAndFilterCreators[0]);
        connectSelectButtonToFileChooserDialog(GTK_BUTTON(outCsvFileSelectButton),
                                               &entriesAndFilterCreators[1]);
        connectSelectButtonToFileChooserDialog(GTK_BUTTON(elanSelectButton),
                                               &entriesAndFilterCreators[2]);

        GtkFileFilter* pyFilter = gtk_file_filter_new();
        gtk_file_filter_add_pattern(pyFilter, "*.py");
        gtk_file_filter_set_name(pyFilter, "Python scripts");
        gtk_file_chooser_add_filter(p.scriptFilename, pyFilter);

        GtkWidget* hbox = gtk_hbox_new(false, 2);
        p.blackBoardClearCheckButton = GTK_CHECK_BUTTON(gtk_check_button_new_with_label("Empty the black board"));
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p.blackBoardClearCheckButton),
                                     true);
        gtk_container_add(GTK_CONTAINER(hbox), GTK_WIDGET(p.blackBoardClearCheckButton));

        GtkWidget* button = gtk_button_new_with_label("Process");
        gtk_widget_set_size_request(button, 70, -1);
        GtkWidget* buttonAlign = gtk_alignment_new(1,1,0,0);

        gtk_container_add(GTK_CONTAINER(hbox), button);
        gtk_container_add(GTK_CONTAINER(buttonAlign),hbox);

        gtk_container_add(GTK_CONTAINER(vbox), buttonAlign);

        // processCancelButtonHandlerId = g_signal_connect(button, "clicked", 
        //                                                 G_CALLBACK(&process), &p);
        processCancelButtonHandlerId = g_signal_connect(button, "clicked", 
                                                        G_CALLBACK(&process), this);
        gtk_box_pack_end(GTK_BOX(vbox), GTK_WIDGET(statusBar), false, false, false);
 
        gtk_widget_show_all(window);

        if (checkAndCreateDefaultConfiguration(GTK_WINDOW(window)))
          p.configFile = DEFAULT_CONF_FILE_FULL_PATH;

        g_signal_connect(configureMenuItem, "activate", G_CALLBACK(&configure),
                         &p.configFile);

        gtk_main();
        return 0;
      }

#elif defined(SLMOTION_WITH_QT)
    class SlmotionQtCallback : public slmotion::UiCallback {
    public:
      explicit SlmotionQtCallback(MainWindow* mainWindow) : mainWindow (mainWindow) {}

      virtual ~SlmotionQtCallback() {}

      bool operator()(double percentComplete) {
        char percent1[10];
        sprintf(percent1, "%0.2f", percentComplete);
        char percent2[10];
        sprintf(percent2, "%0.2f", (double(getCurrentComponentIndex())/
                                    double(getNComponents()) + percentComplete)/
                getNComponents());
        std::string statusBarText = "Processing " + getCurrentComponentName()
          + ": " + percent1 + " % / total: " + percent2 + " %";
        mainWindow->setStatusBarText(statusBarText);

        QCoreApplication::processEvents();

        return !mainWindow->getShouldQuit();
      }

    private:
      MainWindow* mainWindow;
    };



    int MainWindow::exec(int /* argc */, char* argv[]) {
      resize(640, -1);
      setWindowTitle(QString::fromStdString(title));

      argv0 = argv[0];

      QGridLayout layout;

      QStringListModel inputFilesListModel;
      QLabel inputFilesLabel;
      inputFilesLabel.setText("Input files:");
      inputFilesListView.setModel(&inputFilesListModel);
      // inputFilesList << "file1";
      // inputFilesList << "file2";
      // inputFilesListModel.setStringList(inputFilesList);
      QPushButton addInputFileButton("Add");
      QPushButton removeInputFileButton("Remove");
      QVBoxLayout* addRemoveInputFileButtonsLayout = new QVBoxLayout;
      addRemoveInputFileButtonsLayout->addWidget(&addInputFileButton);
      addRemoveInputFileButtonsLayout->addWidget(&removeInputFileButton);
      layout.addWidget(&inputFilesLabel, 0, 0);
      layout.addWidget(&inputFilesListView, 0, 1);
      layout.addItem(addRemoveInputFileButtonsLayout, 0, 2);
      connect(&removeInputFileButton, SIGNAL(clicked()), this, SLOT(removeCurrentInputFile()));
      connect(&addInputFileButton, SIGNAL(clicked()), this, SLOT(addInputFile()));

      QLabel scriptFileLabel;
      scriptFileLabel.setText("Script file:");
      QPushButton scriptFileSelectButton("Select");
      layout.addWidget(&scriptFileLabel, 1, 0);
      layout.addWidget(&scriptFileEdit, 1, 1);
      layout.addWidget(&scriptFileSelectButton, 1, 2);
      connect(&scriptFileSelectButton, SIGNAL(clicked()), this, SLOT(selectScriptFile()));

      QLabel annotationTemplateFileLabel;
      annotationTemplateFileLabel.setText("Annotation template file:");
      QPushButton annotationTemplateFileSelectButton("Select");
      layout.addWidget(&annotationTemplateFileLabel, 2, 0);
      layout.addWidget(&annotationTemplateFileEdit, 2, 1);
      layout.addWidget(&annotationTemplateFileSelectButton, 2, 2);
      connect(&annotationTemplateFileSelectButton, SIGNAL(clicked()), this, 
              SLOT(selectAnnotationTemplateFile()));

      QLabel outVideoFileLabel;
      outVideoFileLabel.setText("Out video file:");
      QPushButton outVideoFileSelectButton("Select");
      layout.addWidget(&outVideoFileLabel, 3, 0);
      layout.addWidget(&outVideoFileEdit, 3, 1);
      layout.addWidget(&outVideoFileSelectButton, 3, 2);
      connect(&outVideoFileSelectButton, SIGNAL(clicked()), this, 
              SLOT(selectOutVideoFile()));

      QLabel outCsvFileLabel;
      outCsvFileLabel.setText("Out CSV file:");
      QPushButton outCsvFileSelectButton("Select");
      layout.addWidget(&outCsvFileLabel, 4, 0);
      layout.addWidget(&outCsvFileEdit, 4, 1);
      layout.addWidget(&outCsvFileSelectButton, 4, 2);
      connect(&outCsvFileSelectButton, SIGNAL(clicked()), this, 
              SLOT(selectOutCsvFile()));

      QLabel outElanFileLabel;
      outElanFileLabel.setText("Out ELAN file:");
      QPushButton outElanFileSelectButton("Select");
      layout.addWidget(&outElanFileLabel, 5, 0);
      layout.addWidget(&outElanFileEdit, 5, 1);
      layout.addWidget(&outElanFileSelectButton, 5, 2);
      connect(&outElanFileSelectButton, SIGNAL(clicked()), this, 
              SLOT(selectOutElanFile()));

      layout.addWidget(&emptyBlackBoardCheckBox, 6, 1, Qt::AlignRight);
      layout.addWidget(&processButton, 6, 2);
      connect(&processButton, SIGNAL(clicked()), this,
              SLOT(process()));

      QMenu* fileMenu = menuBar()->addMenu("&File");
      QAction* loadBbDumpAction = fileMenu->addAction("&Load blackboard dump");
      connect(loadBbDumpAction, SIGNAL(triggered()), this, 
              SLOT(loadBlackBoardDump()));
      QAction* saveBbDumpAction = fileMenu->addAction("&Save blackboard dump");
      connect(saveBbDumpAction, SIGNAL(triggered()), this, 
              SLOT(saveBlackBoardDump()));
      QAction* quitAction = fileMenu->addAction("&Quit");
      connect(quitAction, SIGNAL(triggered()), qApp, SLOT(quit()));

      QMenu* editMenu = menuBar()->addMenu("&Edit");
      QAction* editBbAction = editMenu->addAction("E&dit blackboard");
      editBbAction->setEnabled(false);
      QAction* configureAction = editMenu->addAction("&Configure");
      configureAction->setMenuRole(QAction::NoRole);
      connect(configureAction, SIGNAL(triggered()), this, SLOT(showConfigureWindow()));
      QAction* getCmdOptsAction = editMenu->addAction("&Get command line options");
      connect(getCmdOptsAction, SIGNAL(triggered()), this, SLOT(getCmdOpts()));

      QMenu* helpMenu = menuBar()->addMenu("&Help");
      QAction* aboutAction = helpMenu->addAction("&About");
      connect(aboutAction, SIGNAL(triggered()), this, SLOT(showAboutBox()));

      statusBarMessage.setText("Welcome to slmotion!");
      statusBar()->addWidget(&statusBarMessage);

      QWidget* central = new QWidget;
      central->setLayout(&layout);
      setCentralWidget(central); // centralWidget()->setLayout(&layout);
      show();

      // check configuration file
      bool fileExists = !std::ifstream(conffile).fail();
      if (!fileExists) {
        std::ofstream ofs(conffile);
        if (!ofs) {
          QMessageBox::warning(this,
                               "Warning",
                               ("Could not create default configuration file\"" 
                                + conffile + "\": " + 
                                strerror(errno)).c_str());
        }
        else
          slmotion::configuration::generateDefaultConfig(ofs, 80);
      }

      return qApp->exec();
    }



    std::vector<std::string> MainWindow::generateCmdOpts() const {
      vector<string> argvStrings;
      for (const auto& s : dynamic_cast<QStringListModel*>(inputFilesListView.model())->stringList())
        argvStrings.push_back(s.toStdString());

      if (scriptFileEdit.text().length() > 0) {
        argvStrings.push_back("--script");
        argvStrings.push_back(scriptFileEdit.text().toStdString());
      }

      std::string s = outVideoFileEdit.text().toStdString();
      if (s.length() > 0) {
        argvStrings.push_back("-o");
        argvStrings.push_back(s);
      }

      s = outCsvFileEdit.text().toStdString();
      if (s.length() > 0) {
        argvStrings.push_back("--feature-out");
        argvStrings.push_back(s);
      }
      
      s = annotationTemplateFileEdit.text().toStdString();
      if (s.length() > 0) {
        argvStrings.push_back("--ann-template-file");
        argvStrings.push_back(s);
      }

      s = outElanFileEdit.text().toStdString();
      if (s.length() > 0) {
        argvStrings.push_back("--elan-out");
        argvStrings.push_back(s);
      }

      if (conffile.length() > 0) {
        argvStrings.push_back("--config");
        argvStrings.push_back(conffile);
      }

      return argvStrings;
    }



    void MainWindow::showAboutBox() {
      QString text = QString() + slmotion::getVersions().c_str() + 
        "Copyright " + QChar(0x00A9) + " Aalto University 2013-2014\n"
        "http://research.ics.aalto.fi/cbir/software/\n"
        "For enquiries, please "
        "contact Matti Karppa at "
        "matti.karppa@aalto.fi";
      QMessageBox* box = new QMessageBox(QMessageBox::Information,
                                         "About slmotion",
                                         text,
                                         QMessageBox::Ok,
                                         this);
      box->show();
    }



    void MainWindow::showConfigureWindow() {
      if (confWin == nullptr) {
        confWin = new ConfigurationWindow(conffile, "Configure SLMotion");
        // confWin->resize(200, 200);
      }
      confWin->show();
    }



    void MainWindow::removeCurrentInputFile() {
      QModelIndex selection = inputFilesListView.currentIndex();
      if (selection.isValid()) 
        inputFilesListView.model()->removeRow(selection.row());
    }



    void MainWindow::addInputFile() {
       QStringList files = QFileDialog::getOpenFileNames(
                         this,
                         "Select one or more files",
                         QString(),
                         "Video files (*.avi *.mpg *.mkv *.mp4 *.mov);;Images (*.png *.bmp *.jpg)");
       auto model = dynamic_cast<QStringListModel*>(inputFilesListView.model());
       model->setStringList(model->stringList() << files);
    }



    void MainWindow::selectScriptFile() {
 
      QString s = QFileDialog::getOpenFileName(this, tr("Select File"), QString(),
                                               tr("Python Script (*.py)"));
      if (s.length() > 0)
        scriptFileEdit.setText(s);       
    }



    void MainWindow::selectAnnotationTemplateFile() {
      QString s = QFileDialog::getOpenFileName(this, tr("Select File"), QString());
      if (s.length() > 0)
        annotationTemplateFileEdit.setText(s);       
    }



    void MainWindow::selectOutVideoFile() {
      QString s = QFileDialog::getSaveFileName(this, tr("Select File"), 
                                               QString(),
                                               tr("Video files (*.avi *.mpg *.mkv *.mp4 *.mov);;Images (*.png *.bmp *.jpg"));
      if (s.length() > 0)
        outVideoFileEdit.setText(s);       
    }



    void MainWindow::selectOutCsvFile() {
      QString s = QFileDialog::getSaveFileName(this, tr("Select File"), 
                                               QString(),
                                               tr("CSV files (*.csv)"));
      if (s.length() > 0)
        outCsvFileEdit.setText(s);       
    }



    void MainWindow::selectOutElanFile() {
      QString s = QFileDialog::getSaveFileName(this, tr("Select File"), 
                                               QString(),
                                               tr("ELAN files (*.eaf)"));
      if (s.length() > 0)
        outElanFileEdit.setText(s);       
    }



    void MainWindow::getCmdOpts() {
      vector<string> opts = generateCmdOpts();
      // add quotation marks around options that contain spaces (e.g. filenames)
      for (size_t i = 0; i < opts.size(); ++i)
        if (opts[i].find(' ') != string::npos)
          opts[i] = '"' + opts[i] + '"';
      QMessageBox* box = new QMessageBox(QMessageBox::Information,
                                         "Command line options",
                                         QString::fromStdString(AdvancedString::join(" ", opts)),
                                         QMessageBox::Ok,
                                         this);
      box->show();
    }



    void MainWindow::process() {
      if (processing) {
        // cancel action
        shouldQuit = true;
        processButton.setText("Process");
        processing = false;
        statusBarMessage.setText("Canceled!");
      }
      else {
        // init process action
        shouldQuit = false;
        processButton.setText("Cancel");
        processing = true;
        statusBarMessage.setText("Processing...");

        // construct parameters
        std::vector<std::string> argvStrings = generateCmdOpts();
        argvStrings.insert(argvStrings.begin(), argv0);

        std::vector<std::vector<char> > argvArrays = stringArrayToCharArrayArray(argvStrings);
        
        std::vector<char*> argv(argvArrays.size());
        for (size_t i = 0; i < argvArrays.size(); ++i)
          argv[i] = &argvArrays[i][0];

        if (emptyBlackBoardCheckBox.isChecked() ||
            blackBoard.get() == nullptr) 
          blackBoard = std::shared_ptr<slmotion::BlackBoard>(new slmotion::BlackBoard);

        SlmotionQtCallback callback(this);

        int returnValue = slmotion::main(argv.size(), &argv[0], blackBoard, &callback);
        
        if (returnValue == 0)
          setStatusBarText("Done!");
        else {
          setStatusBarText("An error occurred.");
          const char* c = getenv("SLMOTION_ERROR_LOG_FILE");
          std::string errMess = "An error occurred while processing files. Please see ";
          errMess += c ? c : "stderr";
          errMess += " for more information.";
          QMessageBox* box = new QMessageBox(QMessageBox::Critical,
                                             "Error",
                                             QString::fromStdString(errMess),
                                             QMessageBox::Ok,
                                             this);
          box->show();
        }
        processButton.setText("Process");
        processing = false;
      }
    }

  

    void MainWindow::setStatusBarText(const std::string& text) {
      statusBarMessage.setText(QString::fromStdString(text));
    }



    /**
     * Shows a dialog and saves the contents of the Python black board to
     * a dump file
     */
    void MainWindow::saveBlackBoardDump() {
      QString filename = QFileDialog::getSaveFileName(this, tr("Select File"),
                                                      QString(),
                                                      tr("GZip files (*.gz);;"
                                                         "Dump files (*.dump);;"
                                                         "Any files (*.*)"));
      if (filename.length() > 0) {
        slmotion::DummyFrameSource dummySource;
        slmotion::BlackBoardDumpWriter bbdw(blackBoard.get(), &dummySource);

        class SaveDumpCallback : public slmotion::UiCallback {
        public:
          explicit SaveDumpCallback(QLabel* statusBarMessage) : 
            statusBarMessage(statusBarMessage) {}

          bool operator()(double percentComplete) {
            char percent[10];
            sprintf(percent, "%0.2f", percentComplete);
            statusBarMessage->setText(QString::fromStdString(std::string("Saving ") + percent +
                                              " %..."));
            QCoreApplication::processEvents();
            return true;
          }

        private:
          QLabel* statusBarMessage;
        };

        bbdw.setFilename(filename.toStdString());
        SaveDumpCallback callback(&statusBarMessage);
        try {
          bbdw.processRange(0, SIZE_MAX, &callback);
        }
        catch (slmotion::IOException& e) {
          QMessageBox* box = new QMessageBox(QMessageBox::Critical,
                                             "Error",
                                             e.what(),
                                             QMessageBox::Ok,
                                             this);
          box->show();
        }

        statusBarMessage.setText("Blackboard dump saved!");
      }
    }



    /**
     * Shows a dialog and loads the contents of the dump file to the
     * Python black board.
     */
    void MainWindow::loadBlackBoardDump() {
      QString filename = QFileDialog::getOpenFileName(this, tr("Select File"),
                                                      QString(),
                                                      tr("GZip files (*.gz);;"
                                                         "Dump files (*.dump);;"
                                                         "Any files (*.*)"));
      if (filename.length() > 0) {
        slmotion::DummyFrameSource dummySource;
        slmotion::BlackBoardDumpReader bbdr(blackBoard.get(),
                                            &dummySource);
        
        class LoadDumpCallback : public slmotion::UiCallback {
        public:
          explicit LoadDumpCallback(QLabel* statusBarMessage) : 
            statusBarMessage(statusBarMessage) {}

          bool operator()(double percentComplete) {
            char percent[10];
            sprintf(percent, "%0.2f", percentComplete);
            statusBarMessage->setText(QString::fromStdString(std::string("Loading ") + percent +
                                                             " %..."));
            QCoreApplication::processEvents();

            return true;
          }
          QLabel* statusBarMessage;
        };

        bbdr.setFilenames(std::vector<std::string>{ filename.toStdString() });
        LoadDumpCallback callback(&statusBarMessage);
        try {
          bbdr.processRange(0, SIZE_MAX, &callback);
        }
        catch (slmotion::IOException& e) {
          QMessageBox* box = new QMessageBox(QMessageBox::Critical,
                                             "Error",
                                             e.what(),
                                             QMessageBox::Ok,
                                             this);
          box->show();
        }
        statusBarMessage.setText("Blackboard dump loaded!");
      }
    }
#else // no gui
    int MainWindow::exec(int, char*[]) {
      std::cerr << "Not built with a GUI." << std::endl;
      return EXIT_FAILURE;
    }

#endif // SLMOTION_WITH_GTK

    int exec(int argc, char* argv[]) {
#ifdef SLMOTION_WITH_QT
      QApplication app(argc, argv);
#endif // SLMOTION_WITH_QT
      slmotion::gui::MainWindow mw("slmotion");
      return mw.exec(argc, argv);        
    }


    MainWindow::MainWindow(const std::string& title) : title(title), shouldQuit(false)
#if defined(SLMOTION_WITH_GTK)
                                                     , params(new Params)
#elif defined(SLMOTION_WITH_QT)
                                                     , processing(false), 
                                                       conffile(DEFAULT_CONF_FILE_FULL_PATH),
                                                       confWin(nullptr),
                                                       processButton("Process"),
                                                       blackBoard(new BlackBoard),
                                                       emptyBlackBoardCheckBox("Empty the black board")
#endif // SLMOTION_WITH_QT
    {}

    MainWindow::~MainWindow() {
#if defined(SLMOTION_WITH_GTK)
      delete params;
#elif defined(SLOMOTION_WITH_QT)
      delete confWin;
#endif // SLMOTION_WITH_QT
    }
  }
}
