#include "NaiveBayesClassifier.hpp"
#include "Classifier.hpp"

using std::vector;
using std::cerr;
using std::endl;
using cv::Mat;
using cv::Scalar;

namespace slmotion {
  void NaiveBayesClassifier::train(const vector<vector<float> > &negSamples,
				   const vector<vector<float> > &posSamples,
				   const vector<bool> *dimMask,
				   const vector<float>* /*negWeights*/,
				   const vector<float>* /*posWeights*/) {



    vector<vector<size_t> > componentLabels(2);

    size_t tolerance=30;

    if(components[0]>1)
      kmeans(negSamples,dimMask,componentLabels[0],components[0],tolerance);
    else
      componentLabels[0]=vector<size_t>(negSamples.size(),0);

    if(components[1]>1)
      kmeans(posSamples,dimMask,componentLabels[1],components[1],tolerance);
    else
      componentLabels[1]=vector<size_t>(posSamples.size(),0);



    size_t dim=posSamples[0].size();
    vector<size_t> activeDim;

    for(size_t d=0;d<dim;d++)
      if(dimMask==NULL || (*dimMask)[d]) activeDim.push_back(d);

    dim=activeDim.size();


    cerr << "active dimensionality: " << dim << endl;

    cerr << "active components: " << endl;

    for(size_t d=0;d<dim;d++){
      cerr << activeDim[d]<< " ";
    }
    cerr << endl;

    // pointers to data vectors
    vector<const vector<vector<float> > *> trainSamples(2);
    
    trainSamples[0] = &negSamples;
    trainSamples[1] = &posSamples;

    // compute covariance matrices
    inverseCovariances = vector<vector<Mat> >(2);
    vector<vector<Mat> > covarMat(2);
    means = vector<vector<Mat> >(2);
    covDetSqrt = vector<vector<double> >(2);
    componentWeights=vector<vector<double> >(2);

    
    // determine class means


    vector<size_t> compCounts;

    for(size_t cls=0;cls<2;cls++){

      means[cls]=vector<Mat>(components[cls]);
      for(size_t comp=0;comp<components[cls];comp++)
	means[cls][comp]=Mat(dim,1,CV_32FC1,Scalar::all(0));

      compCounts=vector<size_t>(components[cls],0);

      for(size_t i=0;i<trainSamples[cls]->size();i++){
	size_t lbl=componentLabels[cls][i];
	for(size_t d=0;d<dim;d++)
	  means[cls][lbl].at<float>(d,0)+=(*trainSamples[cls])[i][activeDim[d]];
	compCounts[lbl]++;
      }

      for(size_t comp=0;comp<components[cls];comp++)
	if(compCounts[comp])
	  for(size_t d=0;d<dim;d++)
	    means[cls][comp].at<float>(d,0) /= compCounts[comp];

    }
    
    cerr << "determined means:" << endl;
    cerr << "  m0= ";
    for(size_t d=0;d<dim;d++){
      cerr << means[0][0].at<float>(d,0);    
      cerr << " ";
    }
    cerr << endl;

    cerr << "determined means:" << endl;
    cerr << "  m1= ";
    for(size_t d=0;d<dim;d++){
      cerr << means[1][0].at<float>(d,0);    
      cerr << " ";
    }
    cerr << endl;

    // then determine the sample covariances (divide by N instead of N-1)

    for(size_t cls=0;cls<2;cls++){

      covarMat[cls]=vector<Mat>(components[cls]);
      inverseCovariances[cls]=vector<Mat>(components[cls]);
      covDetSqrt[cls]=vector<double>(components[cls]);
      componentWeights[cls]=vector<double>(components[cls]);

      for(size_t comp=0;comp<components[cls];comp++)
	covarMat[cls][comp]=Mat(dim,dim,CV_32FC1,Scalar::all(0));

      compCounts=vector<size_t>(components[cls],0);

      for(size_t i=0;i<trainSamples[cls]->size();i++){
	size_t lbl=componentLabels[cls][i];

	//	cerr << "lbl="<<lbl<<endl;

	for(size_t d1=0;d1<dim;d1++)
	  for(size_t d2=0;d2<=d1;d2++)
	    covarMat[cls][lbl].at<float>(d1,d2)+=
	      ((*trainSamples[cls])[i][activeDim[d1]]-means[cls][lbl].at<float>(d1,0)) *
	      ((*trainSamples[cls])[i][activeDim[d2]]-means[cls][lbl].at<float>(d2,0));
	compCounts[lbl]++;
      }

      for(size_t comp=0;comp<components[cls];comp++){
	if(compCounts[comp])
	  for(size_t d1=0;d1<dim;d1++)
	    for(size_t d2=0;d2<=d1;d2++){
	      covarMat[cls][comp].at<float>(d1,d2) /= compCounts[comp];
	      // augment the matrix using symmetry
	      if(d1!=d2)
		covarMat[cls][comp].at<float>(d2,d1) = 
		  covarMat[cls][comp].at<float>(d1,d2);
	    }

	// covariance matrix ready at this point

	inverseCovariances[cls][comp] = covarMat[cls][comp].inv(cv::DECOMP_SVD);
	covDetSqrt[cls][comp] = sqrt(determinant(covarMat[cls][comp]));
	if (covDetSqrt[cls][comp] == 0) {
	  cerr << "WARNING: The determinant of a covariance matrix was zero. This is most likely due to axes without any variance. Please be advised, the value has been replaced by a bogus value for computation purposes." << endl;
	  covDetSqrt[cls][comp] = 1;
	}

	componentWeights[cls][comp]=compCounts[comp];
	componentWeights[cls][comp] /= trainSamples[cls]->size();

      }

    }

    // set the a priori probability

    aprioriprob=trainSamples[1]->size();

    aprioriprob /= trainSamples[0]->size()+trainSamples[1]->size();

    predictioncache.clear();

  }

  float NaiveBayesClassifier::predict(const std::vector<float>  &sample,
				      const vector<bool> *dimMask){

    const vector<float> *sptr=&sample;
    size_t dim=sample.size();

    vector<float> newSample;
    
    if(dimMask){
      for(size_t d=0;d<dim;d++)
	if((*dimMask)[d])
	  newSample.push_back(sample[d]);
      sptr=&newSample;
    }


    if(predictioncache.count(sample)==0){
      size_t dim=sptr->size();

      assert(dim==(size_t)means[0][0].rows);
      
      vector<float> l(2);

      l[0]=l[1]=0;

      // determine the likelihoods (both scaled with the factor 2*pi^dim/2)
    
      double maxval = -DBL_MAX;
      
      for (size_t k = 0; k < 2; k++) {

	for(size_t comp=0; comp<components[k];comp++){
	  Mat xMinusM;
	  means[k][comp].copyTo(xMinusM);
	  for(size_t d=0;d<dim;d++)
	    xMinusM.at<float>(d,0) -= (*sptr)[d];
	  Mat result = xMinusM.t() * inverseCovariances[k][comp] * xMinusM;
	
	  maxval = std::max(maxval, -0.5 * result.at<float>(0,0));
	  l[k]+=componentWeights[k][comp]*covDetSqrt[k][comp]*exp(maxval);
	}
      }
      
      float p= l[1]*aprioriprob/(l[0]*(1-aprioriprob)+l[1]*aprioriprob);

      //      cerr << "Determined likelihoods l0="<<l[0]<<" l1="<<l[1]<<" -> p="<<p<<endl;

      predictioncache[sample]= p;
    }

    return predictioncache[sample];

  }
}
