#ifdef SLMOTION_ENABLE_OPENNI

#include "OpenNiOniFileSource.hpp"
#include "util.hpp"
// #include <boost/lexical_cast.hpp>
#include <fstream>

static std::string interpretOpenNiErrorCode(XnStatus code) {
  return xnGetStatusString(code);
}

namespace {
  static bool foo() {
    if (getenv("OPEN_NI_INSTALL_PATH") == NULL) {
      const char* c = SLMOTION_OPENNI_INSTALL_PATH;
      static char s[1024];
      strcpy(s, "OPEN_NI_INSTALL_PATH=");
      strcpy(s+strlen("OPEN_NI_INSTALL_PATH="), c);
      if (putenv(s)) {
        fprintf(stderr, "FATAL ERROR: Could not putenv: %s\n", s);
        exit(EXIT_FAILURE);
      }
    }
    return true;
  };

  static bool b = foo();
}

namespace slmotion {
  class OpenNiOniFileSource::OniTrack : public FrameSource {
  public:
    virtual double getFps() const {
      return parent->getFps();
    }

    virtual frame_number_type size() const {
      return parent->size();
    }

    OniTrack(size_t cacheSize, OpenNiOniFileSource* parent,
             const TrackInfo& trackInfo) : 
      FrameSource(cacheSize, trackInfo), parent(parent) {}

    virtual const cv::Mat& operator[](frame_number_type frameNumber) {
#ifdef SLMOTION_THREADING
      std::lock_guard<std::mutex> lock(parent->mutex);
#endif
      
      if (frameInCache(frameNumber))
        return getFrameFromCache(frameNumber);

      
      parent->seekToFrame(frameNumber);     
      parent->context.WaitAnyUpdateAll();
      
      cacheFrame(frameNumber, getFrame());
      if (frameInCache(frameNumber))
        return getFrameFromCache(frameNumber);
      else 
        throw FrameSourceException("Failed to fetch frame.");
      
      const static cv::Mat bogus;
      return bogus;
    }

  protected:
    OpenNiOniFileSource* parent;

  private:
    virtual cv::Mat getFrame() = 0;
  };



  class OpenNiOniFileSource::OniImageTrack : public OniTrack {
  public:
    OniImageTrack(size_t cacheSize, OpenNiOniFileSource* parent) : 
      OniTrack(cacheSize, parent, 
               TrackInfo { TrackInfo::Type::BGR_IMAGE, "OniImageTrack"})
    {}    
    
  private:
    virtual cv::Mat getFrame() {
      xn::ImageGenerator& imageGenerator = parent->imageGenerator;
      const XnRGB24Pixel* pixelData = imageGenerator.GetRGB24ImageMap();
      
      cv::Mat m(parent->imageGeneratorMetadata.YRes(),
                parent->imageGeneratorMetadata.XRes(), CV_8UC3,
                const_cast<XnRGB24Pixel*>(pixelData));
      
      cv::Mat m2;
      cv::cvtColor(m, m2, CV_RGB2BGR);
      return m2;
    }
  };

  

  class OpenNiOniFileSource::OniDepthTrack : public OniTrack {
  public:
    OniDepthTrack(size_t cacheSize, OpenNiOniFileSource* parent) : 
      OniTrack(cacheSize, parent,
               TrackInfo { TrackInfo::Type::DEPTH_DATA, "OniDepthTrack"}) 
    {}    
    
  private:
    virtual cv::Mat getFrame() {
      xn::DepthGenerator& depthGenerator = parent->depthGenerator;
      const XnDepthPixel* pixelData = depthGenerator.GetDepthMap();
      
      return cv::Mat(parent->depthGeneratorMetadata.YRes(),
                     parent->depthGeneratorMetadata.XRes(), CV_16UC1,
                     const_cast<XnDepthPixel*>(pixelData)).clone();
    }
  };



  class OpenNiOniFileSource::OniSkeletonTrack : public OniTrack {
  public:
    OniSkeletonTrack(size_t cacheSize, OpenNiOniFileSource* parent) : 
      OniTrack(cacheSize, parent,
               TrackInfo { TrackInfo::Type::RAW_VECTOR, "OniSkeletonTrack"}) 
    {}    
    
  private:
    /**
     * Returns a reference to the current skeleton matrix. Warning:
     * The matrix may become invalid at some point.
     */
    virtual cv::Mat getFrame() {
      assert(!parent->skeletons.empty());
      size_t frameNumber = parent->getCurrentDepthFrame();
      cv::Mat skeleton(15*2, 1, CV_64FC1);
      for (size_t j = 0; j < 15; ++j) {
        skeleton.at<double>(0, 2*j) = parent->skeletons.at<double>(frameNumber, 2*j);
        skeleton.at<double>(0, 2*j+1) = parent->skeletons.at<double>(frameNumber, 2*j+1);
      }
      return skeleton;
     }
  };



  OpenNiOniFileSource::OpenNiOniFileSource(const std::string& filename,
                                           size_t cacheSize) :
    FrameSource(cacheSize, 
                TrackInfo { TrackInfo::Type::META_TRACK, "OniMetaTrack" }),
    imageFrameCount(0), depthFrameCount(0),
    imageTrack(new OniImageTrack(cacheSize, this)),
    depthTrack(new OniDepthTrack(cacheSize, this)),
    skeletonTrack(new OniSkeletonTrack(cacheSize, this))
  {
    uint32_t r;

    if ((r = context.Init()))
      throw FrameSourceException("Could not initialise OpenNI context: " +
                                 interpretOpenNiErrorCode(r));
    
    if ((r = context.OpenFileRecording(filename.c_str(), player))) {
      throw FrameSourceException("Could not open ONI recording '" + filename + "': " + interpretOpenNiErrorCode(r));
    }

    if ((r = context.FindExistingNode(XN_NODE_TYPE_IMAGE, imageGenerator)))
      throw FrameSourceException("Could not create Image Generator: " +
                                 interpretOpenNiErrorCode(r));

    if ((r = context.FindExistingNode(XN_NODE_TYPE_DEPTH, depthGenerator)))
      throw FrameSourceException("Could not create Depth Generator: " +
                                 interpretOpenNiErrorCode(r));

    imageGenerator.GetMetaData(imageGeneratorMetadata);
    depthGenerator.GetMetaData(depthGeneratorMetadata);

    imageGeneratorNodeName = imageGenerator.GetInfo().GetInstanceName();
    depthGeneratorNodeName = depthGenerator.GetInfo().GetInstanceName();

    if (imageGenerator.GetPixelFormat() != XN_PIXEL_FORMAT_RGB24)
      imageGenerator.SetPixelFormat(XN_PIXEL_FORMAT_RGB24);

    size(); // to get depthFrameCount

    // try to read the skeleton
    std::string inputcsvfile = filename;
    inputcsvfile.replace(inputcsvfile.end()-4, 
                         inputcsvfile.end(), 
                         ".csv");
    std::ifstream csvstr(inputcsvfile);
    if (csvstr.good()) {
      skeletons = cv::Mat(depthFrameCount, 15*2, CV_64FC1, double(0));
      std:: string line;
      size_t ii = 0, i;
      while (getline(csvstr, line)) {
        std::vector<std::string> parts;
        boost::split(parts, line, boost::is_any_of(","));
        if (parts.size()==(1+15*6)) {
          i = atoi(parts[0].c_str())-1;
          for (size_t j=0; j<15; j++) {
            skeletons.at<double>(i, 2*j+0) = atof(parts[1+3+0+j*6].c_str());
            skeletons.at<double>(i, 2*j+1) = atof(parts[1+3+1+j*6].c_str());
            }
        } 
        else {
          printWarning("OpenNiOniFileSource: " + inputcsvfile + ":" + 
                       boost::lexical_cast<std::string>(ii + 1) + 
                       ": unrecognised CSV line: " + line);
        }
        ii++;
      }
    }
  }



  OpenNiOniFileSource::frame_number_type OpenNiOniFileSource::size() const {
    if (imageFrameCount == 0 && depthFrameCount == 0) {
      XnUInt32 nFrames;
      XnStatus r = player.GetNumFrames(imageGeneratorNodeName.c_str(),
                                       nFrames);
      XnUInt32 nFrames2;
      if ((r = player.GetNumFrames(depthGeneratorNodeName.c_str(),
                                   nFrames2)))
        throw FrameSourceException("Could not read the number of frames in "
                                   "ONI file: " +
                                   interpretOpenNiErrorCode(r));
      if (nFrames != nFrames2)
        printWarning("Frame count mismatch: the image generator"
                     " reported " + 
                     boost::lexical_cast<std::string>(nFrames) +
                     " frames but the depth generator "
                     "reported " +
                     boost::lexical_cast<std::string>(nFrames2)
                     + " frames!");
      imageFrameCount = nFrames;
      depthFrameCount = nFrames2;
    }
    return imageFrameCount;
  }



  const cv::Mat& OpenNiOniFileSource::operator[](frame_number_type frameNumber) {
    return (*imageTrack)[frameNumber];
  }



  size_t OpenNiOniFileSource::getCurrentDepthFrame() {
      XnUInt32 currentFrame = 1;
      XnStatus r = player.TellFrame(depthGeneratorNodeName.c_str(),
                                    currentFrame);
      if (r) 
        throw FrameSourceException("Could not tell the current frame number"
                                   " in the ONI file: " + 
                                   interpretOpenNiErrorCode(r));    
      return currentFrame;
  }



  size_t OpenNiOniFileSource::getCurrentFrame() {
      XnUInt32 currentFrame = 1;
      // XnUInt32 currentFrame2 = 2;

      // while (currentFrame != currentFrame2) {
        XnStatus r = player.TellFrame(imageGeneratorNodeName.c_str(),
                                      currentFrame);
        if (r) 
          throw FrameSourceException("Could not tell the current frame number"
                                     " in the ONI file: " + 
                                     interpretOpenNiErrorCode(r));

        // r = player.TellFrame(depthGeneratorNodeName.c_str(),
        //                      currentFrame2);
        // if (r) 
        //   throw FrameSourceException("Could not tell the current frame number"
        //                              " in the ONI file: " + 
        //                              interpretOpenNiErrorCode(r));

        // if (currentFrame != currentFrame2)
          // seekToFrame(std::min(currentFrame, currentFrame2));
          // throw FrameSourceException("Could not deduce current frame: Image "
          //                            "generator reported " + 
          //                            boost::lexical_cast<std::string>(currentFrame) +
          //                            ", but the depth generator reported " +
          //                            boost::lexical_cast<std::string>(currentFrame2));
      // }
      
      return currentFrame;
  }


  
  void OpenNiOniFileSource::seekToFrame(frame_number_type rgbFrameNumber) {
    XnStatus r = player.SeekToFrame(imageGeneratorNodeName.c_str(),
                                    rgbFrameNumber+1, 
                                    XN_PLAYER_SEEK_SET);
    if (r)
      throw FrameSourceException("ONI file seek failure: " +
                                 interpretOpenNiErrorCode(r));

    XnUInt32 currentFrame, currentFrame2;
    r = player.TellFrame(imageGeneratorNodeName.c_str(), currentFrame);
    if (r) 
      throw FrameSourceException("Could not tell the current frame number"
                                 " in the ONI file: " + 
                                 interpretOpenNiErrorCode(r));
    r = player.TellFrame(depthGeneratorNodeName.c_str(),
                         currentFrame2);
    if (r) 
      throw FrameSourceException("Could not tell the current frame number"
                                 " in the ONI file: " + 
                                 interpretOpenNiErrorCode(r));

  }



  FrameSource& OpenNiOniFileSource::getTrack(size_t trackNumber) {
    switch(trackNumber) {
    case 0:
      return *imageTrack;
      break;

    case 1:
      return *depthTrack;
      break;

    case 2:
      if (!skeletons.empty())
        return *skeletonTrack;
      // otherwise fall through to the default case

    default:
      throw std::out_of_range("The given track number exceeds the number of tracks.");
    }
    
    return *this;
  }



  double OpenNiOniFileSource::getFps() const {
    return imageGeneratorMetadata.FPS();
  }
}

#endif // SLMOTION_ENABLE_OPENNI

