#include "PHOG.hpp"
#include "SLMotionException.hpp"
#include <stdexcept>
#include <numeric>

namespace slmotion {
  double PHOG::PyramidLevel::distanceTo(const PyramidLevel& that) {
    if (that.level != level)
      throw std::invalid_argument("Square chi distance is defined only among equal levels!");

    auto it = descriptorValuesBegin;
    auto jt = that.descriptorValuesBegin;

    double distance = 0.0;

    while (it != descriptorValuesEnd) {
      double x = *it++;
      double y = *jt++;
      distance += (x-y)*(x-y)/(x+y);
    }

    if (jt != descriptorValuesEnd)
      throw SLMotionException("Square chi computation failed: histogram bin count mismatch!");

    return distance / 2.0;
  }



  PHOG::PyramidLevel PHOG::getLevel(int l) const {
    if (l < 0)
      throw std::invalid_argument("Pyramid level must be non-negative!");
    if (l >= nLevels)
      throw std::invalid_argument("Pyramid level exceeds the total number of levels!");

    assert(l < 10);
    static int cumulativeCellIndices[] = { 0, 1, 5, 21, 85, 341, 1365, 5461, 
                                           21845, 87381, 349525 };
    return PyramidLevel(descriptor.cbegin() + cumulativeCellIndices[l]*nBins,
                        descriptor.cbegin() + cumulativeCellIndices[l+1]*nBins, l);
  }



  static cv::Mat expandToRectangular(const cv::Mat& input, int newSize) {
    assert(newSize >= input.cols && newSize >= input.rows);
    assert(input.type() == CV_8UC3);

    // cv::Mat newImg(newSize, newSize, CV_8UC3, cv::Scalar::all(0));    
    // cv::Mat roi(newImg(cv::Rect((newSize-input.cols)/2, (newSize-input.rows)/2,
    //                             input.cols, input.rows)));
    // input.copyTo(roi);
    // cv::imshow("", newImg);
    // cv::waitKey(0);
    cv::Mat newImg;
    int left = (newSize-input.cols)/2;
    int right = newSize - left - input.cols;
    int top = (newSize-input.rows)/2;
    int bottom = newSize - top - input.rows;

    cv::copyMakeBorder(input, newImg, top, bottom, left, right, cv::BORDER_REPLICATE);
    // cv::imshow("", newImg);
    // cv::waitKey(0);

    return newImg;
  }



  PHOG::PHOG(const cv::Mat& input, int nBins, int maxLevels) : nBins(nBins) {
    if (maxLevels <= 0)
      throw std::invalid_argument("maxLevels must be greated than zero!");
    if (nBins <= 0)
      throw std::invalid_argument("nBins must be greated than zero!");

    // expand the image to become rectangular and a power of two
    int oldSize = std::max(input.cols, input.rows);
    int k = std::ceil(log2(oldSize));
    nLevels = std::min(maxLevels, k);
    int newSize = 1 << k;
    cv::Mat paddedInput = expandToRectangular(input, newSize);

    // start at level 0
    int l = 0;
    cv::Size cellSize = paddedInput.size();
    cv::Size blockSize = paddedInput.size();
    cv::Size winSize = paddedInput.size();
    cv::Size& blockStride = cellSize;
    cv::HOGDescriptor hog(winSize, blockSize, blockStride, cellSize, nBins);
    hog.compute(paddedInput, descriptor);

    for (l = 1; l < nLevels; ++l) {
      k -= 1;
      cellSize.height = cellSize.width = 1 << k;
      hog = cv::HOGDescriptor(winSize, blockSize, blockStride, cellSize, nBins);
      std::vector<float> hogValues;
      hog.compute(paddedInput, hogValues);
      descriptor.insert(descriptor.end(), hogValues.cbegin(), hogValues.cend());
    }

    // normalise to sum up to one
    float sum = std::accumulate(descriptor.cbegin(), descriptor.cend(), 0.0);
    if (sum > 0) {
      for (size_t i = 0; i < descriptor.size(); ++i) {
        descriptor[i] /= sum;
      }
    }
  }
}
