#ifndef SLMOTION_PHOG
#define SLMOTION_PHOG

#include <opencv2/opencv.hpp>

namespace slmotion {
  /**
   * This class represents a pyramidal HOG, inspired by the work of
   * Bosch et al.
   *
   * See:
   *
   * Bosch, Anna and Zisserman, Andrew and Munoz, Xavier. Representing
   * Shape with a Spatial Pyramid Kernel. 2007. Proceedings of the 6th
   * ACM International Conference on Image and Video Retrieval (CIVR
   * '07). doi: 10.1145/1282280.1282340
   */
  class PHOG {
  public:
    /**
     * Represents one level in the descriptor pyramid.
     */
    class PyramidLevel {
    public:
      /**
       * Returns the square chi distance to another EQUAL level
       */
      double distanceTo(const PyramidLevel& that);

      inline std::vector<float>::const_iterator begin() const {
        return descriptorValuesBegin;
      }

      inline std::vector<float>::const_iterator cbegin() const {
        return descriptorValuesBegin;
      }

      inline std::vector<float>::const_iterator end() const {
        return descriptorValuesEnd;
      }

      inline std::vector<float>::const_iterator cend() const {
        return descriptorValuesEnd;
      }

      inline size_t size() const {
        return descriptorValuesEnd - descriptorValuesBegin;
      }

      friend class PHOG;

    private:
      PyramidLevel(std::vector<float>::const_iterator descriptorValuesBegin,
                   std::vector<float>::const_iterator descriptorValuesEnd,
                   int level) :
        level(level),
        descriptorValuesBegin(descriptorValuesBegin),
        descriptorValuesEnd(descriptorValuesEnd) {}

      
      int level;
      std::vector<float>::const_iterator descriptorValuesBegin;
      std::vector<float>::const_iterator descriptorValuesEnd;
    };



    /**
     * Returns level l of the pyramid
     */
    PyramidLevel getLevel(int l) const;



    /**
     * Constructs the PHOG from regular HOGs using the following algorithm:
     * Given image I with width = height = 2^k, and
     * An algorithm for computing HOGs (such as the one in OpenCV), 
     * parametrised wrt.
     *   HOG(I, cellSize, blockSize, nBins) where
     *   cellSize is assumed rectangular, and a power of two
     *   blockSize is assumed rectangular, and a power of two, multiple
     *   of cellSize
     *   nBins the same as given as input for PHOG
     *
     *   blockStride is implicitly assumed to be equal to cellSize
     *
     * set blockSize = 2^k
     * For l in 0, ..., maxLevels:
     *   set cellSize = 2^(k-l)
     *   compute HOG(I, 2^(k-l), 2^k, nBins)
     * 
     * Concatenate HOGs and normalise to sum up to one.
     */
    PHOG(const cv::Mat& input, int nBins, int maxLevels);



    inline const std::vector<float>& getRawValues() const {
      return descriptor;
    }



    inline cv::Mat getPhogVector() {
      return cv::Mat(descriptor.size(), 1, CV_32FC1, &descriptor[0]).clone();
    }



  private:
    int nLevels;
    int nBins;
    std::vector<float> descriptor;
  };
}

#endif
