#include "Blob.hpp"
#include "Pdm.hpp"
#include "util.hpp"
#include <numeric>

using std::vector;
using cv::Point;
using cv::Mat;
using cv::Scalar;
using cv::Mat_;
using std::cout;
using std::endl;
using cv::Point2d;

namespace slmotion {
  const Pdm::PoseParameter Pdm::IDENTITY { 0, 1, 0, 0 };

  // returns
  static size_t findAnchorPointIndex(const vector<Point>& points) {
    assert(points.size() > 0);
    size_t j = 0;
    for (size_t i = 1; i < points.size(); ++i)
      if (points[i].y > points[j].y)
        j = i;
    return j;
  }



  // zero-meanifies the co-ordinate pair vector
  static Mat zeroMeanify(const Mat& vector) {
    assert(vector.cols == 1 && vector.rows > 0 && vector.rows % 2 == 0 &&
           vector.type() == CV_64FC1);

    double meanX = 0;
    double meanY = 0;

    for (int i = 0; i < vector.rows; ++i)
      if (i%2 == 0)
        meanX += vector.at<double>(i,0);
      else
        meanY += vector.at<double>(i,0);

    meanX /= (vector.rows/2);
    meanY /= (vector.rows/2);

    Mat newVector(vector.rows, 1, CV_64FC1);

    for (int i = 0; i < vector.rows; ++i)
      if (i%2 == 0)
        newVector.at<double>(i,0) = vector.at<double>(i,0) - meanX;
      else
        newVector.at<double>(i,0) = vector.at<double>(i,0) - meanY;

    return newVector;
  }



  void drawLandmarks(Mat& img, const cv::Mat& landmarkVector,
                     const Point& anchor, const Scalar& colour) {
    assert(landmarkVector.rows > 0 && landmarkVector.rows % 2 == 0 &&
           landmarkVector.cols == 1 && landmarkVector.type() == CV_64FC1);
    
    double offsetX = anchor.x - landmarkVector.at<double>(0,0);
    double offsetY = anchor.y - landmarkVector.at<double>(1,0);

    for (int i = 0; i < landmarkVector.rows; i += 2) {
      double x = landmarkVector.at<double>(i,0) + offsetX;
      double y = landmarkVector.at<double>(i+1,0) + offsetY;
      circle(img, Point(x,y), 3, colour, 1);
    }
  }
  
  
  
  void drawLandmarks(Mat& img, const cv::Mat& landmarkVector,
                     const Scalar& colour) {
    assert(landmarkVector.rows > 0 && landmarkVector.rows % 2 == 0 &&
           landmarkVector.cols == 1 && landmarkVector.type() == CV_64FC1);
    
    for (int i = 0; i < landmarkVector.rows; i += 2) {
      double x = landmarkVector.at<double>(i,0);
      double y = landmarkVector.at<double>(i+1,0);
      circle(img, Point(x,y), 3, colour, 1);
    }
  }



  /**
   * Applies the new pose to the PDM instance
   */
  cv::Mat transform(const cv::Mat& input, const Pdm::PoseParameter& pose) {
    double a1 = pose.scale*cos(pose.theta); // a1 = s cos theta
    double a2 = pose.scale*sin(pose.theta); // a2 = s sin theta
    double tx = pose.tx; // translation
    double ty = pose.ty;
    
    Mat landmarks(input.size(), input.type());
    for (int i = 0; i < input.rows/2; i++) {
      double x2x = input.at<double>(2*i, 0);
      double x2y = input.at<double>(2*i+1, 0);
      double newx2x = x2x * a1 - x2y * a2 + tx;
      double newx2y = x2x * a2 + x2y * a1 + ty;
      landmarks.at<double>(2*i) = newx2x;
      landmarks.at<double>(2*i+1) = newx2y;
    }
    return landmarks;
  }



  /**
   * Aligns the given landmark vector x2 with respect to the target
   * vector x1 as well as possible, using least squares method (as in
   * Cootes et al., 1992) without a weight term.
   *
   * Both x1 and x2 should have an equal amount of landmarks. This
   * operation can be performed in-place.
   *
   * Returns an error figure, and optionally outputs the minimised
   * parameters.
   *
   * @param x1 Reference vector. Should be a column vector with an
   * even number of double elements.
   * @param x2 The alignee, have the same properties as x1
   * @param out Optional output vector, NULL if not required.  
   * @param s Optional scaling parameter output, NULL if not required 
   * @param theta Optional rotation angle output, NULL if not required
   * @param tx Optional horizontal translation, NULL if not required 
   * @param ty Optional vertical translation, NULL if not required
   * @return Sum of squared differences between the original x2 and the 
   * realigned vector.
   * @return Sum of absolute differences with respect to parameters 
   * required for identity transformation (i.e. |tx| + |ty| + |a1 - 1| + 
   * |a2|) that can be used for testing for convergence.
   */
  double align(const Mat& x1, const Mat& x2, Mat* out, double* out_s,
               double* out_theta, double* out_tx, double* out_ty) {
    Mat landmarks(x1.size(), CV_64FC1);
    // Problem: align x2 to x1
    // 
    // by minimising expression
    // E = (x1 - M(x2))^T(x1 - M(x2))
    //  where ( x_jk )   ( (s cos theta) x_jk - (s sin theta) x_jk + t_x )
    //       M(      ) = (                                               )
    //        ( y_jk )   ( (s sin theta) x_jk + (s cos theta) x_jk + t_y )
    //
    // I.e. M is defined as a scaling/translation/rotation for each 
    // coordinate pair separately.
    //
    // Writing a_1 = s cos theta and a_2 = s sin theta,
    // differentiating with respect to each variable, we get
    //
    // ( X_2  -Y_2    N    0  ) ( a_1 )   ( X_1 )
    // ( Y_2   X_2    0    N  ) ( a_2 ) = ( Y_1 )
    // (  Z     0    X_2  Y_2 ) ( t_x )   ( C_1 )
    // (  0     Z   -Y_2  X_2 ) ( t_y )   ( C_2 )
    //
    // where X_i = sum x_ik (from k = 0 to k = N-1, assuming N landmarks)
    //       Y_i = sum y_ik
    //        Z  = sum (x^2_2k + y^2_2k)
    //       C_1 = sum (x_1k x_2k + y_1k y_2k)
    //       C_2 = sum (y_1k x_2k - x_1k y_2k)

    assert(x1.size() == x2.size() && x1.type() == CV_64FC1 && 
           x2.type() == x1.type() && x1.cols == 1 && x2.rows % 2 == 0);
    
    double X1 = 0, X2 = 0, Y1 = 0, Y2 = 0, Z = 0, C1 = 0, C2 = 0;
    int N = x1.rows/2;
    double x1x = 0, x1y = 0, x2x = 0, x2y = 0;
    
    for (int i = 0; i < x1.rows; i += 2) {
      x1x = x1.at<double>(i,0);
      x2x = x2.at<double>(i,0);
      x1y = x1.at<double>(i+1,0);
      x2y = x2.at<double>(i+1,0);
      
      X1 += x1x;
      X2 += x2x;
      
      Y1 += x1y;
      Y2 += x2y;
      
      Z += x2x * x2x + x2y * x2y;
      
      C1 += x1x * x2x + x1y * x2y;
      C2 += x1y * x2x - x1x * x2y;
    }

    // Mat A = (Mat_<double>(4,4) << 
    //          X2, Y2, Z, 0,
    //          -1 * Y2, X2, 0, Z,
    //          N, 0, X2, -1*Y2,
    //          0, N, Y2, X2);
    Mat A = (Mat_<double>(4,4) <<
             X2, -1*Y2, N, 0,
             Y2, X2, 0, N,
             Z, 0, X2, Y2,
             0, Z, -1*Y2, X2);
    
    Mat b = (Mat_<double>(4,1) <<
             X1, Y1, C1, C2);

    Mat c; // the result

    c = A.inv() * b;

    // Now that we have all necessary terms in c, we can compute transformed
    // landmarks

    double a1 = c.at<double>(0, 0); // a1 = s cos theta
    double a2 = c.at<double>(1, 0); // a2 = s sin theta
    double tx = c.at<double>(2, 0); // translation
    double ty = c.at<double>(3, 0);
    
    double squaredError = 0;
    for (int i = 0; i < N; i++) {
      x2x = x2.at<double>(2*i, 0);
      x2y = x2.at<double>(2*i+1, 0);
      double newx2x = x2x * a1 - x2y * a2 + tx;
      double newx2y = x2x * a2 + x2y * a1 + ty;
      landmarks.at<double>(2*i) = newx2x;
      landmarks.at<double>(2*i+1) = newx2y;
    }

    Mat m = x1 - landmarks;
    Mat m2;
    cv::pow(m, 2, m2);
    squaredError = cv::sum(m2)[0];
    
    // Since a1 = s cos theta, a2 = s sin theta
    // --> s = sqrt(a1^2 + a2^2)
    // theta = arctan(a2/a1)
    if (out_s)
      *out_s = sqrt(a1*a1 + a2*a2);
    
    if (out_theta)
      *out_theta = atan2(a2, a1);
    
    if (out_tx)
      *out_tx = tx;
    
    if (out_ty)
      *out_ty = ty;
    
    if (out) 
      *out = landmarks;
    
    return squaredError;
  }



  double align(const cv::Mat& x1, const cv::Mat& x2,
               Pdm::PoseParameter& newPose, cv::Mat* out) {
    return align(x1, x2, out, &newPose.scale, &newPose.theta,
                 &newPose.tx, &newPose.ty);
  }



  void getBlobLandmarks(const std::vector<Blob>& blobs, 
                        size_t nLandmarks, 
                        vector<Mat>& blobLandmarks,
                        cv::Point2d* outMeanAnchor) {
    blobLandmarks.clear();
    if (outMeanAnchor)
      *outMeanAnchor = Point2d(0,0);
    // Get blob points, create corresponding landmarks
    for (auto it = blobs.cbegin(); it != blobs.cend(); ++it) {
      // start by locating the bottommost point
      vector<Point> contourPoints = it->getContour();
      size_t anchorIndex = findAnchorPointIndex(contourPoints);

      // rotate so that the anchor becomes landmark 0
      std::rotate(contourPoints.begin(), 
                  contourPoints.begin() + anchorIndex, 
                  contourPoints.end());

      if (outMeanAnchor)
        *outMeanAnchor += Point2d(contourPoints.front().x,
                                  contourPoints.front().y);

      Mat landmarks(nLandmarks*2, 1, CV_64FC1); // for the blob
      // landmarks are stores as a 2N vector (x0,y0,x1,y1,...,xn,yn)^T
      for (size_t i = 0; i < nLandmarks; ++i) {
        size_t j = (i * contourPoints.size()) / nLandmarks;
        landmarks.at<double>(2*i,0) = contourPoints[j].x;
        landmarks.at<double>(2*i+1,0) = contourPoints[j].y;
      } 
      blobLandmarks.push_back(zeroMeanify(landmarks));
    }

    if (outMeanAnchor) {
      outMeanAnchor->x /= blobs.size();
      outMeanAnchor->y /= blobs.size();
    }
  }



  /**
   * Approximate alignment of similar training shapes (Cootes)
   */
  static void approximateAlign(const vector<Mat>& landmarks,
                               Mat& outMean, 
                               vector<Mat>& outAlignedShapes) {
    assert(landmarks.size() > 0);
    assert(landmarks.front().rows > 0 && landmarks.front().cols == 1 &&
           landmarks.front().type() == CV_64FC1);

    // align each shape vector wrt. the first shape vector
    outAlignedShapes = vector<Mat>(landmarks.size());
    for (size_t i = 0; i < landmarks.size(); ++i)
      align(landmarks.front(), landmarks[i], &outAlignedShapes[i],
            0, 0, 0, 0);

    auto computeMean = [&]() {
      Mat mean(landmarks.front().rows, 1, CV_64FC1, Scalar::all(0));
      return std::accumulate(outAlignedShapes.begin(), 
                             outAlignedShapes.end(), mean) / 
      static_cast<double>(landmarks.size());
    };

    double cumulativeError = DBL_MAX;
    double oldError = DBL_MAX;
    do {
      oldError = cumulativeError;
      align(landmarks.front(), computeMean(), &outMean, 0, 0, 0, 0);

      cumulativeError = 0;
      for (size_t i = 1; i < outAlignedShapes.size(); ++i)
        cumulativeError += align(outMean, outAlignedShapes[i], 
                                 &outAlignedShapes[i]);

      cumulativeError /= landmarks.size();
    } while(oldError - cumulativeError > 0.1);
  }



  Pdm::Pdm(const std::vector<Blob>& blobs, size_t nLandmarks) {
    this->nLandmarks = nLandmarks;
    assert(blobs.size() > 0);

    // landmarks for each blob
    vector<Mat> blobLandmarks;
    getBlobLandmarks(blobs, nLandmarks, blobLandmarks, &meanAnchor);

    vector<Mat> alignedShapes;
    approximateAlign(blobLandmarks, mean, alignedShapes);

    // delta vectors = \hat{x}^i - mean(x)
    // construct the 2N×2N covariance matrix
    Mat covarianceMatrix(nLandmarks*2, nLandmarks*2, CV_64FC1, 
                         Scalar::all(0));

    for (size_t i = 0; i < alignedShapes.size(); ++i) {
      Mat delta = mean - alignedShapes[i];
      covarianceMatrix += delta*delta.t();
    }
    covarianceMatrix /= alignedShapes.size();

    cv::eigen(covarianceMatrix, eigenValues, eigenVectors);
    eigenVectors = eigenVectors.t();
  }



  Mat Pdm::generateShape(const Mat& b) const {
    if (b.rows == 0)
      return mean;

    assert(b.type() == CV_64FC1);
    assert(b.cols == 1);

    int t = b.rows; // number of principal components to consider
    Mat Pt = eigenVectors.colRange(0, t);

    return mean + Pt*b;
  }



  cv::Mat Pdm::generateApproximateShapeDifference(const cv::Mat& dx,
                                                  size_t t,
                                                  double maxShapeParameterDeviation) const {
    Mat Pt = eigenVectors.colRange(0, t);    
    Mat db(Pt.t() * dx);
    // limit the components
    for (int i = 0; i < db.rows; ++i) {
      double limit = maxShapeParameterDeviation*sqrt(eigenValues.at<double>(i,0));
      
      if (db.at<double>(i,0) > limit)
        db.at<double>(i,0) = limit;
      if (db.at<double>(i,0) < -limit)
        db.at<double>(i,0) = -limit;
    }

    return db;
  }



  Pdm& Pdm::operator=(const Pdm& other) {
      if (&other != this) {
        Pdm temp(other);
        this->nLandmarks = temp.nLandmarks;
        this->mean = temp.mean;
        this->eigenVectors = temp.eigenVectors;
        this->eigenValues = temp.eigenValues;
      }
      return *this;
    }



  bool Pdm::operator==(const Pdm& other) const {
    return this->nLandmarks == other.nLandmarks &&
      equal(this->mean, other.mean) &&
      equal(this->eigenVectors, other.eigenVectors) &&
      equal(this->eigenValues, other.eigenValues) &&
      this->meanAnchor == other.meanAnchor;
  }
}

#if 0
// THIS IS DEFUNCT GARBAGE FROM THE OLD IMPLEMENTATION

#include "PDM.hpp"
#include "SLIO.hpp"
#include "util.hpp"

using cv::Mat;
using cv::Vec3b;
using cv::Point;
using std::vector;
using cv::Vec4i;
using cv::norm;
using cv::Scalar;
using std::cerr;
using std::endl;
using slmotion::PDM;
using cv::Point2d;
using cv::Size;



namespace slmotion {
  extern int debug;



  /**
   * Draws a colourised a map of gradients with respect to the given angle.
   *
   * @param gradX Horizontal gradient magnitudes for each pixel
   * @param gradY Vertical gradient magnitudes for each pixel
   * @param theta An angle in which to project the gradients
   *
   * @return An image of gradients projected to the direction represented by theta
   */
  Mat drawDebug2dOrientedGradientMap(const Mat& gradX, const Mat& gradY, double theta) {
    Mat temp(576, 720, CV_8UC3);
    double min1, min2, max1, max2, minimum, maximum;

    minMaxLoc(gradX, &min1, &max1);
    minMaxLoc(gradY, &min2, &max2);
      
    minimum = sqrt(min1*min1 + min2*min2);
    maximum = sqrt(max1*max1 + max2*max2);

    for (int i = 0; i < temp.size().height; i++) {
      for (int j = 0; j < temp.size().width; j++) {
	Vec3b& elem = temp.at<Vec3b>(i, j);
	double x = gradX.at<short>(i, j);
	double y = gradY.at<short>(i, j);
	double d = cos( atan2(-y, x) - theta ) * sqrt(x*x + y*y);
	elem[0] = d < 0 ? abs(d*255) / std::max(fabs(minimum), maximum) : 0;
	elem[1] = 0;
	elem[2] = d > 0 ? (d*255) / std::max(fabs(minimum), maximum) : 0;
      }
    }

    return temp;
  }

  

  bool PDM::operator==(const PDM& other) const {
    if (this->landmarks != other.landmarks)
      return false;
    //    std::cout << "landmarks are equal" << endl;

    if (!equal(this->transformation,other.transformation))
      return false;
    //    std::cout << "Transformation matrices are equal" << endl;

    if (!equal(this->eigenvalues,other.eigenvalues))
      return false;
    //    std::cout << "Eigen values are equal" << endl;

    if (this->prevS != other.prevS)
      return false;
    //    std::cout << "prevS is equal" << std::endl;
    
    if (this->prevT != other.prevT)
      return false;
    //    std::cout << "prevT is equal" << std::endl;

    if (!equal(this->prevBt,other.prevBt))
      return false;
    //    std::cout << "prevBt is equal" << std::endl;

    return true;
  }



  int PDM::intersectionArea(const PDM& a, const PDM& b,
                            const cv::Point& anchA, 
                            const cv::Point& anchB) {
    Size size(std::max(anchA.x, anchB.x)+1,
              std::max(anchA.y, anchB.y)+1);
    // kludge
    for (size_t i = 0; i < a.landmarks.size(); ++i) {
      Point p = a.landmarks[i] - a.landmarks[0] + anchA;
      if (p.x >= size.width)
        size.width = p.x + 1;
      if (p.y >= size.height)
        size.height = p.y + 1;
    }
    for (size_t i = 0; i < b.landmarks.size(); ++i) {
      Point p = b.landmarks[i] - b.landmarks[0] + anchB;
      if (p.x >= size.width)
        size.width = p.x + 1;
      if (p.y >= size.height)
        size.height = p.y + 1;
    }

    cv::Mat m1(size, CV_8UC1, cv::Scalar::all(0));
    cv::Mat m2(size, CV_8UC1, cv::Scalar::all(0));
    cv::Mat m3(size, CV_8UC1, cv::Scalar::all(0));
    a.drawLine(m1, anchA, cv::Scalar::all(255), CV_FILLED);
    b.drawLine(m2, anchB, cv::Scalar::all(255), CV_FILLED);
    min(m1, m2, m3);
    return countNonZero(m3);
  }
}



slmotion::PDM::PDM(const Mat& img, const Point& anchor, size_t nlandmarks) :
  landmarks(vector<Point>(nlandmarks)),
  prevS(1),
  prevT(0)
{
  assert(img.type() == CV_8UC1);

  Mat temp;
  img.copyTo(temp);

  vector< vector<Point> > contours;
  vector<Vec4i> hierarchy;
  findContours(temp, contours, hierarchy, CV_RETR_EXTERNAL,
	       CV_CHAIN_APPROX_NONE);
  
  if (contours.size() != 1)
    throw PDMException("Tried to create a point distribution model from a matrix that had an invalid number of contours in it");

  // Locate the anchor point
  int anchorIndex = -1;
 
  for (size_t i = 0; i < contours[0].size(); i++) {
    if (anchorIndex == -1 ||
 	norm(contours[0][i] - anchor) <
 	norm(contours[0][anchorIndex] - anchor)) {
      anchorIndex = i;
    }
  }
 
  if (anchorIndex == -1)
    throw PDMException("Anchor lookup failed while creating a PDM");


  for (size_t j = 0; j < nlandmarks; j++) {
    size_t i = (anchorIndex + (j * contours[0].size()) / nlandmarks) % contours[0].size();
    landmarks[j] = contours[0][i] - anchor;
  }
}



double slmotion::PDM::align(const PDM& x1, const PDM& x2, PDM* out, double* out_s,
		  double* out_theta, double* out_tx, double* out_ty) {
  vector<Point> landmarks(x1.landmarks.size());
  // Problem: align x2 to x1
  // 
  // by minimising expression
  // E = (x1 - M(x2))^T(x1 - M(x2))
  //  where ( x_jk )   ( (s cos theta) x_jk - (s sin theta) x_jk + t_x )
  //       M(      ) = (                                               )
  //        ( y_jk )   ( (s sin theta) x_jk + (s cos theta) x_jk + t_y )
  //
  // I.e. M is defined as a scaling/translation/rotation for each coordinate
  // pair separately.
  //
  // Writing a_1 = s cos theta and a_2 = s cos theta,
  // differentiating with respect to each variable, we get
  //
  // ( X_2  -Y_2    N    0  ) ( a_1 )   ( X_1 )
  // ( Y_2   X_2    0    N  ) ( a_2 ) = ( Y_1 )
  // (  Z     0    X_2  Y_2 ) ( t_x )   ( C_1 )
  // (  0     Z   -Y_2  X_2 ) ( t_y )   ( C_2 )
  //
  // where X_i = sum x_ik (from k = 0 to k = N-1, assuming N landmarks)
  //       Y_i = sum y_ik
  //        Z  = sum (x^2_2k + y^2_2k)
  //       C_1 = sum (x_1k x_2k + y_1k y_2k)
  //       C_2 = sum (y_1k x_2k - x_1k y_2k)

  int X1 = 0, X2 = 0, Y1 = 0, Y2 = 0, Z = 0, C1 = 0, C2 = 0;

  if (x1.landmarks.size() != x2.landmarks.size())
    throw PDMException("Cannot align two PDMs with different number of landmarks");

  int x1x, x1y, x2x, x2y;
  for (size_t i = 0; i < landmarks.size(); i++) {
    x1x = x1.landmarks[i].x;
    x2x = x2.landmarks[i].x;
    x1y = x1.landmarks[i].y;
    x2y = x2.landmarks[i].y;
    
    X1 += x1x;
    X2 += x2x;

    Y1 += x1y;
    Y2 += x2y;
    
    Z += x2x * x2x + x2y * x2y;

    C1 += x1x * x2x + x1y * x2y;
    C2 += x1y * x2x - x1x * x2y;
  }

  Mat A(4, 4, CV_64FC1);
  A.at<double>(0, 0) = X2;
  A.at<double>(1, 0) = Y2;
  A.at<double>(2, 0) = Z;
  A.at<double>(3, 0) = 0;

  A.at<double>(0, 1) = -1 * Y2;
  A.at<double>(1, 1) = X2;
  A.at<double>(2, 1) = 0;
  A.at<double>(3, 1) = Z;

  A.at<double>(0, 2) = landmarks.size();
  A.at<double>(1, 2) = 0;
  A.at<double>(2, 2) = X2;
  A.at<double>(3, 2) = -1*Y2;

  A.at<double>(0, 3) = 0;
  A.at<double>(1, 3) = landmarks.size();
  A.at<double>(2, 3) = Y2;
  A.at<double>(3, 3) = X2;

  Mat b(4, 1, CV_64FC1);
  Mat c; // the result

  b.at<double>(0, 0) = X1;
  b.at<double>(1, 0) = Y1;
  b.at<double>(2, 0) = C1;
  b.at<double>(3, 0) = C2;

  c = A.inv() * b;

  // Now that we have all necessary terms in c, we can compute transformed
  // landmarks

  double a1 = c.at<double>(0, 0); // a1 = s cos theta
  double a2 = c.at<double>(1, 0); // a2 = s sin theta
  double tx = c.at<double>(2, 0); // translation
  double ty = c.at<double>(3, 0);

  for (size_t i = 0; i < landmarks.size(); i++) {
    x2x = x2.landmarks[i].x;
    x2y = x2.landmarks[i].y;
    landmarks[i] = Point(x2x * a1 - x2y * a2 + tx + 0.5,
			 x2x * a2 + x2y * a1 + ty + 0.5);
  }

  // Since a1 = s cos theta, a2 = s sin thete
  // --> s = sqrt(a1^2 + a2^2)
  // theta = arctan(a2/a1)
  if (out_s)
    *out_s = sqrt(a1*a1 + a2*a2);

  if (out_theta)
    *out_theta = atan(a2/a1);

  if (out_tx)
    *out_tx = tx;

  if (out_ty)
    *out_ty = ty;

  if (out) {
    if (x2.transformation.empty())
      *out = PDM(landmarks);
    else
      *out = PDM(landmarks, x2.transformation, x2.eigenvalues);
  }
  return abs(tx) + abs(ty) + abs(a1 - 1) + abs(a2);
}



void slmotion::PDM::approximateAlign(const std::deque<PDM>& inputVectors, double convThresh, unsigned int maxIterations, PDM& outMean) {
  if (slmotion::debug > 1)
    cerr << "Performing PDM alignment approximation" << endl;


  if (slmotion::debug > 1)
    cerr << "Zeroing vector means" << endl;

  vector<PDM> vectors(inputVectors.size());
  copy(inputVectors.begin(), inputVectors.end(), vectors.begin());

  for (vector<PDM>::iterator it = vectors.begin();
       it != vectors.end(); it++)
    it->zeroMeanify();

  if (vectors.size() == 0) {
    slmotion::SLIO::getWarningOut() << "Could not perform alignment approximation: empty list of vectors supplied. An empty PDM was created." << endl;
    outMean = PDM();
    return;
  }
  if (vectors.size() == 1) {
    if (slmotion::debug > 1)
      cerr << "Input vector of vectors of size 1. Approximation is not necessary." << endl;
    outMean =  PDM(vectors[0].landmarks);
    return;
  }

  // Step one: align all vectors to vector number one

  vector<PDM> aligned(vectors.size());
  aligned[0] = vectors[0];
  PDM mean = vectors[0];


  for (size_t i = 1; i < vectors.size(); i++) 
    PDM::align(mean, vectors[i], &aligned[i]);

  double convergence = DBL_MAX;
  double oldConvergence = 0;
  double newConvergence = 0;
  unsigned int iterations = 0;

  while (convergence > convThresh && iterations < maxIterations) {
    // Step two: calculate mean
    mean = PDM::mean(aligned);

    // Step three: align mean to vector number one

    newConvergence = PDM::align(vectors[0], mean, &mean);
    convergence = abs(newConvergence - oldConvergence);
    oldConvergence = newConvergence;

    for (size_t i = 1; i < aligned.size(); i++) {
      PDM::align(mean, aligned[i], &aligned[i]);
    }

    if (slmotion::debug > 2)
      cerr << "Convergence factor: " << convergence << endl;
    iterations++;
  }

  if (slmotion::debug > 2) {
    if (convThresh > convergence)
      cerr << "No more iterations are necessary." << endl;
    else
      cerr << "Maximum number of iterations exceeded." << endl;
  }

  // Mean vector, [x1, y1, x2, y2, ..., xN, yN]^T
  Mat meanVector(mean.landmarks.size() * 2, 1, CV_64FC1);
  for (size_t i = 0; i < mean.landmarks.size(); i++) {
    meanVector.at<double>(2*i, 0) = mean.landmarks[i].x;
    meanVector.at<double>(2*i + 1, 0) = mean.landmarks[i].y;
  }

  // A difference vector where each element is an x or an y component of a
  // point in an aligned with the mean substracted
  Mat deltaX(mean.landmarks.size()*2, 1, CV_64FC1);

  // Covariance matrix
  Mat coVariance(mean.landmarks.size()*2, mean.landmarks.size()*2, CV_64FC1,
		 Scalar(0));

  Mat deltaXdeltaXT(mean.landmarks.size()*2, mean.landmarks.size()*2,
		    CV_64FC1);

  for (size_t i = 0; i < aligned.size(); i++) {
    for (size_t j = 0; j < aligned[i].landmarks.size(); j++) {
      deltaX.at<double>(2*j, 0) = aligned[i].landmarks[j].x;
      deltaX.at<double>(2*j+1, 0) = aligned[i].landmarks[j].y;
    }

    deltaX -= meanVector;
    deltaXdeltaXT = deltaX * deltaX.t();

    coVariance += deltaXdeltaXT;
  }

  coVariance *= 1./aligned.size();

  if (slmotion::debug > 5) {
    cerr << "Computed the following covariance matrix:" << endl;
    for (int i = 0; i < coVariance.rows; i++) {
      for (int j = 0; j < coVariance.cols; j++) {
	cerr << coVariance.at<double>(i, j) << " ";
      }
      cerr << endl;
    }
  }
 
  Mat eigenValues(1, mean.landmarks.size()*2, CV_64FC1);
  Mat eigenVectors(mean.landmarks.size() * 2, mean.landmarks.size() * 2, CV_64FC1);

  eigen(coVariance, eigenValues, eigenVectors);

  if (slmotion::debug > 5) {
    cerr << "Computed following eigenvalues:" << endl;
    for (int i = 0; i < eigenValues.cols; i++)
      cerr << eigenValues.at<double>(0, i) << endl;
  }
 
  outMean = PDM(mean.landmarks, eigenVectors.t(), eigenValues.t());
}



PDM slmotion::PDM::mean(const vector<PDM>& vectors) {
  if (vectors.size() == 0)
    return PDM();

  if (vectors.size() == 1)
    return vectors[0];

  vector<Point> landmarks(vectors[0].landmarks.size());

  for (size_t i = 0; i < vectors.size(); i++) {
    assert(vectors[i].landmarks.size() == landmarks.size());

    for (size_t j = 0; j < landmarks.size(); j++) {
      landmarks[j] += vectors[i].landmarks[j];
    }
  }

  double a = 1.0 / vectors.size();
  for (size_t j = 0; j < landmarks.size(); j++) 
    landmarks[j] *= a;

  return PDM(landmarks);
}



void slmotion::PDM::zeroMeanify() {
  Point mean(0,0);
  for (vector<Point>::const_iterator it = landmarks.begin();
       it != landmarks.end(); it++)
    mean += *it;

  mean *= 1. / landmarks.size();
  
  for (vector<Point>::iterator it = landmarks.begin();
       it != landmarks.end(); it++)
    *it -= mean;
}



void slmotion::PDM::drawLine(Mat& img, const Point& anchor, const Scalar& colour,
		   int thickness) const {
  if (landmarks.size() == 0)
    return;

  Point offset = anchor - landmarks[0];

  Point* points = new Point[landmarks.size()];
  int i = landmarks.size();

  for (size_t i = 0; i < landmarks.size(); i++)
    points[i] = landmarks[i] + offset;

  if (thickness == CV_FILLED)
    fillPoly(img, (const Point**) &points, &i, 1, colour);
  else
    polylines(img, (const Point**)(&points), &i, 1, true, colour, thickness);
  delete[] points;
}



slmotion::PDM::PDM(const vector<Point>& landmarks, const Mat& trans,
	 const Mat& eigenv) :
  landmarks(landmarks),
  prevS(1),
  prevT(0)
{ 
  trans.copyTo(transformation);
  eigenv.copyTo(eigenvalues);

  if (transformation.cols != static_cast<int>(landmarks.size()) * 2 ||
      transformation.rows != static_cast<int>(landmarks.size()) * 2)
    throw PDMException("A transformation matrix of unexpected dimensions provided");

  if (eigenvalues.cols != static_cast<int>(landmarks.size()) * 2 ||
      eigenvalues.rows != 1)
    throw PDMException("An eigenvalue vector of unexpected dimensions provided");
}



PDM slmotion::PDM::createApproximate(const Mat& shape) const {
  if (shape.cols != 1 || shape.rows < 1 || shape.rows > transformation.rows)
    throw PDMException("Tried to create a PDM approximate with a shape vector of invalid dimensions");

  assert(shape.type() == CV_64FC1);
  assert(transformation.type() == CV_64FC1);

  Mat tempTrans = transformation.colRange(0, shape.rows);



  Mat temp = tempTrans * shape;
  vector<Point> tempP(landmarks.size());
  for (int i = 0; i < static_cast<int>(tempP.size()); i++)
    tempP[i] = landmarks[i] + Point(temp.at<double>(2*i,0),
				    temp.at<double>(2*i+1,0));

  return PDM(tempP);
}



void slmotion::PDM::fitASM(const Mat& target, const Mat& mask, 
		 const Point& anchorGuess, PDM& bestFit, Point& anchor,
		 const AsmFittingContext& fittingContext, 
		 const Mat* blackOutMask) {
  assert(target.size().width == mask.size().width);
  assert(target.size().height == mask.size().height);
  assert(mask.type() == CV_8UC1);
  if (blackOutMask) {
    assert(blackOutMask->type() == CV_8UC1);
    assert(blackOutMask->size() == target.size());
  }

  // imshow("output", *blackOutMask);
  // waitKey(0);

  assert(this->landmarks.size() > 0);
  assert(!this->transformation.empty());
  assert(!this->eigenvalues.empty());

  if (slmotion::debug > 1)
    cerr << "Fitting ASM" << endl;

  // Step one: Choose initial coordinates by
  // - Looking for the optimal angle, and rotating the mean accordingly
  // - Looking at the anchor guess, and translating accordingly


  // Find the initial pose by adjusting rotation (at 30 degree intervals)
  // Scaling is approximated with respect to the rotation 
  // Translation is obtained through the anchor parameter

  double theta, s;


  if (!prevBt.empty() && fittingContext.initialPoseEstimationMethod == 
      AsmFittingContext::POSE_AND_SHAPE_HISTORY)
    bestFit = this->createApproximate(prevBt);
  else
    bestFit = PDM(*this);

  if (fittingContext.initialPoseEstimationMethod != 
      AsmFittingContext::PROBE_ONLY && 
      bestFit.evaluatePoseGoodness(mask, prevS, prevT, anchorGuess) > 0.7) {
    theta = prevT;
    s = prevS;
    if (slmotion::debug > 2)
      cerr << "Using historical data for initial pose" << endl;
  }
  else {
    if (fittingContext.initialPoseEstimationMethod == 
	AsmFittingContext::POSE_AND_SHAPE_HISTORY)
      bestFit = PDM(*this);
      
    if (slmotion::debug > 2)
      cerr << "Probing initial pose" << endl;

    findInitialThetaS(mask, anchorGuess, theta, s, fittingContext.scaleApproximationFactor);
  }

  if (slmotion::debug > 2)
    cerr << "Estimated s = " << s << ", theta = " << theta << " for the initial pose." << endl;

  Point oldCentroid = bestFit.getCentroid();

  PDM tempPDM = bestFit.transform(1, theta, 0, 0);

  Point offset = anchorGuess - tempPDM.landmarks[0]; // Offset = the initial translation that takes the anchor to its guessed location

  double tx = offset.x;
  double ty = offset.y;

  assert(landmarks.size() > 1);

  // Compute the gradient image
  Mat maskedTarget(target.size(), CV_8UC3, Scalar::all(0));
  if (blackOutMask)
    target.copyTo(maskedTarget, *blackOutMask);
  else
    target.copyTo(maskedTarget);

  vector<Point> newMarks = bestFit.landmarks;

  double ds, dtx, dty, dtheta; // Differences

  double convergence = DBL_MAX;

  int iterations = 0;
  
  // The approximate shape for the best fit
  Mat bt(fittingContext.pcaComponentCount, 1, CV_64FC1, Scalar(0)); 

  Mat gradX, gradY, gradientImage;
  if (fittingContext.targetMethod == AsmFittingContext::DIRECTIONAL) 
    computeGradientMap2d(maskedTarget, gradX, gradY,
			 fittingContext.sobelApertureSize,
			 fittingContext.equaliseHistogram,
			 fittingContext.gradientMethod);
  else if (fittingContext.targetMethod == AsmFittingContext::GRADIENT)
    gradientImage = computeGradientMap(maskedTarget, fittingContext.sobelApertureSize, fittingContext.equaliseHistogram, fittingContext.gradientMethod);
  else if (fittingContext.targetMethod == AsmFittingContext::CANNY) {
      Mat gsi;
      cvtColor(maskedTarget, gsi, CV_BGR2GRAY);
      if (fittingContext.equaliseHistogram)
	equalizeHist(gsi, gsi);

      Canny(gsi, gradientImage, fittingContext.cannyThreshold1, fittingContext.cannyThreshold2, 7, true);

  }
  else {
    cerr << "slmotion: WARNING: Attempted to use an unimplemented target selection method" << endl;
  }


  while (convergence > fittingContext.convergenceThreshold && 
	 iterations < fittingContext.maxIterations) {

    // Step two: Go through each landmark point, scan points along the line
    //           (normal to the boundary)

    if (slmotion::debug > 3) {
      cerr << "A new ASM iteration:" << endl;
      cerr << "Initial pose parameters: " << endl << "s = " << s <<
	", theta = " << theta << ", tx = " << tx << ", ty = " << ty << endl;
    }

    tempPDM = bestFit.transform(s, theta, tx, ty);

    // debug
    /*
    if (slmotion::gDebugFrameNumber == 20 ||
	slmotion::gDebugFrameNumber == 21 || 
	slmotion::gDebugFrameNumber == 22) {
      Mat temp(target.size(), CV_8UC3, Scalar::all(0));

      double minimum, maximum;
      minMaxLoc(gradientImage, &minimum, &maximum);
    
      for (int i = 0; i < temp.size().height; i++) {
	for (int j = 0; j < temp.size().width; j++) {
	  Vec3b& elem = temp.at<Vec3b>(i, j);
	  double d = gradientImage.at<short>(i, j);
	  elem[0] =  d < 0 ? abs(d*255) / max(abs(minimum), maximum) : 0;
	  elem[1] = 0;
	  elem[2] = d > 0 ? (d * 255) / max(abs(minimum), maximum) : 0;
	}
      }

      for (size_t i = 0; i < bestFit.landmarks.size(); i++)
	circle (temp, tempPDM.landmarks[i], 3, Scalar::all(255), 1);
    
      imshow("output", temp);
      waitKey(0);
    }
    */

    /*
    for (size_t i = 0; i < tempPDM.landmarks.size(); i++)
      circle(coloured, tempPDM.landmarks[i], 3, Scalar::all(85), 1);
    */

    if (fittingContext.targetMethod == AsmFittingContext::DIRECTIONAL) 
      // Compute gradient images
      findDirectionalTargetLandmarks(tempPDM.landmarks, newMarks, 
				     gradX, gradY, fittingContext);
    
    else if (fittingContext.targetMethod == AsmFittingContext::GRADIENT) 
      findTargetLandmarks(tempPDM.landmarks, newMarks, gradientImage, fittingContext.intensityFactor, fittingContext.maxLookDistance, fittingContext.minIntensity, fittingContext.targetType);
    else if (fittingContext.targetMethod == AsmFittingContext::CANNY)
      findCannyTargetLandmarks(gradientImage, tempPDM.landmarks, newMarks,
			       fittingContext);
    else
      cerr << "slmotion: WARNING: Attempted to use an unimplemented target selection method" << endl;
    

    /*
    for (size_t i = 0; i < newMarks.size(); i++)
      circle(coloured, newMarks[i], 3, Scalar::all(170), 1);
    */

    Mat displacement1(newMarks.size() * 2, 1, CV_64FC1); 
    // Displacement vector (delta x in the book), displacement between each
    // component and their target
    for (size_t i = 0; i < newMarks.size(); i++) {
      displacement1.at<double>(2*i, 0) = newMarks[i].x - tempPDM.landmarks[i].x;
      displacement1.at<double>(2*i+1, 0) = newMarks[i].y - tempPDM.landmarks[i].y;

      /*
      line(coloured, tempPDM.landmarks[i],
	   Point(tempPDM.landmarks[i].x + displacement1.at<double>(2*i, 0),
		 tempPDM.landmarks[i].y + displacement1.at<double>(2*i+1,0)),
		 Scalar(0,0,127));*/
    }

    // Step three: adjust the pose to make points best fit their targets
    align(PDM(newMarks), bestFit, &tempPDM, &ds, &dtheta, &dtx, &dty);

    ds = ds/s - 1.0;
    dtheta -= theta;
    dtx -= tx;
    dty -= ty;

    /*
    for (size_t i = 0; i < tempPDM.landmarks.size(); i++)
      circle(coloured, tempPDM.landmarks[i], 3, Scalar::all(255), 1);
    */

    // Step four: compute displacement vector dx~
    Mat displacement2(newMarks.size() * 2, 1, CV_64FC1); 

    // Formula from the book:
    // dx~ = M(-(theta + dtheta),[s(1+ds)]^-1)[M(theta,s)x~ + dx - (dtx,dty)] - x~

    Mat xtilde = bestFit.toVector();
    Mat tempM = transform(xtilde, s, theta, -dtx, -dty) + displacement1;
    Mat dxtilde = transform(tempM, 1./(s+s*ds), -1.*(theta + dtheta),0,0) -
      xtilde;

    assert(dxtilde.rows == static_cast<int>(2*landmarks.size()));
    assert(dxtilde.cols == 1);
    assert(dxtilde.type() == CV_64FC1);


    // for (int i = 0; i < dxtilde.rows; i++)
    //  cerr << dxtilde.at<double>(i, 0) << endl;

    // Step five: compute dbt
    Mat dbt = transformation.colRange(0, fittingContext.pcaComponentCount).t() * dxtilde;

    if (slmotion::debug > 3) {
      cerr << "dbt:" << endl;
      for (int i = 0; i < dbt.rows; i++)
	cerr << dbt.at<double>(i, 0) << endl;
    }

    bt += dbt;

    // Constrain component values
    for (int i = 0; i < bt.rows; i++) {
      if (abs(bt.at<double>(i, 0)) > fittingContext.maxShapeDeviation * 
	      sqrt(abs(eigenvalues.at<double>(0, i)))) {
	
	if (slmotion::debug > 3) {
	  cerr << "Constraining";
	  if (bt.at<double>(i, 0) < 0)
	    cerr << " negative";
	  cerr << " bt component " << i << ": " <<
	    bt.at<double>(i, 0) << "->";
}

	bt.at<double>(i, 0) =
	  copysign(fittingContext.maxShapeDeviation * sqrt(abs(eigenvalues.at<double>(0, i))), bt.at<double>(i, 0));

	if (slmotion::debug > 3)
	  cerr << bt.at<double>(i, 0) << endl;
	
      }
    }

    bestFit = this->createApproximate(bt);

    if (slmotion::debug > 3) {
      cerr << "ds = " << ds << endl;
      cerr << "dtheta = " << dtheta << endl;
      cerr << "dtx = " << dtx << endl;
      cerr << "dty = " << dty << endl;
    }

    s *= 1 + ds;
    theta += dtheta;
    tx += dtx;
    ty += dty;

    convergence = sum(abs(dbt))[0];

    if (slmotion::debug > 3)
      cerr << "sum |dbt_i| = " << convergence << endl;

    iterations++;
  }

  if (slmotion::debug > 3) {
    if (convergence < fittingContext.convergenceThreshold)
      cerr << "No more iterations are necessary" << endl;
    else
      cerr << "Maximum number of iterations exceeded" << endl;
  }

  bestFit = bestFit.transform(s, theta, tx, ty);
  anchor = bestFit.landmarks[0];
  prevS = s;
  prevT = theta;
  bt.copyTo(prevBt);
}



void slmotion::PDM::findTargetLandmarks(const vector<Point>& src, vector<Point>& newTargets, const Mat& gradImg, double intensityFactor, int maxLookDistance, double minIntensity, AsmFittingContext::TargetType type) const {
  assert(gradImg.type() == CV_16SC1);
  assert(newTargets.size() == src.size());

  // Since absolute values cannot be < 0, use mean
  if (minIntensity < 0) {
    minIntensity = cv::mean(abs(gradImg))[0];
  }

  Point2d p, nrml;
  Point maxIntP; // The point where intensity gradient reaches its maximum
  short maxInt;
  short initialInt;
  for (size_t i = 0; i < landmarks.size(); i++) {
    // circle(coloured, src[i], 3, Scalar::all(127), 1);
    nrml = normal(static_cast<int>(i));
    p = src[i];
    // line(coloured, p + nrml * maxLookDistance,
    //	 p - nrml * maxLookDistance, Scalar::all(255));

    assert(maxLookDistance > 0);
    maxIntP = p;
    maxInt = gradImg.at<short>(maxIntP);
    if (type == AsmFittingContext::ABSOLUTE)
      maxInt = abs(maxInt);
    else if (type == AsmFittingContext::NEGATIVE &&
	     maxInt > 0)
      maxInt = 0;
    else if (type == AsmFittingContext::POSITIVE &&
	     maxInt < 0)
      maxInt = 0;

    initialInt = maxInt;

    for (int j = -1 * maxLookDistance; j < maxLookDistance; j++) {
      // The intensity factor is there to make sure that the new target is
      // clearly more intensive
      // Regardless of the type, we can always consider absolute values
      short s = gradImg.at<short>(p + nrml * j);
      if (abs(s) > intensityFactor * abs(initialInt) &&
	  abs(s) > abs(minIntensity)) {

	// Check the type here
	if ((type == AsmFittingContext::NEGATIVE && 
	     s < maxInt) ||
	    (type == AsmFittingContext::POSITIVE &&
	     s > maxInt) ||
	    (type == AsmFittingContext::ABSOLUTE &&
	     abs(s) > maxInt)) {
	  if (slmotion::debug > 4)
	    cerr << "New landmark with the following absolute intensity: " <<
	      abs(gradImg.at<short>(p + nrml * j)) << endl;
	  maxIntP = p + j * nrml;
	  maxInt = s;
	}
      }
    }

    // circle(coloured, maxIntP, 3, Scalar::all(255), 2);

    newTargets[i] = maxIntP;
  }  
}


 
void slmotion::PDM::findInitialThetaS(const Mat& mask, const Point& anchorGuess,
			    double& outTheta, double& outS, 
				      std::pair<double, double> scaleFactor) const {
  assert(mask.type() == CV_8UC1);

  if (countNonZero(mask) == mask.rows * mask.cols) {
    outTheta = 0;
    outS = 1;
    return;
  }

  double initTheta = 0;
  double initS = 1;
  int initThetaGoodness = 0;
  Mat temp(mask.size(), CV_8UC1, Scalar(0));
  PDM tempPDM;
  Point offset;

  // Find the extreme dimensions for the image in the frame
  int maxx = 0, maxy = 0, minx = mask.cols, miny = mask.rows;
  for (int y = 0; y < mask.rows; y++) {
    for (int x = 0; x < mask.cols; x++) {
      if (mask.at<uchar>(y, x) != 0) {
	if (x < minx)
	  minx = x;
	if (y < miny)
	  miny = y;
	if (x > maxx)
	  maxx = x;
	if (y > maxy)
	  maxy = y;
      }
    }
  }
  Size curSize(maxx-minx, maxy - miny);


  assert(this->landmarks.size() > 0);

  for (double theta = 0; theta < 2*M_PI; theta += 2*M_PI/36) {
    tempPDM = this->transform(1, theta, 0, 0);
    offset = anchorGuess - tempPDM.landmarks[0];
    tempPDM = tempPDM.transform(1, 0, offset.x, offset.y);
    min(tempPDM.toFilledMatrix(mask.size(), anchorGuess), mask, temp);
    int i = countNonZero(temp);
    if (i > initThetaGoodness) {
      initTheta = theta;
      initThetaGoodness = i;
    }
  }

  if (scaleFactor.first > 0 && scaleFactor.second > 0) {
    for (double s = 0.1; s < 1.5; s += 0.05) {
      tempPDM = this->transform(s, initTheta, 0, 0);

      // offset = anchorGuess - tempPDM.landmarks[0];
      // tempPDM = tempPDM.transform(1, 0, offset.x, offset.y);
    
      // Grow it just until either its width or height exceeds that of the
      // reference
      Size siz = tempPDM.computeSize();

      mask.copyTo(temp);
      // tempPDM.drawLandmarks(temp, 3, 1, Scalar::all(127), anchorGuess);

      initS = s;

      if (siz.width > curSize.width*scaleFactor.first || 
	  siz.height > curSize.height*scaleFactor.second ||
	  (siz.width > curSize.width && siz.height > curSize.height))
	break;    
    }
  }

  outTheta = initTheta;
  outS = initS;
}



Mat slmotion::PDM::toFilledMatrix(const Size& size, const Point& anchor) const {
  assert(landmarks.size() > 0);

  Mat newMatrix(size, CV_8UC1, Scalar(0));
  Point offset = anchor - landmarks[0]; 

  // I'm not sure if this is a good idea, but a vector should be guaranteed
  // to be contiguous
  vector<Point> fixedPoints(landmarks.size());
  
  for (size_t i = 0; i < landmarks.size(); i++)
    fixedPoints[i] = landmarks[i] + offset;

  int nPoints = fixedPoints.size();
  const Point* fixedPointsPtr = &fixedPoints[0];
  fillPoly(newMatrix, &fixedPointsPtr, &nPoints, 1, Scalar::all(255));

  return newMatrix;
}



Size slmotion::PDM::computeSize() const {
  // Find extreme points
  Point leftMost = landmarks[0];
  Point rightMost = landmarks[0];
  Point topMost = landmarks[0];
  Point bottomMost = landmarks[0];

  for (vector<Point>::const_iterator it = landmarks.begin();
       it != landmarks.end(); it++) {

    if (it->x < leftMost.x)
      leftMost = *it;

    if (it->x > rightMost.x)
      rightMost = *it;

    if (it->y < topMost.y)
      topMost = *it;

    if (it->y > bottomMost.y)
      bottomMost = *it;
  }

  return Size(rightMost.x - leftMost.x,
	      bottomMost.y - topMost.y);
}



Mat slmotion::PDM::computeGradientMap(const Mat& src, int apertureSize, bool equaliseHistogram, AsmFittingContext::GradientType gradientType) {
  Mat gsi;
  Mat gradientImage;
  Mat gradx, grady;

  switch(gradientType) {

  case AsmFittingContext::LAPLACIAN:
  case AsmFittingContext::LAPLACIAN4:
  case AsmFittingContext::LAPLACIAN8:
  
    cvtColor(src, gsi, CV_BGR2GRAY);
    if (equaliseHistogram)
      equalizeHist(gsi, gsi);

    if (gradientType == AsmFittingContext::LAPLACIAN)
      Laplacian(gsi, gradientImage, CV_16SC1, apertureSize);
    else {
      Mat kernel = Mat(3, 3, CV_32FC1);
      if (gradientType == AsmFittingContext::LAPLACIAN4) {
	kernel.at<float>(0, 0) = 0;
	kernel.at<float>(0, 1) = 1;
	kernel.at<float>(0, 2) = 0;

	kernel.at<float>(1, 0) = 1;
	kernel.at<float>(1, 1) = -4;
	kernel.at<float>(1, 2) = 1;

	kernel.at<float>(2, 0) = 0;
	kernel.at<float>(2, 1) = 1;
	kernel.at<float>(2, 2) = 0;
      }
      else if (gradientType == AsmFittingContext::LAPLACIAN8) {
	kernel.at<float>(0, 0) = 1;
	kernel.at<float>(0, 1) = 1;
	kernel.at<float>(0, 2) = 1;

	kernel.at<float>(1, 0) = 1;
	kernel.at<float>(1, 1) = -8;
	kernel.at<float>(1, 2) = 1;

	kernel.at<float>(2, 0) = 1;
	kernel.at<float>(2, 1) = 1;
	kernel.at<float>(2, 2) = 1;
      }

      filter2D(gsi, gradientImage, CV_16SC1, kernel);
    }

    break;

  case AsmFittingContext::PREWITT:
  case AsmFittingContext::ROBERTS:
  case AsmFittingContext::ROBINSON:
  case AsmFittingContext::KIRSCH:
  case AsmFittingContext::SOBEL:
    computeGradientMap2d(src, gradx, grady, apertureSize, equaliseHistogram,
			 gradientType);
    gradientImage = Mat(src.rows, src.cols, CV_16SC1);
    for (int i = 0; i < src.rows; i++) {
      for (int j = 0; j < src.cols; j++) {
	gradientImage.at<short>(i, j) = 
	  sqrt(gradx.at<short>(i,j) * gradx.at<short>(i,j) +
	       grady.at<short>(i,j) * grady.at<short>(i,j));
      }
    }
    break;

  default:
    cerr << "slmotion: WARNING: Tried to approximate the gradient with a " <<
      "method that has not been implemented yet." << endl;
    break;
  }

  return gradientImage;
}



void slmotion::PDM::computeGradientMap2d(const Mat& src, Mat& outX, Mat& outY, 
			       int apertureSize, bool equaliseHistogram,
			       AsmFittingContext::GradientType gradientType) {
  Mat gsi;

  cvtColor(src, gsi, CV_BGR2GRAY);

  if (equaliseHistogram)
    equalizeHist(gsi, gsi);

  Mat kernel;

  switch(gradientType) {

  case AsmFittingContext::SOBEL:
    Sobel(gsi, outX, CV_16SC1, 1, 0, apertureSize);
    Sobel(gsi, outY, CV_16SC1, 0, 1, apertureSize);
    break;

  case AsmFittingContext::PREWITT:
    kernel = Mat(3, 3, CV_32FC1);
    kernel.at<float>(0, 0) = -1;
    kernel.at<float>(0, 1) = -1;
    kernel.at<float>(0, 2) = -1;

    kernel.at<float>(1, 0) = 0;
    kernel.at<float>(1, 1) = 0;
    kernel.at<float>(1, 2) = 0;

    kernel.at<float>(2, 0) = 1;
    kernel.at<float>(2, 1) = 1;
    kernel.at<float>(2, 2) = 1;

    filter2D(gsi, outY, CV_16SC1, kernel);

    kernel.at<float>(0, 0) = -1;
    kernel.at<float>(0, 1) = 0;
    kernel.at<float>(0, 2) = 1;

    kernel.at<float>(1, 0) = -1;
    kernel.at<float>(1, 1) = 0;
    kernel.at<float>(1, 2) = 1;

    kernel.at<float>(2, 0) = -1;
    kernel.at<float>(2, 1) = 0;
    kernel.at<float>(2, 2) = 1;

    filter2D(gsi, outX, CV_16SC1, kernel);
    break;

  case AsmFittingContext::ROBERTS:
    kernel = Mat(2, 2, CV_32FC1);
    kernel.at<float>(0, 0) = -1;
    kernel.at<float>(0, 1) = 0;

    kernel.at<float>(1, 0) = 0;
    kernel.at<float>(1, 1) = 1;

    filter2D(gsi, outY, CV_16SC1, kernel);

    kernel.at<float>(0, 0) = 0;
    kernel.at<float>(0, 1) = 1;

    kernel.at<float>(1, 0) = -1;
    kernel.at<float>(1, 1) = 0;

    filter2D(gsi, outX, CV_16SC1, kernel);
    break;

  case AsmFittingContext::ROBINSON:
    kernel = Mat(3, 3, CV_32FC1);
    kernel.at<float>(0, 0) = -1;
    kernel.at<float>(0, 1) = -1;
    kernel.at<float>(0, 2) = -1;

    kernel.at<float>(1, 0) = -1;
    kernel.at<float>(1, 1) = 2;
    kernel.at<float>(1, 2) = -1;

    kernel.at<float>(2, 0) = 1;
    kernel.at<float>(2, 1) = 1;
    kernel.at<float>(2, 2) = 1;

    filter2D(gsi, outY, CV_16SC1, kernel);

    kernel.at<float>(0, 0) = -1;
    kernel.at<float>(0, 1) = 1;
    kernel.at<float>(0, 2) = 1;

    kernel.at<float>(1, 0) = -1;
    kernel.at<float>(1, 1) = -2;
    kernel.at<float>(1, 2) = 1;

    kernel.at<float>(2, 0) = -1;
    kernel.at<float>(2, 1) = 1;
    kernel.at<float>(2, 2) = 1;

    filter2D(gsi, outX, CV_16SC1, kernel);
    break;

  case AsmFittingContext::KIRSCH:
    kernel = Mat(3, 3, CV_32FC1);
    kernel.at<float>(0, 0) = -3;
    kernel.at<float>(0, 1) = -3;
    kernel.at<float>(0, 2) = -3;

    kernel.at<float>(1, 0) = -3;
    kernel.at<float>(1, 1) = 0;
    kernel.at<float>(1, 2) = -3;

    kernel.at<float>(2, 0) = 5;
    kernel.at<float>(2, 1) = 5;
    kernel.at<float>(2, 2) = 5;

    filter2D(gsi, outY, CV_16SC1, kernel);

    kernel.at<float>(0, 0) = -5;
    kernel.at<float>(0, 1) = 3;
    kernel.at<float>(0, 2) = 3;

    kernel.at<float>(1, 0) = -5;
    kernel.at<float>(1, 1) = 0;
    kernel.at<float>(1, 2) = 3;

    kernel.at<float>(2, 0) = -5;
    kernel.at<float>(2, 1) = 3;
    kernel.at<float>(2, 2) = 3;

    filter2D(gsi, outX, CV_16SC1, kernel);
    break;
    
  default:
    cerr << "slmotion: WARNING: Tried to approximate the gradient with a " <<
      "method that has not been implemented yet." << endl;
    break;
  }
}



void slmotion::PDM::findDirectionalTargetLandmarks(const vector<Point>& srcPoints, vector<Point>& newTargets, const Mat& gradX, const Mat& gradY, const AsmFittingContext& fittingContext) const {
  assert(gradX.type() == CV_16SC1);
  assert(gradY.type() == CV_16SC1);
  assert(srcPoints.size() == newTargets.size());
  assert(fittingContext.targetMethod == AsmFittingContext::DIRECTIONAL);
  cv::Rect frameBoundaries(0, 0, std::min(gradX.cols, gradY.cols),
                                 std::min(gradY.rows, gradY.rows));

  Point2d p, nrml, initialP;
  double maxInt;
  Point maxIntP;
  double initialInt;
  double t;
  double x, y;

  for (size_t i = 0; i < srcPoints.size(); i++) {
    p = forcePointIntoRect(srcPoints[i], frameBoundaries);
    assert(p.x < gradX.cols);
    assert(p.x < gradY.cols);
    assert(p.y < gradY.rows);
    assert(p.y < gradX.rows);
    assert(p.x >= 0);
    assert(p.y >= 0);
    initialP = p;

    nrml = normal(i);

    x = gradX.at<short>(p);
    y = gradY.at<short>(p);
    t = atan2(-y, x) - atan2(nrml.y, nrml.x);

    initialInt = cos(t) * sqrt(x*x + y*y);

    if (fittingContext.targetType == AsmFittingContext::ABSOLUTE)
      initialInt = abs(initialInt);
    else if (fittingContext.targetType == AsmFittingContext::NEGATIVE &&
	     initialInt > 0)
      initialInt = 0;
    else if (fittingContext.targetType == AsmFittingContext::POSITIVE &&
	     initialInt < 0)
      initialInt = 0;


    maxInt = initialInt;
    maxIntP = p;

    for (int j = -1 * fittingContext.maxLookDistance; 
	 j < fittingContext.maxLookDistance; j++) {
      p = initialP + nrml * j;
      p = forcePointIntoRect(p, frameBoundaries);
      assert(p.x < gradX.cols);
      assert(p.x < gradY.cols);
      assert(p.y < gradY.rows);
      assert(p.y < gradX.rows);
      assert(p.x >= 0);
      assert(p.y >= 0);
      
      x = gradX.at<short>(p);
      y = gradY.at<short>(p);
      t = atan2(-y, x) - atan2(-nrml.y, nrml.x);
      //      cerr << "for (" << x << ", " << y << "), t = " << atan2(-y,x) << " - " << atan2(-nrml.y, nrml.x) << " = " << t << endl;

      double d = cos(t) * sqrt(x*x + y*y);
      // cerr << "j = " << j << ", d = " << d << endl;

      if ((fittingContext.targetType == AsmFittingContext::ABSOLUTE &&
	   abs(d) > fittingContext.intensityFactor * initialInt &&
	   abs(d) > maxInt) ||
	  (fittingContext.targetType == AsmFittingContext::POSITIVE &&
	   d > fittingContext.intensityFactor * initialInt &&
	   d > maxInt) ||
	  (fittingContext.targetType == AsmFittingContext::NEGATIVE &&
	   d < fittingContext.intensityFactor * initialInt &&
	   d < maxInt)) { 
	maxIntP = p;
	maxInt = d;
      }
    }

    newTargets[i] = maxIntP;


    // debug
    /*
    if (slmotion::gDebugFrameNumber == 1) {
      Mat temp = 
	slmotion::drawDebug2dOrientedGradientMap(gradX, gradY,
						 atan2(-nrml.y, nrml.x));

      cerr << "(" << initialP.x << ", " << initialP.y << ")-->" <<
	"(" << maxIntP.x << ", " << maxIntP.y << ")" << endl;

      circle(temp, initialP, 3, Scalar::all(127), 1);
      circle(temp, maxIntP, 3, Scalar::all(255), 1);
      imshow("output", temp);
      waitKey(20);
    }
    */

  }
}




void slmotion::PDM::findCannyTargetLandmarks(const Mat& canny, const vector<Point>& srcPoints, vector<Point>& newTargets, const AsmFittingContext& fittingContext) const {
  assert(canny.type() == CV_8UC1);
  assert(srcPoints.size() == newTargets.size());
  assert(fittingContext.targetMethod == AsmFittingContext::CANNY);

  Point2d p, bestP, nrml;

  // debug stuff
  //  canny.copyTo(gsi);
  for (size_t i = 0; i < srcPoints.size(); i++) {
    // circle(gsi, srcPoints[i], 3, Scalar::all(127));

    p = srcPoints[i];
    bestP = p;

    nrml = normal(i);

    for (int j = 0; j < 2 * fittingContext.maxLookDistance; j++) {
      if (canny.at<uchar>(p) != 0) {
	bestP = p;
	break;
      }
      else
	p += (nrml * j) * (1 - (j % 2)*2);
    }
   
    newTargets[i] = bestP;
    // circle(gsi, p, 3, Scalar::all(255));
  }
  // imshow("output", gsi);
  //  waitKey(0);
}



Point2d slmotion::PDM::computeCentroid(const Point& p) const {
  if (landmarks.size() == 0)
    return Point2d(-1,-1);

  Point sum(0, 0);
  Point offset = p - landmarks[0];
  for (vector<Point>::const_iterator it = landmarks.begin();
       it != landmarks.end(); it++) {
    sum += (*it + offset);
  }
  return Point2d(sum.x,sum.y) * (1./landmarks.size());
}



double slmotion::PDM::computeOrientation() const {
  if (landmarks.size() == 0)
    return 0;

  size_t farthestIndex = 0;
  double farthestDistance = 0;

  // locate the farthest point
  for (size_t i = 0; i < landmarks.size(); i++) {
    double d = norm(landmarks[i] - landmarks[0]);
    if (d > farthestDistance) {
      farthestIndex = i;
      farthestDistance = d;
    }
  }

  Point v = landmarks[farthestIndex] - landmarks[0];
  return atan2(v.y, v.x);
}
#endif

