#ifndef SLMOTION_PYTHON_COMPONENT_BASE
#define SLMOTION_PYTHON_COMPONENT_BASE

#include "PythonNotImplementedException.hpp"
#include <boost/python.hpp>

namespace slmotion {
  namespace python {
    /**
     * This is the C++ version of the python base class used by Python
     * programmers when implementing their own components. Because this
     * is mainly an interface thing, it does not actually derive from
     * Component.
     */
    class PythonComponentBase {
    public:
      explicit PythonComponentBase() :
        uiCallback(nullptr) {}

      virtual ~PythonComponentBase() {}

      inline void setUiCallback(UiCallback* u) {
        uiCallback = u;
      }

      PythonComponentBase(const PythonComponentBase&) = default;
      PythonComponentBase& operator=(const PythonComponentBase&) = default;

      // this class is supposed to represent an abstraction, so the sole
      // function of this function is to raise an exception
      virtual void process(unsigned int) = 0;



      /**
       * Calls the callback function (whatever that may be). The value
       * should indicate if the process should continue as well as the
       * progress of the process: a negative value will indicate that
       * the process should quit while a positive value is taken to be a
       * percentage complete value (i.e. a number in the range of
       * [0,100])
       *
       * The return value indicates if the user wants to terminate the
       * process.
       */
      bool callback(double value) {
        if (uiCallback != nullptr)
          return (*uiCallback)(value);
        else
          return true;
      }




      /**
       * The default implementation models the behaviour of the
       * Component class: it just goes through every frame, and calls
       * process.
       *
       * Successfully completing the process should return true; false
       * should be returned if something goes wrong or the callback
       * returns false
       */
      virtual inline bool processRange(unsigned int firstFrame, unsigned int lastFrame) {
        for (unsigned int i = firstFrame; i < lastFrame; ++i) {
          process(i);
          if (callback(i*100./(lastFrame-firstFrame)) == false)
            return false;
        }
        return true;
      }

      virtual std::string getShortDescription() const {
        return "A Python component";
      }

      virtual std::string getLongDescription() const {
        return "Python configuration file help text";
      }

      virtual boost::python::object getRequirements() const = 0;

      virtual boost::python::object getProvided() const = 0;

      virtual std::string getShortName() const = 0;

      virtual std::string getComponentName() const = 0; 
      // {
      //   throw PythonNotImplementedException("The base class does not provide an "
      //                                       "implementation for the "
      //                                       "getComponentName() function! The "
      //                                       "user of this class should provide "
      //                                       "one themself.");
      //   return "Python Component";
      // }

    private:
      UiCallback* uiCallback;
    };


    // this wrapper manages the python overrides
    class PythonComponentBaseWrapper : public PythonComponentBase, 
                                       public boost::python::wrapper<PythonComponentBaseWrapper> {
    public:
      virtual inline bool processRange(unsigned int firstFrame, unsigned int lastFrame) {
        if (boost::python::override f = this->get_override("processRange")) 
          return f(firstFrame, lastFrame);
        else {
          return PythonComponentBase::processRange(firstFrame, lastFrame);
        }
      }

      virtual void process(unsigned int frameNumber) {
        if (boost::python::override f = this->get_override("process")) 
          f(frameNumber);
        else {
          throw PythonNotImplementedException("The base class does not provide an "
                                              "implementation for the process "
                                              "function! The user of this class "
                                              "should provide one themself.");
        }
      }

      virtual std::string getShortDescription() const  {
        if (boost::python::override f = this->get_override("getShortDescription")) 
          return f();
        else
          return PythonComponentBase::getShortDescription();
      }

      virtual std::string getLongDescription() const {
        if (boost::python::override f = this->get_override("getLongDescription")) 
          return f();
        else
          return PythonComponentBase::getLongDescription();
      }

      virtual boost::python::object getRequirements() const {
        if (boost::python::override f = this->get_override("getRequirements")) 
          return f();        
        else
          throw PythonNotImplementedException("The base class does not provide an "
                                              "implementation for the "
                                              "getRequirements() function! The "
                                              "user of this class should provide "
                                              "one themself.");
        return boost::python::object();
      }

      virtual boost::python::object getProvided() const {
        if (boost::python::override f = this->get_override("getProvided")) 
          return f();        

        throw PythonNotImplementedException("The base class does not provide an "
                                            "implementation for the "
                                            "getProvided() function! The "
                                            "user of this class should provide "
                                            "one themself.");
        return boost::python::object();
      }

      virtual std::string getShortName() const {
        if (boost::python::override f = this->get_override("getShortName")) 
          return f();        

        throw PythonNotImplementedException("The base class does not provide an "
                                            "implementation for the "
                                            "getShortName() function! The "
                                            "user of this class should provide "
                                            "one themself. (Typically, this "
                                            "function should return the name of "
                                            "the current class on the Python "
                                            "side)");
        return "PythonComponent";
      }

      virtual std::string getComponentName() const {
        if (boost::python::override f = this->get_override("getComponentName")) 
          return f();        

        throw PythonNotImplementedException("The base class does not provide an "
                                            "implementation for the "
                                            "getComponentName() function! The "
                                            "user of this class should provide "
                                            "one themself.");
        return "Python Component";
      }
    };
  }
}

#endif

