#ifndef SLMOTION_PYTHON_ENVIRONMENT
#define SLMOTION_PYTHON_ENVIRONMENT

#include "ComponentSpecification.hpp"
#include "configuration.hpp"

namespace slmotion {
  namespace python {
    class PythonEnvironment {
    public:
      static const char* const SLMOTION_PYTHON_ENVIRONMENT_VARIABLE;

      /**
       * Initialises the global state of the API, i.e. sets the appropriate
       * objects used for analysis on the python side. This function should be 
       * called before processScript, or undefined behaviour may occur.
       *
       * @param commandLineParams Possible command line parameters and
       * such (may be empty)
       *
       * @param frameSource Set this frame source as the global main frame source
       *
       * @param pythonBlackBoard Set this black board as the python
       * black board; if it is a pointer to nullptr, the function will
       * initialise the black board on its own
       */
      PythonEnvironment(std::vector<slmotion::configuration::Job>* jobsPtr,
                        slmotion::configuration::SLMotionConfiguration* optsPtr,
                        const boost::program_options::variables_map* commandLineParams,
                        std::shared_ptr<FrameSource> frameSource,
                        std::shared_ptr<BlackBoard> pythonBlackBoard = 
                        std::shared_ptr<BlackBoard>(nullptr),
                        UiCallback* uiCallback = nullptr);

    
      /**
       * Given the filename of a Python script, fire up the Python interpreter, and
       * execute the script, providing all necessary auxiliary functions the script
       * may need.
       *
       * @param Possible UiCallback that is passed to the components
       */
      void processScript(const std::string& scriptfile);



      /**
       * Sets the current main frame source
       */
      inline void setFrameSource(std::shared_ptr<FrameSource> fs) {
        pythonMainFrameSource = fs;
      }

      inline const std::vector<slmotion::configuration::Job>* getJobs() const {
        return jobs;
      }

      inline slmotion::configuration::SLMotionConfiguration* getOpts() {
        return opts;
      }

      inline const std::vector<ComponentSpecification>& getComponentChain() const {
        return componentChain;
      }

      inline std::vector<ComponentSpecification>* getComponentChainPtr() {
        return &componentChain;
      }

      inline std::shared_ptr<BlackBoard> getBlackBoard() {
        return pythonBlackBoard;
      }

      inline void setBlackBoard(std::shared_ptr<BlackBoard> bb) {
        pythonBlackBoard = bb;
      }

      inline std::shared_ptr<FrameSource> getMainFrameSource() {
        return pythonMainFrameSource;
      }

      inline const boost::program_options::variables_map* getCommandLineParameters() const {
        return commandLineParameters;
      }

      inline UiCallback* getUiCallback() {
        return callback;
      }
      
    private:
      // initialises the global name space by clearing the dictionary,
      // injecting built-ins, and setting the environment pointer
      void initGlobals();



      slmotion::configuration::SLMotionConfiguration* opts;
      std::vector<slmotion::configuration::Job>* jobs;
      const boost::program_options::variables_map* commandLineParameters;
      std::shared_ptr<FrameSource> pythonMainFrameSource;
      std::shared_ptr<BlackBoard> pythonBlackBoard;
      UiCallback* callback;
      std::vector<ComponentSpecification> componentChain;
      boost::python::dict globalNamespace;
      std::map<std::string, boost::python::object> registeredComponents; ///< name->class object map for registered components
    };



    /**
     * Returns the current environment
     */
    PythonEnvironment* getEnv();



    /**
     * A helper function which will convert boost::any objects to
     * boost::python::objects
     */
    boost::python::object anyToObject(const boost::any& any);



    /**
     * A helper function whih attempts to convert a Python object to a
     * boost::any object containing a valid C++ object of a similar
     * type
     */
    boost::any objectToAny(const boost::python::object& object);



    /**
     * A helper function whih attempts to convert a Python object to a
     * boost::any object containing a valid C++ object of a similar
     * type
     *
     * This version takes a hint as to the type via the typeSpec parameter
     */
    boost::any objectToAnyAs(const boost::python::object& object,
                             const std::string& typeSpec);



    boost::python::object matrixToNumpyArray(const cv::Mat& m);
  }
}

#endif
