#include "RawFaceDetector.hpp"

using std::string;

namespace slmotion {
  static RawFaceDetector DUMMY(true);

  boost::program_options::options_description RawFaceDetector::getConfigurationFileOptionsDescription() const {
    boost::program_options::options_description opts;
    opts.add_options()
      ("RawFaceDetector.cascade", 
       boost::program_options::value<string>()->default_value(getDefaultHaarCascadeFile()),
       "Sets the cascade filename. A valid cascade file is required for the classifier to work "
       "properly.")
      ("RawFaceDetector.minneighbours", boost::program_options::value<int>()->default_value(2),
       "Minimum number of overlapping neighbouring detections required for "
       "a detection to be considered valid. Larger number means more accurate "
       "detections (in terms of false positives), but at the cost of possibly "
       "missing some valid detections.")
      ("RawFaceDetector.scalefactor", 
       boost::program_options::value<double>()->default_value(1.2),
       "How big a jump there is between different scales; accuracy can be "
       "increased by moving this value closer to 1.0, but this will increase "
       "computation time.");
    return opts;
  }



  Component* RawFaceDetector::createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const {
    std::string cascade = getDefaultHaarCascadeFile();
    if (configuration.count("RawFaceDetector.cascade"))
      cascade = configuration["RawFaceDetector.cascade"].as<std::string>();
    RawFaceDetector faceDetector(blackBoard, frameSource, 
                                 cascade);

    if (configuration.count("RawFaceDetector.minneighbours"))
      faceDetector.setMinNeighbours(configuration["RawFaceDetector.minneighbours"].as<int>());

    if (configuration.count("RawFaceDetector.scalefactor"))
      faceDetector.setScaleFactor(configuration["RawFaceDetector.scalefactor"].as<double>());

    return new RawFaceDetector(faceDetector);
  }



  void RawFaceDetector::process(frame_number_t frnumber) {   
    if (frnumber >= getFrameSource().size())
      throw std::out_of_range("Tried to detect face in a frame whose index exceeds the total number of frames!");

    assert(getFrameSource().size() > 0);
    getBlackBoard().set(frnumber, FACEDETECTOR_RAW_FACE_VECTOR_BLACKBOARD_ENTRY, 
                        getRawFaces(frnumber));
  }  
}
