#ifndef SLMOTION_RULE_SKIN_DETECTOR
#define SLMOTION_RULE_SKIN_DETECTOR

#include "SkinDetector.hpp"

namespace slmotion {
  /**
   * This is a very simple skin detector based on simple rules presented in
   *  Jura Kovač, Peter Peer, Franc Solina, "Human Skin Colour Clustering
   *  for Face Detection", 2003
   *
   * A similar detector was apparently used as an initial skin detector in 
   *  Han, Awad, Sutherland, "Automatic skin segmentation and tracking in 
   *  sign language recognition", IET Computer Vision, 2009
   */
  class RuleSkinDetector : public SkinDetector {
  public:
    enum class Ruleset : char {
      /**
       * Ruleset A is as follows:
       *  pixel x = (R,G,B) is skin iff
       *   R > 95 AND G > 40 AND B > 20 AND max{R,G,B} - min{R,G,B} > 15 AND
       *   |R-G| > 15 AND R > G AND R > B
       */
      A,
      /**
       * Ruleset B is as follows:
       * pixel x = (R,G,B) is skin iff
       *   R > 220 AND G > 210 AND B > 170 AND |R-G| <= 15 AND R > B AND 
       *   G > B
       */
      B
    };



    virtual std::string getShortDescription() const {
      return "A rule-based skin detector using a set of predefined rules from Kovač et al.";
    }


    virtual std::string getLongDescription() const {
      return "A rule-based skin detector using a set of predefined rules "
        "from Kovač et al. Ruleset A is defined as follows:\n"
        "pixel x = (R,G,B) is skin iff\n"
        "R > 95 AND G > 40 AND B > 20 AND max{R,G,B} - min{R,G,B} > 15 "
        "AND |R-G| > 15 AND R > G AND R > B\n"
        "Ruleset B is defined as follows:\n"
        "pixel x = (R,G,B) is skin iff\n"
        "R > 220 AND G > 210 AND B > 170 AND |R-G| <= 15 AND R > B AND "
        "G > B";
    }

    virtual property_set_t getRequirements() const {
      return property_set_t();
    }

    virtual std::string getShortName() const {
      return "RuleSkinDetector";
    }

    virtual std::string getComponentName() const {
      return "Rule Skin Detector";
    }

    virtual void reset() {
    }

    virtual boost::program_options::options_description getConfigurationFileOptionsDescription() const;

    /**
     * The ordinary constructor. Colour Space has been fixed.
     */
    RuleSkinDetector(BlackBoard* blackBoard,
                     FrameSource* frameSource,
                     Ruleset ruleSet) :
      SkinDetector(blackBoard, frameSource, *ColourSpace::BGR),
      ruleset(ruleSet) {}

    explicit RuleSkinDetector(bool) : SkinDetector(true) {}

    RuleSkinDetector(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource);

    virtual boost::program_options::options_description getCommandLineOptionsDescription() const {
      return boost::program_options::options_description();
    }

  private:
    virtual Component* createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const;

    void detect(const cv::Mat& inFrame, cv::Mat& outMask);

    Ruleset ruleset;
  };
}

#endif
