#ifndef SLMOTION_SKINFILTER
#define SLMOTION_SKINFILTER

#include "Component.hpp"
#include "ColourSpace.hpp"
#include "Filter.hpp"

namespace slmotion {
  const std::string SKINDETECTOR_BLACKBOARD_MASK_ENTRY = "skinmask"; // cv::Mat CV_8UC1



  /**
   * An abstract base class for skin detectors. Eats frames, outputs
   * binary masks.
   */
  class SkinDetector : public Component {
  public:
    // dummy constructor
    SkinDetector(bool);



    /**
     * Parses the configuration and creates the component
     */
    SkinDetector(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource);



    /**
     * Returns the component name
     */
    inline std::string getComponentName() const {
      return "Skin Detector";
    }



    /**
     * Processes a frame. This may also include pre or postprocessing
     * stages.
     *
     * @param inFrame Input frame
     * @param outMask Output binary mask
     * @param outBlobs Optional output parameter for blobs
     */
    void process(size_t framenr);



    /**
     * Trains the detector using given image data. The default
     * implementation does nothing, this function is only relevant if it is
     * overridden by a subclass implementation. However, the layout should
     * remain the same.
     *
     * @param samples Example images. Assumed to be 24 bit BGR images.
     * @param groundTruths Ground truths for the examples. Assumed to be 8 
     * bit single channel binary images where any value greater than 0 is 
     * used for skin.
     * @param update Whether the model should be updated or utterly
     * retrained. Not all detectors may support this option.
     */
    virtual void train(const std::vector<cv::Mat>& samples, 
                       const std::vector<cv::Mat>& groundTruths,
                       bool update = false);



    /**
     * A pure virtual destructor
     */
    virtual ~SkinDetector() = 0;



    /**
     * A simple constructor
     */
    explicit SkinDetector(BlackBoard* blackBoard, FrameSource* frameSource,
                          const ColourSpace& c) :
      Component(blackBoard, frameSource),
      colourSpace(c.clone())
    { }



    /**
     * Adds a new filter to post-detection phase.
     *
     * @param f A filter. Will be cloned.
     */
    inline void addPostProcessFilter(const PostProcessFilter& f) {
      postProcessFilters.push_back(std::shared_ptr<PostProcessFilter>(static_cast<PostProcessFilter*>(f.clone())));
    }
    


    inline const ColourSpace& getColourSpace() const {
      return *colourSpace;
    }
    


    inline virtual property_set_t getProvided() const {
      return property_set_t { SKINDETECTOR_BLACKBOARD_MASK_ENTRY };
    }



    inline void postprocess(const cv::Mat& original, cv::Mat& target) const { 
      for (auto it = postProcessFilters.cbegin();
           it != postProcessFilters.cend(); ++it)
        (*it)->apply(original, target);
    }



    virtual boost::program_options::options_description getConfigurationFileOptionsDescription() const;



    inline void setColourSpace(const ColourSpace& c) {
      colourSpace = std::shared_ptr<ColourSpace>(c.clone());
    }



    void reset() { }



  private:
    // Prevent copying
    SkinDetector(const SkinDetector&);
    SkinDetector& operator=(const SkinDetector&);



    /**
     * An abstract function to perform the actual detection using whatever
     * algorithm the subclass needs to use. Parameters are as with the
     * process function.
     *
     * @param inFrame Input frame (assumed to be in BGR colour space)
     * @param outMask Output binary mask
     */
    virtual void detect(const cv::Mat& inFrame, cv::Mat& outMask) = 0;



    /**
     * This function pointer member is a hack
     */
    boost::program_options::options_description(*conffileOptionsDescriptionFunction)();



    std::shared_ptr<ColourSpace> colourSpace; ///< Internal colour space    
    std::vector<std::shared_ptr<PostProcessFilter>> postProcessFilters;
  };



  /**
   * Perform K means clustering and store the largest clusters in the
   * clusters parameter.
   *
   * @param samples Input samples
   * @param clusters Output cluster means
   * @param means Cluster means
   * @param k The number of clusters to return
   * @param totalClusters Total number of clusters. If 0, the number is
   * detected automagically.
   */
  void getBiggestClusters(const cv::Mat& samples,
                          std::vector<cv::Mat>& clusters,
                          cv::Mat& means, int k, size_t totalClusters = 0);

  /**
   * Attempt to locate the best number of clusters using K means. Validation
   * set is selected randomly. The number is selected by using Akaike
   * information criterion.
   *
   * @param samples Samples as row vectors
   * @param maxK Maximum number K of clusters to create
   *
   * @return The best K that could be found
   */
  int findBestK(const cv::Mat& samples, int maxK = 10);



  /**
   * Attempt to locate the best number of clusters using K means. The number
   * is set by using the elbow criterion.
   */
  int findBestK2(const cv::Mat& samples, double elbowCriterion = 0.1,
                 int maxK = 10);



  /**
   * Computes validation error for samples in the validation set, defined as
   * error = sum_t=1^N |min_i (x^t - m_i)|^2
   */
  double kMeansValidationError(const cv::Mat& samples,
                               const cv::Mat& clusterMeans);



  /**
   * Returns a new binary matrix where an ellipse has been
   * drawn inside the rectangle, and everything else is 0
   *
   * @param size Desired size
   * @param rect Desired rectangle
   *
   * @return A new matrix
   */
  inline  cv::Mat getEllipseMask(const cv::Size& size,
                                 const cv::Rect& rect) {
    cv::Mat ret(size, CV_8UC1, cv::Scalar(0));
    ellipse(ret, cv::Point(rect.x + rect.width / 2,
                           rect.y + rect.height / 2),
            cv::Size(rect.width / 2, rect.height / 2), 0, 0, 360,
            cv::Scalar::all(255), CV_FILLED);
    return ret;
  }
    


  /**
   * Returns a new binary matrix where a rectangle has been drawn and
   * everything else is 0
   *
   * @param size Desired size
   * @param rect Desired rectangle
   *
   * @return A new matrix
   */
  inline  cv::Mat getRectangleMask(const cv::Size& size,
                                   const cv::Rect& rect) { 
    cv::Mat ret(size, CV_8UC1, cv::Scalar(0));
    rectangle(ret, rect.tl(), rect.br(), cv::Scalar::all(255), CV_FILLED);
    return ret;
  }



  /**
   * Converts pixels from the source matrix whose mask-value is non-zero
   * to positive training samples. Pixels whose mask value is zero are
   * ignored. The template parameter T sets the input matrix element type.
   *
   * @param src Source image. Any necessary colour space conversions 
   * should have been done beforehand.
   * @param mask Binary pixel mask
   * 
   * @return A sample matrix where each row is a colour vector. The
   * corresponding response vector is not provided because it is trivial:
   * every entry is non-zero.
   */
  template<typename T>
  cv::Mat nonZeroToSamples(const cv::Mat& src, const cv::Mat& mask) {
    assert(mask.size() == src.size());
    assert(mask.type() == CV_8UC1);

    int sampleCount = cv::countNonZero(mask);

    // row vectors
    cv::Mat ret(sampleCount, src.channels(), CV_32FC1);

    float* retP = ret.ptr<float>();
    for (int y = 0; y < mask.rows; ++y) {
      const T* srcP = src.ptr<T>(y);
      for (int x = 0; x < mask.cols; ++x) {
        if (mask.at<uchar>(y,x)) {
          for (int k = 0; k < src.channels(); ++k) {
            retP[k] = srcP[k];
          }
          retP += ret.step1();
        }
        srcP += src.channels();
      }
    }

    return ret;
  }



  inline cv::Mat nonZeroToSamples(const cv::Mat& src, const cv::Mat& mask) {
    assert(src.depth() == CV_8U || src.depth() == CV_32F);
    if (src.depth() == CV_8U)
      return nonZeroToSamples<uchar>(src, mask);
    else if (src.depth() == CV_32F)
      return nonZeroToSamples<float>(src, mask);
    else
      return cv::Mat();
  }
}

#endif
