#ifdef SLMOTION_THREADING
#ifndef SLMOTION_THREAD
#define SLMOTION_THREAD

#include <cassert>
#include <iostream>
#include "exceptions.hpp"

#include <thread>

// These helper macros can be used to reduce unnecessary clutter e.g. in 
// inlined functions. The macros are defined even if threads are disabled.

/**
 * Locks this->mutex using a lock_guard
 */
#define LOCK_THIS_MUTEX_WHEN_THREADING() std::lock_guard<std::mutex> lk(this->mutex)

/**
 * Given two objects, locks their mutex members
 */
#define LOCK_THIS_AND_OTHER(A, B)                              \
  lock((A).mutex, (B).mutex);                                  \
  std::lock_guard<std::mutex> lk1((A).mutex, std::adopt_lock); \
  std::lock_guard<std::mutex> lk2((B).mutex, std::adopt_lock);


namespace slmotion {
  /**
   * std::lock has not been implemented yet as of GCC 4.5, so this manual 
   * implementation may be used instead to lock two mutexes in a 
   * deadlock-free manner
   */
  template<typename MUTEX1, typename MUTEX2>
  void lock(MUTEX1& mutex1, MUTEX2& mutex2) {
    static std::mutex internalMutex;
    std::unique_lock<std::mutex> internalLock(internalMutex);
    const int ARBITRARY_INTERVAL = 1000; // ms

    // try to lock the two mutexes, in case of failure, sleep for an 
    // arbitrary interval

    auto tryToLock = [&]() {
      if (mutex1.try_lock()) {
        if (mutex2.try_lock())
          return true;

        mutex1.unlock();
      }
      return false;
    };

    // std::condition_variable cond;
    // cond.wait(internalLock, tryToLock);

    while (!tryToLock()) {
      internalLock.unlock();
      std::this_thread::sleep_for(std::chrono::milliseconds(ARBITRARY_INTERVAL));
      internalLock.lock();
    }
  }



  class ThreadException : public SLMotionException {
  public:
    explicit ThreadException(const char* msg) :
      SLMotionException(msg) {}
  };



  /**
   * This class encapsulates threads, in order to force a maximum number of
   * threads. Other helping facilities may be implemented at a later time.
   */
  class Thread {
  public:
    inline static size_t getThreadCount() {
      std::lock_guard<std::mutex> lk(threadMutex);
      return threadCount;
    }

    

    inline static void setMaxThreads(size_t s) {
      std::lock_guard<std::mutex> lk(threadMutex);
      maxThreads = s;
    }



    /**
     * Returns the maximum number of threads allowed.
     */
    inline static size_t getMaxThreads() {
      return maxThreads;
    }



    /**
     * Dynamically reserves memory and returns a pointer to a new thread 
     * object, constructed using the parameters given (exactly as with 
     * std::thread), unless the number of threads is equal to or greater 
     * than the maximum number of threads, in which case a NULL pointer is 
     * returned.
     */
    template<typename ...Args>
    static Thread* createThreadIfPossible(Args&&... params) {
      std::lock_guard<std::mutex> lk(threadMutex);
      if (threadCount >= maxThreads)
        return NULL;

      return new Thread(std::forward<Args>(params)...);
    }



    bool joinable() {
      return thread.joinable();
    }



    void join() {
      thread.join();
    }



    ~Thread() {
      std::lock_guard<std::mutex> lk(threadMutex);
      assert(threadCount > 0);
      --threadCount;
    }



    /**
     * Move constructor. Increments the number of threads but does not 
     * enforce the maximum (since the true thread instance is moved).
     */
    Thread(Thread&& other) : thread(std::move(other.thread)) {
      std::lock_guard<std::mutex> lk(threadMutex);
      ++threadCount;
    }



  private:
    template<typename ...Args>
    Thread(Args&&... params) {
      if (threadCount < maxThreads) {
        thread = std::thread(std::forward<Args>(params)...);
        ++threadCount;
      }
      else
        throw ThreadException("Maximum number of threads exceeded!");
    }



    // no copies
    Thread(const Thread&) = delete;
    Thread& operator=(const Thread&) = delete;
    Thread& operator=(Thread&&) = delete;

    static size_t threadCount; ///< Total number of of threads in existence
    static size_t maxThreads; ///< Maximum number of threads allowed
    static std::mutex threadMutex; ///< Mutex for changing any of the two fields above

    std::thread thread; ///< Thread associated with the object
  };
}

#endif
#else
#define LOCK_THIS_MUTEX_WHEN_THREADING()
#define LOCK_THIS_AND_OTHER(X,Y)
#endif
