#ifndef SLMOTION_TRACKED_POINT
#define SLMOTION_TRACKED_POINT

#include "BodyPart.hpp"

namespace slmotion {
  /**
   * This class represents a point that is tracked by the tracker. A
   * point is characterised by an ID, and a location vector. A copy
   * with an updated location is stored for each frame. Lost points
   * should be deleted.
   *
   * Points are mostly immutable. The only way to alter the object
   * is through copy-assignment. Point identities can be compared,
   * but they have no order.
   */
  class TrackedPoint {
  public:
    /**
     * Internal point ID type
     */
    typedef unsigned long long int point_id_type_t;



    /**
     * Point (location vector) type that is wrapped within
     */
    typedef cv::Point2f point_t;



    /**
     * Converts the TrackedPoint to an ordinary point. The
     * identity is lost in the conversion.
     */
    inline explicit operator point_t() const {
      return location;
    }



    /**
     * Returns true if the two points have the same ID
     */
    inline bool operator==(const TrackedPoint& other) const {
      return other.id == id;
    }



    /**
     * Returns true if the two points do NOT have the same ID
     */
    inline bool operator!=(const TrackedPoint& other) const {
      return other.id != id;
    }



    /**
     * Returns true if ID of the LHS goes before RHS
     *
     * Mostly useful with ordered associative STL containers
     */
    inline bool operator<(const TrackedPoint& other) const {
      return id < other.id;
    }



    /**
     * Creates a new point and assigns a unique ID to it
     *
     * @param point Point to associate with
     */
    explicit TrackedPoint(const point_t& point) :
      id(uniqueIdCounter++), location(point), 
      bodyPartName(BodyPart::UNKNOWN) {}



    /**
     * Creates a copy of the point with an updated location (e.g. the 
     * location in the next frame). ID will be preserved.
     *
     * @param point New location
     */
    inline TrackedPoint relocate(const point_t& point) const {
      return TrackedPoint(this->id, point, this->bodyPartName);
    }



    /**
     * Default copy constructor
     */
    TrackedPoint(const TrackedPoint&) = default;



    /**
     * Default copy assignment
     */
    TrackedPoint& operator=(const TrackedPoint&) = default;



    inline void setPartName(BodyPart::PartName newPartName) {
      bodyPartName = newPartName;
    }



    inline BodyPart::PartName getPartName() const {
      return bodyPartName;
    }
      


    /**
     * Returns the unique ID number. The ID has no particular meaning; it is
     * only used to tell different features apart.
     */
    inline point_id_type_t getId() const {
      return id;
    }



    /**
     * Computes the vector difference between the two points and returns a 
     * copy as internal point type (identity is lost)
     */
    inline point_t operator-(const TrackedPoint& other) const {
      return location - other.location;
    }



    /**
     * Computes the vector sum between the two points and returns a 
     * copy as internal point type (identity is lost)
     */
    inline point_t operator+(const TrackedPoint& other) const {
      return location + other.location;
    }



    /**
     * Creates a desired object manually (BEWARE!)
     */
    TrackedPoint(point_id_type_t id, const point_t& point, 
                 BodyPart::PartName part) :
      id(id), location(point), bodyPartName(part) {}



  private:
    /**
     * Not default-constructible
     */
    TrackedPoint() = delete;



    /**
     * Internal ID. These should never be assigned manually, it is done 
     * automagically. Users should not meddle with them, except to tell
     * points apart
     */
    point_id_type_t id;
    point_t location; ///< Internal location
    BodyPart::PartName bodyPartName;

    static point_id_type_t uniqueIdCounter; ///< Used to keep track of IDs.
  };
}

#endif
