#include "TrackingSkinDetector.hpp"
#include "BlackBoardDumpWriter.hpp"
#include "PropagationSkinDetector.hpp"
#include <boost/lexical_cast.hpp>

#include <opencv2/legacy/legacy.hpp>

#include "FacialLandmarkDetector.hpp"

using cv::Mat;
using cv::TermCriteria;
using boost::lexical_cast;
using cv::Rect;
using cv::Size;
using cv::Vec3b;
using cv::Vec3f;
using cv::Vec2f;
using cv::Scalar;
using cv::Point;
using cv::Point2f;
using std::string;
using std::vector;
using std::set;
using std::map;
using std::cerr;
using std::endl;
using std::cout;



namespace slmotion {

  extern string outputprefix;

  extern int debug;

  // this is here only to register the writer types
  static struct Foo {
    Foo() {
      typedef std::map<int,std::vector<cv::Rect> > T;
      BlackBoardDumpWriter::registerAnyWriter<T>();
    }
  } foo;

  int intensity_image_zeropad_margin=150;

  BlackBoard* vvi_blackBoard;
  FrameSource* vvi_frameSource;

 void TrackingSkinDetector::detect(const cv::Mat& /* inFrame*/,  cv::Mat& /*outMask*/)
  {
    assert(false); // detection cannot be performed for frame-by-frame
  }


  bool TrackingSkinDetector::processRangeImplementation(frame_number_t first, 
							   frame_number_t last, 
							   UiCallback* uiCallback){

    //    cerr << "TrackingSkinDetector called for range ["<<first<<","<<last<<")"<<endl;

    // loop and process the range in comfortable size chunks

    int minlen=100;
    int minfacecount=6;
 
    int firstframe=first,lastframe; 

   
    do{

      int facecount=0;

      for(lastframe=firstframe;lastframe<(int)last&&(lastframe-firstframe+1<=minlen || facecount<minfacecount);lastframe++){
	//	cerr << "trying firstframe,lastframe="<<firstframe<<","<<lastframe<<" facecount="<<facecount<<endl; 
	if(!getBlackBoard().has(lastframe, FACEDETECTOR_BLACKBOARD_ENTRY)) continue;
	if(*getBlackBoard().get<Rect>(lastframe, FACEDETECTOR_BLACKBOARD_ENTRY)== FaceDetector::INVALID_FACE)
	  continue;
	facecount++;
      }

      lastframe--;
      
      if(facecount<minfacecount) // not enough faces in the chunk (because we hit the end)
	for(;firstframe>=(int)first&&facecount<minfacecount;firstframe--){

	  if(!getBlackBoard().has(firstframe, FACEDETECTOR_BLACKBOARD_ENTRY)) continue;
	  if(*getBlackBoard().get<Rect>(firstframe, FACEDETECTOR_BLACKBOARD_ENTRY)== FaceDetector::INVALID_FACE)
	    continue;
	  facecount++;
	  
	}
      

      if(firstframe<(int)first) firstframe=first;

      if(facecount<minfacecount)
	throw string("Could not find enough faces for reliable skin detection");

      if(uiCallback!=NULL)
	if(!(*uiCallback)(firstframe)) return false;
      trackRange(firstframe,lastframe,firstframe==(int)first);
      // train the skin color model only for the first chunk


      firstframe=lastframe+1;
    } while(firstframe < (int)last);

    return true;

  }

  void TrackingSkinDetector::trackRange(int firstframe,int lastframe, bool trainColourModel){
    if (slmotion::debug > 1)
      cerr << "trackRange("<<firstframe<<","<<lastframe<<","<<(int)trainColourModel<<")"<<endl;

    cerr << "output filename prefix: " << outputprefix << endl;


    localBlackBoardType localbb;

    vector<size_t> fdframes;

    int framew,frameh;

    // store scale all frames to local bb
    // scale also the face detections from the global bb


    //     int orgrows,orgcols;

    for(int frnumber=firstframe;frnumber <=lastframe;frnumber++){

      //	cerr << "Copying frames to blackboard." << endl;
   
      Mat frame=getFrameSource()[frnumber];

      size_t maxdim=fmax(frame.rows,frame.cols);
      float factor=750.0/maxdim;

//       orgrows=frame.rows;
//       orgcols=frame.cols;

      if(factor>1) factor=1; // do not scale up

      Mat minFrame;
	 
      //	 downsampleFrame(frame,minFrame,factor);
      cv::resize(frame,minFrame, cv::Size(0,0), factor, factor, CV_INTER_AREA );

      if(frnumber==firstframe){
	framew=minFrame.cols;
	frameh=minFrame.rows;
      }

      //	 minFrame=frame;
	 
      //      cerr << "scaled frame to size " << minFrame.cols << " x " << minFrame.rows << endl;

      localbb[std::pair<int,std::string>(frnumber,"frame")]=minFrame.clone();
      
      if(getBlackBoard().has(lastframe, FACEDETECTOR_BLACKBOARD_ENTRY)){

	Rect faceLocation = *getBlackBoard().get<Rect>(frnumber, FACEDETECTOR_BLACKBOARD_ENTRY);
	if (faceLocation != FaceDetector::INVALID_FACE) {         
	  // scale the rectangle

	  faceLocation.x *= factor;
	  faceLocation.y *= factor;
	  faceLocation.width *= factor;
	  faceLocation.height *= factor;

	  localbb[std::pair<int,std::string>(frnumber,"facerect")]=faceLocation;
	  fdframes.push_back(frnumber);
	  
// 	  for(int i=faceLocation.y;i<faceLocation.y+faceLocation.height;i++)
// 	    for(int j=faceLocation.x;j<faceLocation.x+faceLocation.width;j++){
// 	      //	      cerr << "setting pixel "<<i<<","<<j<<" minFrame.size="<<minFrame.rows<<" x " << minFrame.cols << endl;
// 	      minFrame.at<Vec3b>(i,j)=Vec3b(128,0,0);
// 	    }
	}
      }

     //  cv::imshow("scaled frame", minFrame);
//       cv::waitKey(0);      

    }

  
    //    showFeatureTracks(firstframe,lastframe,localbb);


    // frames and face detections now on blackboard


    // fill in the gaps in face detections by 
    // tracking the face rectangle using affine transform

    // first track the first face detection backwards in time 
    // until the start of the sequence
    
    Rect faceLocation=*boost::any_cast<cv::Rect>(&localbb[std::pair<int,std::string>(fdframes[0],"facerect")]);

    for(int frnumber=fdframes[0]-1;frnumber>=firstframe;frnumber--){

      cerr << "backward face propagation from frame" <<fdframes[0] << " to " <<
	frnumber << endl;

      Mat frame=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"frame")]);
      Mat faceframe=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(fdframes[0],"frame")]);

       Mat patch1(faceframe,faceLocation);
       Mat patch2(frame,faceLocation);


       // estimate transform between the frames

       bool fullAffine=false; // rot/trans/unif. scaling will do?

       Mat tMat=estimateRigidTransform(patch1, patch2, fullAffine);

       // assume the rotation and scaling part of tMat be negligible
       // so that we can take just the translation part

       Point translation(tMat.at<float>(0,2),tMat.at<float>(1,2));

       cerr << "estimated translation " << translation.x << "," << 
	 translation.y << endl;

       Rect newrect=faceLocation+translation;

       Mat visframe1=faceframe.clone();
       Mat visframe2=frame.clone();

       cv::rectangle(visframe1, faceLocation, Scalar::all(255));
       cv::rectangle(visframe2, newrect, Scalar::all(255));

       cv::waitKey(0); 

       localbb[std::pair<int,std::string>(frnumber,"facerect")]=newrect;      

    }

    // now continue filling forward in time

    int srcfaceframenr=fdframes[0];

    for(int frnumber=fdframes[0]+1;frnumber<=lastframe;frnumber++){


      if(localbb.count(std::pair<int,std::string>(frnumber,"facerect"))>0){
	faceLocation=*boost::any_cast<cv::Rect>(&localbb[std::pair<int,std::string>(frnumber,"facerect")]);
	srcfaceframenr=frnumber;
      }	else { // not face frame

	cerr << "forward face propagation from frame" << srcfaceframenr << " to " <<
	frnumber << endl;

      Mat frame=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"frame")]);
      Mat faceframe=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(srcfaceframenr,"frame")]);

       Mat patch1(faceframe,faceLocation);
       Mat patch2(frame,faceLocation);


       // estimate transform between the frames

       bool fullAffine=false; // rot/trans/unif. scaling will do?

       Mat tMat=estimateRigidTransform(patch1, patch2, fullAffine);

       // assume the rotation and scaling part of tMat be negligible
       // so that we can take just the translation part

       Point translation(tMat.at<float>(0,2),tMat.at<float>(1,2));

       cerr << "estimated translation " << translation.x << "," << 
	 translation.y << endl;

       Rect newrect=faceLocation+translation;

       localbb[std::pair<int,std::string>(frnumber,"facerect")]=newrect;      

       Mat visframe1=faceframe.clone();
       Mat visframe2=frame.clone();

       cv::rectangle(visframe1, faceLocation, Scalar::all(255));
       cv::rectangle(visframe2, newrect, Scalar::all(255));

       cv::waitKey(0); 
      }      

    }

    cerr << "face detections filled" << endl;

    // from now on, assume that all frames have face detections


    // estimate homography in the face area, 
    // flag points that do not fit to that homography

    if(false){

    Rect faceLocation=*boost::any_cast<cv::Rect>(&localbb[std::pair<int,std::string>(firstframe,"facerect")]);

    for(int fr=firstframe;fr<lastframe;fr++){

      Mat frame=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(fr,"frame")]);

      Mat nextframe=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(fr+1,"frame")]);
      
      Mat flowmat;
      
      // update face location if face found in the first frame

      //      if(localbb.count(std::pair<int,std::string>(fr,"facerect"))>0)
	faceLocation=*boost::any_cast<cv::Rect>(&localbb[std::pair<int,std::string>(fr,"facerect")]);

      // cut the face locations to separate matrices

      //int shrinkX=0.2*faceLocation.width;
      //int shrinkY=0.2*faceLocation.height;


      Rect patchRect=faceLocation;
      // patchRect -= Size(shrinkX,shrinkY);
      

      Mat patch1(frame,patchRect);
      Mat patch2(nextframe,patchRect);


//      cv::imshow("facerect in the first frame",patch1);
//      cv::imshow("facerect in the second frame",patch2);

//      cv::waitKey(0);      

     // estimate transform between the frames

     bool fullAffine=true;


     // evaluate in rectangle around the estimated face location
     

     Rect evalRect=faceLocation;

     evalRect.x -= 0.75*faceLocation.width;
     evalRect.y -= 0.6*faceLocation.height;
     evalRect.width += 1.5*faceLocation.width;
     evalRect.height += 2*faceLocation.height;

     evalRect.x=fmax(0,evalRect.x);
     evalRect.y=fmax(0,evalRect.y);
     evalRect.width=fmin(frame.cols-evalRect.x,evalRect.width);
     evalRect.height=fmin(frame.rows-evalRect.y,evalRect.height);


     Mat eval1(frame,evalRect);
     Mat eval2(nextframe,evalRect);


     Mat tMat=estimateRigidTransform(patch1, patch2, fullAffine);

     Mat edgemagn1;
     Mat edgemagn2;

     computeEdgeMagnitude(eval1,edgemagn1);
     computeEdgeMagnitude(eval2,edgemagn2);

     floatimagesc("edge magnitude in the first frame",edgemagn1);
     floatimagesc("edge magnitude in the second frame",edgemagn2);
     cv::waitKey(0);      

     float thr=findUpperQuantileFromFloatImage(edgemagn1,0.2);
     Mat edgemask(eval1.rows,eval1.cols,CV_8UC1,Scalar::all(0));

     for(int i=0;i<eval1.rows;i++)
       for(int j=0;j<eval1.cols;j++)
	 if(edgemagn1.at<float>(i,j)>thr)
	   edgemask.at<uchar>(i,j)=255;

     cv::imshow("edge mask",edgemask);
     cv::waitKey(0);      

     Mat tPatch;

     warpAffine(eval1, tPatch, tMat, eval2.size());

     cv::imshow("first frame",eval1);
     cv::imshow("second frame",eval2);
     cv::imshow("reconstructed second frame",tPatch);

     cv::waitKey(0);      

     // calculate and display error images

     float maxerr=0;
     float maxdiff=0,mindiff=0;


     Mat rawerr(eval1.rows,eval1.cols,cv::DataType<float>::type,cv::Scalar::all(0));
     Mat transerr(eval1.rows,eval1.cols,cv::DataType<float>::type,cv::Scalar::all(0));

     Mat errdiff(eval1.rows,eval1.cols,cv::DataType<float>::type,cv::Scalar::all(0));

     int evalmargin=10;

     for(int i=evalmargin;i+evalmargin<eval1.rows;i++)
       for(int j=evalmargin;j+evalmargin<eval1.cols;j++){
	 
	 //	 cerr << "i,j="<<i<<","<<j<<endl;

	 //	 if(edgemask.at<uchar>(i,j)>0)
	 //	   continue;


	 float e=0;
	 for(int c=0;c<3;c++){
	   float d=eval1.at<Vec3b>(i,j)[c]-eval2.at<Vec3b>(i,j)[c];
	   e += d*d;
	 }
	 e=sqrt(e);
	 rawerr.at<float>(i,j)=e;
	 errdiff.at<float>(i,j)=-e;
	 maxerr=fmax(maxerr,e);

	 e=0;
	 for(int c=0;c<3;c++){
	   float d=tPatch.at<Vec3b>(i,j)[c]-eval2.at<Vec3b>(i,j)[c];
	   e += d*d;
	 }
	 e=sqrt(e);
	 transerr.at<float>(i,j)=e;
	 errdiff.at<float>(i,j)+=e;
	 maxerr=fmax(maxerr,e);

	 mindiff=fmin(mindiff,errdiff.at<float>(i,j));
	 maxdiff=fmax(maxdiff,errdiff.at<float>(i,j));

       }

     

     for(int i=0;i<eval1.rows;i++)
       for(int j=0;j<eval1.cols;j++){
	 rawerr.at<float>(i,j) /= maxerr*0.5;
	 transerr.at<float>(i,j) /= maxerr*0.5;
	 errdiff.at<float>(i,j) -= mindiff;
	 errdiff.at<float>(i,j) /= maxdiff-mindiff;

       }

     cv::imshow("raw difference between frames",rawerr);
     cv::imshow("difference w/ affine transformation",transerr);
     cv::imshow("difference of errors",errdiff);

     cv::waitKey(0);      

    }
    }

    //    showOpticalFlow(firstframe,lastframe,localbb);

    // estimateBackgroundWatershed(firstframe,lastframe,localbb);

    // place potential propagation seed mask on blackboard

    if(false){

    string tgtid("mmask");
    string fgid("fgmask");

    //    extractMovementMask(localbb,tgtid,firstframe,lastframe);
    findNonBackgroundBlocks(firstframe,lastframe,localbb,fgid);

    // expand the motion mask a few frames backwrds and forwards in time

    for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){
      
      Mat  frame=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"frame")]);
      
      localbb[std::pair<int,std::string>(frnumber,"diffmask")]=Mat(frame.rows,frame.cols,cv::DataType<float>::type,cv::Scalar::all(0));

      Mat  diffmask=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"diffmask")]);

      Mat  fgmask=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"fgmask")]);
	


// 	for(int i = 0; i < frame.rows; i++)
// 	  for(int j = 0; j < frame.cols; j++){
	    
// 	    bool movementfound=false;
// 	    int tdist=6;
	    
// 	    for(int ff=frnumber-tdist;ff<=frnumber+tdist&&ff<=lastframe;ff++){
// 	      if(ff<firstframe) continue;
// 	      if(boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(ff,"mmask")])->at<float>(i,j)>0){

// 		break;
// 	      }
// 	    }

// 	    if(movementfound && fgmask.at<float>(i,j)>0){
// 	      diffmask.at<float>(i,j)==1;
// 	    }

// 	  }

	int margin=60;

	cv::dilate(fgmask, diffmask, 
      		 cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(margin, margin)));


	//	cv::imshow("frame", frame);
	//	cv::imshow("evalmask", diffmask);

	//	cv::waitKey(0);
	


    }
    }

    


    if(trainColourModel){

      cerr << "starting to train the colour model" << endl;

      // otherwise assume that the colour classifier has already been trained

      delete cls;
      cls=new ELMClassifier;

      cls->maxcachedim=2;


      vector<size_t> sampledframes;

       size_t nrFramesToSample=20;

       vector<vector<float> > negSamples,posSamples;
       vector<float> negWeights,posWeights;
       
       FeatureExtractor fe;

       if(nrFramesToSample>=fdframes.size())
	 sampledframes=fdframes;
       else{
	 float tgtdelta=fdframes.size();
	 tgtdelta /= nrFramesToSample+1;
	 
	 float tgtsampleidx=tgtdelta*0.5;
	 
	 while(tgtsampleidx<=fdframes.size()-1){
	   sampledframes.push_back(fdframes[(size_t)tgtsampleidx]);
	   tgtsampleidx += tgtdelta;
	 }
       }

       sampleFrames(localbb,"dummyid","",false,sampledframes,negSamples,posSamples,negWeights,posWeights,fe,false,false,true,false,false,true,true);

       size_t samplesize=5000;

       if(posSamples.size()>samplesize) posSamples.resize(samplesize);
       if(negSamples.size()>samplesize) negSamples.resize(samplesize);

       // at the moment this truncation is a way to prevent numerical
       // problems in ELM training (should eventually fix properly)

       // hard coded: cls uses feature 0

       vector<bool> dimMask=fe.getFeatureMask(0);
       cls->train(negSamples,posSamples,&dimMask,&negWeights,
		 &posWeights);

              cerr << "...classifier trained" << endl;

    
    } // trainColourModel

    // from now on, assume that cls points to a valid colour classifier

    // seed the skin detection by confirming skin in areas specified by the seed mask

    // go through the frames and evaluate the skin prob. of pixels 
    // indicated by the seed mask

   
    vector<float> samplevec;
    //    size_t maxseedcount=0;
    // size_t maxseedframe=firstframe;
 

    Mat possibleskinmask(frameh,framew,CV_8UC1,Scalar::all(255));

    //    int maxskinmovement=0.2*sqrt(framew*frameh);
    float skinthreshold=0.5;


    size_t seedstep=1;
    for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){



      cerr << "evaluating pmap for frame " << frnumber << endl;
	
      // size_t frameseedcount=0;
	//	cerr << "seeding frame " << frnumber << endl;
	
	Mat  frame=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"frame")]);

	localbb[std::pair<int,std::string>(frnumber,"skinpmap")]=Mat(frame.rows,frame.cols,cv::DataType<float>::type,cv::Scalar::all(0));

	Mat  pmap=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"skinpmap")]);


	localbb[std::pair<int,std::string>(frnumber,"skinmask")]=Mat(frame.rows,frame.cols,CV_8UC1);

	Mat  skinmask=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"skinmask")]);


	//Mat  diffmask=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"diffmask")]);

	//Mat  fgmask=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"fgmask")]);
	
	if(frnumber%seedstep==0){


	for(int i = 0; i < frame.rows; i++)
	  for(int j = 0; j < frame.cols; j++)
	    if(possibleskinmask.at<uchar>(i,j)){

	      vector<float> samplevec;

	      fe.extractFeatures(samplevec,i,j,frnumber,localbb);
	      float p=pmap.at<float>(i,j)=cls->predict_lean(samplevec);

	      skinmask.at<uchar>(i,j)= p > skinthreshold ? 255 : 0;
	    }
	    
	}

	// post-process the skin mask

	// opening to get rid of thin structures 

	int minthickness=6;

	cv::morphologyEx(skinmask, skinmask, cv::MORPH_OPEN, 
		    cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(minthickness, minthickness)), 
		    cv::Point(-1,-1), 1);

	// fill small holes

	cv::morphologyEx(skinmask, skinmask, cv::MORPH_CLOSE, 
		    cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(4, 4)), 
		    cv::Point(-1,-1), 1);

	// prepare possible skin mask for the next frame


// 	cv::dilate(skinmask, possibleskinmask, 
//       		 cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(maxskinmovement, maxskinmovement)));


	// show the result

	//		cv::imshow("frame", frame);
	//		cv::imshow("gated pmap", pmap);

	//		cv::waitKey(0);
	
    }

    // determine the average shape of the skin region surrounding the face

    // parametrise the shape as set of lengths of rays that can be 
    // drawn from the face center that are wholly contained within the area
    // step the direction of rays w/ fixed angular increment

    // determine the threshold for skin probability from first
    // frame w/ succesful face detection

  //   Rect faceLocation=*boost::any_cast<cv::Rect>(&localbb[std::pair<int,std::string>(fdframes[0],"facerect")]);

//     Mat  pmap=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(fdframes[0],"skinpmap")]);

//     vector<float> faceprobs;

//     float aspectratio=faceLocation.width/faceLocation.height;

//     int facemargin=4;

//     for(int x=faceLocation.x+facemargin;x<faceLocation.br().x-facemargin;x++)
//       for(int y=faceLocation.y+facemargin;y<faceLocation.br().y-facemargin;y++){

// 	int dx=x-0.5*(faceLocation.x+faceLocation.br().x);
// 	int dy=y-0.5*(faceLocation.y+faceLocation.br().y);

// 	if(dx*dx+aspectratio*aspectratio*dy*dy> 0.25*faceLocation.width*faceLocation.width)
// 	  continue; // use only elliptical region inside face
	
// 	cerr << "sampling face prob from loc ("<<x<<","<<y<<")"<<endl;

// 	faceprobs.push_back(pmap.at<float>(y,x));

//       }

//     cerr << "collected "<< faceprobs.size() << " skin probabilities" << endl;


//     sort(faceprobs.begin(),faceprobs.end());


//     float faceskinfraction=0.75;

//    float skinthreshold=0.9*faceprobs[(1-faceskinfraction)*faceprobs.size()];

//    cerr << "determined the skin threshold to be " << skinthreshold;

    // show the thresholded pmap

    if(false){



     

  //   for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){

//       if(localbb.count(std::pair<int,std::string>(frnumber,"facerect"))==0)
// 	continue; // skip frames w/ no face detection result

//       Rect faceLocation=*boost::any_cast<cv::Rect>(&localbb[std::pair<int,std::string>(frnumber,"facerect")]);
      


// 	Mat  frame=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"frame")]);

// 	Mat  pmap=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"skinpmap")]);

// 	Mat  skinmask=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"skinmask")]);

//  	Mat visframe=frame.clone();


//  	for(int i = 0; i < frame.rows; i++)
//  	  for(int j = 0; j < frame.cols; j++)
//  	    if(skinmask.at<uchar>(i,j)==0){
//  	      visframe.at<Vec3b>(i,j)=Vec3b(0,0,0);
//  	    }
// 	    else skincount.at<float>(i,j)++;


	//			cv::imshow("thresholded skin area", visframe);
	//	 	cv::waitKey(0);      
 

// 	vector<int> shapevec(ndirs);

// 	float mx=0.5*(faceLocation.x+faceLocation.br().x-1);
// 	float my=0.5*(faceLocation.y+faceLocation.br().y-1);

// 	for(int dir=0;dir<ndirs;dir++){
	  
// 	  float angle=dir*2*3.14156536/ndirs;
	  
// 	  float yfrac=cos(angle);
// 	  float xfrac=sin(angle);

// 	  int r;
	  
// 	  for(r=1;skinmask.at<uchar>(my+r*yfrac,mx+r*xfrac)>0;r++);

// 	  r--;
// 	  shapevec[dir]=r;
// 	  cv::line(visframe, Point(mx,my),
// 		   Point(mx+r*xfrac,my+r*yfrac),
// 		   cv::Scalar(255, 255, 255));


// 	}

// 	cv::imshow("face shape descriptor", visframe);
// 	cv::waitKey(0);      

//	  }



    //	floatimagesc("skin count map",skincount);
    //	cv::waitKey(0);      
    
    }

	//recordBlobProgression(firstframe,lastframe,localbb);

	extractHandBlobs(firstframe,lastframe,localbb);

	return;

	// show diffmask

     Mat  fr1=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(fdframes[0],"frame")]);
     
     Mat skincount(fr1.rows,fr1.cols,cv::DataType<float>::type,cv::Scalar::all(0));

	for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){

	  Mat  skinmask=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"skinmask")]);
	  Mat diffmask=skinmask.clone();

	  for(int i=0;i<skinmask.rows;i++)
	    for(int j=0;j<skinmask.cols;j++){
	      if(skinmask.at<uchar>(i,j)){

		float frac=1.0-skincount.at<float>(i,j)/(lastframe-firstframe+1);
		diffmask.at<uchar>(i,j)=frac*255;
		   
	      }
	    }

	  cv::imshow("skin difference mask",diffmask);
	  cv::waitKey(0);      
	}
	      


	vector<Point> means;

	clusterSkinCount(skincount,means);

	// assign the skin pixels in each frame to the clusters

	int k=means.size();

	for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){

	  Mat  skinmask=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"skinmask")]);
	  
	  Mat lbl32(skinmask.rows,skinmask.cols,CV_32SC1,Scalar::all(k));
	  for(int i=0;i<lbl32.rows;i++)
	    for(int j=0;j<lbl32.cols;j++)
	      if(skinmask.at<uchar>(i,j)>0){
		Point pt(j,i);
		float mindist=sqrdist(means[0],pt);
		int mincl=0;
		for(int cl=1;cl<k;cl++){
		  float d=sqrdist(means[cl],pt);
		  if(d<mindist){
		    mincl=cl;
		    mindist=d;
		  }
		}
		lbl32.at<int>(i,j)=mincl;
	      }

	  Mat vismask;

	  falseColour32BitImg(lbl32,vismask);

	  cv::imshow("clustering result", vismask);
	  cv::waitKey(0);

	}


	// estimate the average head blob shape
	// parametsised as skin pixel count in discretised directions
	// linearly interpolate between the directions
	

	int ndirs=30;
	
	vector<vector<float> > faceextent(ndirs);

	for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){

	  //	  cerr << "measuring head shape in frame " << frnumber << endl;

	  vector<float> faceshape(ndirs,0);

	   Mat  skinmask=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"skinmask")]);


	   Mat headblobmask;
	   
	   Rect faceLocation=*boost::any_cast<cv::Rect>(&localbb[std::pair<int,std::string>(frnumber,"facerect")]);

	   labelConnectedComponents(skinmask,headblobmask);

	   Point pt(0.5*(faceLocation.x+faceLocation.br().x-1),0.5*(faceLocation.y+faceLocation.br().y-1));
	  
	   while(headblobmask.at<uchar>(pt.y,pt.x)==0 && pt.x<headblobmask.cols-1)
	     pt.x++;

	   uchar headlbl=headblobmask.at<uchar>(pt.y,pt.x);

	   pt.x=0.5*(faceLocation.x+faceLocation.br().x-1);
	   pt.y=0.5*(faceLocation.y+faceLocation.br().y-1);

	   for(int i=0;i<headblobmask.rows;i++)
	     for(int j=0;j<headblobmask.cols;j++)
	       if(headblobmask.at<uchar>(i,j)==headlbl){
		 headblobmask.at<uchar>(i,j)=255;

	// 	 int dx=j-pt.x;
// 		 int dy=i-pt.y;

// 		 float pi=3.1415926536;

// 		 float angle;
// 		 if(dy==0){
// 		   if(dx>0)
// 		     angle=0;
// 		   else
// 		     angle=pi;
// 		 } else{
// 		   angle=atan2(dy,dx);
// 		   if(angle<0) angle += 2*pi;
// 		 }
		 
// 		 float angleidx=0.5*ndirs*angle/pi;
// 		 int intpart=angleidx;
// 		 float fracpart=angleidx-intpart;

// 		 faceshape[intpart%ndirs]+= 1-fracpart;
// 		 faceshape[(intpart+1)%ndirs]+= fracpart;

	       }
	       else
		 headblobmask.at<uchar>(i,j)=0;
	   
	 

// 	   for(int i=0;i<ndirs;i++)
// 	     faceextent[i].push_back(faceshape[i]);

 	   localbb[std::pair<int,std::string>(frnumber,"headblobmask")]
 	     =headblobmask;

	}

	// find medians of the head blob shape

// 	cerr << "face extents collected calculting medians" << endl;

// 	vector<float> medianfaceshape(ndirs,0);

// 	for(int i=0;i<ndirs;i++){

// 	  int medidx=0.5*faceextent[i].size();
// 	  nth_element(faceextent[i].begin(), faceextent[i].begin()+medidx,
// 		      faceextent[i].end() );

// 	  medianfaceshape[i]=faceextent[i][medidx];
	  
// 	}

// 	cerr << "median shape calculated:" << endl;
// 	for(int i=0;i<ndirs;i++){
// 	  cerr << medianfaceshape[i] << " ";
// 	}
// 	cerr << endl;


	// now identify the parts of the head blob that 
	// fits to the typical extent



	for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){


	  Mat  frame=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"frame")]);

	  //	  vector<float> faceshape(ndirs,0);

	  Mat  headblobmask=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"headblobmask")]);

	  // Mat visframe=frame.clone();


	  Rect faceLocation=*boost::any_cast<cv::Rect>(&localbb[std::pair<int,std::string>(frnumber,"facerect")]);
	  
	  Point pt(0.5*(faceLocation.x+faceLocation.br().x-1),0.5*(faceLocation.y+faceLocation.br().y-1));
	  
// 	  Mat typicalhead(headblobmask.rows,headblobmask.cols,CV_8UC1,Scalar::all(0));

	  int pixelcount=0;
	  
	  for(int i=0;i<headblobmask.rows;i++)
	    for(int j=0;j<headblobmask.cols;j++)
	      if(headblobmask.at<uchar>(i,j)>0)
		pixelcount++;

	  cerr << "head blob pixel count: " << pixelcount << endl;

	  cv::imshow("frame",frame);
	  cv::waitKey(0);      
	  
// 	  float pi=3.1415926536;


// 	  for(int i=0;i<ndirs;i++){

// 	    float r=sqrt(medianfaceshape[i]*ndirs/pi);

// 	    cerr << "i=" << i << " r="<<r<<endl;

// 	    float angle=i*2*pi/ndirs;

// 	    int dx=r*cos(angle);
// 	    int dy=r*sin(angle);

// 	    cv::line(visframe, pt, Point(pt.x+dx,pt.y+dy), Scalar::all(255));

// 	  }

// 	  cv::imshow("median head shape",visframe);
// 	  cv::imshow("headblobmask",headblobmask);
// 	  cv::waitKey(0);      
	  



// 	  list<pair<int,int> > stack;
// 	  stack.push_back(pair<int,int>(pt.x,pt.y));

// 	  while(stack.size()){
// 	    pair<int,int> loc=stack.front();
// 	    stack.pop_front();

// 	    if(typicalhead.at<uchar>(loc.second,loc.first))
// 	      continue;

// 	    float dx=loc.first-pt.x;
// 	    float dy=loc.second-pt.y;

// 	    float angle;
// 	    if(dy==0){
// 	      if(dx>0)
// 		angle=0;
// 	      else
// 		angle=pi;
// 	    } else{
// 	      angle=atan2(dy,dx);
// 	      if(angle<0) angle += 2*pi;
// 	    }
		 
// 	    float angleidx=0.5*ndirs*angle/pi;
// 	    int intpart=angleidx;
// 	    float fracpart=angleidx-intpart;

// 	    faceshape[intpart%ndirs]+= 1-fracpart;
// 	    faceshape[(intpart+1)%ndirs]+= fracpart;
	    
// 	    if(faceshape[intpart%ndirs]<=1.2*medianfaceshape[intpart%ndirs] &&
// 	       faceshape[(intpart+1)%ndirs]<=1.2*medianfaceshape[(intpart+1)%ndirs]){

// 	  //     cerr << "accepted (x,y)=("<<loc.first<<","<<
// // 		loc.second<<"), faceshape now "<<endl;

// // 	      for(int i=0;i<ndirs;i++)
// // 		cerr << faceshape[i] << " "

// 	      typicalhead.at<uchar>(loc.second,loc.first)=255;
// 	      for(int dx=-1;dx<=1;dx++)
// 		for(int dy=-1;dy<=1;dy++)
// 		  if(typicalhead.at<uchar>(loc.second+dy,loc.first+dx)==0 &&
// 		     headblobmask.at<uchar>(loc.second+dy,loc.first+dx)>0)
// 		    stack.push_back(pair<int,int>(loc.first+dx,loc.second+dy));


// 	    }
	    
// 	  }

// 	  // display the head blob mask & its typical part

// 	   cv::imshow("head blob mask",headblobmask);
// 	   cv::imshow("typical part of head blob mask",typicalhead);
// 	   cv::waitKey(0);      

	}


 //    // calculate and show the average image of skin pixels

//     Mat  frame1=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(firstframe,"frame")]);

//     Mat avemat(frame1.rows,frame1.cols,CV_32FC3,cv::Scalar::all(0));

//     Mat countmat(frame1.rows,frame1.cols,cv::DataType<int>::type,cv::Scalar::all(0));

//     for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){

//       if(localbb.count(std::pair<int,std::string>(frnumber,"skinmask"))==0)
// 	continue; // skip frames w/ no previously created skinmap

//       Mat  frame=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"frame")]);

//       Mat  skinmask=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"skinmask")]);


//       cerr << "accumulating ave for frame " << frnumber << endl;

//       //      cv::imshow("skin mask",skinmask);
//       //      cv::waitKey(0);      

//       for(int i = 0; i < frame.rows; i++)
// 	for(int j = 0; j < frame.cols; j++)
// 	  if(skinmask.at<uchar>(i,j)){
// 	    countmat.at<int>(i,j)++;
// 	    avemat.at<Vec3f>(i,j)[0] += frame.at<Vec3b>(i,j)[0];
// 	    avemat.at<Vec3f>(i,j)[1] += frame.at<Vec3b>(i,j)[1];
// 	    avemat.at<Vec3f>(i,j)[2] += frame.at<Vec3b>(i,j)[2];
// 	  }
//     }
    
//      for(int i = 0; i < avemat.rows; i++)
//        for(int j = 0; j < avemat.cols; j++){
//  	int c=countmat.at<int>(i,j);

// 	// 	cerr << "(i,j)=" << i << "," << j << " count="<<c<<endl;

//  	if(c>0){
//  	  avemat.at<Vec3f>(i,j)[0] /= c*255;
//  	  avemat.at<Vec3f>(i,j)[1] /= c*255;
//  	  avemat.at<Vec3f>(i,j)[2] /= c*255;
//  	}
	
//        }

  

//     cv::imshow("average within skin area",avemat);
//     cv::waitKey(0);      

   

    string flowid("farnebackflow");

    calcOpticalFlow(firstframe,lastframe,localbb,flowid);


  
  }



  void showOpticalFlow(int firstframe,int lastframe,localBlackBoardType &bb){

    for(int fr=firstframe;fr<lastframe;fr++){

      Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(fr,"frame")]);

      Mat nextframe=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(fr+1,"frame")]);
      
      Mat flowmat;
      

      Mat fri(frame.rows,frame.cols,CV_8UC1,cv::Scalar::all(0));
      Mat nexti(frame.rows,frame.cols,CV_8UC1,cv::Scalar::all(0));

      for(int i=0;i<frame.rows;i++)
	for(int j=0;j<frame.cols;j++){
	  fri.at<uchar>(i,j)=(frame.at<Vec3b>(i,j)[0]+
			      frame.at<Vec3b>(i,j)[1]+
			      frame.at<Vec3b>(i,j)[2])/3;
	  nexti.at<uchar>(i,j)=(nextframe.at<Vec3b>(i,j)[0]+
			      nextframe.at<Vec3b>(i,j)[1]+
			      nextframe.at<Vec3b>(i,j)[2])/3;
	}

      //      cv::imshow("frame (i)", fri);
      //      cv::imshow("nextframe (i)", nexti);
      //      cv::waitKey(0);

      int niter=40;

      cv::calcOpticalFlowFarneback(fri, nexti, flowmat,
				   0.5, 1, 10,niter,5,1.1,0);

      Mat visimgx(frame.rows,frame.cols,cv::DataType<float>::type,
		  cv::Scalar::all(0));

      Mat visimgy(frame.rows,frame.cols,cv::DataType<float>::type,
		  cv::Scalar::all(0));

      Mat visimgmagn(frame.rows,frame.cols,cv::DataType<float>::type,
		  cv::Scalar::all(0));

      Mat vispcol(frame.rows,frame.cols,CV_8UC3,
		  cv::Scalar::all(0));

      float maxy,miny;
      float maxx,minx;

      maxy=miny=flowmat.at<Vec2f>(0,0)[1];
      maxx=minx=flowmat.at<Vec2f>(0,0)[0];

      for(int i=0;i<frame.rows;i++)
	for(int j=0;j<frame.cols;j++){
	  visimgx.at<float>(i,j)=flowmat.at<Vec2f>(i,j)[0];
	  visimgy.at<float>(i,j)=flowmat.at<Vec2f>(i,j)[1];

	  visimgmagn.at<float>(i,j)=fmin(1.0,sqrt(visimgx.at<float>(i,j)*visimgx.at<float>(i,j)+visimgy.at<float>(i,j)*visimgy.at<float>(i,j))/50);

	  //	  cerr << "flow("<<i<<","<<j<<")=("<< visimgx.at<float>(i,j) << ","
	  //   << visimgy.at<float>(i,j) << ")" << endl;
	  
	  if( visimgx.at<float>(i,j)<minx) minx=visimgx.at<float>(i,j);
	  if( visimgx.at<float>(i,j)>maxx) maxx=visimgx.at<float>(i,j);

	  if( visimgy.at<float>(i,j)<miny) miny=visimgy.at<float>(i,j);
	  if( visimgy.at<float>(i,j)>maxy) maxy=visimgy.at<float>(i,j);

	  vispcol.at<Vec3b>(i,j)[0]=fmax(0,fmin(255,128*(1+visimgx.at<float>(i,j)/30)));
	  vispcol.at<Vec3b>(i,j)[1]=fmax(0,fmin(255,128*(1+visimgy.at<float>(i,j)/30)));

	  
	}

      cerr << "flow etrema: minx="<<minx<<" maxx="<<maxx<<
	" miny="<<miny<<" maxy="<<maxy<<endl; 

      cv::imshow("frame", frame);
      cv::imshow("flow magnitude", visimgmagn);
      cv::imshow("flow (pcol)", vispcol);
      floatimagesc("flow (x)",visimgx);
      floatimagesc("flow (y)",visimgy);
      
      cv::waitKey(0);


    }

  }

  void calcOpticalFlow(int firstframe,int lastframe,localBlackBoardType &bb,string &tgtid){

    for(int fr=firstframe;fr<lastframe;fr++){

      Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(fr,"frame")]);

      Mat nextframe=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(fr+1,"frame")]);
      
      Mat flowmat;
      

      Mat fri(frame.rows,frame.cols,CV_8UC1,cv::Scalar::all(0));
      Mat nexti(frame.rows,frame.cols,CV_8UC1,cv::Scalar::all(0));

      for(int i=0;i<frame.rows;i++)
	for(int j=0;j<frame.cols;j++){
	  fri.at<uchar>(i,j)=(frame.at<Vec3b>(i,j)[0]+
			      frame.at<Vec3b>(i,j)[1]+
			      frame.at<Vec3b>(i,j)[2])/3;
	  nexti.at<uchar>(i,j)=(nextframe.at<Vec3b>(i,j)[0]+
			      nextframe.at<Vec3b>(i,j)[1]+
			      nextframe.at<Vec3b>(i,j)[2])/3;
	}

      int niter=40;

      cv::calcOpticalFlowFarneback(fri, nexti, flowmat,
				   0.5, 1, 10,niter,5,1.1,0);


      bb[std::pair<int,std::string>(fr,tgtid)]=flowmat.clone();

    }
  }

  void showFeatureTracks(int firstframe,int lastframe,localBlackBoardType &bb){

    for(int fr=firstframe;fr<lastframe;fr++){

      Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(fr,"frame")]);

      Mat nextframe=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(fr+1,"frame")]);

      Mat fri(frame.rows,frame.cols,CV_8UC1,cv::Scalar::all(0));
      Mat nexti(frame.rows,frame.cols,CV_8UC1,cv::Scalar::all(0));

      // create B&W versions of the frames

      for(int i=0;i<frame.rows;i++)
	for(int j=0;j<frame.cols;j++){
	  fri.at<uchar>(i,j)=(frame.at<Vec3b>(i,j)[0]+
			      frame.at<Vec3b>(i,j)[1]+
			      frame.at<Vec3b>(i,j)[2])/3;
	  nexti.at<uchar>(i,j)=(nextframe.at<Vec3b>(i,j)[0]+
			      nextframe.at<Vec3b>(i,j)[1]+
			      nextframe.at<Vec3b>(i,j)[2])/3;
	}

      
      Mat featuremat;

      int maxCorners=2000;
      double qualityLevel=0.001;
      double minDistance=5;

      cv::goodFeaturesToTrack(fri, featuremat, maxCorners, qualityLevel, 
			  minDistance);

      cerr << "found " << featuremat.rows << " " << featuremat.cols
	   << " dimensional corners" << endl;


      Mat visframe=frame.clone();

      for(int i=0;i<featuremat.rows;i++){
	cv::circle(visframe, Point(featuremat.at<float>(i,0),
				   featuremat.at<float>(i,1)),
		   1, Scalar::all(255));
      }

      cv::imshow("detected corner points", visframe);
      cv::waitKey(0);

      // track the detected points

      Mat trackedmat;

      vector<uchar> status;
      vector<float> err;

      //      Mat status;
      //      Mat err;
      
      calcOpticalFlowPyrLK(fri, nexti, featuremat, trackedmat, status, err);

      // visualise the result

      visframe=frame.clone();

      int trackcount=0;

      for(int i=0;i<featuremat.rows;i++){

	if(status[i]>0){
	  trackcount++;
	  cv::circle(visframe, Point(featuremat.at<float>(i,0),
				   featuremat.at<float>(i,1)),
		     1, Scalar::all(255));
	  //	  cv::circle(visframe, Point(trackedmat.at<float>(i,0),
	  //				     trackedmat.at<float>(i,1)),
	  //		     1, cv::Scalar(0, 255, 0));

	  cv::line(visframe, Point(featuremat.at<float>(i,0),
				   featuremat.at<float>(i,1)),
		   Point(trackedmat.at<float>(i,0),
			 trackedmat.at<float>(i,1)),
		   cv::Scalar(0, 255, 0));


	}
      }

      cerr << "tracked " << trackcount << " points" << endl;

      cv::imshow("tracked corner points", visframe);
      cv::waitKey(0);


    }

  }

  void fillHoles(Mat &m){

    // m should be of type CV_8UC1

    assert(m.type() == CV_8UC1);

    Mat lbl;

    labelConnectedComponents(m,lbl);

    // collect the labels of background that touches the borders of m

    set<int> borderlbl;

    for(int x=0;x<m.cols;x++){
      if(m.at<uchar>(0,x)==0)
	borderlbl.insert(lbl.at<uchar>(0,x));
      if(m.at<uchar>(m.rows-1,x)==0)
	borderlbl.insert(lbl.at<uchar>(m.rows-1,x));
    }

    for(int y=0;y<m.rows;y++){
      if(m.at<uchar>(y,0)==0)
	borderlbl.insert(lbl.at<uchar>(y,0));
      if(m.at<uchar>(y,m.cols-1)==0)
	borderlbl.insert(lbl.at<uchar>(y,m.cols-1));
    }

    // set all the elements of m to 255
    // that have labels different from border touching background

    for(int y=0;y<m.rows;y++)
      for(int x=0;x<m.cols;x++){
	if(borderlbl.count(lbl.at<uchar>(y,x))==0)
	  m.at<uchar>(y,x)=255;
      }

  }

  void labelConnectedComponents(Mat &m,Mat &lbl){

    //   cv::imshow("incoming mask", m);
    //   cv::waitKey(0);      

    int nextlbl=1;
    vector<int> corresp;
    corresp.push_back(0);

    // hard code background as 0

    int minx=m.cols,maxx=0,miny=m.rows,maxy=0;

    lbl=Mat(m.rows,m.cols,CV_8UC1,cv::Scalar::all(0));

    Mat worklbl(m.rows,m.cols,cv::DataType<int>::type,cv::Scalar::all(0));

    for(int y=0;y<m.rows;y++)
      for(int x=0;x<m.cols;x++){

	if(m.at<uchar>(y,x)==0){
	  worklbl.at<int>(y,x)=0;
	  continue;
	}

	minx=fmin(x,minx);
	maxx=fmax(x,maxx);
	miny=fmin(y,miny);
	maxy=fmax(y,maxy);

	vector<int> nbrlbl;
	vector<int> nbrval;

	vector<int> samelbl;

	if(y>0){ //inspect the neighbours on row above
	  if(x>0){
	    nbrlbl.push_back(worklbl.at<int>(y-1,x-1));
	    nbrval.push_back(m.at<uchar>(y-1,x-1));
	  }

	  nbrlbl.push_back(worklbl.at<int>(y-1,x));
	  nbrval.push_back(m.at<uchar>(y-1,x));

	  if(x<m.cols-1){
	    nbrlbl.push_back(worklbl.at<int>(y-1,x+1));
	    nbrval.push_back(m.at<uchar>(y-1,x+1));
	  }
	    
	}

	// nbr in west

	if(x>0){
	  nbrlbl.push_back(worklbl.at<int>(y,x-1));
	  nbrval.push_back(m.at<uchar>(y,x-1));
	}
	
	for(int i=0;i<(int)nbrval.size();i++){
	  if(nbrval[i]==(int)m.at<uchar>(y,x))
	    samelbl.push_back(nbrlbl[i]);
	}

	
	if(samelbl.size()==0){
	  
	  // no neighbours with same val found -> start new lbl

	  worklbl.at<int>(y,x)=nextlbl;
	  corresp.push_back(nextlbl); // initialise the correspondence
	  // table entry to point to the label itself
	  nextlbl++;
	}
	else if(samelbl.size()==1){
	  // only one pixel w/ same val found -> use its label
	  worklbl.at<int>(y,x)=samelbl[0];
	}
	else{
	  
	  // otherwise use the minimum of the labels
	  // mark the rest of the labels to be equivalent

	  sort(samelbl.begin(),samelbl.end());
	  worklbl.at<int>(y,x)=samelbl[0];

	  for(int i=1;i<(int)samelbl.size();i++)
	    corresp[samelbl[i]]=smallestCorresponding(samelbl[0],corresp);
	}

      }

    // first pass complete, now compact the corresp-array

    for(int i=0;i<(int)corresp.size();i++)
      corresp[i]=smallestCorresponding(i,corresp);

    //     

    //           Mat vislbl=worklbl;

    for(int y=miny;y<=maxy;y++)
      for(int x=minx;x<=maxx;x++){
	//				vislbl.at<int>(y,x)=
	  lbl.at<uchar>(y,x)=corresp[worklbl.at<int>(y,x)];
      }

//         Mat vismat;

//         falseColour32BitImg(vislbl,vismat);

//         cv::imshow("connected components", vismat);
//         cv::waitKey(0);      


  }

  int smallestCorresponding(int lbl,vector<int> corresp){

    int r=lbl;

    while(r!=corresp[r]) r=corresp[r];

    return r;

  }

  void computeEdgeMagnitude(Mat &src,Mat &eimg){

    // sums the squares of vertical & horisontal sobel edge strengths
    // on three channels

    eimg=Mat(src.rows,src.cols,CV_32FC1);

    Mat hore,verte;

    cv::Sobel(src,hore,CV_32F,1,0);
    cv::Sobel(src,verte,CV_32F,0,1);

    // sum the responses

    for(int i=0;i<eimg.rows;i++)
      for(int j=0;j<eimg.cols;j++){
	float sqrsum=0;
	for(int ch=0;ch<3;ch++){
	  float v = hore.at<Vec3f>(i,j)[ch];
	  sqrsum += v*v;
	  v = verte.at<Vec3f>(i,j)[ch];
	  sqrsum += v*v;
	}
	eimg.at<float>(i,j)=sqrt(sqrsum);
      }

  }

  float findUpperQuantileFromFloatImage(Mat &img, float quantile){

    //    int samplingfactor=10;
    vector<float> vals;

    for(int i=0;i<img.rows;i++)
      for(int j=0;j<img.cols;j++)
	vals.push_back(img.at<float>(i,j));

    int n=vals.size()*(1-quantile)-1;
    if(n<0) n=0;

    nth_element(vals.begin(),vals.begin()+n,vals.end());
    return vals[n];

  }

  void clusterSkinCount(Mat &skincount, vector<Point> &means){

    // use k means clustering

    int k=15;

    means=vector<Point>(k,Point(skincount.cols/2,skincount.rows/2));

    int r=0.2*sqrt(skincount.rows*skincount.cols);

    for(int i=0;i<k;i++){
      float pi=3.1415926536;
      float angle=i*2*pi/k;
      int dx=r*cos(angle);
      int dy=r*sin(angle);
      
      means[i].x += dx;
      means[i].y += dy;

    }

    Mat lbl(skincount.rows,skincount.cols,CV_8UC1,Scalar::all(0));

    int changecount;

    do{

      changecount=0;

      for(int i=0;i<skincount.rows;i++)
	for(int j=0;j<skincount.cols;j++)
	  if(skincount.at<float>(i,j)>0){
	    uchar oldlbl=lbl.at<uchar>(i,j);
	    Point pt(j,i);
	    float mindist=sqrdist(means[0],pt);
	    int mincl=0;
	    for(int cl=1;cl<k;cl++){
	      float d=sqrdist(means[cl],pt);
	      if(d<mindist){
		mincl=cl;
		mindist=d;
	      }
	    }
	    if(mincl != oldlbl)
	      changecount++;
	    lbl.at<uchar>(i,j)=mincl;
	  }
      
      vector<float> wsum(k,0);
      means= vector<Point>(k,Point(0,0));
      for(int i=0;i<skincount.rows;i++)
	for(int j=0;j<skincount.cols;j++)
	  if(skincount.at<float>(i,j)>0){
	    uchar l=lbl.at<uchar>(i,j);
	    float c=skincount.at<float>(i,j);
	    wsum[l] += c;
	    means[l] += Point(c*j,c*i);
	  }

      for(int cl=0;cl<k;cl++){
	if(wsum[cl]){
	  means[cl].x /= wsum[cl];
	  means[cl].y /= wsum[cl];
	}
      }

      cerr << "kmeans: changecount=" << changecount << endl;


    } while(changecount>0);

    Mat lbl32(lbl.rows, lbl.cols, CV_32SC1,Scalar::all(0));

    for(int i=0;i<skincount.rows;i++)
      for(int j=0;j<skincount.cols;j++)
	lbl32.at<int>(i,j)=lbl.at<uchar>(i,j);

    Mat vismask;
    falseColour32BitImg(lbl32,vismask);

    cv::imshow("clustering result", vismask);
    cv::waitKey(0);
  }

  float sqrdist(Point &p1, Point &p2){
    float r;

    r=p1.x-p2.x;
    r*=r;

    r+=(p1.y-p2.y)*(p1.y-p2.y);

    return r;
  }

  void recordBlobProgression(int firstframe,int lastframe,localBlackBoardType &bb){

    // cerr << "now recording blob progression" << endl;

    // obtain a measure of characteristic scale by 
    // determining the average size of the face rectangles


    // this function places two results on blacboard:

    //     the uniquely labelled blobs are stored on bb w/
    // key uniqueblobintmask

    // blob progression is stored on bb w/ key
    // blobprogression




    float avefacesize=0;

    //    int framesize_rows=0;
    //    int framesize_cols=0;

    for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){

      // assume the existence of face detections
      // for all frames

      // (missing faces should have already been interpolated/estimated)

      Rect faceLocation=*boost::any_cast<cv::Rect>(&bb[std::pair<int,std::string>(frnumber,"facerect")]);    
      avefacesize += faceLocation.width*faceLocation.height;
    }

    avefacesize /= (lastframe-firstframe+1);

    int minblobsize=0.1*avefacesize;


    BlobProgression bp;

    bp.firstframe=firstframe;

    // assume that skinmask -entry on the blackboard specifies the
    // blobs whose progress is recorded in bp

    // the uniquely labelled blobs are stored on bb w/
    // key uniqueblobintmask

    // process the first frame

    BlobInfo bi;
    vector<BlobInfo> frameBlobInfo;
    map<int,int> framelbl2blobidx;

    int frnumber=firstframe;

    Mat skinmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"skinmask")]);

 //    framesize_rows=skinmask.rows;
//     framesize_cols=skinmask.cols;
    

    Mat unique;

    labelConnectedComponents(skinmask,unique);
    removeSmallBlobs(unique,skinmask,minblobsize);

    bb[std::pair<int,std::string>(frnumber,"uniqueblobintmask")]=
      Mat(unique.rows,unique.cols,cv::DataType<int>::type,cv::Scalar::all(0));

    Mat uniqueblobintmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"uniqueblobintmask")]);

    int nextlbl=1;

    map<int,int> unlbl2intlbl;

    for(int i=0;i<unique.rows;i++)
      for(int j=0;j<unique.cols;j++)
	if(skinmask.at<uchar>(i,j)){
	  int lbl=unique.at<uchar>(i,j);

	  if(unlbl2intlbl.count(lbl)==0){
	    unlbl2intlbl[lbl]=nextlbl++;
	    bi.frame=firstframe;
	    bi.label=unlbl2intlbl[lbl];
	    bi.seedpoint=Point(j,i);
	    framelbl2blobidx[unlbl2intlbl[lbl]]=frameBlobInfo.size();
	    frameBlobInfo.push_back(bi);
	  }

	  uniqueblobintmask.at<int>(i,j)=unlbl2intlbl[lbl];
	}

    bp.blobs.push_back(frameBlobInfo);
    bp.lbl2blobidx.push_back(framelbl2blobidx);

    // bp.dump(bp.blobs.size()-1);

    // first frame processed, process subsequent frames

    for(frnumber=firstframe+1;frnumber<=lastframe;frnumber++){

      //      cerr << "analysing the overlaps in frame " << frnumber << endl;

      frameBlobInfo.clear();
      framelbl2blobidx.clear();
      bi.overlappinglabels_prevframe.clear();
      bi.overlappinglabels_nextframe.clear();




      Mat skinmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"skinmask")]);

      Mat unique;

      labelConnectedComponents(skinmask,unique);
      removeSmallBlobs(unique,skinmask,minblobsize);

      bb[std::pair<int,std::string>(frnumber,"uniqueblobintmask")]=
	Mat(unique.rows,unique.cols,cv::DataType<int>::type,cv::Scalar::all(0));

      Mat uniqueblobintmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"uniqueblobintmask")]);

      Mat prevuniqueblobintmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber-1,"uniqueblobintmask")]);
      
      //      cv::imshow("skin mask",skinmask);      
      //      showDeterministicFalseColour32bit("previous labels",prevuniqueblobintmask);
      //      showDeterministicFalseColour32bit("unique labeling",unique);
      //      cv::waitKey(0);


      // evaluate the overlap of blobs with the previous frame

      map<int,set<int> > prevlbl2newun;
      map<int,set<int> > newun2prevlbl;

      for(int i=0;i<unique.rows;i++)
	for(int j=0;j<unique.cols;j++)
	  if(skinmask.at<uchar>(i,j)){
	    uchar newun=unique.at<uchar>(i,j);

	    if(newun2prevlbl.count(newun)==0)
	      newun2prevlbl[newun]=set<int>();
	    
	    int prevlbl=prevuniqueblobintmask.at<int>(i,j);
	    if(bp.lbl2blobidx[bp.lbl2blobidx.size()-1].count(prevlbl)){

	      // exclude background of prev. frame



	      //	    cerr << "recording overlap of prevlbl " << prevlbl 
	      //		 << " with newun " << (int)newun << endl;

	      prevlbl2newun[prevlbl].insert(newun);
	      newun2prevlbl[newun].insert(prevlbl);
	    }

	  }

        // cerr << "dumping overlaps between frames " << frnumber-1 << " and " 
        // 	   << frnumber << endl;

        // for(auto it=prevlbl2newun.begin(); it != prevlbl2newun.end(); it++){
        // 	cerr << "previous label " << it->first << " overlaps with new "
        // 	     << " unique labels";
        // 	for(auto it2=it->second.begin(); it2 != it->second.end();it2++)
        // 	  cerr << " " << *it2 << endl;
        // }



      // select labeling for  the new frame
      // idea: if blob overlaps w/ exactly one lbl in prev frame
      // and the converse is also true (1->1 mapping), use 
      // the old label

      // otherwise start a fresh label

      set<int> encounteredlbl;
      map<int,int> newun2newlbl;
      map<int,int> newlbl2newun;

      for(auto it=newun2prevlbl.begin();it!=newun2prevlbl.end();it++){
	if(it->second.size()==1 && prevlbl2newun[*it->second.begin()].size()==1){
	  newun2newlbl[it->first]=*it->second.begin();
	  newlbl2newun[*it->second.begin()]=it->first;
	}
	else{
	  newlbl2newun[nextlbl]=it->first;
	  newun2newlbl[it->first]=nextlbl++;
	}
      }
	
      // cerr << "selected unique->newlabel label mapping:" << endl;

      // for(auto it=newun2newlbl.begin(); it != newun2newlbl.end(); it++)
      // 	cerr << it->first << " -> " << it->second << endl;


      // cerr << "inverse: newlabel->unique mapping:" << endl;

      // for(auto it=newlbl2newun.begin(); it != newlbl2newun.end(); it++)
      // 	cerr << it->first << " -> " << it->second << endl;


      // go through the new unique array and translate labels
      // according to the selected scheme

      for(int i=0;i<unique.rows;i++)
	for(int j=0;j<unique.cols;j++)
	  if(skinmask.at<uchar>(i,j)){
	    int lbl=unique.at<uchar>(i,j);
	    if(encounteredlbl.count(lbl)==0){
	      encounteredlbl.insert(lbl);
	      //	      cerr << "encounterd new label " << lbl << endl;

	      bi.frame=frnumber;
	      bi.label=newun2newlbl[lbl];
	      bi.seedpoint=Point(j,i);
	      framelbl2blobidx[newun2newlbl[lbl]]=frameBlobInfo.size();
	      frameBlobInfo.push_back(bi);
	    }

	  uniqueblobintmask.at<int>(i,j)=newun2newlbl[lbl];
	}

      bp.blobs.push_back(frameBlobInfo);
      bp.lbl2blobidx.push_back(framelbl2blobidx);


      // finally update the overlap information 
      // to frameBlobInfos

      // first previous frame

      for(auto it=bp.blobs[bp.blobs.size()-2].begin();it!=bp.blobs[bp.blobs.size()-2].end();it++){

	it->overlappinglabels_nextframe.clear();
	
	for(auto newunit=prevlbl2newun[it->label].begin();
	    newunit!=prevlbl2newun[it->label].end();
	    newunit++)
	  it->overlappinglabels_nextframe.insert(newun2newlbl[*newunit]);
      }

      // then current frame

      for(auto it=bp.blobs[bp.blobs.size()-1].begin();it!=bp.blobs[bp.blobs.size()-1].end();it++){

	it->overlappinglabels_prevframe=newun2prevlbl[newlbl2newun[it->label]];
      }

      //      bp.dump(bp.blobs.size()-1);
      //      cv::waitKey(0);

    }

    bb[std::pair<int,std::string>(0,"blobprogression")]=bp;

  }


  void interpretBlobs(int firstframe,int lastframe,localBlackBoardType &bb){

    int framesize_rows=0;
    int framesize_cols=0;


    // based on blackboard entries "uniqueblobintmask" and "blobprogression"

    // places blob interpretations on blackboard with key 
    // "blobinterpretation"
    // type map<int,blobinterpretation_type>

    BlobProgression bp=*boost::any_cast<BlobProgression>(&bb[std::pair<int,std::string>(0,"blobprogression")]);

    map<int,blobinterpretation_type> interp;

    // initialise the interpretation to empty for all labels in bp

    for(size_t frameidx=0; frameidx<bp.blobs.size();frameidx++){
      for(size_t blobidx=0;blobidx<bp.blobs[frameidx].size();blobidx++){
	BlobInfo *bi=&(bp.blobs[frameidx][blobidx]);

	interp[bi->label]=blobinterpretation_type();
	interp[bi->label].isolated=true;
	interp[bi->label].uniquelabel=bi->label;
      }
    }




    // identify blobs that may be isolated hands

    // tag invalid all labels that either are a merge of
    // several previous labels
    // or split into several labels

    set<int> invalidlabels;

    for(size_t frameidx=0; frameidx<bp.blobs.size();frameidx++){
      for(size_t blobidx=0;blobidx<bp.blobs[frameidx].size();blobidx++){
	BlobInfo *bi=&(bp.blobs[frameidx][blobidx]);

	//	bi->dump();

	if(bi->overlappinglabels_prevframe.size()>1)
	  {
	  invalidlabels.insert(bi->label);
	  interp[bi->label].isolated=false;
	  }
	if(bi->overlappinglabels_nextframe.size()>1){
	  invalidlabels.insert(bi->label);
	  interp[bi->label].isolated=false;
	}
      }
    }


    float avefacex;
    int maxfacey=0;
    int facecount=0;

    // invalidate also blobs overlapping with face detections

    for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){

      Rect faceLocation=*boost::any_cast<cv::Rect>(&bb[std::pair<int,std::string>(frnumber,"facerect")]);    

      avefacex += faceLocation.x + faceLocation.br().x;
      facecount +=2;

      if(faceLocation.br().y>maxfacey) maxfacey=faceLocation.br().y;

      Mat uniqueblobintmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"uniqueblobintmask")]);

      framesize_rows=uniqueblobintmask.rows;
    framesize_cols=uniqueblobintmask.cols;

      bool facelblmarked=false;

      for(int i=faceLocation.y;facelblmarked==false && i<faceLocation.br().y;i++)
	for(int j=faceLocation.x;facelblmarked==false && j<faceLocation.br().x;j++)
	  if(uniqueblobintmask.at<int>(i,j)){
	    int lbl=uniqueblobintmask.at<int>(i,j);
	    invalidlabels.insert(lbl);
	    
	    interp[lbl].face_overlap=true;
	    interp[lbl].head=true;
	    
	    facelblmarked=true;
	  }

    }

    avefacex /= facecount;


     // perform filtering of labels tagged as invalid

//   for(frnumber=firstframe;frnumber<=lastframe;frnumber++){

//        cerr << "showing minimal blobs in frame " <<frnumber << endl; 

//        Mat uniqueblobintmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"uniqueblobintmask")]);
      
//        Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"frame")]);
      
//        bool minimalinframe=false;

//        for(int i=0;i<uniqueblobintmask.rows;i++)
//      	for(int j=0;j<uniqueblobintmask.cols;j++){
//      	  int lbl=uniqueblobintmask.at<int>(i,j);
//      	  if(invalidlabels.count(lbl))
//      	    uniqueblobintmask.at<int>(i,j)=0;
//      	  else if(lbl>0)
//      	    minimalinframe=true;
//      	}

//        //if(false && minimalinframe){
//                showDeterministicFalseColour32bit("minimal blobs",uniqueblobintmask);
//          cv::imshow("frame",frame);
//          cv::waitKey(0);
// 	 //}
//     }
     

    // here we try to decide which of the minimal blobs are left and
    // which right hands

    // simple logic:

    // initialise the left and right hand vote counters of each labels to zero
    // for each frame w/ exactly 2 non-face isolated blobs, increase the 
    // left hand counter of the leftmost and vice versa 

    // after going through all the frames do a majority decision
    // in case of a tie, use the average x-coordinate of the 
    // blob to decide (whether smaller than the average face enterline)
    
    // this step could be supplemented/replace with appearance based 
    // detection of right/left hands


    map<int,int> lhvotes,rhvotes;
    set<int> minimalblobs;

    map<int,int> xsum;
    map<int,int> count;


    // collect statistics of blob sizes
    // andx coordinates

    map<int, map<int,int> > blobsizes;

    map<int, map<int,int> > minx;
    map<int, map<int,int> > miny;
    map<int, map<int,int> > maxx;
    map<int, map<int,int> > maxy;

    // semantics: blobsizes[label][frame] 

    for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){

      set<int> framelabels;
      map<int,int> framexsum;
      map<int,int> framecount;

      Mat uniqueblobintmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"uniqueblobintmask")]);

      //      showDeterministicFalseColour32bit("unique labeling",uniqueblobintmask);
      // cv::waitKey(0);



      for(int i=0;i<uniqueblobintmask.rows;i++)
	for(int j=0;j<uniqueblobintmask.cols;j++){
	  int lbl=uniqueblobintmask.at<int>(i,j);
	  if(lbl!=0 && !invalidlabels.count(lbl)){
	 
	    framelabels.insert(lbl);
	    minimalblobs.insert(lbl);
	    xsum[lbl] += j;
	    count[lbl]++;
	    framexsum[lbl] += j;
	    framecount[lbl]++;

	    if(minx.count(lbl)==0 || minx[lbl].count(frnumber)==0
	       || j < minx[lbl][frnumber]) minx[lbl][frnumber]=j;
	    if(miny.count(lbl)==0 || miny[lbl].count(frnumber)==0
	       || i < miny[lbl][frnumber]) miny[lbl][frnumber]=i;
	    if(maxx.count(lbl)==0 || maxx[lbl].count(frnumber)==0
	       || j > maxx[lbl][frnumber]) maxx[lbl][frnumber]=j;
	    if(maxy.count(lbl)==0 || maxy[lbl].count(frnumber)==0
	       || i > maxy[lbl][frnumber]) maxy[lbl][frnumber]=i;

	  }
	}

      for(auto it=framecount.begin(); it!=framecount.end();it++)
	blobsizes[it->first][frnumber]=it->second;

      if(framelabels.size()==2){

	// voting frame
	
	int l1,l2;

	l1=*(framelabels.begin());
	l2=*(++framelabels.begin());

	if(framexsum[l1]/framecount[l1]<framexsum[l2]/framecount[l2]){
	  lhvotes[l1]++;
	  rhvotes[l2]++;
// 	  cout << "frame " << frnumber << ": lbl " << l1 << " is voted left and lbl " << l2 << " right hand " << endl; 
	} else{
	  lhvotes[l2]++;
	  rhvotes[l1]++;
// 	  cout << "frame " << frnumber << ": lbl " << l2 << " is voted left and lbl " << l1 << " right hand " << endl; 
	}
      }

    }

    set<int> lhlabels,rhlabels;

    for(auto it=minimalblobs.begin();it!=minimalblobs.end();it++){
      int lbl=*it;

      interp[lbl].right_hand_score=rhvotes[lbl];
      interp[lbl].left_hand_score=lhvotes[lbl];


      if(xsum[lbl]<avefacex*count[lbl])
	interp[lbl].left_hand_score +=0.5;
      else
	interp[lbl].right_hand_score +=0.5;

      if(lhvotes[lbl]>rhvotes[lbl])
	lhlabels.insert(lbl);
      else if(rhvotes[lbl]>lhvotes[lbl])
	rhlabels.insert(lbl);
      else{
	
	// decide by average x-coordinate

	if(xsum[lbl]<avefacex*count[lbl])
	  lhlabels.insert(lbl);
	  
	else
	  rhlabels.insert(lbl);
      }

//       cout << "considering lbl " << lbl << ": ";
//       cout << "left hand score: " << interp[lbl].left_hand_score
// 	   << " right hand score: " << interp[lbl].right_hand_score << endl;

      if(interp[lbl].right_hand_score>interp[lbl].left_hand_score)
	interp[lbl].right_hand=true;
      else
	interp[lbl].left_hand=true;

    }

    // voting results ready

  //    cerr << "minimalblobs: ";
//      for(auto it=minimalblobs.begin();it!=minimalblobs.end();it++){
//        cerr << (*it) << " "; 
//      }
//      cerr << endl;

//       cerr << "lhlabels: ";
//       for(auto it=lhlabels.begin();it!=lhlabels.end();it++){
//         cerr << (*it) << " "; 
//       }
//       cerr << endl;

//       cerr << "rhlabels: ";
//       for(auto it=rhlabels.begin();it!=rhlabels.end();it++){
//         cerr << (*it) << " "; 
//       }
//       cerr << endl;

//       cout << "interp after voting: " << endl;

//       for(auto it=interp.begin(); it!=interp.end();it++)
// 	it->second.show();

    // in this stage isolated non-face labels have been
    // decided to be either left or right hands 

    // the integral part of interpretations {left,right}_hand_score
    // record the votes and 0.5 is added to either score according
    // to whether the blob's average x is left or right of the face


    // now retract some isolated hand interpretations if
    // the blob looks like it could be the merger of two hands

    // here we could include also appearance-based analysis

    // determine median hand blobs size

    vector<int> handsizes;

    for(auto it=lhlabels.begin();it!=lhlabels.end();it++)
      for( auto it2=blobsizes[*it].begin();
	   it2 != blobsizes[*it].end(); it2++){

	int touchmargin=6;

	if(minx[*it][it2->first] < touchmargin 
	   || maxx[*it][it2->first] > framesize_cols-touchmargin-1
	   || miny[*it][it2->first] < touchmargin 
	   || maxy[*it][it2->first] > framesize_rows-touchmargin-1)
	  continue;

	// exclude hands only partially in the image 
	// from median size determination

	if( miny[*it][it2->first]<maxfacey+50)
	  continue;
	
	handsizes.push_back(it2->second);
      }

    for(auto it=rhlabels.begin();it!=rhlabels.end();it++)
      for( auto it2=blobsizes[*it].begin();
	   it2 != blobsizes[*it].end(); it2++){

	int touchmargin=6;

	if(minx[*it][it2->first] < touchmargin 
	   || maxx[*it][it2->first] > framesize_cols-touchmargin-1
	   || miny[*it][it2->first] < touchmargin 
	   || maxy[*it][it2->first] > framesize_rows-touchmargin-1)
	  continue; 
	// exclude hands only partially in the image 
	// from median size determination

	if( miny[*it][it2->first]<maxfacey+50)
	  continue;
	

	handsizes.push_back(it2->second);
      }
  //   cerr << "dumping hand sizes: " << endl;
 //    for(int i=0;i<handsizes.size();i++)
//       cerr << " " << handsizes[i];
//     cerr << endl;

    int medianhandsize;

    if(handsizes.size()){
      nth_element( handsizes.begin(), handsizes.begin()+handsizes.size()/2,
		   handsizes.end() );

      medianhandsize=handsizes[handsizes.size()/2];
      // go through the blob size list and invalidate 
      // all labels whose blob is too big in some frame
      // in comparison with the median hand size
      
      float handsizemultiplier=2.4;
      
      for(auto it=blobsizes.begin();it!=blobsizes.end();it++)
	for( auto it2=it->second.begin();
	     it2 != it->second.end(); it2++){


	    int lbl=it->first;

	    interp[lbl].isolated_hand_score=fmin(interp[lbl].isolated_hand_score,-it2->second/medianhandsize);
	  //	cerr << "label " << it->first << " size=" << it2->second << endl;
	  if(it2->second>handsizemultiplier*medianhandsize){
	    // tag the label as invalid
	    invalidlabels.insert(lbl);
	    interp[lbl].isolated=false;

	    interp[lbl].left_hand=true;
	    interp[lbl].right_hand=true;
	    
	    //    cerr << "size filtering invalidating label " << it->first << endl;
	    
	    break; // skip to next label
	  }
	}
    }

//       cout << "interp after filterings: " << endl;

//       for(auto it=interp.begin(); it!=interp.end();it++)
// 	it->second.show();

    // here logic should be added that propagates isolated 
    // hands to composite blobs to which they merge or 
    // from which they leave

    bool changes;
    do{
      changes=false;

      for(size_t frameidx=0; frameidx<bp.blobs.size();frameidx++){
	for(size_t blobidx=0;blobidx<bp.blobs[frameidx].size();blobidx++){
	  BlobInfo *bi=&(bp.blobs[frameidx][blobidx]);

	  if(bi->overlappinglabels_prevframe.size()>1) // current blob is 
	    // composite of previous blobs
	  for(auto it=bi->overlappinglabels_prevframe.begin();
	      it!=bi->overlappinglabels_prevframe.end(); it++){

	    int idx=bp.lbl2blobidx[frameidx-1][*it];
	    const BlobInfo &bi2=bp.blobs[frameidx-1][idx];

	    if(interp[bi2.label].left_hand && interp[bi->label].left_hand==false){
	      interp[bi->label].left_hand=true;
	      changes=true;
	    }

	    if(interp[bi2.label].right_hand && interp[bi->label].right_hand==false){
	      interp[bi->label].right_hand=true;
	      changes=true;
	    }
	  }

	  if(bi->overlappinglabels_nextframe.size()>1) // current blob is 
	    // composite of next blobs
	  for(auto it=bi->overlappinglabels_nextframe.begin();
	      it!=bi->overlappinglabels_nextframe.end(); it++){

	    int idx=bp.lbl2blobidx[frameidx+1][*it];
	    const BlobInfo &bi2=bp.blobs[frameidx+1][idx];

	    if(interp[bi2.label].left_hand && interp[bi->label].left_hand==false){
	      interp[bi->label].left_hand=true;
	      changes=true;
	    }

	    if(interp[bi2.label].right_hand && interp[bi->label].right_hand==false){
	      interp[bi->label].right_hand=true;
	      changes=true;
	    }
	  }
	}
      }

    } while(changes);

//     cout << "interp after hand propagation: " << endl;

//     for(auto it=interp.begin(); it!=interp.end();it++)
//       it->second.show();


    bb[std::pair<int,std::string>(0,"blobinterpretation")]=interp;

  }



 void extractIsolatedHands(int firstframe,int lastframe,localBlackBoardType &bb){


    // based on blackboard entries "uniqueblobintmask" and "blobinterpretation"


   set<int> lhlabels,rhlabels;

   map<int,blobinterpretation_type> interp=*boost::any_cast<map<int,blobinterpretation_type> >(&bb[std::pair<int,std::string>(0,"blobinterpretation")]);


   // extract the remaining blobs from the mask   
    

   for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){

     Mat uniqueblobintmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"uniqueblobintmask")]);
      
     Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"frame")]);

     // collect statistics of blob sizes
     map<int, map<int,int> > blobsizes;
    
     // semantics: blobsizes[label][frame]

     for(int i=0;i<uniqueblobintmask.rows;i++)
       for(int j=0;j<uniqueblobintmask.cols;j++){
	 int lbl=uniqueblobintmask.at<int>(i,j);
	 blobsizes[frnumber][lbl]++;
       } 

//      cout << "frame " << frnumber << " blobsizes:" << endl;
//      for(auto it=blobsizes[frnumber].begin();it!=blobsizes[frnumber].end();it++)
//        cout <<"blobsizes["<<it->first<<"]="<<it->second<<endl;

      set<int> lbl;

      map<int,int> minx;
      map<int,int> miny;
      map<int,int> maxx;
      map<int,int> maxy;

      for(int i=0;i<uniqueblobintmask.rows;i++)
	for(int j=0;j<uniqueblobintmask.cols;j++){
	 
	  int label=uniqueblobintmask.at<int>(i,j);
	  
	  if(label==0) continue;

	  if(interp[label].isolated==false || 
	     (interp[label].left_hand==false && 
	      interp[label].right_hand==false)){
	    //cerr << "rejecting label " << label << endl;
	    continue;
	  }

	  if(interp[label].left_hand)
	    lhlabels.insert(label);
	  else
	    rhlabels.insert(label);
	  
	  lbl.insert(label);
	  if(minx.count(label)==0 || j < minx[label]) minx[label]=j;
	  if(maxx.count(label)==0 || j > maxx[label]) maxx[label]=j;
	  if(miny.count(label)==0 || i < miny[label]) miny[label]=i;
	  if(maxy.count(label)==0 || i > maxy[label]) maxy[label]=i;
	}


      //            cerr << "frame " << frnumber << " has " << lbl.size() << "blobs" << endl;

      if(lbl.size()>2) continue;

      int acceptedlabels=0;

      for(auto it=lbl.begin();it!=lbl.end();it++){

	int touchmargin=6;

	if(minx[*it] < touchmargin 
	   || maxx[*it] > uniqueblobintmask.cols-touchmargin-1
	   || miny[*it] < touchmargin 
	   || maxy[*it] > uniqueblobintmask.rows-touchmargin-1)
	  continue;

	// 	cerr << "size of blob " << *it << ": " << blobsizes[frnumber][*it]
	// 	     << endl;

	if(blobsizes[frnumber][*it]<200) continue; 
	// hard coded heuristic threshold
	
// 	 cerr << "accepted blob " << *it << " in frame "
// 	      << frnumber << endl;
  


	float ex=(maxx[*it]-minx[*it]+1);
	  ex /= (maxy[*it]-miny[*it]+1);

	  //if(ex<1) ex=1/ex;

// 	cerr << "elongatedness=" << ex << endl;

// 	// hard coded heuristic threshold
 	if(ex>3) continue;

	


	//	cerr << "median hand size: " << medianhandsize << endl;

	acceptedlabels++;

	Mat patchmat(maxy[*it]-miny[*it]+1,maxx[*it]-minx[*it]+1,CV_8UC3,
		     Scalar::all(0));
	Mat patchmask(maxy[*it]-miny[*it]+1,maxx[*it]-minx[*it]+1,CV_8UC1,
		      Scalar::all(0));

	for(int py=0;py<patchmat.rows;py++)
	  for(int px=0;px<patchmat.cols;px++){
	    if(uniqueblobintmask.at<int>(py+miny[*it],px+minx[*it])==*it){
	      patchmask.at<uchar>(py,px)=255;
	      patchmat.at<Vec3b>(py,px)=frame.at<Vec3b>(py+miny[*it],px+minx[*it]);
	    }
	  }

	if(!outputprefix.empty()){

	  char frnrstr[30];
	  sprintf(frnrstr,"%d",frnumber);

	  if(lhlabels.count(*it)>0)
	    writeFiles(outputprefix+":"+frnrstr+"_lh",frame,uniqueblobintmask,*it,
		       minx[*it],miny[*it],maxx[*it],maxy[*it],false);
	  else if(rhlabels.count(*it)>0)
	    writeFiles(outputprefix+":"+frnrstr+"_rh",frame,uniqueblobintmask,*it,
		       minx[*it],miny[*it],maxx[*it],maxy[*it],true);

	}

      }
    }

    // visualise and check the result

    if(false){
    for(int frnumber=firstframe+1;frnumber<=lastframe;frnumber++){

      Mat uniqueblobintmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"uniqueblobintmask")]);

      Mat prevuniqueblobintmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber-1,"uniqueblobintmask")]);

      showDeterministicFalseColour32bit("previous labels",prevuniqueblobintmask);
      showDeterministicFalseColour32bit("current labels",uniqueblobintmask);
      cv::waitKey(0);

    }
    }
      


 }
 
  void showDeterministicFalseColour32bit(const string &title, Mat &src){

    int maxind=0;

    Mat visimg(src.rows, src.cols, CV_8UC3,Scalar::all(0));

    for(int i=0;i<src.rows;i++)
      for(int j=0;j<src.cols;j++)
	if(src.at<int>(i,j)>maxind) maxind = src.at<int>(i,j);

    //    cerr << "maximum index is " << maxind  << endl;

    vector<Vec3b> colourtable(maxind+1);
    
    colourtable[0]=Vec3b(0,0,0);

    for(int i=1;i<=maxind;i++)
      colourtable[i]=Vec3b((i*87)&255,(i*101)&255,(i*173)&255);

    for(int i=0;i<src.rows;i++)
      for(int j=0;j<src.cols;j++)
	if(src.at<int>(i,j)>0)
	  visimg.at<Vec3b>(i,j)=colourtable[src.at<int>(i,j)];

    cv::imshow(title,visimg);


  }

  void showDeterministicFalseColour8bit(const string &title, Mat &src){

    int maxind=0;

    Mat visimg(src.rows, src.cols, CV_8UC3,Scalar::all(0));

    for(int i=0;i<src.rows;i++)
      for(int j=0;j<src.cols;j++)
	if(src.at<uchar>(i,j)>maxind) maxind = src.at<uchar>(i,j);

    //    cerr << "maximum index is " << maxind  << endl;

    vector<Vec3b> colourtable(maxind+1);
    
    colourtable[0]=Vec3b(0,0,0);

    for(int i=1;i<=maxind;i++)
      colourtable[i]=Vec3b((i*87)&255,(i*101)&255,(i*173)&255);

    for(int i=0;i<src.rows;i++)
      for(int j=0;j<src.cols;j++)
	if(src.at<uchar>(i,j)>0)
	  visimg.at<Vec3b>(i,j)=colourtable[src.at<uchar>(i,j)];

    cv::imshow(title,visimg);


  }

  void BlobProgression::dump(int frameidx){

    assert((int)blobs.size()>frameidx);
    assert((int)lbl2blobidx.size()>frameidx);

    cerr << "dumping blob progression record for frame " <<
      firstframe+frameidx << " (idx " << frameidx << endl;

    cerr << "blobs:" << endl;
    for(auto it=blobs[frameidx].begin();
	it != blobs[frameidx].end(); it++)
      it->dump();

    cerr << "lbl2blobidx:" << endl;
    for(auto it=lbl2blobidx[frameidx].begin();
	it != lbl2blobidx[frameidx].end(); it++)
      cerr << "  " << it->first << " -> " << it->second << endl;

    

  }


  void BlobInfo::dump(){
    //    string indent("  ");

    cerr << "frame=" << frame << " label= " << label << 
      " seedpoint=("<<seedpoint.x<<","<<seedpoint.y<<")"<<endl;

    cerr << "overlappinglabels_prevframe:";
    for(auto it=overlappinglabels_prevframe.begin();
	it!=overlappinglabels_prevframe.end(); it++)
      cerr << " " << *it;
    cerr << endl;

    cerr << "overlappinglabels_nextframe:";
    for(auto it=overlappinglabels_nextframe.begin();
	it!=overlappinglabels_nextframe.end(); it++)
      cerr << " " << *it;
    cerr << endl;
  }

  void removeSmallBlobs(Mat &m, Mat &mask, int minsize){

    // m is an uchar matrix of unique labels
    // mask is a binary mask (type uchar) of m being nonzero
    
    // removes less than minmsize times occurring labels of m
    // from mask

    // count the labels

//      cv::imshow("original mask", mask);
//      showDeterministicFalseColour8bit("labeling", m);
    vector<int> sizes(256,0);

    for(int i=0;i<m.rows;i++)
      for(int j=0;j<m.cols;j++)
	sizes[m.at<uchar>(i,j)]++;
    
    // second pass: zero labels w/ too small a size

    vector<int> labelmap(256);

    for(int i=0;i<256;i++)
      labelmap[i]= sizes[i]<minsize? 0 : 1;

    for(int i=0;i<m.rows;i++)
      for(int j=0;j<m.cols;j++)
	if(!labelmap[m.at<uchar>(i,j)])
	  mask.at<uchar>(i,j)=0;

//          cerr << "removed small blobs with size threshold " << minsize << endl;

//       cv::imshow("filtered mask", mask);
//       cv::waitKey(0);   



  }


  void TrackingSkinDetector::extractHandBlobs(int firstframe,int lastframe,localBlackBoardType &bb){


 // obtain a measure of characteristic scale by 
    // determining the average size of the face rectangles


    float avefacesize=0;

    for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){

      // assume the existence of face detections
      // for all frames

      // (missing faces should have already been interpolated/estimated)

      Rect faceLocation=*boost::any_cast<cv::Rect>(&bb[std::pair<int,std::string>(frnumber,"facerect")]);    
      avefacesize += faceLocation.width*faceLocation.height;
    }

    avefacesize /= (lastframe-firstframe+1);

    int minblobsize=0.1*avefacesize;

    // filter out too small blobs



    
    // go through frames and pick the ones
    // that fullfill the criteria:

    // 1. blob count is 3
    // 2. one blob overlaps with detected face

    // label the other blobs tentatively as hands 
    // they must

    // 1. have size in [0.3*avefacesize,headblobsize]
    // 2. one of them must have the centroid left of 
    //    the face centerline, the other one to the right

 for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){

      Mat skinmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"skinmask")]);

      Mat unique;

      labelConnectedComponents(skinmask,unique);
      removeSmallBlobs(unique,skinmask,minblobsize);

      map<int,int> labelhist;
      map<int,int> avex;

       map<int,int> minx;
      map<int,int> miny;
      map<int,int> maxx;
      map<int,int> maxy;

      for(int i=0;i<unique.rows;i++)
	for(int j=0;j<unique.cols;j++)
	  if(skinmask.at<uchar>(i,j)){
	    int lbl=unique.at<uchar>(i,j);
	    labelhist[lbl]++;
	    avex[lbl]+=j;

	    if(minx.count(lbl)==0 || j < minx[lbl]) minx[lbl]=j;
	    if(maxx.count(lbl)==0 || j > maxx[lbl]) maxx[lbl]=j;
	    if(miny.count(lbl)==0 || i < miny[lbl]) miny[lbl]=i;
	    if(maxy.count(lbl)==0 || i > maxy[lbl]) maxy[lbl]=i;

	  }

      cerr << "detected " << (int)labelhist.size() << " blobs in frame " << 
	frnumber << endl;

      if(labelhist.size()!=3){
	cerr << " -> frame rejected" << endl;
	continue;
      }


      Rect faceLocation=*boost::any_cast<cv::Rect>(&bb[std::pair<int,std::string>(frnumber,"facerect")]);    

      
      map<int,int> facehist;


      for(int i=faceLocation.y;i<faceLocation.br().y;i++)
	for(int j=faceLocation.x;j<faceLocation.br().x;j++)
	  if(skinmask.at<uchar>(i,j)){
	    int lbl=unique.at<uchar>(i,j);
	    facehist[lbl]++;

	  }

      for(auto it=labelhist.begin();it!=labelhist.end();it++)
	avex[it->first] /= it->second;

      if(facehist.size()!=1){
	cerr << "no single blob overlapping with face -> frame rejected" 
	     << endl;
	continue;
      }

      int headlbl=facehist.begin()->first;
      int lhlbl,rhlbl;

      int headsize=labelhist[headlbl];

      if(headsize<avefacesize){

	cerr << "head blob too small -> frame rejected" 
	     << endl;
	continue;
      }

      bool lhfound=false;
      bool rhfound=false;

      for(auto it=labelhist.begin();it!=labelhist.end();it++){
	if(it->first==headlbl) continue;
	if(avex[it->first]<0.5*(faceLocation.x+faceLocation.br().x)){
	  lhlbl=it->first;
	  lhfound=true;
	}
	else{
	  rhlbl=it->first;
	  rhfound=true;
	}
      }

      if(lhfound==false||rhfound==false){
	cerr << "couldn't determine left and right hands -> frame rejected" 
	     << endl;
	continue;
      }

      // check the size of tentative hands

      if(labelhist[lhlbl]<0.25*avefacesize ||
	 labelhist[rhlbl]<0.25*avefacesize ||
	 labelhist[lhlbl]>3*headsize ||
	 labelhist[rhlbl]>3*headsize){

	cerr << "hand blob sizes out of range -> frame rejected" 
	     << endl;
	continue;
      }

      cerr << "frame is good blob-wise" << endl;

      // now the frame is accepted, extract the hand patches

      
      Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"frame")]);

      Mat pmap=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"skinpmap")]);
      
      int touchmargin=6;

      // bool patchestoshow=false;
      bool writelh=false;
      bool writerh=false;

      // left hand

      int maskexpansion=4;


      Mat lhmat(maxy[lhlbl]-miny[lhlbl]+1+2*maskexpansion,maxx[lhlbl]-minx[lhlbl]+1+2*maskexpansion,CV_8UC3,
		     Scalar::all(0));
      Mat lhmask(maxy[lhlbl]-miny[lhlbl]+1+2*maskexpansion,maxx[lhlbl]-minx[lhlbl]+1+2*maskexpansion,CV_8UC1,
		      Scalar::all(0));

      Mat lhpmap(maxy[lhlbl]-miny[lhlbl]+1+2*maskexpansion,maxx[lhlbl]-minx[lhlbl]+1+2*maskexpansion,CV_32FC1,
		      Scalar::all(0));


      if(!(minx[lhlbl] < touchmargin 
	 || maxx[lhlbl] > unique.cols-touchmargin-1
	 || miny[lhlbl] < touchmargin 
	   || maxy[lhlbl] > unique.rows-touchmargin-1)){

	writelh=true;

      for(int py=0;py<lhmat.rows;py++)
	for(int px=0;px<lhmat.cols;px++){
	  if(unique.at<uchar>(py+miny[lhlbl]-maskexpansion,px+minx[lhlbl]-maskexpansion)==lhlbl){
	    lhmask.at<uchar>(py,px)=255;
	  }
	}

      // refine the detection result by reconsidering pixels within dilated 
      // mask around the hand

      Mat mask=lhmask.clone();

      cv::dilate(mask, mask, 
      		 cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(maskexpansion, maskexpansion)));
      
      for(int py=0;py<lhmat.rows;py++)
	for(int px=0;px<lhmat.cols;px++)
	  if(mask.at<uchar>(py,px)){

	    int fx=px+minx[lhlbl]-maskexpansion;
	    int fy=py+miny[lhlbl]-maskexpansion;

	    if(fx<0 || fy<0 || fx >= pmap.cols || fy >= pmap.rows)
	      continue;

	    //	    vector<float> samplevec;
	    //	    fe.extractFeatures(samplevec,py+miny[lhlbl]-maskexpansion,px+minx[lhlbl]-maskexpansion,frnumber,bb);
	    float prob=pmap.at<float>(fy,fx);

	    lhpmap.at<float>(py,px)=prob;
	    
	      //cls->predict_lean(samplevec);

	    float pthreshold=0.5;

	    if(prob>=pthreshold){
	      lhmask.at<uchar>(py,px)=255;
	    }
	  }

      // slightly smoothen the mask morphologically

	cv::morphologyEx(lhmask, lhmask, cv::MORPH_OPEN, 
			 cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(2,2)), 
		    cv::Point(-1,-1), 1);

	// fill small holes

	cv::morphologyEx(lhmask, lhmask, cv::MORPH_CLOSE, 
		    cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(2, 2)), 
		    cv::Point(-1,-1), 1);


	for(int py=0;py<lhmat.rows;py++)
	  for(int px=0;px<lhmat.cols;px++)
	    if(lhmask.at<uchar>(py,px))
	      lhmat.at<Vec3b>(py,px)=
		frame.at<Vec3b>(py+miny[lhlbl]-maskexpansion,
				px+minx[lhlbl]-maskexpansion);
	    
      }

      // right hand
      Mat rhmat(maxy[rhlbl]-miny[rhlbl]+1+2*maskexpansion,maxx[rhlbl]-minx[rhlbl]+1+2*maskexpansion,CV_8UC3,
		     Scalar::all(0));
      Mat rhmask(maxy[rhlbl]-miny[rhlbl]+1+2*maskexpansion,maxx[rhlbl]-minx[rhlbl]+1+2*maskexpansion,CV_8UC1,
		      Scalar::all(0));

      Mat rhpmap(maxy[rhlbl]-miny[rhlbl]+1+2*maskexpansion,maxx[rhlbl]-minx[rhlbl]+1+2*maskexpansion,CV_32FC1,
		      Scalar::all(0));


      if(!(minx[rhlbl] < touchmargin 
	 || maxx[rhlbl] > unique.cols-touchmargin-1
	 || miny[rhlbl] < touchmargin 
	   || maxy[rhlbl] > unique.rows-touchmargin-1)){

	writerh=true;

      for(int py=0;py<rhmat.rows;py++)
	for(int px=0;px<rhmat.cols;px++){
	  if(unique.at<uchar>(py+miny[rhlbl]-maskexpansion,px+minx[rhlbl]-maskexpansion)==rhlbl){
	    rhmask.at<uchar>(py,px)=255;
	  }
	}

      // refine the detection result by reconsidering pixels within dilated 
      // mask around the hand

      Mat mask=rhmask.clone();

      cv::dilate(mask, mask, 
      		 cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(maskexpansion, maskexpansion)));
      
      for(int py=0;py<rhmat.rows;py++)
	for(int px=0;px<rhmat.cols;px++)
	  if(mask.at<uchar>(py,px)){

	    int fx=px+minx[rhlbl]-maskexpansion;
	    int fy=py+miny[rhlbl]-maskexpansion;

	    if(fx<0 || fy<0 || fx >= pmap.cols || fy >= pmap.rows)
	      continue;

	    //	    vector<float> samplevec;
	    //	    fe.extractFeatures(samplevec,py+miny[rhlbl]-maskexpansion,px+minx[rhlbl]-maskexpansion,frnumber,bb);
	    float prob=pmap.at<float>(fy,fx);

	    rhpmap.at<float>(py,px)=prob;
	    
	      //cls->predict_lean(samplevec);

	    float pthreshold=0.5;

	    if(prob>=pthreshold){
	      rhmask.at<uchar>(py,px)=255;
	    }
	  }

      // slightly smoothen the mask morphologically

	cv::morphologyEx(rhmask, rhmask, cv::MORPH_OPEN, 
			 cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(2,2)), 
		    cv::Point(-1,-1), 1);

	// fill small holes

	cv::morphologyEx(rhmask, rhmask, cv::MORPH_CLOSE, 
		    cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(2, 2)), 
		    cv::Point(-1,-1), 1);


	for(int py=0;py<rhmat.rows;py++)
	  for(int px=0;px<rhmat.cols;px++)
	    if(rhmask.at<uchar>(py,px))
	      rhmat.at<Vec3b>(py,px)=
		frame.at<Vec3b>(py+miny[rhlbl]-maskexpansion,
				px+minx[rhlbl]-maskexpansion);
	    
      }

	char frnrstr[30];
	sprintf(frnrstr,"frame%d",frnumber);

	vector<int> writeparam;

	writeparam.push_back(CV_IMWRITE_PNG_COMPRESSION);
	writeparam.push_back(0);

	if(writelh){
	  string lhfn=outputprefix+"_"+frnrstr+"_lh.png";
	  string lhmaskfn=outputprefix+"_"+frnrstr+"_lhmask.png";

	  cv::imwrite(lhfn,lhmat,writeparam);
	  cv::imwrite(lhmaskfn,lhmask,writeparam);
	}

	if(writerh){
	  string rhfn=outputprefix+"_"+frnrstr+"_rh.png";
	  string rhmaskfn=outputprefix+"_"+frnrstr+"_rhmask.png";

	  cv::imwrite(rhfn,rhmat,writeparam);
	  cv::imwrite(rhmaskfn,rhmask,writeparam);

	}

// 	cv::imshow("frame",frame);
// 	cv::imshow("pmap",pmap);
// 	cv::imshow("skin mask",skinmask);

// 	cv::imshow(string("left hand: ")+lhfn,lhmat);
// 	cv::imshow("left hand pmap",lhpmap);
// 	cv::imshow(string("right hand: ")+rhfn,rhmat);

// 	cv::waitKey(0);
 }
  }

  void writeFiles(const string &fnprefix,
		  Mat frame,Mat uniqueblobintmask, int lbl,
		  int minx, int miny, int maxx, int maxy,bool mirror){

    //    cerr << "writing files for fnprefix " << fnprefix << endl;

    vector<int> writeparam;

    writeparam.push_back(CV_IMWRITE_PNG_COMPRESSION);
    writeparam.push_back(0);

    int extend=4;
     
    minx -= extend; if(minx<0) minx=0;
    miny -= extend; if(minx<0) miny=0;

    maxx += extend; if(maxx>=frame.cols) maxx=frame.cols-1;
    maxy += extend; if(maxy>=frame.rows) maxy=frame.rows-1;

    Mat patchmat(maxy-miny+1,maxx-minx+1,CV_8UC3,Scalar::all(0));
    Mat patchmasked(maxy-miny+1,maxx-minx+1,CV_8UC3,Scalar::all(0));
    Mat patchmask(maxy-miny+1,maxx-minx+1,CV_8UC1,Scalar::all(0));

    for(int i=0;i<patchmat.rows;i++)
      for(int j=0;j<patchmat.cols;j++){

	//int srcy= mirror ? patchmat.rows-i-1+miny : i+miny;
	int srcy=i+miny; // mirroring only in x-direction
       	int srcx= mirror ? patchmat.cols-j-1+minx : j+minx;


	patchmat.at<Vec3b>(i,j)=frame.at<Vec3b>(srcy,srcx);
	if(uniqueblobintmask.at<int>(srcy,srcx)==lbl){
	  patchmask.at<uchar>(i,j)=255;
	  patchmasked.at<Vec3b>(i,j)=patchmat.at<Vec3b>(i,j);
	}

      }	

    snakeRefine(patchmat,patchmask);

    // first write the bitmap within the bounding box as such

    string fn=fnprefix+"-cut.png";
    cv::imwrite(fn,patchmat,writeparam);

    // then masked with the hand mask

    fn=fnprefix+"-masked.png";
    cv::imwrite(fn,patchmasked,writeparam);

    // write out also the mask as bitmap

    fn=fnprefix+"-mask.png";
    cv::imwrite(fn,patchmask,writeparam);

    // finally write the bounding box coordinates
    // as a text file

    fn=fnprefix+"-coords.txt";
    FILE *fp=fopen(fn.c_str(),"w");
    if(fp)
      fprintf(fp,"(%d,%d)-(%d,%d)\n",minx,miny,maxx,maxy);
    fclose(fp);

  }


  void snakeRefine(cv::Mat patch, cv::Mat mask){
    
    Mat masked=patch.clone();

    Mat bw=mask.clone();

    for(int i=0;i<masked.rows;i++)
      for(int j=0;j<masked.cols;j++){
	bw.at<uchar>(i,j)=0.33*(
				patch.at<Vec3b>(i,j)[0]+patch.at<Vec3b>(i,j)[1]+patch.at<Vec3b>(i,j)[2]);
	if(mask.at<uchar>(i,j)==0)
	  masked.at<Vec3b>(i,j)=Vec3b(0,0,0);
      }

    cv::imshow("patch to refine", patch);
    cv::imshow("mask to refine", mask);
    cv::imshow("masked patch to refine", masked);
    
    Mat eimg;

    computeEdgeMagnitude(patch,eimg);

    floatimagesc("edge magnitude", eimg);

    cv::waitKey(0);      



    IplImage *image = 0 ;
    IplImage *image2 = 0 ; 

    int ialpha = 40;
    int ibeta=40; 
    int igamma=20; 


    IplImage img=mask;
    IplImage img2=patch;
    
    image=&img;
    image2=&img2;

    CvMemStorage* storage = cvCreateMemStorage(0);
    CvSeq* contours = 0;

    cvFindContours( image, storage, &contours, sizeof(CvContour), 
        CV_RETR_EXTERNAL , CV_CHAIN_APPROX_SIMPLE );

    int length = contours->total;    
   
    CvPoint* point = new CvPoint[length]; 

    CvSeqReader reader;
    CvPoint pt= cvPoint(0,0);;    
    CvSeq *contour2=contours;    

    cvStartReadSeq(contour2, &reader);
    for (int i = 0; i < length; i++)
    {
        CV_READ_SEQ_ELEM(pt, reader);
        point[i]=pt;
    }
    cvReleaseMemStorage(&storage);

    for( int i=0;i<length;i++)
    {
        int j = (i+1)%length;
        cvLine( image2, point[i],point[j],CV_RGB( 255, 255, 255 ),1,8,0 ); 
    }

    float alpha=ialpha/100.0f; 
    float beta=ibeta/100.0f; 
    float gamma=igamma/100.0f; 

    CvSize size; 
    size.width=5; 
    size.height=5; 
    CvTermCriteria criteria; 
    criteria.type=CV_TERMCRIT_ITER; 
    criteria.max_iter=1000; 
    criteria.epsilon=0.1; 

    IplImage bwi=bw;

    cvSnakeImage( &bwi, point,length,&alpha,&beta,&gamma,CV_VALUE,size,criteria,1 ); 

    for(int i=0;i<length;i++)
    {
        int j = (i+1)%length;
        cvLine( &img2, point[i],point[j],CV_RGB( 0, 255, 0 ),1,8,0 ); 
    }
    delete []point;


    cv::imshow("refined patch", patch);
    cv::waitKey(0);      

  }


  void visualiseInterpretation(int firstframe,int lastframe,localBlackBoardType &bb, 
			       const string &destination){

    // visualises the interpretation of blob progression
    // recorded on blackbord entry "blobinterpretation"

    // also displays "blobtracks" if they exist on bb

    // appends the output in given video sequence if it's specified
    
    // otherwise it is echoed to the screen

    // uses also "uniqueblobintmask" and "frame" 
    // entries stored on the blackboard

    map<int,blobinterpretation_type> interp=*boost::any_cast<map<int,blobinterpretation_type> >(&bb[std::pair<int,std::string>(0,"blobinterpretation")]);


    // go through interp and assign ordinal number corresponding
    // to a colour code to every label avccording to its interpretation

    // key:
    // 0 unknown isolated (bright blue)
    // 1 isolated head (bright green)
    // 2 isolated left hand (bright red)
    // 3 isolated right hand (bright cyan)
    // 4 unknown composite (bright magenta)
    // 5 unknown composite with head (yellow)
    // 6 unknown composite with left hand (white)
    // 7 left hand + head (pastel yellow)
    // 8 unknown composite with right hand (pastel magenta)
    // 9 right hand + head (pastel cyan)
    // 10 left hand + right hand (pastel red)
    // 11 left hand + right hand + head (pastel green)


    

    map<int,int> code;

    for(auto it=interp.begin();it!=interp.end();it++){
      const blobinterpretation_type &bi=it->second;

      if(bi.isolated){
	if(bi.head)
	  code[bi.uniquelabel]=1;
	else if(bi.left_hand)
	  code[bi.uniquelabel]=2;
	else if(bi.right_hand)
	  code[bi.uniquelabel]=3;
	else
	  code[bi.uniquelabel]=0;
      }
      else{

	  code[bi.uniquelabel]=
	    4+(bi.head ? 1 : 0) + (bi.left_hand ?2:0) + (bi.right_hand?4:0);


      }
    }

    map<int,Vec3b> cc; // colour coding

    cc[0]=Vec3b(255,0,0);
    cc[1]=Vec3b(0,255,0);
    cc[2]=Vec3b(0,0,255);
    cc[3]=Vec3b(255,255,0);
    cc[4]=Vec3b(255,0,255);
    cc[5]=Vec3b(0,255,255);
    cc[6]=Vec3b(255,255,255);
    cc[7]=Vec3b(190,255,255);
    cc[8]=Vec3b(255,190,255);
    cc[9]=Vec3b(255,255,190);
    cc[10]=Vec3b(190,190,255);
    cc[11]=Vec3b(190,255,190);

    // show colours for key

//     Mat frame=(*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(firstframe,"frame")])).clone();

//     int dy=frame.rows/12;

//     for(int i=0;i<12;i++){

//       for(int x=0;x<frame.cols;x++)
// 	for(int y=i*dy;y<(i+1)*dy;y++)
// 	  frame.at<Vec3b>(y,x)=cc[i];
// 	  //	  cv::rectangle(frame,Point(0,i*dy), Point(frame.cols,(i+1)*dy-1),
// 	  //		    Scalar(cc[i]));
//     }

//    cv::imshow("key to interpretation", frame);
//    cv::waitKey(0);  
      
    for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){

      Mat uniqueblobintmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"uniqueblobintmask")]);
      
      Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"frame")]);

      Mat visframe=frame.clone();

      Mat magnframe;

      // float factor=1;

      //   cv::resize(frame,magnframe, cv::Size(0,0), factor, factor, CV_INTER_AREA );

      for(int i=0;i<uniqueblobintmask.rows;i++)
	for(int j=0;j<uniqueblobintmask.cols;j++){
	 
	  int label=uniqueblobintmask.at<int>(i,j);
	  
	  if(label==0) continue;

	  visframe.at<Vec3b>(i,j)=cc[code[label]];
	}

      if(bb.count(std::pair<int,std::string>(frnumber,"blobtracks"))){
	map<int, vector<Rect> > *bt=boost::any_cast<map<int,vector<Rect> > >(&bb[std::pair<int,std::string>(frnumber,"blobtracks")]);

	//       // visualise the result

	for(auto it=bt->begin(); it!=bt->end();it++){

	  cv::Scalar c=CV_RGB(((it->first+1)*87)&255,((it->first+1)*101)&255,
			      ((it->first+1)*173)&255);

	  for(auto rit=it->second.begin();rit!=it->second.end();rit++)
	    cv::rectangle(visframe,rit->tl(),rit->br(),c);

	}
      }

      char str[80];
      sprintf(str,"frame %d",frnumber);

      putText(visframe,str,Point(0,15),  cv::FONT_HERSHEY_PLAIN, 1,CV_RGB(255,255,255));


      //      cv::imshow("frame", magnframe);
      if(destination==""){
	cv::imshow("interpreted frame", visframe);
	cv::waitKey(0);  
      } else
      appendFrameToOutputVideo(visframe,destination,bb);
      cout << "frame " << frnumber << endl; 



    }




  }


  void updateMaskByTrackedPoints(Mat &oldmask,Mat &newmask,vector<Point2f> &oldpoints, vector<Point2f> &newpoints){

    Mat wsum(oldmask.rows,oldmask.cols,cv::DataType<float>::type,cv::Scalar::all(0));
    Mat summat(oldmask.rows,oldmask.cols,cv::DataType<float>::type,cv::Scalar::all(0));

    int radius=30;

    Mat wmat(2*radius+1,2*radius+1,cv::DataType<float>::type,cv::Scalar::all(1));
    for(int dx=-radius;dx<=radius;dx++)
      for(int dy=-radius;dy<=radius;dy++)
	if(dx!=0 || dy!=0)
	  wmat.at<float>(dy+radius,dx+radius)=1.0/(sqrt(dx*dx+dy*dy));

    //    cout << "weight matrix formed" << endl;

    for(size_t i=0;i<oldpoints.size();i++){

      //cout << "processing point " << i << endl;

      int minx=newpoints[i].x-radius;
      int miny=newpoints[i].y-radius;

      int maxx=newpoints[i].x+radius;
      int maxy=newpoints[i].y+radius;

      if(minx<0) minx=0;
      if(miny<0) miny=0;

      if(maxx>=oldmask.cols) maxx=oldmask.cols-1;
      if(maxy>=oldmask.rows) maxy=oldmask.rows-1;

      int dx=oldpoints[i].x-newpoints[i].x;
      int dy=oldpoints[i].y-newpoints[i].y;

      if(minx+dx<0) minx=-dx;
      if(miny+dy<0) miny=-dy;

      if(maxx+dx>=oldmask.cols) maxx=oldmask.cols-1-dx;
      if(maxy+dy>=oldmask.rows) maxy=oldmask.rows-1-dy;

      for(int tgtx=minx;tgtx<=maxx;tgtx++)
	for(int tgty=miny;tgty<=maxy;tgty++){
	  float w=wmat.at<float>(tgty-newpoints[i].y+radius,
				 tgtx-newpoints[i].x+radius);

	  wsum.at<float>(tgty,tgtx) += w;
	  summat.at<float>(tgty,tgtx) += (1.0/255)*w*oldmask.at<uchar>(tgty+dy,
								       tgtx+dx);	  

	}

    }

    //    cout << "all points processed" << endl;

    newmask=oldmask.clone();
    
    for(int x=0;x<oldmask.cols;x++)
      for(int y=0;y<oldmask.rows;y++){

	//	cout << "(x,y)=("<<x<<","<<y<<")"<<endl;
	
	  newmask.at<uchar>(y,x)=
	    (wsum.at<float>(y,x)==0)? 0 : 
	    255*summat.at<float>(y,x)/wsum.at<float>(y,x);
      }

    cout << " returning" << endl;
  }

  void estimateSmoothMotionField(vector<Point2f> &oldpoints, vector<Point2f> 
				 &newpoints, Mat &motionfield){

    Mat wsum(motionfield.rows,motionfield.cols,cv::DataType<float>::type,cv::Scalar::all(0));
    Mat summat_x(motionfield.rows,motionfield.cols,cv::DataType<float>::type,cv::Scalar::all(0));
    Mat summat_y(motionfield.rows,motionfield.cols,cv::DataType<float>::type,cv::Scalar::all(0));

    motionfield=Mat(motionfield.rows,motionfield.cols,cv::DataType<Point2f>::type,cv::Scalar::all(0));

    int radius=50;

    Mat wmat(2*radius+1,2*radius+1,cv::DataType<float>::type,cv::Scalar::all(1));
    for(int dx=-radius;dx<=radius;dx++)
      for(int dy=-radius;dy<=radius;dy++)
	if(dx!=0 || dy!=0)
	  wmat.at<float>(dy+radius,dx+radius)=1.0/(sqrt(dx*dx+dy*dy));

    for(size_t i=0;i<oldpoints.size();i++){

      int minx=oldpoints[i].x-radius;
      int miny=oldpoints[i].y-radius;

      int maxx=oldpoints[i].x+radius;
      int maxy=oldpoints[i].y+radius;

      if(minx<0) minx=0;
      if(miny<0) miny=0;

      if(maxx>=motionfield.cols) maxx=motionfield.cols-1;
      if(maxy>=motionfield.rows) maxy=motionfield.rows-1;

      int dx=oldpoints[i].x-newpoints[i].x;
      int dy=oldpoints[i].y-newpoints[i].y;

      for(int tgtx=minx;tgtx<=maxx;tgtx++)
	for(int tgty=miny;tgty<=maxy;tgty++){
	  float w=wmat.at<float>(tgty-oldpoints[i].y+radius,
				 tgtx-oldpoints[i].x+radius);

	  wsum.at<float>(tgty,tgtx) += w;
	  summat_x.at<float>(tgty,tgtx) += w*dx;	  
	  summat_y.at<float>(tgty,tgtx) += w*dy;	  

	}
    }


    for(int x=0;x<motionfield.cols;x++)
      for(int y=0;y<motionfield.rows;y++){
	
	motionfield.at<Point2f>(y,x).x=(wsum.at<float>(y,x)==0)? 0 : 
	  summat_x.at<float>(y,x)/wsum.at<float>(y,x);
	
	motionfield.at<Point2f>(y,x).y=(wsum.at<float>(y,x)==0)? 0 : 
	  summat_y.at<float>(y,x)/wsum.at<float>(y,x);

	//	cout << "smoothed y motion:" << val << endl;

      }


  }

  void evaluateExtrapolatedMotionFieldAtPoints(vector<Point2f> &oldpoints, vector<Point2f> &newpoints, vector<Point2f> &evalpoints,vector<Point2f> &evalres,Size imgsize){

    evalres.clear();

   Mat idxmat(imgsize.width,imgsize.height,
	      cv::DataType<int>::type,cv::Scalar::all(-1));


    for(size_t i=0;i<oldpoints.size();i++)
      idxmat.at<int>(oldpoints[i].x,oldpoints[i].y)=i;

    vector<Point> delta;

    int maxdim=fmax(idxmat.cols,idxmat.cols);

    delta.push_back(Point(0,0));

    for(int d=1;d<=maxdim/2;d++){
      for(int x=-d;x<=d;x++){
	delta.push_back(Point(x,-d));
	delta.push_back(Point(x,d));
      }
      
      for(int y=-d+1;y<d;y++){
	delta.push_back(Point(-d,y));
	delta.push_back(Point(d,y));
      }
    }

    // now delta contains ordered enumeration of relative locations

    for(auto it=evalpoints.begin();it!=evalpoints.end();it++){

      int x=it->x;
      int y=it->y;

      //      cout << "target point (" <<x<<","<<y<<")"<<endl;
      
      int ptctr=0;
      float wsum=0;
      float xsum=0;
      float ysum=0;

      for(int i=0;ptctr<6;i++){

	int ix=x+delta[i].x;
	int iy=y+delta[i].y;

	if(ix<0 || ix>= idxmat.cols || iy<0 || iy>=idxmat.rows)
	  continue;
	
	int idx=idxmat.at<int>(iy,ix);
	if(idx==-1) continue;
	
	int dx=oldpoints[idx].x-newpoints[idx].x;
	int dy=oldpoints[idx].y-newpoints[idx].y;

	//	cout << "neighbouring motion point at ("<<ix<<","<<iy<<")"<<endl; 
	
	float a=1;
	
	float w=1.0/(1+a*sqrt((x-ix)*(x-ix)+(y-iy)*(y-iy)));
	
	//	cout << "dx="<<dx<<" dy="<<dy<<" w="<<w<<endl;

	wsum += w;
	xsum += w*dx;
	ysum += w*dy;

	ptctr++;
      }

      //      cout << "average: ("<<xsum/wsum<<","<<ysum/wsum<<")"<<endl; 

      evalres.push_back((wsum>0) ? Point2f(xsum/wsum,ysum/wsum) : 
			Point2f(0,0));

      }


  }

  void init_grid(Mat &mask, int spacing, vector<gridpoint_type> &grid){

    grid.clear();

    map<int,vector<int> > nbrcoord;

    map<std::pair<int,int>,int> coord2idx;

    gridpoint_type n;

    int idx=0;

    for(int x=0;x<mask.cols;x+=spacing)
      for(int y=0;y<mask.rows;y+=spacing)
	if(mask.at<uchar>(y,x)){
	  n.idx=idx++;
	  n.refx=n.tgtx=x;
	  n.refy=n.tgty=y;
	  n.nbr.clear();
	  n.l0.clear();
	  for(int nbrx=x-spacing;nbrx<=x+spacing;nbrx+=spacing)
	    for(int nbry=y-spacing;nbry<=y+spacing;nbry+=spacing){
	      //if(nbrx!=x && nbry!=y) continue; // uncomment for 4-connectivity

	      if(coord2idx.count(std::pair<int,int>(nbrx,nbry))>0){
		float l=sqrt((nbrx-x)*(nbrx-x)+(nbry-y)*(nbry-y));
		n.nbr.push_back(coord2idx[std::pair<int,int>(nbrx,nbry)]);
		n.l0.push_back(l);
		grid[coord2idx[std::pair<int,int>(nbrx,nbry)]].nbr.push_back(n.idx);
		grid[coord2idx[std::pair<int,int>(nbrx,nbry)]].l0.push_back(l);

	      }
	    }
	  coord2idx[std::pair<int,int>(x,y)]=n.idx;
	  grid.push_back(n);
	}
  }

  void init_grid_ntuple_full(Mat &mask, int n, vector<gridpoint_type> &grid){

    // randomly select the first point within the mask

    //cout << "size of mask:"<<mask.cols<<"x"<<mask.rows<<endl;

//     cv::imshow("mask for grid selection", mask);
//        cv::waitKey(0);

    grid.clear();

    gridpoint_type node;

    node.idx=0;


    for(;;){
      int x=rand(mask.cols);
      int y=rand(mask.rows);

      if(mask.at<uchar>(y,x)>0){
	node.refx=node.tgtx=x;
	node.refy=node.tgty=y;

	grid.push_back(node);

	break;
      }

    }

    //cout << "initial point selected at " << grid[0].tgtx << "," << grid[0].tgty << endl;

    int maxdist=40;
    int mindist=8;

    for(int i=1;i<n;){

      int x=grid[0].tgtx+rand(2*maxdist)-maxdist;
      int y=grid[0].tgty+rand(2*maxdist)-maxdist;

      //cout << "csampled point " << x<<","<<y<<endl;

      if(x<0 || y<0 || x>= mask.cols || y>= mask.rows) continue;
      if(mask.at<uchar>(y,x)==0) continue;

      bool tooclose=false;


      //cout << "about to check distances" << endl;
      for(int ii=0;ii<i;ii++){
	int dx=x-grid[ii].tgtx;
	int dy=y-grid[ii].tgty;

	if(dx*dx+dy*dy<mindist*mindist){
	  tooclose=true;
	  break;
	}
      }

      if(tooclose) continue;

      // cout << "selected "<<i<<"th point at ("<<x<<","<<y<<")"<<endl;

      node.idx=i++;
      node.refx=node.tgtx=x;
      node.refy=node.tgty=y;

      grid.push_back(node);

    }

    // coordinates of the points selected, 
    // setup the neighbour connections

    //    cout << grid.size() << " point selected, constructing neigbourhood" << endl;

    for(int i=0;i<n;i++){
      grid[i].nbr.clear();
      grid[i].l0.clear();
      for(int ii=0;ii<n;ii++){
	if(i==ii) continue;

	grid[i].nbr.push_back(ii);

	int dx=grid[i].tgtx-grid[ii].tgtx;
	int dy=grid[i].tgty-grid[ii].tgty;

	grid[i].l0.push_back(sqrt(dx*dx+dy*dy));
      }
    }

  }

  void showgrid(Mat &img, const vector<gridpoint_type> &grid, bool showref,
		bool showtgt, bool showlinks, bool showtgtlinks){

    for(size_t i=0;i<grid.size();i++){
      if(showref) cv::circle(img, Point(grid[i].refx,grid[i].refy),
			     1, CV_RGB(255,0,0));
      if(showtgt) cv::circle(img, Point(grid[i].tgtx,grid[i].tgty),
		 1, CV_RGB(0,255,0));
      if(showlinks)
	for(size_t ni=0;ni<grid[i].nbr.size();ni++){
	  cv::line(img, Point(grid[i].refx,grid[i].refy), 
		   Point(grid[grid[i].nbr[ni]].refx,grid[grid[i].nbr[ni]].refy),
		   CV_RGB(255,255,255));
	}

      if(showtgtlinks)
	for(size_t ni=0;ni<grid[i].nbr.size();ni++){
	  cv::line(img, Point(grid[i].tgtx,grid[i].tgty), 
		   Point(grid[grid[i].nbr[ni]].tgtx,grid[grid[i].nbr[ni]].tgty),
		   CV_RGB(255,255,255));

	
	}
    }
  }

  
  void detectOccludedArea(int firstreferenceframe,
			  int firstocclusionframe,
			  int lastframe,
			  size_t nr_gridlets,
			  //			  int isolatedheadlbl,
			  const string &constraintmaskid,
			  const string &unoccludedptid,
			  localBlackBoardType &bb,
			  bool show){


    int timedir;

    if(firstreferenceframe<firstocclusionframe &&
       firstocclusionframe<=lastframe)
      timedir=1;
    else if(firstreferenceframe>firstocclusionframe &&
       firstocclusionframe>=lastframe)
      timedir=-1;
    else throw string("detectOccludedArea(): support only forward and backward time flow");


    cout << "timedir="<<timedir<<endl;

    //	 map<int,Mat> vistracks;

	 Mat mask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(firstreferenceframe,constraintmaskid)]);

// 	 for(int x=0;x<mask.cols;x++)
// 	   for(int y=0;y<mask.rows;y++)
// 	     mask.at<uchar>(y,x)=(uniqueblobintmask.at<int>(y,x)==isolatedheadlbl)?255:0;


	 //Mat orgmask=mask.clone();

	//  for(int f=firstreferenceframe;timedir*f<=timedir*lastframe;f+=timedir){
// 	   //	   vistracks[f]=Mat(mask.rows,mask.cols,CV_8UC3,cv::Scalar::all(0));
// 	   vistracks[f]=boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f+1,"frame")])->clone();

// 	 }


	 grid_type grid,best_grid;
	 int spacing=10;

	 for(unsigned int gridlet=0;gridlet<nr_gridlets;gridlet++){

	   //	   	   cv::imshow("mask", mask);
	   //cv::imshow("orgmask", orgmask);
	   //	   cv::waitKey(0);  

	   //	   if(gridlet%1==0)
	     cout << gridlet << endl;

	   //mask=orgmask.clone();
	   
	   init_grid_ntuple_full(mask, 5, grid);
	   
	   cout << "gridlet initialised" << endl;

	   // bool nextgridlet=false;

	   Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(firstreferenceframe,"frame")]);

	   // then do tracking successively for all frame pairs

	   vector<float> referenceEnergies;
	   float threshold;

	   int refframe=firstreferenceframe;

	   for(int framenr=firstreferenceframe;timedir*framenr<timedir*lastframe;framenr+=timedir){
	     
	     //refframe=framenr;


	     //     }

	     if(framenr==firstocclusionframe){
	       float sum=0;
	       float sqrsum=0;

	       for(auto it=referenceEnergies.begin();
		   it != referenceEnergies.end(); it++){
		 sum += *it;
		 sqrsum += (*it)*(*it);
	       }
	       
	       threshold=0;

	       size_t n=referenceEnergies.size();

	       if(n){
		 sum /= n;
		 sqrsum /= n;
		 sqrsum -= sum*sum;
		 
		 threshold=sum + fmax(0.2*sum,fmax(0.0025,1.3*sqrt(sqrsum)));

		 threshold = fmin(0.015,threshold);

		 cout << "determined threshold: " << threshold << endl;
		 bool acceptall=false;
		 if(acceptall)
		   threshold=1;

	       }


	     }



       Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,"frame")]);
       
       Mat nextframe=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(framenr+timedir,"frame")]);

       Mat frif=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,"intensity_float")]);
       
       Mat nextif=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(framenr+timedir,"intensity_float")]);

       //       Mat uniqueblobintmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(framenr+timedir,"uniqueblobintmask")]);

       Mat cmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(framenr+timedir,constraintmaskid)]);
       

	 
       // initialise grid to its old location, hoping
       // that face moves so little that optimisation is able 
       // to compensate for the movement

	 for(auto it=grid.begin();it!=grid.end();it++){
	   it->tgtx=it->refx;
	   it->tgty=it->refy;

	   for(size_t nind=0;nind<it->nbr.size();nind++){
	     int dx=it->refx-grid[it->nbr[nind]].refx;
	     int dy=it->refy-grid[it->nbr[nind]].refy;

	     it->l0[nind]=sqrt(dx*dx+dy*dy);
	   }
	 }

// 	 Mat visframe=nextframe.clone();
// 	 showgrid(visframe,grid,true,false,true);
	 
// 	 cv::imshow("initial grid", visframe);
// 	 cv::waitKey(0);
	     

	 // int templsize=13;
	 int templater=6;
	 float alpha=1.0/4000;

	 float bestE;

	 grid_type oldgrid=grid;

	 

	 updateGridByGreedyLocal(grid,frif,nextif,cmask,
				 templater,2*spacing,alpha,
				 bestE,best_grid);



	 float finalE=evaluateGridEnergy(grid,true,true, -1,
					 frif,nextif,
					 templater,alpha);

	 cout << "finalE="<<finalE<<" (threshold="<<threshold<<")"<<endl;


	 if(timedir*framenr<timedir*firstocclusionframe)
	   referenceEnergies.push_back(finalE);



	 if(timedir*framenr<timedir*firstocclusionframe || finalE<threshold){
	   if(show)	   cout << "tracking accepted" << endl;

	   float validThr=0.015;

	   if(finalE<validThr){

	   if(bb.count(std::pair<int,string>(framenr+timedir,unoccludedptid))==0)
	     bb[std::pair<int,std::string>(framenr+timedir,unoccludedptid)]=vector<Point>();


	   for(auto it=grid.begin();it!=grid.end();it++){

	     vector<Point> &v=*boost::any_cast<vector<Point> >(&bb[std::pair<int,std::string>(framenr+timedir,unoccludedptid)]);
	     //	   vistracks[framenr].at<uchar>(it->tgty,it->tgtx)=255;
	     //	     vistracks[framenr].at<Vec3b>(it->tgty,it->tgtx)=Vec3b(255,255,255);
	     v.push_back(Point(it->tgtx,it->tgty));

	     //	     cout << "vector size now " << v.size() << endl;
	     
	   }
	   }
	   refframe=framenr+timedir;
	 }
	 else{
	   if(show)	   cout << "tracking not accepted" << endl;
	  grid=oldgrid;
	 }
	 
	 // prepare moving to next frame

	 for(auto it=grid.begin();it!=grid.end();it++){
	   it->refx=it->tgtx;
	   it->refy=it->tgty;
	 }


	 // 	 if(gridlet==0){
 	 if(show){
 	   Mat visframe=nextframe.clone();
 	   showgrid(visframe,grid,true,true,false,true);
 	   //appendFrameToOutputVideo(visframe,"outputvideo",bb);
	   cv::imshow("tracked grid", visframe);

	   cv::waitKey(0);
 	 }

	   }
	 }

	 //	 cout << "last gridlet tracked" << endl;

	 // show the final result
// 	 for(int framenr=firstreferenceframe;timedir*framenr<=timedir*lastframe;framenr+=timedir){

 
// 	   Mat nextframe=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(framenr+timedir,"frame")]);

// 	 Mat visframe=nextframe.clone();
// 	 showgrid(visframe,grid,false,true,false,true);
// 	 cv::imshow("tracked grid", visframe);


// 	 cv::imshow("cumulative track", vistracks[framenr]);
// 	 cv::waitKey(0);
// 	 }




  }


 void detectOccludedArea(int firstframe,
			 int lastframe,
			 const string &salientpointid,
			 const string &constraintmaskid,
			 const string &unoccludedptid,
			 //			 const string &motionfieldid,
			 localBlackBoardType &bb,
			 bool show){

   bool orgShow=show;


    int timedir;

    if(firstframe<=lastframe)
      timedir=1;
    else
      timedir=-1;

    cout << "timedir="<<timedir<<endl;

    	 map<int,Mat> vistracks;


	 for(int f=firstframe;timedir*f<=timedir*lastframe-1;f+=timedir){
	    // 		   vistracks[f]=Mat(mask.rows,mask.cols,CV_8UC3,cv::Scalar::all(0));
 	   vistracks[f]=boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f+1,"frame")])->clone();

 	 }
	  
	 //	  Mat refpoints=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(firstframe,salientpointid)]);

	 grid_type grid,best_grid;
	 int spacing=10;

 // then go through the list of salient points and track 
    // a set of points constellations from refframe to target frame

	 int points_in_gridlet=4;
	 
	 int maxdist=20;
	 
	 vector<vector<Point> > gridpoints;
	 
	 enumerateNtuplesAmongPoints(firstframe, salientpointid,
					  points_in_gridlet, maxdist,
					  gridpoints,bb);

	 int pointctr=0;

	 int samplestep=1;

	 for(auto git=gridpoints.begin();git!=gridpoints.end();git++){

	   if(pointctr%50==0) show=orgShow;
	   else show=false;

	   if(pointctr%samplestep!=0) continue;

	   pointctr++;

	   //	   cout << "seed point (" << (*git)[0].x << "," << (*git)[0].y<<")"<< endl;

	   cout << "seed point " << pointctr << " / " << gridpoints.size() << endl;

	   grid_type grid(git->size());
	   for(int i=0;i<(int)git->size();i++){
	     grid[i].idx=i;
	     grid[i].refx=(*git)[i].x;
	     grid[i].refy=(*git)[i].y;
	     
	   }

	   set_nbr_fullconnectivity(grid);
	   set_l0(grid);

	   //     for(int i=0;i<(int)pi.size();i++)
	   //  	grid[i].show();
	   
	   // bool nextgridlet=false;

	   Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(firstframe,"frame")]);

	   // then do tracking successively for all frame pairs

	   vector<float> referenceEnergies;
	   float threshold=-1;

	   int refframes_needed=10;

	   int refframe=firstframe;

	   for(int framenr=firstframe;timedir*framenr<timedir*lastframe;framenr+=timedir){
	     
	     //refframe=framenr;


	     //     }

	     if(threshold<0 && referenceEnergies.size()==(size_t)refframes_needed){
	       float sum=0;
	       float sqrsum=0;

	       for(auto it=referenceEnergies.begin();
		   it != referenceEnergies.end(); it++){
		 sum += *it;
		 sqrsum += (*it)*(*it);
	       }
	       
	       threshold=0;

	       size_t n=referenceEnergies.size();

	       if(n){
		 sum /= n;
		 sqrsum /= n;
		 sqrsum -= sum*sum;
		 
		 threshold=sum + fmax(0.2*sum,fmax(0.0025,1.3*sqrt(sqrsum)));

		 threshold = fmax(0.045,threshold);

		 threshold = fmin(threshold,0.1);

		 cout << "determined threshold: " << threshold << endl;

		 bool acceptall=false;
		 if(acceptall)
		   threshold=1;

	       }

	     }



       Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,"frame")]);
       
       Mat nextframe=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(framenr+timedir,"frame")]);

       Mat frif=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,"intensity_float")]);
       
       Mat nextif=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(framenr+timedir,"intensity_float")]);

       //       Mat uniqueblobintmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(framenr+timedir,"uniqueblobintmask")]);

       Mat cmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(framenr+timedir,constraintmaskid)]);
       
       Mat dummymask(frame.rows,frame.cols,CV_8UC1,Scalar::all(0));

       
	 
       // initialise grid to its old location, hoping
       // that face moves so little that optimisation is able 
       // to compensate for the movement

	 for(auto it=grid.begin();it!=grid.end();it++){
	   it->tgtx=it->refx;
	   it->tgty=it->refy;

	   for(size_t nind=0;nind<it->nbr.size();nind++){
	     int dx=it->refx-grid[it->nbr[nind]].refx;
	     int dy=it->refy-grid[it->nbr[nind]].refy;

	     it->l0[nind]=sqrt(dx*dx+dy*dy);
	   }
	 }

// 	 Mat visframe=nextframe.clone();
// 	 showgrid(visframe,grid,true,false,true);
	 
// 	 cv::imshow("initial grid", visframe);
// 	 cv::waitKey(0);
	     

	 // int templsize=13;
	 int templater=6;
	 float alpha=1.0/4000;

	 float bestE;

	 grid_type oldgrid=grid;

	 

	 updateGridByGreedyLocal(grid,frif,nextif,dummymask,
				 templater,2*spacing,alpha,
				 bestE,best_grid);



	 float finalE=evaluateGridEnergy(grid,true,true, -1,
					 frif,nextif,
					 templater,alpha);

	 //cout << "finalE="<<finalE<<" (threshold="<<threshold<<")"<<endl;


	 if(referenceEnergies.size()<(size_t)refframes_needed)
	   referenceEnergies.push_back(finalE);



	 if(threshold<0 || finalE<threshold){
	   if(show)	   cout << "tracking accepted" << endl;

// 	   float validThr=0.015;

// 	   if(finalE<validThr){

	     if(bb.count(std::pair<int,string>(framenr+timedir,unoccludedptid))==0)
	       bb[std::pair<int,std::string>(framenr+timedir,unoccludedptid)]=vector<Point>();


	     for(auto it=grid.begin();it!=grid.end();it++){
	       
	       vector<Point> &v=*boost::any_cast<vector<Point> >(&bb[std::pair<int,std::string>(framenr+timedir,unoccludedptid)]);
	       //	   vistracks[framenr].at<uchar>(it->tgty,it->tgtx)=255;
	       vistracks[framenr].at<Vec3b>(it->tgty,it->tgtx)=Vec3b(255,255,255);
	       v.push_back(Point(it->tgtx,it->tgty));
	       
	       //	     cout << "vector size now " << v.size() << endl;
	       
	     }
// 	   }
	   refframe=framenr+timedir;

	   // prepare moving to next frame

	   for(auto it=grid.begin();it!=grid.end();it++){
	     it->refx=it->tgtx;
	     it->refy=it->tgty;
	   }



	 }
	 else{
	   if(show)	   cout << "tracking not accepted" << endl;
	  grid=oldgrid;
	 }
	 
	 // prepare moving to next frame

// 	 for(auto it=grid.begin();it!=grid.end();it++){
// 	   it->refx=it->tgtx;
// 	   it->refy=it->tgty;
// 	 }


	 // 	 if(gridlet==0){
 	 if(show){
 	   Mat visframe=nextframe.clone();
 	   showgrid(visframe,grid,true,true,false,true);
 	   //appendFrameToOutputVideo(visframe,"outputvideo",bb);
	   cv::imshow("tracked grid", visframe);
	   cv::imshow("cumulative track", vistracks[framenr]);

	   cv::waitKey(0);
 	 }

	   }
	 }

	 //	 cout << "last gridlet tracked" << endl;

	 // show the final result
// 	 for(int framenr=firstreferenceframe;timedir*framenr<=timedir*lastframe;framenr+=timedir){

 
// 	   Mat nextframe=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(framenr+timedir,"frame")]);

// 	 Mat visframe=nextframe.clone();
// 	 showgrid(visframe,grid,false,true,false,true);
// 	 cv::imshow("tracked grid", visframe);


// 	 cv::imshow("cumulative track", vistracks[framenr]);
// 	 cv::waitKey(0);
// 	 }




  }




   void trackAreaWithGrid(int firstframe,int lastframe,
			  size_t nr_gridlets,
			  // int initiallbl,
			  const string &constraintmaskid,
			  //  const string &unoccludedbkgndid,
			  const string &trackedptid,
			  localBlackBoardType &bb){

     // initialise the grid by mask

     cout << "tracking frame range ["<<firstframe<<","<<lastframe<<"]"<<endl;

 int timedir;

    if(firstframe<lastframe)
      timedir=1;
    else if(firstframe>lastframe)
      timedir=-1;
    else throw string("detectOccludedArea(): support only forward and backward time flow");


    cout << "timedir="<<timedir<<endl;

    //	 map<int,Mat> vistracks;

	 Mat uniqueblobintmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(firstframe,
											"uniqueblobintmask")]);

	 Mat cmask0=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(firstframe,constraintmaskid)]);

// 	  	 cv::imshow("cmask0",cmask0);
// 	 	 cv::waitKey(0);

// (uniqueblobintmask.rows,uniqueblobintmask.cols,
// 		  CV_8UC1,cv::Scalar::all(0));

// 	 for(int x=0;x<mask.cols;x++)
// 	   for(int y=0;y<mask.rows;y++)
// 	     mask.at<uchar>(y,x)=(uniqueblobintmask.at<int>(y,x)==initiallbl)?255:0;



	 map<int,Mat> intensityimages_char;
	 map<int,Mat> motionfield; // motionfield[f] is the motion field from frame f to the next frame 
	 // in the direction of the time flow
	 map<int,Mat> vistracks;


	 for(int f=firstframe;timedir*f<=timedir*lastframe;f+=timedir){

	   Mat frif=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"intensity_float")]);

	   //cout << "trying to allocate matri of size " << cmask0.rows << "x" << cmask0.cols << endl;

	   intensityimages_char[f]=Mat(cmask0.rows,cmask0.cols,CV_8UC1,cv::Scalar::all(0));

	   for(int i=0;i<cmask0.rows;i++)
	     for(int j=0;j<cmask0.cols;j++){
	       intensityimages_char[f].at<uchar>(i,j)=frif.at<float>(i+intensity_image_zeropad_margin,j+intensity_image_zeropad_margin);

	   }

	 }

	 //	 cout << "about to estimate motion field" << endl;

	 for(int f=firstframe;timedir*f<timedir*lastframe;f+=timedir){
	     motionfield[f]=cmask0.clone();

	     //  cout << "frame " << f << endl;

	      Mat cmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,constraintmaskid)]);
	      Mat cmask_next=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f+timedir,
										      constraintmaskid)]);

	      Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f+timedir,"frame")]);

	      //    cout << "got masks" << endl;

	     estimateMotionFieldbyGrid(cmask,motionfield[f],f,f+timedir,intensityimages_char,
				       cmask_next,bb); 
	 

	     vistracks[f]=frame.clone();

   
// 	     cout << f<<endl;
// 	     cv::imshow("motion field",motionfield[f]);
// 	     cv::waitKey(0);

	 }




     //     Mat orgmask(mask.rows,mask.cols,CV_8UC1,cv::Scalar::all(0));

    //  for(int x=0;x<mask.cols;x++)
//        for(int y=0;y<mask.rows;y++)
// 	 orgmask.at<uchar>(y,x)=mask.at<uchar>(y,x);
     
	 grid_type grid,initgrid,best_grid;

     int spacing=10;

     for(unsigned int gridlet=0;gridlet<nr_gridlets;gridlet++){

//        cv::imshow("mask", mask);
//        cv::imshow("orgmask", orgmask);
//        cv::waitKey(0);  

       cout << "gridlet " << gridlet << endl;

       //       mask=orgmask.clone();

       //       init_grid(mask,spacing,initgrid);

     //     for(int i=0;i<20;i++){

       init_grid_ntuple_full(cmask0, 5, grid);

       // bool nextgridlet=false;

       Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(firstframe,"frame")]);

     // then do tracking successively for all frame pairs

     for(int framenr=firstframe;timedir*framenr<timedir*lastframe;framenr+=timedir){

       //       cout << "frame " << framenr << endl;

  //      Mat visframe=frame.clone();
//        showgrid(visframe,grid,true,false,true);
     
//        cv::imshow("initial grid", visframe);
//        cv::waitKey(0);
       //     }



       // initialise the tracking by smoothed estimate of the motion 
       // of individual points
       
       Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(framenr,"frame")]);
       
       Mat nextframe=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(framenr+timedir,"frame")]);

       Mat frif=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(framenr,"intensity_float")]);
       
       Mat nextif=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(framenr+timedir,"intensity_float")]);

       Mat uniqueblobintmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(framenr+timedir,"uniqueblobintmask")]);

       Mat cmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(framenr+timedir,constraintmaskid)]);

	 
//        showDeterministicFalseColour32bit("unique labeling",uniqueblobintmask);
//        cv::waitKey(0);



//	 initTrackByGridMotion(mask,framenr,framenr+1,constraintlbl,bb,grid,1);

	 initTrackByMotionField(grid,cmask,bb,
				motionfield[framenr],1);

       // initialise grid to its old location, hoping
       // that the target moves so little that optimisation is able 
       // to compensate for the movement

// 	 for(auto it=grid.begin();it!=grid.end();it++){
// 	   it->tgtx=it->refx;
// 	   it->tgty=it->refy;

// 	   for(size_t nind=0;nind<it->nbr.size();nind++){
// 	     int dx=it->refx-grid[it->nbr[nind]].refx;
// 	     int dy=it->refy-grid[it->nbr[nind]].refy;

// 	     it->l0[nind]=sqrt(dx*dx+dy*dy);
// 	   }
// 	 }


	 //	 cout << "track initialised" << endl;
	 
	 // grid.show();
// 	 if(framenr==lastframe){
//        visframe=frame.clone();
//        showgrid(visframe,grid,true,true,true);
     
//        cv::imshow("grid", visframe);
//        cv::waitKey(0);
	 // }

	 // int iter=0;


	 // int nextoperation=0;

	 // int templsize=13;

	 int templater=6;
	 float alpha=1.0/400;
	 float T=-0.01/log(0.5); // 2% energy increase is accepted w/ p=0.5
	 float tgtaccprob=0.2;
	 
	 float leakfactor=0.906;

	 float wsum=1/(1-leakfactor);

	 float acceptancesum=wsum*tgtaccprob;

	 best_grid=grid;
// 	 if(framenr==lastframe){
//        floatimagesc("frif", frif);
//        floatimagesc("nextif", nextif);
//        cv::waitKey(0);
// 	 }

	 float bestE=evaluateGridEnergy(grid,true,true, -1,
					frif,
					nextif,
					templater,alpha);


// 	   // randomly choose the node to update
// 	   iter++;
	   //	       cout << "about to start fitting iteration" << endl;
	   	   for(int ii=0;ii<3;ii++){
				     
	     updateGridByGreedyLocal(grid,frif,nextif,cmask,
				     templater,2*spacing,alpha,
				     bestE,best_grid);
				     
	     //  cout << "greedy local update done" << endl;
	     updateGridGlobalpos(grid,frif,
				 nextif, 
				 25,T,bestE,best_grid,
				 leakfactor,wsum,acceptancesum,
				 tgtaccprob,
				 templater,alpha,cmask);
	     
	     
	     
 	     grid=best_grid;
 	   }
	   
	 //   if(gridlet%45==0){
//  	     Mat visframe=nextframe.clone();
//  	     showgrid(visframe,grid,false,true,false,true);
// // 	     appendFrameToOutputVideo(visframe,"outputvideo",bb);	     
// 	     cv::imshow("point to update", visframe);
// // 	   }

	 //   cout << "final grid energy" << evaluateGridEnergy(grid,true,true, 
// 							     -1,
// 							     frif,
// 							     nextif,
// 							     templater,alpha) << endl;

                   if(bb.count(std::pair<int,string>(framenr+timedir,trackedptid))==0)
	     bb[std::pair<int,std::string>(framenr+timedir,trackedptid)]=vector<Point>();

	   
	   for(auto it=grid.begin();it!=grid.end();it++){
	     //	   vistracks[framenr].at<uchar>(it->tgty,it->tgtx)=255;
	     vistracks[framenr].at<Vec3b>(it->tgty,it->tgtx)=Vec3b(255,255,255);

	     vector<Point> &v=*boost::any_cast<vector<Point> >(&bb[std::pair<int,std::string>(framenr+timedir,
											      trackedptid)]);
	     v.push_back(Point(it->tgtx,it->tgty));
	   }

// 	   if(gridlet%45==1){

// 	     // cv::imshow("cumulative track", vistracks[framenr]);
// 	     appendFrameToOutputVideo(vistracks[framenr],"outputvideo",bb);
// 	   } 


	   // prepare moving to next frame


	   //	   mask=cmask.clone();
	   for(auto it=grid.begin();it!=grid.end();it++){
	     it->refx=it->tgtx;
	     it->refy=it->tgty;

	     for(size_t nind=0;nind<it->nbr.size();nind++){
	       int dx=it->refx-grid[it->nbr[nind]].tgtx;
	       int dy=it->refy-grid[it->nbr[nind]].tgty;
	       
	       it->l0[nind]=sqrt(dx*dx+dy*dy);
	     }

	   }

	 


     }
     }


   }


  int rand(int n){

    // returns evenly distributed random integer from [0,n-1]
    
    double r;
    do{
      r=::rand()/((double)RAND_MAX);
    } while (r>= 1.0);
    return (int)(r*n);
  }

  void estimateMotionFieldbyGrid(Mat &mask, Mat &motionfield,int refframe, int tgtframe,map<int,Mat> &intensityimages, Mat &constraintmask_tgt, localBlackBoardType &bb){


    // estimates the motion field from reference frame to target frame by tracking a raster of points
    // selected within a mask in the source frame

    // such tracks are discarded that either
    // a) go outside the specified mask in the target frame
    // b) differ too much from the estimated motion field
    
    // arguments

    // mask : the mask in source frame
    // motionfield: place for the result: Mat with element type Point2f
    // refframe: reference frame number
    // tgtframe: target frame number
    // intensityimages: map of uchar intensity with frame number as the key
    //                  must contain intensity images for the ref and tgt frames
    // constraintmask_tgt: mask in the target image outside which point tracks are discarded

    // bb must contain the following entries for the frames:

    // "frame"



 int spacing=8;

    vector<Point2f> pt;

    for(int x=0;x<mask.cols;x+=spacing)
      for(int y=0;y<mask.rows;y+=spacing)
	if(mask.at<uchar>(y,x)>0)
	  pt.push_back(Point2f(x,y));

        Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,"frame")]);

    Mat nextframe=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"frame")]);
    
    //    Mat uniqueblobintmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"uniqueblobintmask")]);
    
    vector<Point2f> trackedpt;
    
    vector<Point2f> pt_clean;
    vector<Point2f> trackedpt_clean;
    
    vector<uchar> status;
    vector<float> err;
    
    
//     cout << "estimating motion field" << endl;

//     cv::imshow("ref intensity", intensityimages[refframe]);
//     cv::imshow("tgt intensity", intensityimages[tgtframe]);
//     cv::waitKey(0);  



    calcOpticalFlowPyrLK(intensityimages[refframe],
			 intensityimages[tgtframe], 
			 pt, trackedpt, status, err);



    // Mat 	visframe=nextframe.clone();

   for(size_t i=0;i<pt.size();i++){

      if(status[i]>0 && constraintmask_tgt.at<uchar>(trackedpt[i].y,trackedpt[i].x)>0){
	pt_clean.push_back(pt[i]);
	trackedpt_clean.push_back(trackedpt[i]);

   // 	       cv::circle(visframe, Point(pt[i].x,pt[i].y),
//    			1, Scalar::all(255));

//    	       cv::line(visframe, Point(pt[i].x,pt[i].y),
//    			Point(trackedpt[i].x,trackedpt[i].y),
//    			cv::Scalar(0, 255, 0));
	     }
	 }
	 
	 

//       	 cv::imshow("tracked corner points", visframe);
//     	 cv::waitKey(0);

    motionfield=frame.clone();

    estimateSmoothMotionField(pt_clean,trackedpt_clean,motionfield);

	 // visualise the motion field

    float dev=5;

    GaussianBlur(motionfield, motionfield, cv::Size( 0,0 ),dev);

// Mat	visframe=nextframe.clone();
// 	for(int i=0;i<visframe.rows;i++)
// 	  for(int j=0;j<visframe.cols;j++){
// 	    visframe.at<Vec3b>(i,j)[0]=128+motionfield.at<Point2f>(i,j).x;
// 	    visframe.at<Vec3b>(i,j)[1]=128+motionfield.at<Point2f>(i,j).y;
// 	    visframe.at<Vec3b>(i,j)[2]=0;

// 	  }


//          cv::imshow("estimated motion field", visframe);
//  	cv::waitKey(0);  


	 // filter out outliers and redo the estimation

    pt=pt_clean;
    trackedpt=trackedpt_clean;
    
    pt_clean.clear();
    trackedpt_clean.clear();

    //		 visframe=frame.clone();
      

    for(size_t i=0;i<pt.size();i++){
      
      int mx=pt[i].x-trackedpt[i].x;
      int my=pt[i].y-trackedpt[i].y;

	   //	   Point2f ave=motionvec[i];
      Point2f ave=motionfield.at<Point2f>(pt[i].y,pt[i].x);

	 //   cout << "size filter point " << i << ":" ;
// 	   cout << "(mx,my)=("<<mx<<","<<my<<") ave="<<
// 	     ave.x<<","<<ave.y<<")"<<endl;
	  
      float dx=mx-ave.x;
      float dy=my-ave.y;
      
      float diffmargin=10;
      
      if(dx*dx+dy*dy<=diffmargin*diffmargin){
	pt_clean.push_back(pt[i]);
	trackedpt_clean.push_back(trackedpt[i]);
	
// 	 	       cv::circle(visframe, Point(pt[i].x,pt[i].y),
// 	 			1, Scalar::all(255));
	
// 	 	       cv::line(visframe, Point(pt[i].x,pt[i].y),
// 	 			Point(trackedpt[i].x,trackedpt[i].y),
// 	 			cv::Scalar(0, 255, 0));
      }
    }
	 


//      	 cv::imshow("tracked corner points with outliers removed", visframe);
//      	 cv::waitKey(0);	 

    estimateSmoothMotionField(pt_clean,trackedpt_clean,motionfield);

// 	visframe=nextframe.clone();
// 	for(int i=0;i<visframe.rows;i++)
// 	  for(int j=0;j<visframe.cols;j++){
// 	    visframe.at<Vec3b>(i,j)[0]=128+motionfield.at<Point2f>(i,j).x;
// 	    visframe.at<Vec3b>(i,j)[1]=128+motionfield.at<Point2f>(i,j).y;
// 	    visframe.at<Vec3b>(i,j)[2]=0;

// 	  }


//          cv::imshow("re-estimated motion field", visframe);
//  	cv::waitKey(0);  
    
//     cout << "done" << endl;


  }

  vector<Point> sp;


  void initTrackByMotionField(grid_type &grid,
			      Mat & constraintmask, 
			      localBlackBoardType& /* bb */,			
			      Mat &motionfield,int initmode){
    
	 // initialize the track according to the motion field


   //  imshow("constraint mask",constraintmask);
//     cv::waitKey(0);


	 Point2f avemotion(0,0);

 	 for(size_t i=0;i<grid.size();i++){

 	   int x=grid[i].refx;
 	   int y=grid[i].refy;
	   if(initmode==0){
	     grid[i].tgtx=x-motionfield.at<Point2f>(y,x).x;
	     grid[i].tgty=y-motionfield.at<Point2f>(y,x).y;

// 	     cout << "motion field at ("<<x<<","<<y<<")=("<<motionfield.at<Point2f>(y,x).x
// 		  <<","<<motionfield.at<Point2f>(y,x).y<<")"<<endl;

	   } else if(initmode==1){
	     avemotion.x += motionfield.at<Point2f>(y,x).x;
	     avemotion.y += motionfield.at<Point2f>(y,x).y;
	   }
 	 }
	 
	 if(initmode==1){
	   avemotion.x /= grid.size();
	   avemotion.y /= grid.size();

	   for(size_t i=0;i<grid.size();i++){
	     grid[i].tgtx=grid[i].refx-avemotion.x;
	     grid[i].tgty=grid[i].refy-avemotion.y;
	   }
	 }




	 // try to bring any points outside the constraint into the constraint


	 if(sp.empty())
	   spiralDeltas(sp,fmax(constraintmask.cols,constraintmask.rows));
	   
 	 for(size_t i=0;i<grid.size();i++){

	   Point org(grid[i].tgtx,grid[i].tgty);

	   //	   cout << "org=("<<org.x<<","<<org.y<<")" << endl;

	   for(size_t spind=0;
	       (grid[i].tgtx<0||grid[i].tgtx>=constraintmask.cols||
		 grid[i].tgty<0||grid[i].tgty>=constraintmask.rows||
		constraintmask.at<uchar>(grid[i].tgty,grid[i].tgtx)==0)&&spind<sp.size();spind++){
	       int x=org.x+sp[spind].x;
	       int y=org.y+sp[spind].y;

	       if(x>=0&&x<constraintmask.cols&&y>=0&&y<constraintmask.rows){
		 grid[i].tgtx=x;
		 grid[i].tgty=y;
	       }
	     }
	 }

// 	 for(size_t i=0;i<grid.size();i++){

// 	   Point org(grid[i].tgtx,grid[i].tgty);

// 	   cout << "final=("<<org.x<<","<<org.y<<")" << endl;
//	 }
  }
	 
  void spiralDeltas(vector<Point> &v,int maxdev){

    // produces enumeration of xoordinate deltas spiralling away
    // from the origin

    v.clear();

    Point c(0,0);

    // enumeration of dirs : S=0 E=1 N=2 W=3

    int dx[]={0,1,0,-1};
    int dy[]={1,0,-1,0};

    int dir=0;

    v.push_back(c);

    for(int steplen=1;fabs(c.x)<=maxdev&&fabs(c.y)<=maxdev;steplen++){
      
      for(int i=0;i<steplen;i++){
	c.x += dx[dir];
	c.y += dy[dir];
	v.push_back(c);
      }
      dir=(dir+1)%4;

      for(int i=0;i<steplen;i++){
	c.x += dx[dir];
	c.y += dy[dir];
	v.push_back(c);
      }
      dir=(dir+1)%4;
    }

  }


  void updateGridPointGreedyLocal(grid_type &grid,
				  int gind, Mat &frif, Mat &nextif,
				  Mat &constraintmask,int templsize, int searchr,float alpha,
				  float power, float maxdist,bool show){

	     
	     Rect rect;

	     rect.x=grid[gind].refx-templsize/2+intensity_image_zeropad_margin;
	     rect.y=grid[gind].refy-templsize/2+intensity_image_zeropad_margin;

	     rect.width=templsize;
	     rect.height=templsize;

	     Mat templ(frif,rect);

	     int matchwindowr=templsize/2+searchr;

	     rect.x=grid[gind].tgtx-matchwindowr+intensity_image_zeropad_margin;
	     rect.y=grid[gind].tgty-matchwindowr+intensity_image_zeropad_margin;

	     rect.width=2*matchwindowr+1;
	     rect.height=2*matchwindowr+1;


	     Mat tgtmat(nextif,rect);

	     Mat result;

	     matchTemplate(tgtmat,templ,result,CV_TM_SQDIFF);
	     //
 // 	     	     floatimagesc("template",templ);
//  	     	     floatimagesc("tgtmat",tgtmat);

	     if(show){
  	     	     imshow("template",templ);
  	     	     imshow("tgtmat",tgtmat);
	     }

	     // combine neighbourhood form penalty with template 
	     // matching result
		     if(show){
		     float maxcost=0;
		     
 	     for(int x=0;x<result.cols;x++)
 	       for(int y=0;y<result.rows;y++)
		 maxcost=fmax(maxcost,result.at<float>(y,x));

	     Mat viscost=result;

	     viscost.at<float>(matchwindowr-templsize/2,matchwindowr-templsize/2)
	       = maxcost;

          floatimagesc("matching cost",viscost);

	  cv::waitKey(0);
		     }

	     float maxdiff=templ.rows*templ.cols*255*255;

	     Mat geomcost=result.clone();

	     int largecost=100000;

	     for(int x=0;x<result.cols;x++)
	       for(int y=0;y<result.rows;y++){
		 geomcost.at<float>(y,x)=0;

		 int x0=grid[gind].tgtx-matchwindowr+templsize/2;
		 int y0=grid[gind].tgty-matchwindowr+templsize/2;

		 if(constraintmask.at<uchar>(y+y0,x+x0)==0){
		   geomcost.at<float>(y,x)=largecost;
		   continue;
		 }
	       

		 for(size_t ni=0; ni<grid[gind].nbr.size();ni++){
		   int nbri=grid[gind].nbr[ni];
		   
		   int dx=grid[nbri].tgtx-x-x0;
		   int dy=grid[nbri].tgty-y-y0;
		   
		   float diff=fabs(sqrt(dx*dx+dy*dy)-grid[gind].l0[ni]);
		   
		   if(maxdist>0 && diff>maxdist) diff=maxdist;

		   geomcost.at<float>(y,x) += pow(diff,power);
	       
		 }

		 
	       }
  	     
	     // 	     Mat viscost=geomcost.clone();

// 	     floatimagesc("geometric cost within constraint mask", viscost);
// 	     cv::waitKey(0);      

// 	     for(int x=0;x<result.cols;x++)
// 	       for(int y=0;y<result.rows;y++)
// 		 if(geomcost.at<float>(y,x)>=largecost)
// 		   viscost.at<float>(y,x)=0;


	     // combine costs

	     Mat ccost=geomcost.clone();

	     //	     float alpha=1.0/4000;

	     	     
	     for(int x=0;x<result.cols;x++)
	       for(int y=0;y<result.rows;y++)
		 ccost.at<float>(y,x)=alpha*geomcost.at<float>(y,x)
		   + result.at<float>(y,x)/maxdiff;	     

	     if(show){
 	     floatimagesc("combined cost", ccost);
 	     cv::waitKey(0);      
	     
	     Mat viscost=ccost.clone();

	    vector<float> costvec;

 	     for(int x=0;x<result.cols;x++)
 	       for(int y=0;y<result.rows;y++)
		 costvec.push_back(ccost.at<float>(y,x));

	     sort(costvec.begin(),costvec.end());

	     float thrcost=costvec[0.06*costvec.size()];

 	     for(int x=0;x<result.cols;x++)
 	       for(int y=0;y<result.rows;y++)
// 		 if(geomcost.at<float>(y,x)>=largecost)
		 if(ccost.at<float>(y,x)>thrcost)
 		   viscost.at<float>(y,x)=0;

	     // mark the origin
	     
	     viscost.at<float>(matchwindowr-templsize/2,matchwindowr-templsize/2)
	       = thrcost;

 	     floatimagesc("minimum parts of combined cost", viscost);
 	     cv::waitKey(0);      
	     }

	     double minval;
	     Point loc;

	     minMaxLoc(ccost,&minval,NULL,&loc);
	     
	     grid[gind].tgtx+=loc.x-matchwindowr+templsize/2;
	     grid[gind].tgty+=loc.y-matchwindowr+templsize/2;


  }


  float evaluateGridEnergy(grid_type &grid, bool include_matchenergy,
			   bool include_linkenergy, int idx, Mat &fri, Mat &nexti, int templater, float alpha){

    // E is calculated as \sum matcherrors + alpha*\sum (linklen-l0)^2

    float E=0;
    Rect rect;

      if(idx>=0){
	if(include_matchenergy){

	  rect.x=grid[idx].refx-templater+intensity_image_zeropad_margin;
	  rect.y=grid[idx].refy-templater+intensity_image_zeropad_margin;
	  rect.width=rect.height=2*templater+1;

	  Mat refmat(fri,rect);

	  rect.x=grid[idx].tgtx-templater+intensity_image_zeropad_margin;
	  rect.y=grid[idx].tgty-templater+intensity_image_zeropad_margin;

	  Mat tgtmat(nexti,rect);

	  Mat result;

	  matchTemplate(tgtmat,refmat,result,CV_TM_SQDIFF);

	  float maxdiff=rect.width*rect.height*255*255;

	  E += result.at<float>(0,0)/maxdiff;

	}

	if(include_linkenergy){
	  for(size_t ni=0;ni<grid[idx].nbr.size();ni++){
	    int dx=grid[idx].tgtx-grid[grid[idx].nbr[ni]].tgtx;
	    int dy=grid[idx].tgty-grid[grid[idx].nbr[ni]].tgty;

	    float ldiff=sqrt(dx*dx+dy*dy)-grid[idx].l0[ni];

	    E += alpha*ldiff*ldiff;
	  }
	}

      } else for(size_t i=0;i<grid.size();i++){

	if(include_matchenergy){

	  rect.x=grid[i].refx-templater+intensity_image_zeropad_margin;
	  rect.y=grid[i].refy-templater+intensity_image_zeropad_margin;
	  rect.width=rect.height=2*templater+1;

	  Mat refmat(fri,rect);

	  rect.x=grid[i].tgtx-templater+intensity_image_zeropad_margin;
	  rect.y=grid[i].tgty-templater+intensity_image_zeropad_margin;

	  Mat tgtmat(nexti,rect);

	  Mat result;

	  matchTemplate(tgtmat,refmat,result,CV_TM_SQDIFF);

	  float maxdiff=rect.width*rect.height*255*255;

	  E += result.at<float>(0,0)/maxdiff;
	}

	if(include_linkenergy){
	  for(size_t ni=0;ni<grid[i].nbr.size();ni++){
	    int dx=grid[i].tgtx-grid[grid[i].nbr[ni]].tgtx;
	    int dy=grid[i].tgty-grid[grid[i].nbr[ni]].tgty;

	    float ldiff=sqrt(dx*dx+dy*dy)-grid[i].l0[ni];

	    E += 0.5*alpha*ldiff*ldiff;
	  }
	}
      }
  
      return E;


  }

  bool gridAnnealStepGlobalTranslation(grid_type &grid, 
				       cv::Mat &fri, cv::Mat &nexti, 
				       int templater, float alpha, float T,
				       Mat &constraintmask){
    

    // randomly select translation ~N(0,dev)

    float dev=5;

    int dx=dev*randn();
    int dy=dev*randn();

    // reject the translation right away if it takes any point 
    // outside the image area or the constraint area

    for(auto it=grid.begin();it!=grid.end();it++){
      int newx=it->tgtx+dx;
      int newy=it->tgty+dy;

      if(newx<templater || newx >= constraintmask.cols-templater || newy<templater || newy >= constraintmask.rows-templater)
	return false;

      if(constraintmask.at<uchar>(newy,newx)==0)
	return false;

    }

    // create new candidate grid

    grid_type grid_new=grid;

    for(auto it=grid_new.begin();it!=grid_new.end();it++){
      it->tgtx += dx;
      it->tgty += dy;
    }

    double E_old= evaluateGridEnergy(grid,true,false, -1,fri,nexti,templater,alpha);

    double E_new= evaluateGridEnergy(grid_new,true,false, -1,fri,nexti,templater,alpha);

    //    cout << "E_old="<<E_old<<" E_new="<<E_new<<endl;

    if(E_new<E_old){
      // always accept steps that reduce the energy
      grid=grid_new;
      return true;
    }

    double p=exp((E_old-E_new)/T*E_old);

    //    cout << "uphill move with p="<<p<<" (E_old="<<E_old<<" E_new="<<E_new<<")";

    if(rand(1000)<1000*p){
      //     cout << " accepted" << endl;
      grid=grid_new;
      return true;
    } else{
      //cout << " rejected" << endl;
    }

    return false;

  }

  double randn(){

    // uses the Box-Mueller method

    // wastefully throws away half of the result

    double r1,r2;

    double pi=3.1415926536;

    r1=::rand()/((double)RAND_MAX);
    r2=::rand()/((double)RAND_MAX);

    if(r1==0) return 0;

    return sqrt(-2*log(r1))*cos(2*pi*r2);

  }


  void updateGridGlobalpos(grid_type &grid,Mat &frif, Mat &nextif, int iter, float &T, float &bestE, grid_type &best_grid,float leakfactor, float &wsum, float &acceptancesum,float tgtaccprob,int templater,float alpha,Mat &constraintmask){
 
   for(int i=0;i<iter;i++){

      float Eold= evaluateGridEnergy(grid,true,false, -1,frif,nextif,templater,alpha);
      bool accepted=gridAnnealStepGlobalTranslation(grid,frif,nextif, 
						    templater, alpha, T,constraintmask);
      float Enew= evaluateGridEnergy(grid,true,false, -1,frif,nextif,templater,alpha);

      if(Enew>Eold||accepted==false){
	acceptancesum *= leakfactor;

	if(accepted) acceptancesum += 1;

	if(acceptancesum > wsum*tgtaccprob)
	  T *= 0.92;
	else
	  T *= 1.08;
	
	//cout << "adjusted T to " << T << endl;
	//cout << "(acceptancesum="<<acceptancesum<<" wsum="<<wsum<<")"<<endl; 
	
      }
	       
      float Efull= evaluateGridEnergy(grid,true,true, -1,frif,nextif,templater,alpha);

      if(Efull<bestE){
	bestE=Efull;
	best_grid=grid;
      }
    }

  }
  
  void updateGridByGreedyLocal(grid_type & grid,Mat &frif, Mat &nextif,
			       Mat &constraintmask,
			       int templater,int searchr,float alpha,
			       float &bestE, grid_type &best_grid){

    vector<bool> needs_update(grid.size(),true);

    int needupdatecount=grid.size();

//     float oldE=evaluateGridEnergy(grid,true,true, -1,frif,nextif,templater,alpha);

//     float newE=oldE;


    do{
//       oldE=newE;

      int r;

      do{
	r=rand(grid.size());
      } while(!needs_update[r]);

      Point oldtgt(grid[r].tgtx,grid[r].tgty);

      updateGridPointGreedyLocal(grid,r,frif,nextif,constraintmask,2*templater+1,searchr,alpha);

//       cout << "moved grid point " << r<< " from (" << oldtgt.x<<","<<
// 	oldtgt.y<<") to (" <<grid[r].tgtx<<","<<grid[r].tgty<<")"<<endl;  

//       Mat visframe=nextif.clone();
//       showgrid(visframe,grid,false,true,false,false);

//       cv::imshow("updated grid",visframe);
//       cv::waitKey(0);

      //      for(int i;i<(int)grid.size();i++)



      needs_update[r]=false;
      needupdatecount--;

      


      if(oldtgt.x!=grid[r].tgtx || oldtgt.y !=grid[r].tgty)
      for(auto it=grid[r].nbr.begin(); it!=grid[r].nbr.end();it++)
	if(needs_update[*it]==false){
	  needupdatecount++;
	  needs_update[*it]=true;
	}

      float newE=evaluateGridEnergy(grid,true,true, -1,frif,nextif,templater,alpha);

      if(newE<bestE){
	bestE=newE;
	best_grid=grid;
     }
      

      //      cout << "needupdatecount="<<needupdatecount<<" newE="<<newE<<" oldE="<<oldE<<endl;
      
    } while(needupdatecount>0);
    
  }


  void placeConstraintMaskOnBlackboard(int frame,localBlackBoardType &bb,
				       const string &maskid,
				       const string &unique,
				       set<int> &constraintlbl){

    // assumes existence of intmask w/ id unique on blackboard
    // for the frame in question

    Mat umask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frame,unique)]);

    bb[std::pair<int,std::string>(frame,maskid)]=Mat(umask.rows,umask.cols,CV_8UC1,cv::Scalar::all(0));
	
    Mat cmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frame,maskid)]);


    for(int x=0;x<cmask.cols;x++)
      for(int y=0;y<cmask.rows;y++)
	cmask.at<uchar>(y,x)=(constraintlbl.count(umask.at<int>(y,x))>0)?255:0;
    
  }

  void placeFloatIntensityOnBlackboard(int frame,localBlackBoardType &bb){

    // assumes the esistence of frames (id "frame") on the blackboard
    // places float version of the intensity image on bb under key 
    // "intensity_float"

    // note that the image is extended with zero-padding
    // the image with a constant ("intensity_image_zeropad_margin") margin
    // in order to avoid problems w/ template matching near the image boundaries


    Mat fr=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frame,"frame")]);

    bb[std::pair<int,std::string>(frame,"intensity_float")]=Mat(fr.rows+2*intensity_image_zeropad_margin,
								fr.cols+2*intensity_image_zeropad_margin,cv::DataType<float>::type,cv::Scalar::all(0));
	
    Mat I=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frame,"intensity_float")]);

    for(int i=0;i<fr.rows;i++)
      for(int j=0;j<fr.cols;j++){
	I.at<float>(i+intensity_image_zeropad_margin,j+intensity_image_zeropad_margin)=
	  (fr.at<Vec3b>(i,j)[0]+
	   fr.at<Vec3b>(i,j)[1]+
	   fr.at<Vec3b>(i,j)[2])/3.0;
      }
  }

  void ensurePaddedFramesOnBlackboard(int frame,localBlackBoardType &bb){

    // assumes the esistence of frames (id "frame") on the blackboard
    // if necessary, places zero padded version of the frames on bb under key 
    // "frame_padded"

    // note that the image is extended with zero-padding
    // the image with a constant ("intensity_image_zeropad_margin") margin
    // in order to avoid problems w/ template matching near the image boundaries
    if(bb.count(std::pair<int,std::string>(frame,"frame_padded"))>0)
      return;

    Mat fr=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frame,"frame")]);

    Mat padded(fr.rows+2*intensity_image_zeropad_margin,
	       fr.cols+2*intensity_image_zeropad_margin,
	       CV_8UC3,cv::Scalar::all(0));

    for(int i=0;i<fr.rows;i++)
      for(int j=0;j<fr.cols;j++){
	padded.at<Vec3b>(i+intensity_image_zeropad_margin,j+intensity_image_zeropad_margin)=fr.at<Vec3b>(i,j);
      }
 
    bb[std::pair<int,std::string>(frame,"frame_padded")]=padded;

  }

  void ensureIntensityOnBB_uchar(int frame,localBlackBoardType &bb){

    // ensures the existence of uchar intensity image for the frame on bb

    // note: not zero-padded around the image

    if(bb.count(std::pair<int,std::string>(frame,"intensity_uchar"))>0)
      return;

    // assumes the esistence of frames (id "frame") on the blackboard
    // places uchar version of the intensity image on bb under key 
    // "intensity_uchar"

    Mat fr=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frame,"frame")]);

    bb[std::pair<int,std::string>(frame,"intensity_uchar")]=Mat(fr.rows,fr.cols,
								CV_8UC1,cv::Scalar::all(0));
	
    Mat I=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frame,"intensity_uchar")]);

    for(int i=0;i<fr.rows;i++)
      for(int j=0;j<fr.cols;j++){
	I.at<uchar>(i,j)=(fr.at<Vec3b>(i,j)[0]+
			  fr.at<Vec3b>(i,j)[1]+
			  fr.at<Vec3b>(i,j)[2])/3.0;
      }
  }




 void constructFloatMaskFromPointCloud(int firstframe,int lastframe,const string &ptcloudid,
				       const string &floatmaskid,localBlackBoardType &bb, bool show){
	 float charlen;
	 bool first=true;
	 
	 for(int f=firstframe;f<=lastframe;f++){
	   if(bb.count(std::pair<int,string>(f,ptcloudid))==0) continue;
	   Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"frame")]);
	   Mat visframe=frame.clone();
	   vector<Point> &v=*boost::any_cast<vector<Point> >(&bb[std::pair<int,std::string>(f,ptcloudid)]);

	   if(v.size()==0) continue;

	   Point minpt=v[0],maxpt=v[0];

	   for(auto it=v.begin();it!=v.end();it++){
	     visframe.at<Vec3b>(it->y,it->x)=Vec3b(255,255,255);
	     minpt.x=fmin(minpt.x,it->x);
	     minpt.y=fmin(minpt.y,it->y);
	     maxpt.x=fmax(maxpt.x,it->x);
	     maxpt.y=fmax(maxpt.y,it->y);
	   }

	   if(first){
	     charlen=sqrt(0.5*(maxpt.x-minpt.x+1)*(maxpt.y-minpt.y+1)/v.size());
	     cout << "charlen=" << charlen << endl;
	     first=false;
	   }

	   int maxmargin=10;
	   int margin=fmin(maxmargin,fmin(minpt.x,fmin(minpt.y,fmin(frame.cols-maxpt.x-1,frame.rows-maxpt.y-1))));

	   int dx=minpt.x-margin;
	   int dy=minpt.y-margin;

	   Mat m(maxpt.y-minpt.y+1+2*margin,maxpt.x-minpt.x+1+2*margin,CV_8UC1,cv::Scalar::all(0));

	   for(auto it=v.begin();it!=v.end();it++){
	     m.at<uchar>(it->y-dy,it->x-dx)=255;
	   }

	   cv::morphologyEx(m, m, cv::MORPH_CLOSE, 
	   		    cv::getStructuringElement(cv::MORPH_ELLIPSE, cv::Size(7*charlen, 7*charlen)), 
	   		    cv::Point(-1,-1), 1);

	   Mat m2;

	   erode(m, m2, Mat(),cv::Point(-1,-1), 4);


	   bb[std::pair<int,std::string>(f,floatmaskid)]=Mat(frame.rows,frame.cols,
							    cv::DataType<float>::type,cv::Scalar::all(0));

	   Mat unoccludedheadmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,floatmaskid)]);

	   float val=1;

	   for(int i=0;i<7;i++){
	     for(int x=0;x<m.cols;x++)
	       for(int y=0;y<m.rows;y++)
		 if(unoccludedheadmask.at<float>(y+dy,x+dx)==0 &&m2.at<uchar>(y,x)!=0 )
		   unoccludedheadmask.at<float>(y+dy,x+dx)=val;
	     dilate(m2, m2, Mat(),cv::Point(-1,-1), 1);
	     val *= 0.92;
	   }


	   if(show){
	   
 	   for(int x=0;x<visframe.cols;x++)
 	     for(int y=0;y<visframe.rows;y++)
 	       visframe.at<Vec3b>(y,x)[0]=255*unoccludedheadmask.at<float>(y,x);

 	    cv::imshow("unoccluded head points",visframe);
	   //cv::imshow("morphologically filtered unoccluded area mask",m);
	   //cv::imshow("smoothed unoccluded area mask",m2);
	   //   cv::imshow("morphologically smoothed unoccluded area mask",unoccludedheadmask);

	   //	   	   appendFrameToOutputVideo(visframe,"outputvideo",bb);

	   	   	   	   cv::waitKey(0);   
	   } // if show
	   

	 }

 }

  void appendFrameToOutputVideo(Mat &m, const string &outid, localBlackBoardType &bb){

    int nextframe=0;

    if(bb.count(std::pair<int,std::string>(0,outid+"_nextframe"))>0)
      nextframe=*boost::any_cast<int>(&bb[std::pair<int,std::string>(0,outid+"_nextframe")]);

    bb[std::pair<int,std::string>(nextframe,outid)]=m.clone();
    nextframe++;
    bb[std::pair<int,std::string>(0,outid+"_nextframe")]=nextframe;

  }

  void writeOutputVideo(const string &outid, const string &filename, int fps, localBlackBoardType &bb){

    cout << "writing video" << endl;

    cv::VideoWriter videorecord;


    if(bb.count(std::pair<int,std::string>(0,outid))==0) return;

    Mat fr=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(0,outid)]);

    Size orgsize=fr.size();

    videorecord.open(filename+".avi", CV_FOURCC('D','I','V','X'), fps, fr.size(), true);


    int frnumber=0;

    while(true){
    if(bb.count(std::pair<int,std::string>(frnumber,outid))==0) return;
    Mat fr=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,outid)]);

    if(fr.size() != orgsize) return;

    videorecord << fr;

    frnumber++;
    }
  }

  int findHeadBlobSize(int f, localBlackBoardType &bb){

    if(bb.count(std::pair<int,std::string>(f,"uniqueblobmask"))==0){
      Mat  skinmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"skinmask")]);
      
      Mat ulbl;

      labelConnectedComponents(skinmask,ulbl);
      bb[std::pair<int,std::string>(f,"uniqueblobmask")]=ulbl;
    }	     
    
    Mat ulbl=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"uniqueblobmask")]);

    Rect faceLocation=*boost::any_cast<cv::Rect>(&bb[std::pair<int,std::string>(f,"facerect")]);

    uchar facelbl=ulbl.at<uchar>(faceLocation.y+0.5*faceLocation.height,
				 faceLocation.x+0.5*faceLocation.width);

    int facecount=0;
    for(int x=0;x<ulbl.cols;x++)
      for(int y=0;y<ulbl.rows;y++)
	if(ulbl.at<uchar>(y,x)==facelbl)
	  facecount++;

    return facecount;

  }

  void selectSalientPointsOfMask(int f,
				 const string &maskid,
				 const string &pointsetid,
			       localBlackBoardType &bb){


    Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"frame")]);
    Mat  skinmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"skinmask")]);

    //    Rect faceLocation=*boost::any_cast<cv::Rect>(&bb[std::pair<int,std::string>(f,"facerect")]);
    
    Mat cmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,maskid)]);
    
    Mat fri(frame.rows,frame.cols,CV_8UC1,cv::Scalar::all(0));
    for(int i=0;i<frame.rows;i++)
      for(int j=0;j<frame.cols;j++){
	fri.at<uchar>(i,j)=(frame.at<Vec3b>(i,j)[0]+
			    frame.at<Vec3b>(i,j)[1]+
			    frame.at<Vec3b>(i,j)[2])/3;
      }
    
      
    Mat featuremat;

    int maxCorners=2000;
    double qualityLevel=0.001;
    double minDistance=5;

    
    Mat fmask(frame.rows,frame.cols,CV_8UC1,cv::Scalar::all(0));
    cv::goodFeaturesToTrack(fri, featuremat, maxCorners, qualityLevel, 
				   minDistance);

    for(int i=0;i<featuremat.rows;i++){
      if(cmask.at<uchar>(featuremat.at<float>(i,1),featuremat.at<float>(i,0))){
	//       cv::circle(visframe, Point(featuremat.at<float>(i,0),
	       //			  featuremat.at<float>(i,1)),
	       //	  1, Scalar::all(255));
	fmask.at<uchar>(featuremat.at<float>(i,1),featuremat.at<float>(i,0))=255;
      }
    }
    
    Mat pointmask=fmask.clone();

	   // featuremat contains now point to track

	   // augment the set of points in two phases

	   // as preparation, prepare mask f that prevents augmentation too close to existing points

	   // use closeness threshold t1

    int t1=7;

    cv::morphologyEx(fmask, fmask, cv::MORPH_DILATE, 
			    cv::getStructuringElement(cv::MORPH_ELLIPSE, cv::Size(t1,t1)), 
		     cv::Point(-1,-1), 1);
    
    // 1. select points near the borders of cmask

    // find border pixels

    Mat bmask(frame.rows,frame.cols,CV_8UC1,cv::Scalar::all(0));

    for(int x=0;x<frame.cols;x++)
      for(int y=0;y<frame.rows;y++){
	if(cmask.at<uchar>(y,x)!=0 &&
	   ((y>0&&cmask.at<uchar>(y-1,x)==0) ||
	    (x>0&&cmask.at<uchar>(y,x-1)==0) ||
	    (y<frame.rows-1&&cmask.at<uchar>(y+1,x)==0) ||
	    (x<frame.cols-1&&cmask.at<uchar>(y,x+1)==0)))
	  bmask.at<uchar>(y,x)=255;
	
      }
    // dilate the mask
    
    cv::morphologyEx(bmask, bmask, cv::MORPH_DILATE, 
		     cv::getStructuringElement(cv::MORPH_ELLIPSE, cv::Size(t1,t1)), 
		     cv::Point(-1,-1), 1);

    // then go through raster t1*2-1 pixel spacing and

    // accept raster points if both near to border and not too near to 
    // existing points (as told by mask)

    for(int x=0;x<frame.cols;x+=t1*1.2)
      for(int y=0;y<frame.rows;y+=t1*1.2){
	//	 pointmask.at<uchar>(y,x)=128;

	if(bmask.at<uchar>(y,x)>0 && fmask.at<uchar>(y,x)==0)
	  pointmask.at<uchar>(y,x)=255;
	bmask.at<uchar>(y,x)=128;
      }

    // dilate the prohibited zone -> t2

    int t2=9-t1;

	   cv::morphologyEx(fmask, fmask, cv::MORPH_DILATE, 
			    cv::getStructuringElement(cv::MORPH_ELLIPSE, cv::Size(t2,t2)), 
		      cv::Point(-1,-1), 1);


// 	   cv::imshow("forbidden mask",fmask);
// 	   cv::waitKey(0); 

	   // 2. select points in areas without points nearby:

	   // by going through raster w/ 9 pixel spacing

	   // accept points, if theu're not forbidden by the mask

	   for(int x=0;x<frame.cols;x+=9)
	     for(int y=0;y<frame.rows;y+=9)
	       if(cmask.at<uchar>(y,x) && fmask.at<uchar>(y,x)==0)
		 pointmask.at<uchar>(y,x)=255;


// 	   cv::rectangle(visframe,faceLocation,CV_RGB( 255, 255, 255 ));

// 	   for(int x=0;x<frame.cols;x++)
// 	     for(int y=0;y<frame.rows;y++)
// 	       if(pointmask.at<uchar>(y,x)>0)
// 	       cv::circle(visframe, Point(x,y),
// 			  1, Scalar::all(255));

	   bb[std::pair<int,std::string>(f,pointsetid)]=pointmask;

// 	   cv::imshow("selected pivot face points",visframe);
// 	   cv::waitKey(0); 


  
  }

   void matchingEnergyVsSet(int f, set<int> &pivotheadframes,
			    const string &pointsetid,localBlackBoardType &bb){

     vector<float> totalEnergy(1,0);

     for(auto it=pivotheadframes.begin();it!=pivotheadframes.end();it++){
       vector<float> E=matchingEnergyVsFrame(f,*it,pointsetid,bb);

       for(size_t d=0;d<E.size();d++)
	 totalEnergy[d]=E[d];
     }
       
   }

  vector<float> matchingEnergyVsFrame(int f, int reff,
			     const string &pointsetid,localBlackBoardType &bb){

    vector<float> ret(1,0);

    Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"frame")]);
    Mat refframe=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(reff,"frame")]);
    Mat cmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"headconstraintmask")]);
    Mat refcmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(reff,"headconstraintmask")]);


    Mat visframe=refframe.clone();

    Mat refpoints=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(reff,pointsetid)]);

    for(int x=0;x<frame.cols;x++)
      for(int y=0;y<frame.rows;y++)
	if(refpoints.at<uchar>(y,x)>0)
	  cv::circle(visframe, Point(x,y),
		     1, Scalar::all(255));

    cout << "comparing frame " <<f<<" with reference frame " << reff << endl;

    // start by estimating the smoothed motion field 
    // from reference frame to f

    ensureIntensityOnBB_uchar(f,bb);
    ensureIntensityOnBB_uchar(reff,bb);

    cout << "intensity images ensured" << endl;

    map<int,Mat> intensityimages;

    Mat motionfield;

    intensityimages[f]=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"intensity_uchar")]);
    intensityimages[reff]=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(reff,"intensity_uchar")]);

    cout << "intensity images put in map " << endl;

    

    //estimateMotionFieldbyGrid(cmask,motionfield,f,reff,intensityimages,refcmask,bb); 

    // visualise the motion field

    //Mat mframe=frame.clone();

   //  for(int x=0;x<mframe.cols;x++)
//       for(int y=0;y<mframe.rows;y++)
// 	mframe.at<Vec3b>(y,x)=Vec3b(10*motionfield.at<Point2f>(y,x).x+128,
// 				    10*motionfield.at<Point2f>(y,x).y+128,
// 				    0);
	

 
    //    // select number of gridlets among the pivot points and 
    // evaluate the energies of their best matches


    


    //initTrackByMotionField(grid,cmask,bb,
    //			   motionfield[framenr],1);


    cv::imshow("frame",frame);
    cv::imshow("reference frame",visframe);
    //    cv::imshow("motion field",mframe);  

    cv::waitKey(0); 



    return ret;
  }

  void correspondenceFieldByConstellationMatching(int refframe, int tgtframe,
						  const string &pointsetid,
						  const string &constraintmaskid,
						  Mat &displacementfield,
						  Mat &errorfield,
						  localBlackBoardType &bb){

    

    Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,"frame")]);

    Mat tgtfmat=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"frame")]);

    Mat refcmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,constraintmaskid)]);
    Mat tgtcmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,constraintmaskid)]);

    Mat dummymask=tgtcmask.clone();

    for(int y=0;y<dummymask.rows;y++)
      for(int x=0;x<dummymask.cols;x++)
	dummymask.at<uchar>(y,x)=255;

    Mat refif=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,"intensity_float")]);
       
       Mat tgtif=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"intensity_float")]);


       displacementfield=Mat(refcmask.rows,refcmask.cols,cv::DataType<Point2f>::type,cv::Scalar::all(0));
       errorfield=Mat(refcmask.rows,refcmask.cols,cv::DataType<float>::type,cv::Scalar::all(0));
       
       Mat countmat(refcmask.rows,refcmask.cols,CV_8UC1,cv::Scalar::all(0));

    // fetch the set of salient points in refframe from the bb


    Mat refpoints=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,pointsetid)]);

    // estimate and smooth the motion field between the frames
    // by matching single points
    
    ensureIntensityOnBB_uchar(refframe,bb);
    ensureIntensityOnBB_uchar(tgtframe,bb);

    map<int,Mat> intensityimages;

    Mat motionfield;

    intensityimages[refframe]=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,"intensity_uchar")]);
    intensityimages[tgtframe]=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"intensity_uchar")]);


    estimateMotionFieldbyGrid(refcmask,motionfield,refframe,tgtframe,
			      intensityimages,tgtcmask,bb); 

    // visualise the motion field

//     Mat mframe=frame.clone();

//      for(int x=0;x<mframe.cols;x++)
//        for(int y=0;y<mframe.rows;y++)
//  	mframe.at<Vec3b>(y,x)=Vec3b(10*motionfield.at<Point2f>(y,x).x+128,
//  				    10*motionfield.at<Point2f>(y,x).y+128,
//  				    0);


//       cv::imshow("motion field", mframe);
//       cv::waitKey(0);
	


    // then go through the list of salient points and track 
    // a set of points constellations from refframe to target frame

    vector<Point> pointvec;

    for(int x=0;x<refpoints.cols;x++)
      for(int y=0;y<refpoints.rows;y++){
	if(refpoints.at<uchar>(y,x)>0)
	   pointvec.push_back(Point(x,y));
      }

    int npoints=pointvec.size();

    cout << npoints << " salient points" << endl;

    int npoints_in_constellation=4;

    for(int p1=0;p1<npoints;p1++){
      cout << "seed point " << p1 << endl;
      vector<int> pi;
      pi.push_back(p1);
      for(int i=1;i<npoints_in_constellation;i++){
	int r=rand(npoints);
	bool ok=true;
	for(int j=0;j<i;j++){
	  if(r==pi[j]){
	    ok=false;
	    continue;
	  }
	}

	int maxsqrdist=20*20;

	int dx=pointvec[p1].x-pointvec[r].x;
	int dy=pointvec[p1].y-pointvec[r].y;

	if(dx*dx+dy*dy>maxsqrdist) ok=false;

	if(!ok){
	  i--;
	  continue;
	}


	//	cout << "accepted point " << r << endl;
	pi.push_back(r);
      }

      grid_type grid(pi.size());
      for(int i=0;i<(int)pi.size();i++){
	grid[i].idx=i;
	grid[i].refx=pointvec[pi[i]].x;
	grid[i].refy=pointvec[pi[i]].y;

      // initialise track by the motion field

	grid[i].tgtx=grid[i].refx -
	  motionfield.at<Point2f>(grid[i].refy,grid[i].refx).x;
	grid[i].tgty=grid[i].refy -
	  motionfield.at<Point2f>(grid[i].refy,grid[i].refx).y;
      }
      set_nbr_fullconnectivity(grid);
      set_l0(grid);

//     for(int i=0;i<(int)pi.size();i++)
//  	grid[i].show();



      // fine tune with greedy local optimisation

      // visualise


//       Mat visframe=frame.clone();
//       Mat visframe2=tgtfmat.clone();


//       showgrid(visframe, grid,
// 	       true, false,true,false);

//       showgrid(visframe2, grid,
// 	       true,true,false,true);
      
//       cv::imshow("grid in reference frame", visframe);
// //       cv::imshow("track", visframe2);
//       cv::waitKey(0);

      // somewhat heuristic parameters for the tracker

      int templater=6;
      float alpha=1.0/4000;
      int spacing=10;
      
      float bestE;
      
      grid_type best_grid;
      
	 

      updateGridByGreedyLocal(grid,refif,tgtif,dummymask,
			      templater,2*spacing,alpha,
			      bestE,best_grid);


    //   visframe2=tgtfmat.clone();
//       showgrid(visframe2, grid,
// 	       true,true,false,true);
//       cv::imshow("track", visframe2);
//       cv::waitKey(0);

	 float finalE=evaluateGridEnergy(grid,true,true, -1,
					 refif,tgtif,
					 templater,alpha);

	 //	 cout << "finalE="<<finalE<<endl;

	 for(int i=0;i<(int)grid.size();i++){
	   errorfield.at<float>(grid[i].tgty,grid[i].tgtx) += finalE;
	   displacementfield.at<Point2f>(grid[i].tgty,grid[i].tgtx).x += 
	     grid[i].tgtx - grid[i].refx;
	   displacementfield.at<Point2f>(grid[i].tgty,grid[i].tgtx).y += 
	     grid[i].tgty - grid[i].refy;

	   countmat.at<uchar>(grid[i].tgty,grid[i].tgtx)++;
	 }

    }

    // normalise errorfield by count

    float maxerr=0;

    for(int x=0;x<errorfield.cols;x++)
      for(int y=0;y<errorfield.rows;y++)
	if(countmat.at<uchar>(y,x)>0){
	  errorfield.at<float>(y,x) /= countmat.at<uchar>(y,x);
	  displacementfield.at<Point2f>(y,x).y /= countmat.at<uchar>(y,x);
	  displacementfield.at<Point2f>(y,x).x /= countmat.at<uchar>(y,x);
	  maxerr=fmax(maxerr,errorfield.at<float>(y,x));
	}


    Mat mframe(refcmask.rows,refcmask.cols,CV_8UC3,cv::Scalar::all(0));

     for(int x=0;x<mframe.cols;x++)
       for(int y=0;y<mframe.rows;y++)
	if(countmat.at<uchar>(y,x)>0)
	  mframe.at<Vec3b>(y,x)=Vec3b(10*displacementfield.at<Point2f>(y,x).x+128,
				      10*displacementfield.at<Point2f>(y,x).y+128,
				      0);


     // calculate and visualise the displacement field discontinuity

     Mat dframe(refcmask.rows,refcmask.cols,cv::DataType<float>::type,cv::Scalar::all(0));

     int r=20;

     float maxdisc=0;

     for(int x=0;x<displacementfield.cols;x++)
       for(int y=0;y<displacementfield.rows;y++){

	if(countmat.at<uchar>(y,x)==0) continue;

	int minx,maxx,miny,maxy;

	minx=fmax(0,x-r);
	maxx=fmin(displacementfield.cols-1,x+r);
	
	miny=fmax(0,y-r);
	maxy=fmin(displacementfield.rows-1,y+r);
	
	int ctr=0;
	float e=0;
	for(int x2=minx;x2<=maxx;x2++)
	  for(int y2=miny;y2<=maxy;y2++){
	    if(countmat.at<uchar>(y2,x2)==0) continue;
	    ctr++;

	    float d=displacementfield.at<Point2f>(y,x).x-displacementfield.at<Point2f>(y2,x2).x;

// 	    if(x==258 && y==215){
// 	      cout << "diff1 betw ("<<x<<","<<y<<") and ("<<x2<<","<<y2<<") is "
// 		   << d << endl;
// 	    }

	    e += d*d;
	    d=displacementfield.at<Point2f>(y,x).y-displacementfield.at<Point2f>(y2,x2).y;
	    e += d*d;

// 	    if(x==258 && y==215){
// 	      cout << "diff2 betw ("<<x<<","<<y<<") and ("<<x2<<","<<y2<<") is "
// 		   << d << endl;
// 	    }

	  }

	e /= ctr;

// 	cout << "(x,y) = ("<<x<<","<<y<<") e="<<e<<" ctr="<<ctr<<endl; 

	dframe.at<float>(y,x)=e;
	maxdisc=fmax(maxdisc,e);
       }

     // now form the modulated error

     Mat modeframe(refcmask.rows,refcmask.cols,cv::DataType<float>::type,cv::Scalar::all(0));

     float maxmode=0;

     for(int x=0;x<modeframe.cols;x++)
       for(int y=0;y<modeframe.rows;y++)
	 if(countmat.at<uchar>(y,x)){
	   modeframe.at<float>(y,x)=
	     dframe.at<float>(y,x) * errorfield.at<float>(y,x);
	   maxmode=fmax(maxmode,modeframe.at<float>(y,x));
	 }

    cout << "maxerr="<<maxerr<<endl;
    cout << "maxdisc="<<maxdisc<<endl;
    cout << "maxmode="<<maxmode<<endl;
    cv::imshow("frame", tgtfmat);
    cv::imshow("estimated motion field", mframe);
    floatimagesc("energy field", errorfield);
    floatimagesc("displacement discontinuity field", dframe);
    floatimagesc("discontinuity modulated error field", modeframe);
    cv::waitKey(0);


  }

  void correspondenceFieldByTemplateMatching(int refframe, int tgtframe,
					     int windowr, int spacing,
					     int maxdisplacement, Point minpt, 
					     Point maxpt,
					     Mat &displacementfield,
					     Mat &errorfield,
					     localBlackBoardType &bb){

    ensureIntensityOnBB_uchar(refframe,bb);    
    ensureIntensityOnBB_uchar(tgtframe,bb);


    Mat refi=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,"intensity_uchar")]);
    Mat tgti=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"intensity_uchar")]);

    displacementfield=Mat(refi.rows,refi.cols,cv::DataType<Point2f>::type,cv::Scalar::all(0));
    errorfield=Mat(refi.rows,refi.cols,cv::DataType<float>::type,cv::Scalar::all(0));

    for(int refx=fmax(windowr,minpt.x);refx<fmin(refi.cols-windowr,maxpt.x+1);refx += spacing)
      for(int refy=fmax(windowr,minpt.y);refy<fmin(refi.rows-windowr,maxpt.y+1);refy += spacing){
	 Rect rect;

	 rect.x=refx-windowr;
	 rect.y=refy-windowr;

	 rect.width=2*windowr+1;
	 rect.height=2*windowr+1;
	 
// 	 cout << "template coordinates " << rect.x << " " << rect.y << " " <<
// 	   rect.width << " " << rect.height << endl;
 
	 Mat templ(refi,rect);

	 rect.x=refx-windowr-maxdisplacement;
	 rect.y=refy-windowr-maxdisplacement;

	 rect.width=2*(windowr+maxdisplacement)+1;
	 rect.height=2*(windowr+maxdisplacement)+1;

	 int xcut=0;
	 int ycut=0;

	 if(rect.x<0){
	   xcut=-rect.x;
	   rect.x=0;
	   rect.width -= xcut;
	 }

	 if(rect.y<0){
	   ycut=-rect.y;
	   rect.y=0;
	   rect.height -= ycut;
	 }

	 if(rect.br().x>=refi.cols){
	   rect.width=refi.cols-rect.x;
	 }

	 if(rect.br().y>=refi.rows){
	   rect.height=refi.rows-rect.y;
	 }

// 	 cout << "target coordinates " << rect.x << " " << rect.y << " " <<
// 	   rect.width << " " << rect.height << endl;

// 	 cout << "frame size " << refi.cols <<"x"<<refi.rows<<endl;

	 Mat tgtmat(tgti,rect);

	 Mat result;

	 matchTemplate(tgtmat,templ,result,CV_TM_SQDIFF);
	 //
	 
	 double minval;
	 Point loc;

	 minMaxLoc(result,&minval,NULL,&loc);
	     
	 for(int x=refx-spacing/2;x<refx+spacing-spacing/2;x++)
	   for(int y=refy-spacing/2;y<refy+spacing-spacing/2;y++){
	     errorfield.at<float>(y,x)=minval;
	     displacementfield.at<Point2f>(y,x)=Point2f(loc.x-maxdisplacement+xcut,
							  loc.y-maxdisplacement+ycut);
;
	   }


      }

  }

  void calculateDiscontinuityEnergy(Mat &dfield, Mat &cmask,int r, Mat &res){
    
    res=Mat(dfield.rows,dfield.cols,cv::DataType<float>::type,cv::Scalar::all(0));

//     for(int x=0;x<dfield.cols;x++)
//       for(int y=0;y<dfield.rows;y++){
// 	cout << "dfield("<<x<<","<<y<<")=("<<dfield.at<Point2f>(y,x).x<<","<<
// 	  dfield.at<Point2f>(y,x).y<<")"<<endl;
//       }

    for(int x=0;x<dfield.cols;x++)
      for(int y=0;y<dfield.rows;y++){

	if(cmask.at<uchar>(y,x)==0) continue;

	int minx,maxx,miny,maxy;

	minx=fmax(0,x-r);
	maxx=fmin(dfield.cols-1,x+r);
	
	miny=fmax(0,y-r);
	maxy=fmin(dfield.rows-1,y+r);
	
	int ctr=0;
	float e=0;
	for(int x2=minx;x2<=maxx;x2++)
	  for(int y2=miny;y2<=maxy;y2++){
	    if(cmask.at<uchar>(y2,x2)==0) continue;
	    ctr++;

	    float d=dfield.at<Point2f>(y,x).x-dfield.at<Point2f>(y2,x2).x;

// 	    if(x==258 && y==215){
// 	      cout << "diff1 betw ("<<x<<","<<y<<") and ("<<x2<<","<<y2<<") is "
// 		   << d << endl;
// 	    }

	    e += d*d;
	    d=dfield.at<Point2f>(y,x).y-dfield.at<Point2f>(y2,x2).y;
	    e += d*d;

// 	    if(x==258 && y==215){
// 	      cout << "diff2 betw ("<<x<<","<<y<<") and ("<<x2<<","<<y2<<") is "
// 		   << d << endl;
// 	    }

	  }

	e /= ctr;

// 	cout << "(x,y) = ("<<x<<","<<y<<") e="<<e<<" ctr="<<ctr<<endl; 

	res.at<float>(y,x)=e;
      }

  }

  void recordHeadAreaCorrespondenceByConstellations(int first, int last, const string &maskid,
				 const string &displacementfieldid,
				 const string &errorfieldid,
				 const string &discontinuityfieldid,
				 const string &moderrorfieldid,
				 const string &interrorid,
				 const string &intdiscontinuityid,
				 const string &intmoderrorid,
				 localBlackBoardType &bb){
    // follows area specified by maskid on bb
    // in frame range [first,last)
    // with template matching

    // records various statistics of matching/mismatch 
    // on bb with the given keys

    // any of the keys may be left empty, in which case
    // the corresponding statistic is not evaluated (if avoidable)

    
     for(int f=first+1;f<last;f++){
       Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"frame")]);
       Mat cmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,maskid)]);

       Mat prevframe=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f-1,"frame")]);

       Mat dfield,efield;

//        int windowr=10;
//        int spacing=4;
//        int maxdisplacement=40; 

       correspondenceFieldByConstellationMatching(f-1,f,"pivotheadpoints",
						  "headconstraintmask",
						  dfield,efield,bb);

       for(int x=0;x<cmask.cols;x++)
	 for(int y=0;y<cmask.rows;y++){
	   if(cmask.at<uchar>(y,x)==0){
	     efield.at<float>(y,x)=0;
	     dfield.at<Point2f>(y,x)=Point2f(0,0);
	   }
	 }

       if(displacementfieldid!="")
	 bb[std::pair<int,std::string>(f,displacementfieldid)]=dfield;
       if(errorfieldid!="")
	 bb[std::pair<int,std::string>(f,errorfieldid)]=efield;
       

//         cv::imshow("frame",frame);
//         floatimagesc("error field",efield);
// 	cv::waitKey(0);
     
       //       Mat mframe=frame.clone();

       bool calculatediscontinuity=false;
       bool calculatemoderror=false;

       if(discontinuityfieldid!="" || moderrorfieldid != "" || intdiscontinuityid != "" || intmoderrorid != "")
	 calculatediscontinuity=true;

       if(moderrorfieldid != "" || intmoderrorid != "")
	 calculatemoderror=true;

       Mat res;
       int r=20;
       
       if(calculatediscontinuity){
	 calculateDiscontinuityEnergy(dfield,cmask,r,res);

//         floatimagesc("displacement discontinuity",res);
// 	cv::waitKey(0);

	 if(discontinuityfieldid!="")
	   bb[std::pair<int,std::string>(f,discontinuityfieldid)]=res;






       }

       Mat mode;

       if(calculatemoderror){

	 mode=efield.clone();
	 
	 for(int x=0;x<mode.cols;x++)
	   for(int y=0;y<mode.rows;y++)
	     mode.at<float>(y,x) *= res.at<float>(y,x);

//         floatimagesc("error modulated by displacement discontinuity",mode);
// 	cv::waitKey(0);

	 if(moderrorfieldid!="")
	   bb[std::pair<int,std::string>(f,moderrorfieldid)]=mode;
       }

       float eint=0,emodint=0,dint=0;

       bool inteflag=interrorid!="";
       bool intdflag=intdiscontinuityid!="";
       bool intmeflag=intmoderrorid!="";

       int ctr=0;

       for(int x=0;x<efield.cols;x++)
	 for(int y=0;y<efield.rows;y++){
	   if(cmask.at<uchar>(y,x)){
	     ctr++;
	     if(inteflag)
	       eint += efield.at<float>(y,x);
	     if(intmeflag)
	       emodint += mode.at<float>(y,x);
	     if(intdflag)
	       dint += res.at<float>(y,x);
	   }
	 }

       if(ctr){
	 eint /= ctr;
	 emodint /= ctr;
	 dint /= ctr;
       }

       if(interrorid!="")
	 bb[std::pair<int,std::string>(f,interrorid)]=eint;

       if(intmoderrorid!="")
	 bb[std::pair<int,std::string>(f,intmoderrorid)]=emodint;

       if(intdiscontinuityid!="")
	 bb[std::pair<int,std::string>(f,intdiscontinuityid)]=dint;

//        cv::imshow("frame",frame);
//        cv::imshow("prevframe",prevframe);
//        cv::imshow("displacement field",mframe);  
//        floatimagesc("error field",efield);
//        floatimagesc("displacement discontinuity",res);
//        floatimagesc("error modulated by displacement discontinuity",mode);

//        cout << "maxerr="<<maxerr<<endl;
//        cout << "maxdiscerr="<<maxdiscerr<<endl;

//        cout << "integrated errors: eint="<<eint<<" mod. eint="<<emodint << 
// 	 " dint="<<dint<<endl;

       cout << "correspondence recorded for frame " << f << endl;

     }
     cout << "all correspondences recorded" << endl;
  }


  void recordHeadAreaCorrespondence(int first, int last, const string &maskid,
				 const string &displacementfieldid,
				 const string &errorfieldid,
				 const string &discontinuityfieldid,
				 const string &moderrorfieldid,
				 const string &interrorid,
				 const string &intdiscontinuityid,
				 const string &intmoderrorid,
				 localBlackBoardType &bb){
    // follows area specified by maskid on bb
    // in frame range [first,last)
    // with template matching

    // records various statistics of matching/mismatch 
    // on bb with the given keys

    // any of the keys may be left empty, in which case
    // the corresponding statistic is not evaluated (if avoidable)

    
     for(int f=first+1;f<last;f++){
       Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"frame")]);
       Mat cmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,maskid)]);

       Mat prevframe=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f-1,"frame")]);

       Mat dfield,efield;

       int windowr=10;
       int spacing=4;
       int maxdisplacement=40; 

       Point minpt(frame.cols,frame.rows),maxpt(0,0);

       for(int x=0;x<minpt.x;x++)
	 for(int y=0;y<frame.rows;y++)
	   if(cmask.at<uchar>(y,x)){
	     minpt.x=x;
	     break;
	   }

       for(int x=frame.cols-1;x>maxpt.x;x--)
	 for(int y=0;y<frame.rows;y++)
	   if(cmask.at<uchar>(y,x)){
	     maxpt.x=x;
	     break;
	   }

       for(int y=0;y<minpt.y;y++)
	 for(int x=0;x<frame.cols;x++)
	   if(cmask.at<uchar>(y,x)){
	     minpt.y=y;
	     break;
	   }

       for(int y=frame.rows-1;y>maxpt.y;y--)
	 for(int x=0;x<frame.cols;x++)
	   if(cmask.at<uchar>(y,x)){
	     maxpt.y=y;
	     break;
	   }

       minpt.x -= windowr;
       minpt.y -= windowr;

       maxpt.x += windowr;
       maxpt.y += windowr;


       correspondenceFieldByTemplateMatching(f-1,f,windowr,spacing,
						  maxdisplacement, minpt, 
						  maxpt,
						  dfield, efield, bb);
       for(int x=0;x<cmask.cols;x++)
	 for(int y=0;y<cmask.rows;y++){
	   if(cmask.at<uchar>(y,x)==0){
	     efield.at<float>(y,x)=0;
	     dfield.at<Point2f>(y,x)=Point2f(0,0);
	   }
	 }

       if(displacementfieldid!="")
	 bb[std::pair<int,std::string>(f,displacementfieldid)]=dfield;
       if(errorfieldid!="")
	 bb[std::pair<int,std::string>(f,errorfieldid)]=efield;
       

//         cv::imshow("frame",frame);
//         floatimagesc("error field",efield);
// 	cv::waitKey(0);
     
       //       Mat mframe=frame.clone();

       bool calculatediscontinuity=false;
       bool calculatemoderror=false;

       if(discontinuityfieldid!="" || moderrorfieldid != "" || intdiscontinuityid != "" || intmoderrorid != "")
	 calculatediscontinuity=true;

       if(moderrorfieldid != "" || intmoderrorid != "")
	 calculatemoderror=true;

       Mat res;
       int r=20;
       
       if(calculatediscontinuity){
	 calculateDiscontinuityEnergy(dfield,cmask,r,res);

//         floatimagesc("displacement discontinuity",res);
// 	cv::waitKey(0);

	 if(discontinuityfieldid!="")
	   bb[std::pair<int,std::string>(f,discontinuityfieldid)]=res;






       }

       Mat mode;

       if(calculatemoderror){

	 mode=efield.clone();
	 
	 for(int x=0;x<mode.cols;x++)
	   for(int y=0;y<mode.rows;y++)
	     mode.at<float>(y,x) *= res.at<float>(y,x);

//         floatimagesc("error modulated by displacement discontinuity",mode);
// 	cv::waitKey(0);

	 if(moderrorfieldid!="")
	   bb[std::pair<int,std::string>(f,moderrorfieldid)]=mode;
       }

       float eint=0,emodint=0,dint=0;

       bool inteflag=interrorid!="";
       bool intdflag=intdiscontinuityid!="";
       bool intmeflag=intmoderrorid!="";

       int ctr=0;

       for(int x=0;x<efield.cols;x++)
	 for(int y=0;y<efield.rows;y++){
	   if(cmask.at<uchar>(y,x)){
	     ctr++;
	     if(inteflag)
	       eint += efield.at<float>(y,x);
	     if(intmeflag)
	       emodint += mode.at<float>(y,x);
	     if(intdflag)
	       dint += res.at<float>(y,x);
	   }
	 }

       if(ctr){
	 eint /= ctr;
	 emodint /= ctr;
	 dint /= ctr;
       }

       if(interrorid!="")
	 bb[std::pair<int,std::string>(f,interrorid)]=eint;

       if(intmoderrorid!="")
	 bb[std::pair<int,std::string>(f,intmoderrorid)]=emodint;

       if(intdiscontinuityid!="")
	 bb[std::pair<int,std::string>(f,intdiscontinuityid)]=dint;

//        cv::imshow("frame",frame);
//        cv::imshow("prevframe",prevframe);
//        cv::imshow("displacement field",mframe);  
//        floatimagesc("error field",efield);
//        floatimagesc("displacement discontinuity",res);
//        floatimagesc("error modulated by displacement discontinuity",mode);

//        cout << "maxerr="<<maxerr<<endl;
//        cout << "maxdiscerr="<<maxdiscerr<<endl;

//        cout << "integrated errors: eint="<<eint<<" mod. eint="<<emodint << 
// 	 " dint="<<dint<<endl;

       cout << "correspondence recorded for frame " << f << endl;

     }
     cout << "all correspondences recorded" << endl;
  }


  void set_nbr_fullconnectivity(grid_type &g){
    for(int i=0;i<(int)g.size();i++){
      g[i].nbr.clear();
      for(int j=0;j<(int)g.size();j++){
	if(j==i) continue;
	g[i].nbr.push_back(j);
      }
    }
  }


  void set_nbr_randomconnectivity(grid_type &g,int nbrcount){
    for(int i=0;i<(int)g.size();i++){
      g[i].nbr.clear();
      int nbr_assigned=0;
      for(int j=0;nbr_assigned<nbrcount && j<(int)g.size();j++){
	if(j==i) continue;
	int toselect=nbrcount-nbr_assigned;
	int left=g.size()-j;
	
	if(rand(left)<toselect){
	  g[i].nbr.push_back(j);
	  nbr_assigned++;
	}
      }
    }
  }


  void set_l0(grid_type &g){
    for(auto it=g.begin();it<g.end();it++){
      it->l0.clear();
      for(int ni=0;ni<(int)it->nbr.size();ni++){
	int dx=it->refx-g[it->nbr[ni]].refx;
	int dy=it->refy-g[it->nbr[ni]].refy;
	it->l0.push_back(sqrt(dx*dx+dy*dy));
      }
    }
  }

  void trackNonfaceArea(int first,int last,
			const string &constraintmaskid,
			const string &nonfacemaskid,
			const string &salientpointid,
			const string &motionfieldid,
			const string &trackedpointid,
			localBlackBoardType &bb){

    //vector<vector<Point> > *trackedgridpoints=new vector<vector<Point> >;
    vector<Point> *trackedgridpoints=new vector<Point>;

    // these variables don't seem to be used anywhere? //Matti
    // int points_in_gridlets=2;
    // int maxdist=20;



    for(int f=first;f<last;f++){

      Mat nonfacemask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,nonfacemaskid)]);
      Mat nonfacemask_next=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f+1,nonfacemaskid)]);

      Mat cmask_next=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f+1,constraintmaskid)]);

      Mat headcmask_next=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f+1,"headconstraintmask")]);

      Mat skinmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"skinmask")]);
      // track the points from f to f+1

      Mat reff=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"frame")]);
      Mat tgtf=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f+1,"frame")]);
      
      vector<Point> res;

      // trackedgridpoints already contains the point n-tuples that were tracked outside
      // the nonfacemask

      // append the salient points of non-face area to the mix

      //      enumerateNtuplesAmongPoints(f, salientpointid,points_in_gridlets,maxdist,
      //				  *trackedgridpoints,bb);

      Mat pointmat=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,salientpointid)]);

      for(int x=0;x<pointmat.cols;x++)
	for(int y=0;y<pointmat.rows;y++)
	  if(pointmat.at<uchar>(y,x))
	    trackedgridpoints->push_back(Point(x,y));

      cout << "tracking " << trackedgridpoints->size() << " gridlets from frame " << f << " to frame " << f+1 << endl;

      Mat visframe=reff.clone();

      for(int x=0;x<visframe.cols;x++)
	for(int y=0;y<visframe.rows;y++)
	  if(nonfacemask.at<uchar>(y,x)==0)
	    visframe.at<Vec3b>(y,x)[0]=0;


      for(auto git=trackedgridpoints->begin(); git != trackedgridpoints->end();git++)
	//	for(auto it=git->begin();it!=git->end();it++)
	  visframe.at<Vec3b>(git->y,git->x)=Vec3b(255,255,255);

      imshow("points to track",visframe);
      imshow("skin mask",skinmask);
      cv::waitKey(0);

      //      vector<vector<Point> > *retainedgridpoints=new vector<vector<Point> >;
      vector<Point> *retainedgridpoints=new vector<Point>;

      int pointctr=0;

      for(auto git=trackedgridpoints->begin();git!=trackedgridpoints->end();git++){
	//	float finalE=trackGridletBetweenFrames(*git,f,f+1,constraintmaskid,motionfieldid,bb);

       	float finalE=trackSinglePointBetweenFrames(*git,res,f,f+1,constraintmaskid,motionfieldid,bb);
	// the coordinates in *git get replaced by the new ones in the next frame

	pointctr++;

	cout << "point #"<<pointctr<<endl;

	float thr=1;

	if(finalE>=0 && finalE<thr){
	  // output the accepted track
	  if(bb.count(std::pair<int,string>(f+1,trackedpointid))==0)
	    bb[std::pair<int,std::string>(f+1,trackedpointid)]=vector<Point>();

	  bool outside=false; // set true if any of the tracked points drifts outside the
	                      // nonface mask 

	  //	  for(auto it=git->begin();it!=git->end();it++){

	    vector<Point> &v=*boost::any_cast<vector<Point> >(&bb[std::pair<int,std::string>(f+1,trackedpointid)]);
	    
	    v.push_back(*git);

	    if(nonfacemask_next.at<uchar>(git->y,git->x)==0)
	      outside=true;
	    //}

	  if(outside)
	    retainedgridpoints->push_back(*git);
	}

      }

    //   delete trackedgridpoints;

      trackedgridpoints=new vector<Point>;
    
      Mat pointmask(reff.rows,reff.cols,CV_8UC1,Scalar::all(0));

      for(auto it=trackedgridpoints->begin();it!=trackedgridpoints->end();it++)
	if(nonfacemask_next.at<uchar>(it->y,it->x)==0)
	  pointmask.at<uchar>(it->y,it->x)=255;

      for(auto it=res.begin();it!=res.end();it++)
	if(nonfacemask_next.at<uchar>(it->y,it->x)==0)
	  pointmask.at<uchar>(it->y,it->x)=255;

      int pointcount=0;

      int tgtpointcount=500;

      for(int x=0;x<pointmask.cols;x++)
	for(int y=0;y<pointmask.rows;y++)
	  if(pointmask.at<uchar>(y,x))
	    pointcount++;

      float p;
      
      if(pointcount>tgtpointcount){
	p=tgtpointcount;
        p /= pointcount;
      }

      cout << "p="<<p<<endl;


      delete retainedgridpoints;
      retainedgridpoints=NULL;

      for(int x=0;x<pointmask.cols;x++)
	for(int y=0;y<pointmask.rows;y++)
	  if(pointmask.at<uchar>(y,x)&& (p==1 || rand(1000)<1000*p))
	    trackedgridpoints->push_back(Point(x,y));
      
    
      // visualise the tracking results
	
      visframe=tgtf.clone();

      for(int x=0;x<visframe.cols;x++)
	for(int y=0;y<visframe.rows;y++){
	  if(nonfacemask_next.at<uchar>(y,x)==0)
	    visframe.at<Vec3b>(y,x)[0]=0;
	  if(cmask_next.at<uchar>(y,x))
	    visframe.at<Vec3b>(y,x)[0]=255;
	}

      if(bb.count(std::pair<int,string>(f+1,trackedpointid))){
	vector<Point> &v=*boost::any_cast<vector<Point> >(&bb[std::pair<int,std::string>(f+1,trackedpointid)]);
	for(auto it=v.begin();it!=v.end();it++)
	  visframe.at<Vec3b>(it->y,it->x)=Vec3b(255,255,255);	  
      }

      //      for(auto git=trackedgridpoints->begin(); git != trackedgridpoints->end();git++)
	//for(auto it=git->begin();it!=git->end();it++)
	for(auto it=res.begin();it!=res.end();it++)
	  visframe.at<Vec3b>(it->y,it->x)=Vec3b(255,255,0);

      Mat floatpointmask(reff.rows,reff.cols,cv::DataType<float>::type,Scalar::all(0));

      for(auto it=trackedgridpoints->begin();it!=trackedgridpoints->end();it++)
	floatpointmask.at<float>(it->y,it->x)=1;


      boxFilter(floatpointmask,floatpointmask,floatpointmask.depth(),Size(13,13),Point(-1,-1),false);

      float maxpointcount=0;

      for(int x=0;x<floatpointmask.cols;x++)
	for(int y=0;y<floatpointmask.rows;y++)
	  maxpointcount=fmax(maxpointcount,floatpointmask.at<float>(y,x));

      cout << "maxpointcount=" << maxpointcount << endl;
      
      retainedgridpoints=new vector<Point>;


      int minpointcount=3;

      for(auto it=trackedgridpoints->begin();it!=trackedgridpoints->end();it++)
	if(floatpointmask.at<float>(it->y,it->x)>=minpointcount)
	  retainedgridpoints->push_back(*it);

      delete trackedgridpoints;
      trackedgridpoints=retainedgridpoints;

      imshow("tracking results",visframe);
      floatimagesc("retained point density",floatpointmask);

      Mat retainedmat=tgtf.clone();
      for(auto it=trackedgridpoints->begin();it!=trackedgridpoints->end();it++)
	retainedmat.at<Vec3b>(it->y,it->x)=Vec3b(255,255,255);

      Mat classification;

      bool trackallpoints=true;

      if(trackallpoints){
	bb[std::pair<int,std::string>(f+1,trackedpointid)]=res;
      }

      vector<string> classptids;

      classptids.push_back("unoccludedheadpoints");
      classptids.push_back(trackedpointid);

 //      approximateNNClassification(f+1,classptids,headcmask_next, 
// 				  classification,bb);

//       Mat visclass=tgtf.clone();
//       for(int x=0;x<visclass.cols;x++)
// 	for(int y=0;y<visclass.rows;y++)
// 	  visclass.at<Vec3b>(y,x)[0]=(classification.at<uchar>(y,x)==2)?255:0;

      imshow("retained points",retainedmat);
 //      imshow("nn classification",visclass);
      cv::waitKey(0);

    }

    delete trackedgridpoints;

  }

  void trackNonfaceAreaGridlet(int first,int last,
			const string &constraintmaskid,
			const string &nonfacemaskid,
			const string &salientpointid,
			const string &motionfieldid,
			const string &trackedpointid,
			localBlackBoardType &bb){

    vector<vector<Point> > *trackedgridpoints=new vector<vector<Point> >;
    //    vector<Point> *trackedgridpoints=new vector<Point>;

    int points_in_gridlets=2;
    int maxdist=20;



    for(int f=first;f<last;f++){

      Mat nonfacemask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,nonfacemaskid)]);
      Mat nonfacemask_next=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f+1,nonfacemaskid)]);

      Mat cmask_next=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f+1,constraintmaskid)]);

      Mat headcmask_next=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f+1,"headconstraintmask")]);

      Mat skinmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"skinmask")]);

      Mat skinmask_next=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f+1,"skinmask")]);
      // track the points from f to f+1

      Mat reff=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"frame")]);
      Mat tgtf=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f+1,"frame")]);
      
      vector<vector<Point> > res;

      // trackedgridpoints already contains the point n-tuples that were tracked outside
      // the nonfacemask

      // append the salient points of non-face area to the mix

      enumerateNtuplesAmongPoints(f, salientpointid,points_in_gridlets,maxdist,
      				  *trackedgridpoints,bb);

      cout << "tracking " << trackedgridpoints->size() << " gridlets from frame " << f << " to frame " << f+1 << endl;

      Mat visframe=reff.clone();

      for(int x=0;x<visframe.cols;x++)
	for(int y=0;y<visframe.rows;y++)
	  if(nonfacemask.at<uchar>(y,x)==0)
	    visframe.at<Vec3b>(y,x)[0]=0;


      for(auto git=trackedgridpoints->begin(); git != trackedgridpoints->end();git++)
	for(auto it=git->begin();it!=git->end();it++)
	  visframe.at<Vec3b>(it->y,it->x)=Vec3b(255,255,255);

      imshow("points to track",visframe);
      imshow("skin mask",skinmask);
      cv::waitKey(0);

            vector<vector<Point> > *retainedgridpoints=new vector<vector<Point> >;
	    //vector<Point> *retainedgridpoints=new vector<Point>;

      int pointctr=0;

      for(auto git=trackedgridpoints->begin();git!=trackedgridpoints->end();git++){



	float finalE=trackGridletBetweenFrames(*git,f,f+1,constraintmaskid,motionfieldid,false,res,bb);

	//       	float finalE=trackSinglePointBetweenFrames(*git,res,f,f+1,constraintmaskid,motionfieldid,bb);
	// the coordinates in *git get replaced by the new ones in the next frame

	pointctr++;

	cout << "point #"<<pointctr<<endl;

	float thr=1;

	if(finalE>=0 && finalE<thr){
	  // output the accepted track
	  if(bb.count(std::pair<int,string>(f+1,trackedpointid))==0)
	    bb[std::pair<int,std::string>(f+1,trackedpointid)]=vector<Point>();

	  bool outside=false; // set true if any of the tracked points drifts outside the
	                      // nonface mask 

	  for(auto it=git->begin();it!=git->end();it++){

	    vector<Point> &v=*boost::any_cast<vector<Point> >(&bb[std::pair<int,std::string>(f+1,trackedpointid)]);
	    
	    v.push_back(*it);

	    if(nonfacemask_next.at<uchar>(it->y,it->x)==0)
	      outside=true;
	  }

	  if(outside)
	    retainedgridpoints->push_back(*git);

	  // repeat for the additional results

	  for(auto rit=res.begin();rit!=res.end();rit++){

	    bool outside=false; // set true if any of the tracked points drifts outside the
	  // nonface mask 

	    for(auto it=rit->begin();it!=rit->end();it++){
	      
	      vector<Point> &v=*boost::any_cast<vector<Point> >(&bb[std::pair<int,std::string>(f+1,trackedpointid)]);
	      
	      v.push_back(*it);
	      
	      if(nonfacemask_next.at<uchar>(it->y,it->x)==0)
		outside=true;
	    }
	  //   if(outside)
// 	      retainedgridpoints->push_back(*rit);
	  }


	}

      }

      delete trackedgridpoints;
      trackedgridpoints=retainedgridpoints;

    
      // visualise the tracking results
	
      visframe=tgtf.clone();

      for(int x=0;x<visframe.cols;x++)
	for(int y=0;y<visframe.rows;y++){
	  if(nonfacemask_next.at<uchar>(y,x)==0)
	    visframe.at<Vec3b>(y,x)[0]=0;
	  if(cmask_next.at<uchar>(y,x))
	    visframe.at<Vec3b>(y,x)[0]=255;
	}

      if(bb.count(std::pair<int,string>(f+1,trackedpointid))){
	vector<Point> &v=*boost::any_cast<vector<Point> >(&bb[std::pair<int,std::string>(f+1,trackedpointid)]);
	for(auto it=v.begin();it!=v.end();it++)
	  visframe.at<Vec3b>(it->y,it->x)=Vec3b(255,255,255);	  
      }

      imshow("tracking results",visframe);


      // suppress points that do not receive enough support
      // from their neighbours
      
      Mat floatpointmask(reff.rows,reff.cols,cv::DataType<float>::type,Scalar::all(0));

      for(auto git=trackedgridpoints->begin();git!=trackedgridpoints->end();git++)
	for(auto it=git->begin();it!=git->end();it++)
	  floatpointmask.at<float>(it->y,it->x)=1;


      boxFilter(floatpointmask,floatpointmask,floatpointmask.depth(),Size(13,13),Point(-1,-1),false);

      float maxpointcount=0;

      for(int x=0;x<floatpointmask.cols;x++)
	for(int y=0;y<floatpointmask.rows;y++)
	  maxpointcount=fmax(maxpointcount,floatpointmask.at<float>(y,x));

      cout << "maxpointcount=" << maxpointcount << endl;
      
      retainedgridpoints=new vector<vector<Point>>;

      int minpointcount=2;

      for(auto git=trackedgridpoints->begin();git!=trackedgridpoints->end();git++){
	float pointcount=minpointcount;
	for(auto it=git->begin();it!=git->end();it++)
	  pointcount=fmin(pointcount,floatpointmask.at<float>(it->y,it->x));

	if(pointcount>=minpointcount)
	  retainedgridpoints->push_back(*git);
      }

      delete trackedgridpoints;
      trackedgridpoints=retainedgridpoints;

      Mat classification;

     vector<string> classptids;

     classptids.push_back("unoccludedheadpoints");
     classptids.push_back(trackedpointid);

     approximateNNClassification(f+1,classptids,headcmask_next, 
				 classification,bb);   
       Mat visclass=tgtf.clone();
       for(int x=0;x<visclass.cols;x++)
 	for(int y=0;y<visclass.rows;y++)
 	  visclass.at<Vec3b>(y,x)[0]=(classification.at<uchar>(y,x)==2)?255:0;

      imshow("nn classification",visclass);

      // augment the list of tracked gridlets with gridlets
      // generated among the salient points within the non-face labelled area
      // outside the nonfacemask
      
   //    Mat ptmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f+1,"pivotskinpoints")]);
//       for(int x=0;x<visclass.cols;x++)
//  	for(int y=0;y<visclass.rows;y++)
// 	  if(ptmask.at<uchar>(y,x) !=0 && (classification.at<uchar>(y,x)!=2 || 
// 					  nonfacemask_next.at<uchar>(y,x)!=0))
// 	    ptmask.at<uchar>(y,x)=0;

//       bb[std::pair<int,std::string>(f+1,"ptmask")]=ptmask;

//       enumerateNtuplesAmongPoints(f+1, "ptmask",points_in_gridlets,maxdist,
//       				  *trackedgridpoints,bb);

//       imshow("refreshing point set",ptmask);
//       cv::waitKey(0);

    }

    delete trackedgridpoints;

  }


  void enumerateNtuplesAmongPoints(int f, const string &pointsetid,
				   int n, int maxdist,
				   vector<vector<Point> > &res,localBlackBoardType &bb){
  
    // note: points are appended to res, which is not cleared of its old contents

    Mat refpoints=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,pointsetid)]);

    //res.clear();

    vector<Point> pointvec;

    Mat idxmat(refpoints.rows, refpoints.cols,cv::DataType<int>::type,cv::Scalar::all(0));

    for(int x=0;x<refpoints.cols;x++)
      for(int y=0;y<refpoints.rows;y++){
	if(refpoints.at<uchar>(y,x)>0){
	   pointvec.push_back(Point(x,y));
	   idxmat.at<int>(y,x)=pointvec.size()-1;
	}
      }

    int npoints=pointvec.size();

    // this value seems to go unused? //Matti
    // int maxsqrdist=maxdist*maxdist;
  
    for(int p1=0;p1<npoints;p1++){

      vector<int> pi;
      pi.push_back(p1);

      Point minpt,maxpt;
      minpt.x=fmax(0,pointvec[p1].x-maxdist);
      minpt.y=fmax(0,pointvec[p1].y-maxdist);

      maxpt.x=fmin(refpoints.cols-1,pointvec[p1].x+maxdist);
      maxpt.y=fmin(refpoints.rows-1,pointvec[p1].y+maxdist);

      vector<int> validpoints;

      for(int x=minpt.x;x<=maxpt.x;x++)
	for(int y=minpt.y;y<=maxpt.y;y++)
	  if(refpoints.at<uchar>(y,x)>0 && idxmat.at<int>(y,x)!=p1)
	    validpoints.push_back(idxmat.at<int>(y,x));

      if(static_cast<int>(validpoints.size())<n-1) continue;

      for(int i=0;i<n-1;i++){
	int r=rand(validpoints.size()-i);
	pi.push_back(validpoints[r]);
	validpoints[r]=validpoints[validpoints.size()-1-i];
      }

      vector<Point> v;
      for(auto it=pi.begin();it!=pi.end();it++)
	v.push_back(pointvec[*it]);

      res.push_back(v);
    }
  }


  float trackGridletBetweenFrames(vector<Point> &pt,int refframe, int tgtframe,
				  const string &constraintmaskid,
				  const string &motionfieldid,
				  bool posanneal,
				  vector<vector<Point> > &additionalresults,
				  localBlackBoardType &bb,
				  bool show){

    // the coordinates in pt get replaced by the new ones in the next frame

    Mat reff=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,"frame")]);
    Mat tgtf=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"frame")]);


    Mat *motionfield=NULL;
    if(motionfieldid!="")
      motionfield=boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,motionfieldid)]);

    Mat cmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,constraintmaskid)]);

    Mat frif=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,"intensity_float")]);
       
    Mat nextif=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"intensity_float")]);


    vector<Point> refpt=pt;

    // set up the gridlet according to pt

    grid_type grid(pt.size());

    for(int i=0;i<(int)pt.size();i++){
      grid[i].idx=i;
      grid[i].refx=pt[i].x;
      grid[i].refy=pt[i].y;


    }

    set_nbr_fullconnectivity(grid);
    set_l0(grid);

      // initialise the track according to motion field
    // if its specified
    if(motionfield){
      if(show)
	showMotionField(*motionfield);
      initTrackByMotionField(grid,cmask,bb,
			     *motionfield,0);
    }   else // otherwise initialise the grid to the reference frame location
         // and hope the movement be so small that optimisation takes care of it
      for(int i=0;i<(int)pt.size();i++){
	grid[i].tgtx=pt[i].x;
	grid[i].tgty=pt[i].y;
      }

    if(show){
     Mat visframe=reff.clone();
     Mat visframe2=tgtf.clone();


     showgrid(visframe, grid,
 	     true, false,true,false);

     showgrid(visframe2, grid,
 	     true,true,false,true);
      
     cv::imshow("grid in reference frame", visframe);
     cv::imshow("initialistion of the tracked grid", visframe2);
     cv::waitKey(0);
    }

    // fine tune the track with greedy local optimisation

    int templater=6;
    float alpha=1.0/4000;
    
    float bestE;
    
    grid_type best_grid;

    int spacing=10;
	 

    updateGridByGreedyLocal(grid,frif,nextif,cmask,
 				 templater,2*spacing,alpha,
 				 bestE,best_grid);

    for(int i=0;i<(int)grid.size();i++)
      if(grid[i].tgtx<0||grid[i].tgtx>=cmask.cols ||
	 grid[i].tgty<0||grid[i].tgty>=cmask.rows)
	return -1;

    // copy the tracking result to pt

    for(int i=0;i<(int)pt.size();i++){
      pt[i].x=grid[i].tgtx;
      pt[i].y=grid[i].tgty;
    }





    if(show){

     Mat visframe2=tgtf.clone();

     showgrid(visframe2, grid,
 	     true,true,false,true);

     cv::imshow("tracked grid", visframe2);
    }


    if(posanneal){
      ensurePaddedFramesOnBlackboard(refframe,bb);
      ensurePaddedFramesOnBlackboard(tgtframe,bb);
 

      Mat reff_zeropad=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,"frame_padded")]);
      Mat tgtf_zeropad=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"frame_padded")]);

      vector<Point> greedypt=pt;

      vector<Point> trackedpt=greedypt;

      Mat sumres;

      int templater=5;

      int maxdev=40;
      int matchwindowr=templater+maxdev;




      for(int i=0;i<(int)pt.size();i++){

	//     // simple idea: find out the template matching neighbourhood 
	//     // of trackedpt 

	Rect rect;

	rect.x=refpt[i].x-templater+intensity_image_zeropad_margin;
	rect.y=refpt[i].y-templater+intensity_image_zeropad_margin;

	rect.width=templater*2+1;
	rect.height=rect.width;

	Mat templ(reff_zeropad,rect);

	rect.x=trackedpt[i].x-matchwindowr+intensity_image_zeropad_margin;
	rect.y=trackedpt[i].y-matchwindowr+intensity_image_zeropad_margin;

	rect.width=2*matchwindowr+1;
	rect.height=2*matchwindowr+1;
	
	Mat tgtmat(tgtf_zeropad,rect);

	Mat result;
	
	matchTemplate(tgtmat,templ,result,CV_TM_SQDIFF);

// 	cout << "point #"<<i<<endl;

// 	floatimagesc("template match neighbourhood of single point",result);
// 	cv::waitKey(0);

	if(i==0)
	  sumres=result;
	else{
	  for(int x=0;x<sumres.cols;x++)
	    for(int y=0;y<sumres.rows;y++)
	      sumres.at<float>(y,x) += result.at<float>(y,x);
	}
      }

      // add radial penalty term beta*r to sumres

      Mat geomres=sumres;

      float beta;

      float oldres=sumres.at<float>(matchwindowr-templater,matchwindowr-templater);

      beta=2*oldres/10;

      // reasoning for selection of beta: distance of 10 pixels from the origin 
      // corresponds to tripling the old matching energy

      for(int x=0;x<sumres.cols;x++)
	for(int y=0;y<sumres.rows;y++){
	  int dx=x-(matchwindowr-templater);
	  int dy=x-(matchwindowr-templater);
	  geomres.at<float>(y,x) += beta*sqrt(dx*dx+dy*dy); 
	}

  //     floatimagesc("combined template match neighbourhood",sumres);
//       floatimagesc("geom. adjusted template match neighbourhood",geomres);
//       cv::waitKey(0);
	
      // now use simulated annealing type approach to produce 
      // new locations around the seed

      Mat accres=tgtf.clone();

      float T=-0.01/log(0.5); // 2% energy increase is accepted w/ p=0.5
      int acceptcount=0;
      int acceptuphillcount=0;
      int rejectcount=0;

    
    T=-0.15/log(0.5);

    int iter=0;


    Point delta(0,0);

    while(acceptcount<10&&iter<10000){

      iter++;

//       cout << "iteration #" << iter << endl;

      Point newdelta=delta;

    // perturb the location randomly

      float dev=10;

      newdelta.x+=dev*randn();
      newdelta.y+=dev*randn();

      //      visframe2=tgtf.clone();

      int withinbounds=true;

      for(int i=0;i<(int)greedypt.size();i++){
	Point newpt=greedypt[i]+newdelta;
	if(newpt.x<0 || newpt.x>=cmask.cols || newpt.y < 0 || newpt.y > cmask.rows 
	   || fabs(newdelta.x)>=maxdev 
	   || fabs(newdelta.y)>=maxdev
	   || cmask.at<uchar>(newpt.y,newpt.x)==0){
	  withinbounds=false;
	  break;
	}
	
      }

      if(!withinbounds)
	continue;

     //  visframe2.at<Vec3b>(newpt.y,newpt.x)=Vec3b(255,255,255);
//       imshow("candidate location",visframe2);
      

      float p;

      float E_old=geomres.at<float>(matchwindowr-templater+delta.y,
				   matchwindowr-templater+delta.x);
      float E_new=geomres.at<float>(matchwindowr-templater+newdelta.y,
				   matchwindowr-templater+newdelta.x);


      //     cout << "E_old="<<E_old<<" E_new="<<E_new<<" relative diff="<<(E_new-E_old)/E_old<<endl;

    // with p=1 if energy lower than that of the old location
    // with p=exp((E_old-E_new)/(T*E_old)) if energy increases


    if(E_old>=E_new) p=1;
    else
      p=exp((E_old-E_new)/(T*E_old));
    
    //     cout << "p="<<p<<endl;
//     cout << "T="<<T<<endl;

     bool accept=false;

    if(p==1){
      accept=true;
    }
    else if(rand(1000)<1000*p){
    // accumulate all the accepted translations

      accept=true;
      acceptuphillcount++;

    // adjust T so that fraction tgtp of uphill moves get accepted

//       cout << "accepted" << endl;

    } else{
      rejectcount++;
      //      cout << "step rejected" << endl;
    }

    if(accept){

      acceptcount++;
      delta=newdelta;

      cout << "step accepted, delta now ("<<delta.x<<","<<delta.y<<")"<<endl;
      cout << "acceptcount="<<acceptcount<<endl;

      vector<Point> v=greedypt;
      for(auto it=v.begin();it!=v.end();it++){
	*it += delta;
	accres.at<Vec3b>(it->y,it->x)=Vec3b(255,255,255);
      }

      additionalresults.push_back(v);

    //   imshow("accumulated result",accres);
//       cv::waitKey(0);

    }

    }
    



    }

      float finalE=evaluateGridEnergy(grid,true,true, -1,
				    frif,nextif,
				    templater,alpha);
    
      return finalE;
  }

  float trackSinglePointBetweenFrames(Point &pt, vector<Point> &res,
				      int refframe, int tgtframe,
				  const string &constraintmaskid,
				  const string &motionfieldid,localBlackBoardType &bb){

    // the coordinates in pt get replaced by the new ones in the next frame

    // multiple tracking results are also appended to res

    Point trackedpt;

    Mat reff=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,"frame")]);
    Mat tgtf=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"frame")]);

    ensurePaddedFramesOnBlackboard(refframe,bb);
    ensurePaddedFramesOnBlackboard(tgtframe,bb);
 

    Mat reff_zeropad=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,"frame_padded")]);
    Mat tgtf_zeropad=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"frame_padded")]);

    Mat motionfield=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,motionfieldid)]);

    Mat cmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,constraintmaskid)]);

    Mat frif=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,"intensity_float")]);
       
    Mat nextif=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"intensity_float")]);

    // initialise tracked pt to the location indicated by motion field

    trackedpt.x = pt.x - motionfield.at<Point2f>(pt.y,pt.x).x;
    trackedpt.y = pt.y - motionfield.at<Point2f>(pt.y,pt.x).y;

    Point motionpt=trackedpt;

    // simple idea: find out the template matching neighbourhood 
    // of trackedpt 

    int templater=5;

    int maxdev=40;

    Rect rect;

    rect.x=pt.x-templater+intensity_image_zeropad_margin;
    rect.y=pt.y-templater+intensity_image_zeropad_margin;

    rect.width=templater*2+1;
    rect.height=rect.width;

    Mat templ(reff_zeropad,rect);

    int matchwindowr=templater+maxdev;

    rect.x=trackedpt.x-matchwindowr+intensity_image_zeropad_margin;
    rect.y=trackedpt.y-matchwindowr+intensity_image_zeropad_margin;

    rect.width=2*matchwindowr+1;
    rect.height=2*matchwindowr+1;

    Mat tgtmat(tgtf_zeropad,rect);

    Mat result;
    
    matchTemplate(tgtmat,templ,result,CV_TM_SQDIFF);

    Mat visframe=reff.clone();
    Mat visframe2=tgtf.clone();

    visframe.at<Vec3b>(pt.y,pt.x)=Vec3b(255,255,255);
    visframe2.at<Vec3b>(trackedpt.y,trackedpt.x)=Vec3b(255,255,255);

//     imshow("point to track",visframe);
//     imshow("track initialised bu motion field",visframe2);

//     floatimagesc("template match neighbourhood",result);
//     cv::waitKey(0);

    Mat accres=visframe2.clone();

    float T=-0.01/log(0.5); // 2% energy increase is accepted w/ p=0.5
    int acceptcount=0;
    int acceptuphillcount=0;
    int rejectcount=0;

    
    T=-0.15/log(0.5);

    int iter=0;


    while(acceptcount<4&&iter<10000){

      iter++;

//       cout << "iteration #" << iter << endl;

      Point newpt=trackedpt;

    // perturb the location randomly

      float dev=5;

      newpt.x+=dev*randn();
      newpt.y+=dev*randn();

      //      visframe2=tgtf.clone();

      if(newpt.x<0 || newpt.x>=cmask.cols || newpt.y < 0 || newpt.y > cmask.rows 
	 || fabs(newpt.x-motionpt.x)>=maxdev 
	 || fabs(newpt.y-motionpt.y)>=maxdev
	 || cmask.at<uchar>(newpt.y,newpt.x)==0
)
	continue;

     //  visframe2.at<Vec3b>(newpt.y,newpt.x)=Vec3b(255,255,255);
//       imshow("candidate location",visframe2);
      

      float p;

      float E_old=result.at<float>(matchwindowr-templater+trackedpt.y-motionpt.y,
				   matchwindowr-templater+trackedpt.x-motionpt.x);
      float E_new=result.at<float>(matchwindowr-templater+newpt.y-motionpt.y,
				   matchwindowr-templater+newpt.x-motionpt.x);


//       cout << "E_old="<<E_old<<" E_new="<<E_new<<" relative diff="<<(E_new-E_old)/E_old<<endl;

    // with p=1 if energy lower than that of the old location
    // with p=exp((E_old-E_new)/(T*E_old)) if energy increases


    if(E_old>=E_new) p=1;
    else
      p=exp((E_old-E_new)/(T*E_old));
    
//     cout << "p="<<p<<endl;
//     cout << "T="<<T<<endl;

    if(p==1){
      acceptcount++;
      trackedpt=newpt;
      if(cmask.at<uchar>(newpt.y,newpt.x))
	res.push_back(newpt);
      accres.at<Vec3b>(trackedpt.y,trackedpt.x)=Vec3b(255,255,255);
//       cout << "accepted" << endl;
    }
    else if(rand(1000)<1000*p){
    // accumulate all the accepted translations

      acceptcount++;
      acceptuphillcount++;
      trackedpt=newpt;
      if(cmask.at<uchar>(newpt.y,newpt.x))
	res.push_back(newpt);
      accres.at<Vec3b>(trackedpt.y,trackedpt.x)=Vec3b(255,255,255);
//       cout << "accepted" << endl;

    // adjust T so that fraction tgtp of uphill moves get accepted
    } else
      rejectcount++;


    }
//  imshow("accumulated result",accres);
//     cv::waitKey(0);

    pt=trackedpt;

    if(cmask.at<uchar>(trackedpt.y,trackedpt.x))
      return 0;
    else
      return -1;

  }



  void approximateNNClassification(int f,vector<string> classptids, Mat &mask, Mat &res,
				   localBlackBoardType &bb){


    // performs an approximate nearest neighbour classification
    // of the image area defined by mask

    // the classes are defined by the prototype point clouds 
    // with the specified ids 

    // the class numbering in res starts from 1

    int nclass=classptids.size();

    res=Mat(mask.rows,mask.cols,CV_8UC1,Scalar::all(0));

    if(nclass<2) return;

    vector<Mat> ptmat;

    // translate point lists into matrix representation
    
    for(int i=0;i<nclass;i++){
      vector<Point> &v=*boost::any_cast<vector<Point> >(&bb[std::pair<int,std::string>(f,classptids[i])]);
      Mat m=Mat(mask.rows,mask.cols,CV_8UC1,Scalar::all(0));
      for(auto it=v.begin();it!=v.end();it++){
	m.at<uchar>(it->y,it->x)=255;
	if(mask.at<uchar>(it->y,it->x))
	  res.at<uchar>(it->y,it->x)=i+1;
      }
      ptmat.push_back(m);
 //      cout << "class " << i << endl;
//       imshow("pointmat",m);
//       cv::waitKey(0);
    }

    // for speeding up the forthcoming operations, find out the 
    // extent of the mask

    Point minpt(mask.cols,mask.rows),maxpt(0,0);

    int unlabelledcount=0;

    for(int x=0;x<mask.cols;x++)
      for(int y=0;y<mask.rows;y++)
	if(mask.at<uchar>(y,x)){
	  minpt.x=fmin(minpt.x,x);
	  minpt.y=fmin(minpt.y,y);
	  maxpt.x=fmax(maxpt.x,x);
	  maxpt.y=fmax(maxpt.y,y);
	  if(res.at<uchar>(y,x)==0)
	    unlabelledcount++;
	}
	
    while(unlabelledcount>0){
      
      for(int i=0;i<nclass;i++){
	// dilate the point matrix #i by 1

	cv::dilate(ptmat[i], ptmat[i], 
		   cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(3, 3)));
	
   //    cout << "class " << i << endl;
//       imshow("dilated pointmat",ptmat[i]);
//       cv::waitKey(0);

	// go through the unlabelled part of mask 

	for(int x=minpt.x;x<=maxpt.x;x++)
	  for(int y=minpt.y;y<=maxpt.y;y++)
	    	if(mask.at<uchar>(y,x) && 
		   res.at<uchar>(y,x)==0 &&
		   ptmat[i].at<uchar>(y,x)){
		  res.at<uchar>(y,x)=i+1;
		  unlabelledcount--;
		}
      }

    }



  }



  void purgeBlackboardFromFrames(int first,int last,localBlackBoardType &bb){
    for(auto it=bb.begin();it!=bb.end();it++)
      if(it->first.first>=first && it->first.first<=last)
	bb.erase(it);
  }


//   void initialiseNonfaceTracking(int frame,localBlackBoardType bb){
//     // the state information of the tracker is kept on bb from the source frame of tracking
//     // under key "nonfacetrackerstate"

//     // the state information is of type vector<vector<Point> > 
//     // that specifies the gridlets that are tracked from frame to frame
//   }


 //  void trackNonFacePoints(int first, int last,localBlackBoardType bb){

//     // uses the single point tracker

//     // however, the tracker state information is still in vector<vector<Point>


//}

  void trackNonFaceGridlet(int srcframe, int tgtframe,localBlackBoardType &bb){



    int points_in_gridlets=4;
    int maxdist=20;

    ensureMasksOnBB(srcframe,bb);
    ensureMasksOnBB(tgtframe,bb);

    // this makes sure that the following masks are on blackboard:
    // "nonfacemask"
    // "nonfaceconstraintmask"
    // "skinmask" 
    // "headconstraintmask" 

    ensureFrameOnBB(srcframe,bb); 
    ensureFrameOnBB(tgtframe,bb); 

    // this ensures the following keys
    // "frame","frame_padded","intensity_uchar","intensity_float"

   ensureSalientPointsOnBB(srcframe,bb);    
   ensureSalientPointsOnBB(tgtframe,bb);    

    // this ensures the following keys
    // "pivotheadpoints","pivotnonfacepoints","pivotskinpoints"

   ensureMotionFieldOnBB(srcframe,bb);
    // this ensures the following keys: "motionfield"

   Mat nonfacemask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(srcframe,"nonfacemask")]);
   Mat nonfacemask_next=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"nonfacemask")]);

   Mat cmask_next=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"nonfaceconstraintmask")]);

   Mat headcmask_next=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"headconstraintmask")]);

   Mat skinmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(srcframe,"skinmask")]);
      // track the points from f to f+1

   Mat reff=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(srcframe,"frame")]);
   Mat tgtf=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"frame")]);
   
   vector<vector<Point> > res;

   vector<vector<Point> > trackedgridpoints;
   
   // fetch from bb to trackedgridpoints the point n-tuples that were tracked outside
   // the nonfacemask

   if(bb.count(std::pair<int,std::string>(srcframe,"trackerstate_nonface")))
     trackedgridpoints=*boost::any_cast<vector<vector<Point> > >(&bb[std::pair<int,std::string>(srcframe,"trackerstate_nonface")]);
   
   // append the salient points of non-face area to the mix
   
   enumerateNtuplesAmongPoints(srcframe, "pivotnonfacepoints",points_in_gridlets,maxdist,
			       trackedgridpoints,bb);

   bool showgridlets=true; // show individual gridlets until this it set to false

   showgridlets=false; // disable debug output

   //  if(tgtframe<28)
//       showgridlets=false; // hack

   if(showgridlets){
  cout << "tracking " << trackedgridpoints.size() << " gridlets from frame " << srcframe << " to frame " << tgtframe << endl;

    Mat visframe=reff.clone();

    for(int x=0;x<visframe.cols;x++)
      for(int y=0;y<visframe.rows;y++)
        if(nonfacemask.at<uchar>(y,x)==0)
 	 visframe.at<Vec3b>(y,x)[0]=0;


    for(auto git=trackedgridpoints.begin(); git != trackedgridpoints.end();git++)
      for(auto it=git->begin();it!=git->end();it++)
        visframe.at<Vec3b>(it->y,it->x)=Vec3b(255,255,255);

    imshow("points to track",visframe);
   imshow("skin mask",skinmask);
    cv::waitKey(0);
  }
   // set up the place to store the tracking results

   if(bb.count(std::pair<int,string>(tgtframe,"trackerstate_nonface"))==0)
     bb[std::pair<int,string>(tgtframe,"trackerstate_nonface")]= vector<vector<Point> >();

   vector<vector<Point> > &retainedgridpoints=*boost::any_cast<vector<vector<Point> > >(&bb[std::pair<int,std::string>(tgtframe,"trackerstate_nonface")]);

;
 bb[std::pair<int,std::string>(tgtframe,"trackednonfacepoints")]=vector<Point>();
 // wipe out old results

   int pointctr=0;

   for(auto git=trackedgridpoints.begin();git!=trackedgridpoints.end();git++){


	float finalE=trackGridletBetweenFrames(*git,srcframe,tgtframe,
					       "nonfaceconstraintmask","motionfield",false,res,bb,showgridlets);

	if(showgridlets){
	  int k=cv::waitKey(0);
	  if(k=='n')
	    showgridlets=false;
	}

	//       	float finalE=trackSinglePointBetweenFrames(*git,res,f,f+1,constraintmaskid,motionfieldid,bb);
	// the coordinates in *git get replaced by the new ones in the next frame

	pointctr++;

	//cout << "point #"<<pointctr<<endl;

	float thr=1;

	if(finalE>=0 && finalE<thr){
	  // output the accepted track
	  if(bb.count(std::pair<int,string>(tgtframe,"trackednonfacepoints"))==0)
	    bb[std::pair<int,std::string>(tgtframe,"trackednonfacepoints")]=vector<Point>();

	  bool outside=false; // set true if any of the tracked points drifts outside the
	                      // nonface mask 

	  for(auto it=git->begin();it!=git->end();it++){

	    vector<Point> &v=*boost::any_cast<vector<Point> >(&bb[std::pair<int,std::string>(tgtframe,"trackednonfacepoints")]);
	    
	    v.push_back(*it);

	    if(nonfacemask_next.at<uchar>(it->y,it->x)==0)
	      outside=true;
	  }

	  if(outside)
	    retainedgridpoints.push_back(*git);

	  // repeat for the additional results

	  for(auto rit=res.begin();rit!=res.end();rit++){

	    bool outside=false; // set true if any of the tracked points drifts outside the
	  // nonface mask 

	    for(auto it=rit->begin();it!=rit->end();it++){
	      
	      vector<Point> &v=*boost::any_cast<vector<Point> >(&bb[std::pair<int,std::string>(tgtframe,"trackednonfacepoints")]);
	      
	      v.push_back(*it);
	      
	      if(nonfacemask_next.at<uchar>(it->y,it->x)==0)
		outside=true;
	    }
	  //   if(outside)
	    //	      retainedgridpoints->push_back(*rit);
	  }


	}

      }

      // visualise the tracking results
	
  //       Mat visframe=tgtf.clone();

//       for(int x=0;x<visframe.cols;x++)
// 	for(int y=0;y<visframe.rows;y++){
// 	  if(nonfacemask_next.at<uchar>(y,x)==0)
// 	    visframe.at<Vec3b>(y,x)[0]=0;
// 	  if(cmask_next.at<uchar>(y,x))
// 	    visframe.at<Vec3b>(y,x)[0]=255;
// 	}

//       if(bb.count(std::pair<int,string>(tgtframe,"trackednonfacepoints"))){
// 	vector<Point> &v=*boost::any_cast<vector<Point> >(&bb[std::pair<int,std::string>(tgtframe,"trackednonfacepoints")]);
// 	for(auto it=v.begin();it!=v.end();it++)
// 	  visframe.at<Vec3b>(it->y,it->x)=Vec3b(255,255,255);	  
//       }

//       imshow("tracking results",visframe);
//       cv::waitKey(0);

      // expand THE TRACKING RESULTS BY SINGLE POINT ANNEALING

      if(bb.count(std::pair<int,string>(tgtframe,"trackednonfacepoints"))){
	vector<Point> &v=*boost::any_cast<vector<Point> >(&bb[std::pair<int,std::string>(tgtframe,"trackednonfacepoints")]);


	// expand only points that are in the head blob
	ensureUniqueBlobMaskOnBB(tgtframe,bb);
	Mat ulbl=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"uniqueblobmask")]);

	Rect faceLocation=*boost::any_cast<cv::Rect>(&bb[std::pair<int,std::string>(tgtframe,"facerect")]);

	uchar facelbl=ulbl.at<uchar>(faceLocation.y+0.5*faceLocation.height,
				     faceLocation.x+0.5*faceLocation.width);

	vector<Point> toExpand;

	for(auto it=v.begin();it!=v.end();it++)
	  if(ulbl.at<uchar>(it->y,it->x)==facelbl)
	    toExpand.push_back(*it);

	expandPointSetBySinglePointAnnealing(toExpand,v,tgtframe,"nonfaceconstraintmask",12,0.5,bb);

// 	Mat visframe=tgtf.clone();
// 	for(auto it=v.begin();it!=v.end();it++)
// 	  visframe.at<Vec3b>(it->y,it->x)=Vec3b(255,255,255);	  

// 	imshow("expanded tracking results",visframe);
// 	cv::waitKey(0);

      }

//       // suppress points that do not receive enough support
//       // from their neighbours

      vector<vector<Point> > gv;
      
       Mat floatpointmask(reff.rows,reff.cols,cv::DataType<float>::type,Scalar::all(0));



       for(auto git=retainedgridpoints.begin();git!=retainedgridpoints.end();git++)
 	for(auto it=git->begin();it!=git->end();it++)
 	  floatpointmask.at<float>(it->y,it->x)=1;

 //       if(retainedgridpoints.size()){
//        Mat fm2=floatpointmask.clone();

//     float dev=20;

//     GaussianBlur(fm2, fm2, cv::Size( 0,0 ),dev);

//     imshow("blurred point density",fm2);
//     floatimagesc("normalised blurred point density",fm2);

//     cv::waitKey(0);
//        }

       boxFilter(floatpointmask,floatpointmask,floatpointmask.depth(),Size(13,13),Point(-1,-1),false);



//       float maxpointcount=0;

//       for(int x=0;x<floatpointmask.cols;x++)
// 	for(int y=0;y<floatpointmask.rows;y++)
// 	  maxpointcount=fmax(maxpointcount,floatpointmask.at<float>(y,x));

//       cout << "maxpointcount=" << maxpointcount << endl;
      
//       retainedgridpoints=new vector<vector<Point>>;

       int minpointcount=2;

       Mat *unoccludedheadmask=NULL;
       if(bb.count(std::pair<int,std::string>(tgtframe,"unoccludedheadfloatmask")))
	 unoccludedheadmask=boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"unoccludedheadfloatmask")]);
       
       //       Mat m=skinmask.clone();

       //       for(int x=0;x<m.cols;x++)
       //	 for(int y=0;y<m.rows;y++)
       //	   m.at<uchar>(y,x)=(unoccludedheadmask.at<float>(


       for(auto git=retainedgridpoints.begin();git!=retainedgridpoints.end();git++){
 	float pointcount=minpointcount;
	bool invalid=false;

 	for(auto it=git->begin();it!=git->end();it++){
	  if(unoccludedheadmask && unoccludedheadmask->at<float>(it->y,it->x)>0.2){
	    invalid=true;
	    break;
	  }
 	  pointcount=fmin(pointcount,floatpointmask.at<float>(it->y,it->x));
	}

 	if(!invalid && pointcount>=minpointcount)
 	  gv.push_back(*git);
       }

       int oldsize=retainedgridpoints.size();

       retainedgridpoints=gv;

       cout << "retained " << gv.size() << " out of " << oldsize << " gridlets" << endl;
       
//       delete trackedgridpoints;
//       trackedgridpoints=retainedgridpoints;



//       Mat classification;

//      vector<string> classptids;

//      classptids.push_back("unoccludedheadpoints");
//      classptids.push_back(trackedpointid);

//      approximateNNClassification(f+1,classptids,headcmask_next, 
// 				 classification,bb);   
//        Mat visclass=tgtf.clone();
//        for(int x=0;x<visclass.cols;x++)
//  	for(int y=0;y<visclass.rows;y++)
//  	  visclass.at<Vec3b>(y,x)[0]=(classification.at<uchar>(y,x)==2)?255:0;

//       imshow("nn classification",visclass);
//       cv::waitKey(0);

//       // augment the list of tracked gridlets with gridlets
//       // generated among the salient points within the non-face labelled area
//       // outside the nonfacemask
      


  }

  void trackFaceGridlet(int srcframe, int tgtframe,localBlackBoardType &bb){



    int points_in_gridlets=4;
    int maxdist=20;

    ensureMasksOnBB(srcframe,bb);
    ensureMasksOnBB(tgtframe,bb);

    // this makes sure that the following masks are on blackboard:
    // "nonfacemask"
    // "nonfaceconstraintmask"
    // "skinmask" 
    // "headconstraintmask" 

    ensureFrameOnBB(srcframe,bb); 
    ensureFrameOnBB(tgtframe,bb); 

    // this ensures the following keys
    // "frame","frame_padded","intensity_uchar","intensity_float"

   ensureMotionFieldOnBB(srcframe,bb);
    // this ensures the following keys: "motionfield"

   Mat headcmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"headconstraintmask")]);

   Mat headcmask_next=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"headconstraintmask")]);

   // track the points from f to f+1

   Mat reff=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(srcframe,"frame")]);
   Mat tgtf=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"frame")]);
   
   // select tracked points to be the salient head area points that either
   // 1) fall in the area of unoccludedheadfloatmask, if it exists
   // 2) are sampled from the area of headconstraintmask otherwise (actually default option)

   // set up the place for tracking results

   if(bb.count(std::pair<int,string>(srcframe,"trackerstate_face"))==0)
     bb[std::pair<int,string>(srcframe,"trackerstate_face")]= vector<TrackingRecord>();

   // the old state for tgtframe is wiped in case it exists
   bb[std::pair<int,string>(tgtframe,"trackerstate_face")]= vector<TrackingRecord>();

   vector<TrackingRecord> &trackerstate=*boost::any_cast<vector<TrackingRecord> >(&bb[std::pair<int,std::string>(srcframe,"trackerstate_face")]);

   vector<TrackingRecord> &trackerstate_new=*boost::any_cast<vector<TrackingRecord> >(&bb[std::pair<int,std::string>(tgtframe,"trackerstate_face")]);

   if(trackerstate.empty()){

     // no tracker state on bb or it's empty -> initialise tracker to salient points 
     // in head area

     ensureSalientPointsOnBB(srcframe,bb);    

    // this ensures the following keys
    // "pivotheadpoints","pivotnonfacepoints","pivotskinpoints"

     vector<vector<Point> > gridpt;

     enumerateNtuplesAmongPoints(srcframe, "pivotheadpoints",points_in_gridlets,maxdist,
			       gridpt,bb);   

     // go through the list of n-tuples of points in gridpt
     // and initialise the tracker state

     TrackingRecord tr;
     tr.referenceframe=srcframe;
     tr.E=0;
     tr.E_threshold=-1;

     for(auto it=gridpt.begin();it!=gridpt.end();it++){
       tr.pt=*it;
       trackerstate.push_back(tr);
     }
   }

   vector<vector<Point> > res; // currently dummy placeholder

   //   cout << "tracking " << trackedgridpoints.size() << " gridlets from frame " << srcframe << " to frame " << tgtframe << endl;

    Mat visframe=reff.clone();

    for(auto git=trackerstate.begin(); git != trackerstate.end();git++)
      for(auto it=git->pt.begin();it!=git->pt.end();it++)
        visframe.at<Vec3b>(it->y,it->x)=(git->referenceframe==srcframe)
 	 ? Vec3b(255,255,255)
 	 : Vec3b(255,0,0);

//     imshow("face points to be tracked",visframe);
//     int k=cv::waitKey(0);

     bool showgridlets=true; // tracking is shown gridlet for gridlet 
                             // until this is set false

//     if(k=='n') showgridlets=false;

   int pointctr=0;

   bb[std::pair<int,std::string>(tgtframe,"unoccludedheadpoints")]=vector<Point>();
   // wipe out old results

   for(auto git=trackerstate.begin();git!=trackerstate.end();git++){

     pointctr++;

     //     cout << "point #"<<pointctr<<endl;

     vector<Point> pt=git->pt;
     vector<Point> orgpt=git->pt;

     float finalE=trackGridletBetweenFrames(pt,git->referenceframe,tgtframe,
					    "skinmask",
					    "",false,res,bb);


     // process the tracking result

     int referencesneeded=6; // this should be in sync with the minimum occlusion
                             // separation in the caller fcn (<= separation)
                             // so that $(referencesneeded) first tracks can
                             // be assumed to come from unoccluded face

     TrackingRecord tr=*git;
     
     if(false && showgridlets){

       Mat visref=boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(git->referenceframe,"frame")])->clone();
       Mat vistgt=tgtf.clone();

       for(auto it=orgpt.begin();it!=orgpt.end();it++)
	 visref.at<Vec3b>(it->y,it->x)=Vec3b(255,255,255);

       for(auto it=pt.begin();it!=pt.end();it++)
	 vistgt.at<Vec3b>(it->y,it->x)=Vec3b(255,255,255);
       
       cout << "finalE="<<finalE<<endl;
       cout << "threshold="<<tr.E_threshold<<endl;
       cout << "referenceEnergies=";
       for(auto it=tr.referenceE.begin();
	   it != tr.referenceE.end(); it++)
	 cout << " " << *it;
       cout << endl;
       

//        imshow("gridlet in refframe",visref);
//        imshow("gridlet in tgtframe",vistgt);
//        int k=cv::waitKey(0);

//        if(k=='n') showgridlets=false;
     }

     if(finalE>=0 && (int)tr.referenceE.size()<referencesneeded){
       tr.referenceE.push_back(finalE);
     }

     if(tr.E_threshold<0 && (int)tr.referenceE.size()==referencesneeded){
       // enough reference energies collected, ready to determine 
       // the threshold energy
       float sum=0;
       float sqrsum=0;

       for(auto it=tr.referenceE.begin();
		   it != tr.referenceE.end(); it++){
	 sum += *it;
	 sqrsum += (*it)*(*it);
       }
	       
       tr.E_threshold=0;

       size_t n=tr.referenceE.size();
       sum /= n;
       sqrsum /= n;
       sqrsum -= sum*sum;
		 
       tr.E_threshold=sum + fmax(0.2*sum,fmax(0.0025,1.3*sqrt(sqrsum)));

       float minThreshold;

       //       minThreshold=0.030; // haastatteluvideot
       minThreshold=0.0045; // suvi

       tr.E_threshold = fmax(minThreshold,tr.E_threshold);

       tr.E_threshold = fmin(tr.E_threshold,0.1);

       tr.E_threshold=minThreshold; // suvi

       //cout << "determined threshold: " << tr.E_threshold << endl;

       bool acceptall=false;
       if(acceptall)
	 tr.E_threshold=1;
       
     }

     if(finalE>=0 && (tr.E_threshold<0 || finalE <= tr.E_threshold)){
       // accept the track if 
       // 1) tracking is not signalled as invalid (finalE<0)
       // and
       // 2) the tracking energy falls below the threshold value or the 
       // threshold has not yet been determined

       //cout << "accepted track w/ E="<<finalE<<" (threshold " << tr.E_threshold <<")"<<endl;

       tr.pt=pt;
       tr.referenceframe=tgtframe;
       tr.E=finalE;

       // output the accepted track
       if(bb.count(std::pair<int,string>(tgtframe,"unoccludedheadpoints"))==0)
	 bb[std::pair<int,std::string>(tgtframe,"unoccludedheadpoints")]=vector<Point>();

       for(auto it=pt.begin();it!=pt.end();it++){
	    vector<Point> &v=*boost::any_cast<vector<Point> >(&bb[std::pair<int,std::string>(tgtframe,"unoccludedheadpoints")]);
	    v.push_back(*it);
	  }
     }
     else{
       //cout << "rejected track w/ E="<<finalE<<" (threshold " << tr.E_threshold <<")"<<endl;
     }

     trackerstate_new.push_back(tr);
   }

//    // visualise the tracking results
	
//    visframe=tgtf.clone();

//    for(int x=0;x<visframe.cols;x++)
//      for(int y=0;y<visframe.rows;y++){
//        if(headcmask_next.at<uchar>(y,x))
// 	 visframe.at<Vec3b>(y,x)[0]=255;
//      }

//    if(bb.count(std::pair<int,string>(tgtframe,"unoccludedheadpoints"))){
//      vector<Point> &v=*boost::any_cast<vector<Point> >(&bb[std::pair<int,std::string>(tgtframe,"unoccludedheadpoints")]);
//      for(auto it=v.begin();it!=v.end();it++)
//        visframe.at<Vec3b>(it->y,it->x)=Vec3b(255,255,255);	  
//    }

//    imshow("face tracking results",visframe);
//    cv::waitKey(0);

  }


  void ensureUniqueBlobMaskOnBB(int f,localBlackBoardType &bb){
    if(bb.count(std::pair<int,std::string>(f,"uniqueblobmask"))==0){
      assert(bb.count(std::pair<int,std::string>(f,"skinmask"))!=0);

      Mat skinmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"skinmask")]);

      Mat ulbl;
      labelConnectedComponents(skinmask,ulbl);
      bb[std::pair<int,std::string>(f,"uniqueblobmask")]=ulbl;
    }	     
  }

  void ensureMasksOnBB(int f,localBlackBoardType &bb){

    // this makes sure that the following masks are on blackboard:
    // "nonfacemask"
    // "nonfaceconstraintmask"
    // "skinmask" 
    // "headconstraintmask" 

    // skinmask

    if(bb.count(std::pair<int,std::string>(f,"skinmask"))==0){
      //      bb[std::pair<int,std::string>(f,"skinmask")]=vvi_blackBoard->get<cv::Mat>(f, SKINDETECTOR_BLACKBOARD_MASK_ENTRY);

      throw string("skinmask not found on local blackboard");

      Mat skinmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"skinmask")]);

      ensureUniqueBlobMaskOnBB(f,bb);
      Mat ulbl=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"uniqueblobmask")]);
      int minblobsize=500;
      removeSmallBlobs(ulbl,skinmask,minblobsize);
    }
      

    ensureUniqueBlobMaskOnBB(f,bb);
    Mat ulbl=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"uniqueblobmask")]);

    Mat skinmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"skinmask")]);
    ensureFaceRectOnBB(f,bb);
    Rect faceLocation=*boost::any_cast<cv::Rect>(&bb[std::pair<int,std::string>(f,"facerect")]);

    uchar facelbl=ulbl.at<uchar>(faceLocation.y+0.5*faceLocation.height,
				 faceLocation.x+0.5*faceLocation.width);


    int rows=skinmask.rows;
    int cols=skinmask.cols;

    if(bb.count(std::pair<int,std::string>(f,"headconstraintmask"))==0){
      Mat headcmask(rows,cols,CV_8UC1,cv::Scalar::all(0));

      for(int x=0;x<cols;x++)
	for(int y=0;y<rows;y++){
	  if(x>faceLocation.x-0.4*faceLocation.width &&
	     x<faceLocation.x+1.4*faceLocation.width &&
	     y>faceLocation.y-0.3*faceLocation.height &&
	     y<faceLocation.y+1.3*faceLocation.height &&
	     skinmask.at<uchar>(y,x)!=0){

	    uchar lbl=ulbl.at<uchar>(y,x);
	  
	    if(lbl==facelbl){
	      headcmask.at<uchar>(y,x)=255;
	    }
	  }
	}
      
      bb[std::pair<int,std::string>(f,"headconstraintmask")]=headcmask;

      // headconstraintmask now constructed as certain rectangle around the face detection
    }


    if(bb.count(std::pair<int,std::string>(f,"nonfacemask"))==0){

      //


      Mat nonfacemask=skinmask.clone();

      // construct nonfacemask as the blobs not overlapping with head area
      
      if(false){

      for(int x=0;x<cols;x++)
	for(int y=0;y<rows;y++){
	  uchar lbl=ulbl.at<uchar>(y,x);
	  if(lbl==facelbl){
	    nonfacemask.at<uchar>(y,x)=0;
	  }
	}
      }

      // alternative: nonfacemask = skinmask \ dilated head

      if(true){
	Mat hcmask=boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"headconstraintmask")])->clone();

	int margin=20;

	cv::dilate(hcmask, hcmask, 
      		 cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(margin, margin)));	

	for(int x=0;x<cols;x++)
	  for(int y=0;y<rows;y++){
	    if(hcmask.at<uchar>(y,x)){
	      nonfacemask.at<uchar>(y,x)=0;
	    }

	  }
	
	// if blobtracks are defined for the frame
	// - invalidate all blobs that do not have trackers
	// - invalidate all areas that are too far away from trackers
	// - add back areas of hcmask that are within tracked rectangles

	if(bb.count(std::pair<int,std::string>(f,"blobtracks"))
&& bb.count(std::pair<int,std::string>(f,"trackedarea"))

){

	    set<uchar> trackedlbl;
	    map<int, vector<Rect> > *bt=boost::any_cast<map<int,vector<Rect> > >(&bb[std::pair<int,std::string>(f,"blobtracks")]);

	    Mat trackedarea=boost::any_cast<Mat>(&bb[std::pair<int,std::string>(f,"trackedarea")])->clone();

	    for(auto it=bt->begin();it!=bt->end();it++)
	      if(it->second.size())
		trackedlbl.insert(ulbl.at<uchar>(it->second[0].y,it->second[0].x));	  

	    for(int x=0;x<ulbl.cols;x++)
	      for(int y=0;y<ulbl.rows;y++){
		if(trackedlbl.count(ulbl.at<uchar>(y,x))==0)
		  nonfacemask.at<uchar>(y,x)=0;
		if(hcmask.at<uchar>(y,x)!=0 && trackedarea.at<uchar>(y,x))
		  nonfacemask.at<uchar>(y,x)=255;
	      }
	    
	    int trackermargin=30;

	    cv::dilate(trackedarea, trackedarea, 
		       cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(trackermargin, trackermargin)));	

	    for(int x=0;x<ulbl.cols;x++)
	      for(int y=0;y<ulbl.rows;y++){
		if(trackedarea.at<uchar>(y,x)==0)
		  nonfacemask.at<uchar>(y,x)=0;
	      }

	    // imshow("nonfacemask",nonfacemask);
	    // cv::waitKey(0);

	  }


      }

      bb[std::pair<int,std::string>(f,"nonfacemask")]=nonfacemask;
    }


    if(bb.count(std::pair<int,std::string>(f,"nonfaceconstraintmask"))==0){

      // prepare constraint mask for tracking nonface points
      // as the skin mask \ unoccludedheadmask

      Mat nonfacecmask=skinmask.clone();

      if(bb.count(std::pair<int,std::string>(f,"unoccludedheadfloatmask"))){

	cout << "turning off parts of nonfaceconstraintmask by unoccludedheadfloatmask" << endl;

	Mat occl=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"unoccludedheadfloatmask")]);

	for(int x=0;x<cols;x++)
	  for(int y=0;y<rows;y++)
	    if(occl.at<float>(y,x)>0.5)
	      nonfacecmask.at<uchar>(y,x)=0;

      }

      bb[std::pair<int,std::string>(f,"nonfaceconstraintmask")]=nonfacecmask;
    }
  }
						 

  void ensureFaceRectOnBB(int f,localBlackBoardType &bb){

    // this makes sure that the following keys on blackboard:
    // "facerect"

    if(bb.count(std::pair<int,std::string>(f,"facerect"))==0)
      //    bb[std::pair<int,std::string>(f,"facerect")]=vvi_black9Board->get<cv::Rect>(f, FACEDETECTOR_BLACKBOARD_ENTRY);
      throw string("facerect not found on local bb");


  }



    void ensureFrameOnBB(int f,localBlackBoardType &bb){

      //      cout << "ensuring frame " << f << endl;

      // this ensures the following keys
      // "frame","frame_padded","intensity_uchar","intensity_float"
      
      if(bb.count(std::pair<int,std::string>(f,"frame"))==0){

	bb[std::pair<int,std::string>(f,"frame")]=(*vvi_frameSource)[f];
      }

      ensurePaddedFramesOnBlackboard(f,bb);

      if(bb.count(std::pair<int,std::string>(f,"intensity_float"))==0){
	placeFloatIntensityOnBlackboard(f,bb);
      }

      //    cout << "ensuring uchar intendity for frame " << f << endl;

      if(bb.count(std::pair<int,std::string>(f,"intensity_uchar"))==0){
	Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"frame")]);
	Mat i=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"intensity_float")]);

      	Mat ic=Mat(frame.rows,frame.cols,CV_8UC1,Scalar::all(0));
	
	for(int x=0;x<frame.cols;x++)
	  for(int y=0;y<frame.rows;y++)
	    ic.at<uchar>(y,x)=i.at<float>(y+intensity_image_zeropad_margin,x+intensity_image_zeropad_margin);

	bb[std::pair<int,std::string>(f,"intensity_uchar")]=ic;

// 	imshow("intensity_uchar",ic);
// 	cv::waitKey(0);

      }
    }

   void ensureSalientPointsOnBB(int f,localBlackBoardType &bb){

    // this ensures the following keys
    // "pivotheadpoints","pivotnonfacepoints","pivotskinpoints"

     bool skinneeded=bb.count(std::pair<int,std::string>(f,"pivotskinpoints"))==0;
     bool headneeded=bb.count(std::pair<int,std::string>(f,"pivotheadpoints"))==0;
     bool nonfaceneeded=bb.count(std::pair<int,std::string>(f,"pivotnonfacepoints"))==0;

     if(!skinneeded && !headneeded && !nonfaceneeded) return;

     ensureFrameOnBB(f,bb);
     ensureMasksOnBB(f,bb);

     // detect 
     
     if(skinneeded)
       selectSalientPointsOfMask(f,"skinmask","pivotskinpoints",bb);

     if(!headneeded && !nonfaceneeded) return;

     Mat skinpt=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"pivotskinpoints")]);

     if(headneeded){
       Mat cmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"headconstraintmask")]);
       Mat headpt=skinpt.clone();

       for(int x=0;x<cmask.cols;x++)
	 for(int y=0;y<cmask.rows;y++)
	   if(cmask.at<uchar>(y,x)==0)
	     headpt.at<uchar>(y,x)=0;
       bb[std::pair<int,std::string>(f,"pivotheadpoints")]=headpt;
     }

     if(nonfaceneeded){
       Mat cmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"nonfacemask")]);
       Mat pt=skinpt.clone();

       for(int x=0;x<cmask.cols;x++)
	 for(int y=0;y<cmask.rows;y++)
	   if(cmask.at<uchar>(y,x)==0)
	     pt.at<uchar>(y,x)=0;
       bb[std::pair<int,std::string>(f,"pivotnonfacepoints")]=pt;
     }
   }


   void ensureMotionFieldOnBB(int f,localBlackBoardType &bb){

     // ensure existence of "motionfield" from frame f to the next frame

     if(bb.count(std::pair<int,std::string>(f,"motionfield"))==0){

       ensureFrameOnBB(f,bb);
       ensureFrameOnBB(f+1,bb);

       ensureMasksOnBB(f,bb);
       ensureMasksOnBB(f+1,bb);

       Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"frame")]);
       Mat  skinmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"skinmask")]);
       Mat  skinmask_next=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f+1,"skinmask")]);

       map<int,Mat> intensityimages;

       intensityimages[f]=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"intensity_uchar")]);
       intensityimages[f+1]=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f+1,"intensity_uchar")]);

       Mat motionfield;

       estimateMotionFieldbyGrid(skinmask,motionfield,f,f+1,intensityimages,skinmask_next,bb); 

       bb[std::pair<int,std::string>(f,"motionfield")]=motionfield;
     }

   }

  bool testIfOccluded(int f,localBlackBoardType &bb){
    // goes through the point list "trackednonfacepoints" 
    // and flags occlusion if any of the points falls into 
    // "headconstraintmask"

    if(bb.count(std::pair<int,std::string>(f,"trackednonfacepoints"))==0)
      return false;

    vector<Point> &v=*boost::any_cast<vector<Point> >(&bb[std::pair<int,std::string>(f,"trackednonfacepoints")]);

    ensureMasksOnBB(f,bb);

    Mat  cmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"headconstraintmask")]);

//     Mat vismask=cmask.clone();

//     for(auto it=v.begin();it!=v.end();it++)
//       vismask.at<uchar>(it->y,it->x)=(cmask.at<uchar>(it->y,it->x))?196:128;

//     imshow("occlusion test mask",vismask);
//     cv::waitKey(0);


    for(auto it=v.begin();it!=v.end();it++)
      if(cmask.at<uchar>(it->y,it->x))
	return true;

    return false;


  }

  bool testIfOccluded(int f,Mat &ptmask, localBlackBoardType &bb){
    // goes through the providedf point mask
    // and flags occlusion if any of the points falls into 
    // "headconstraintmask"

    ensureMasksOnBB(f,bb);

    Mat  cmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"headconstraintmask")]);

    for(int x=0;x<cmask.cols;x++)
      for(int y=0;y<cmask.rows;y++)
	if(cmask.at<uchar>(y,x) && ptmask.at<uchar>(y,x))
	  return true;

    return false;


  }


  void expandPointSetBySinglePointAnnealing(vector<Point> &srcv,vector<Point> &resv,
					  int f, const string &cmaskid,
					  int accCount,float prob,
					  localBlackBoardType &bb){

    // results are appended to res, it's not cleared

    // srcv and resv can be the same

     Mat f_zeropad=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"frame_padded")]);
     Mat cmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,cmaskid)]);

    int n=srcv.size();

    int templater=5;

    int maxdev=40;

    Rect rect;



    for(int i=0;i<n;i++){

      Point pt=srcv[i];

      rect.x=pt.x-templater+intensity_image_zeropad_margin;
      rect.y=pt.y-templater+intensity_image_zeropad_margin;
    
      rect.width=templater*2+1;
      rect.height=rect.width;

      Mat templ(f_zeropad,rect);

      int matchwindowr=templater+maxdev;

      rect.x=pt.x-matchwindowr+intensity_image_zeropad_margin;
      rect.y=pt.y-matchwindowr+intensity_image_zeropad_margin;

      rect.width=2*matchwindowr+1;
      rect.height=2*matchwindowr+1;

      Mat tgtmat(f_zeropad,rect);

      Mat result;
    
      matchTemplate(tgtmat,templ,result,CV_TM_SQDIFF);

      int acceptcount=0;
      int acceptuphillcount=0;
      int rejectcount=0;

    
      float T=-0.15/log(0.5);

      int iter=0;


      while(acceptcount<accCount && iter<1000){

      iter++;

//       cout << "iteration #" << iter << endl;

      Point newpt=pt;

    // perturb the location randomly

      float dev=5;

      newpt.x+=dev*randn();
      newpt.y+=dev*randn();

      //      visframe2=tgtf.clone();

      if(newpt.x<0 || newpt.x>=cmask.cols || newpt.y < 0 || newpt.y > cmask.rows 
	 || fabs(newpt.x-srcv[i].x)>=maxdev 
	 || fabs(newpt.y-srcv[i].y)>=maxdev
	 || cmask.at<uchar>(newpt.y,newpt.x)==0
)
	continue;

     //  visframe2.at<Vec3b>(newpt.y,newpt.x)=Vec3b(255,255,255);
//       imshow("candidate location",visframe2);
      

      float p;
      float E_old=result.at<float>(matchwindowr-templater+pt.y-srcv[i].y,
				   matchwindowr-templater+pt.x-srcv[i].x);
      float E_new=result.at<float>(matchwindowr-templater+newpt.y-srcv[i].y,
				   matchwindowr-templater+newpt.x-srcv[i].x);


//       cout << "E_old="<<E_old<<" E_new="<<E_new<<" relative diff="<<(E_new-E_old)/E_old<<endl;

    // with p=1 if energy lower than that of the old location
    // with p=exp((E_old-E_new)/(T*E_old)) if energy increases


    if(E_old>=E_new) p=1;
    else
      p=exp((E_old-E_new)/(T*E_old));
    
//     cout << "p="<<p<<endl;
//     cout << "T="<<T<<endl;

    bool accept=false;

    if(p==1){
      accept=true;
    }
    else if(rand(1000)<1000*p){
    // accumulate all the accepted translations

      accept=true;
      acceptuphillcount++;
    } else
      rejectcount++;

    if(accept){
      acceptcount++;
      pt=newpt;
      if(rand(1000)<prob*1000)
	resv.push_back(newpt);
    }


      }
    }
  }

  void interpolateLandmarksToBB(int first,int last,localBlackBoardType &bb){

    vector<float> widths;
    vector<float> heights;
    vector<bool> valid_landmarks(last-first+1,true);

   for(int n=first;n<=last;n++){
     if(bb.count(std::pair<int,std::string>(n,"faciallandmarks"))==0)
       throw string("faciallandmarks not found onlocal bb");

     //     const std::vector<cv::Point2f>& facialLandmarks = vvi_blackBoard->get<std::vector<cv::Point2f>>(n, FACIAL_LANDMARK_BLACKBOARD_ENTRY);

     //     BlackBoardPointer< std::vector<cv::Point2f> > facialLandmarks = 
     //       boost::any_cast<BlackBoardPointer<vector<Point2f> > >(bb[std::pair<int,std::string>(n,"faciallandmarks")]);


//   const std::vector<cv::Point2f>& facialLandmarks = 
//        *boost::any_cast<vector<Point2f> >(&bb[std::pair<int,std::string>(n,"faciallandmarks")]);


//      // the commented-out values seem to go unused //Matti
//      // const cv::Point2f* midpoint = &facialLandmarks[0];
//      const cv::Point2f* lEyei = &(*facialLandmarks)[1]; // inner
//      const cv::Point2f* rEyei = &(*facialLandmarks)[2]; // inner
//      const cv::Point2f* mouthl = &(*facialLandmarks)[3]; // lhs   
//      const cv::Point2f* mouthr = &(*facialLandmarks)[4]; // rhs   
//      const cv::Point2f* lEyeo = &(*facialLandmarks)[5]; // outer
//      const cv::Point2f* rEyeo = &(*facialLandmarks)[6]; // outer
//      // const cv::Point2f* nose = &facialLandmarks[7];

    const std::vector<cv::Point2f>& facialLandmarks = 
       *boost::any_cast<vector<Point2f> >(&bb[std::pair<int,std::string>(n,"faciallandmarks")]);

     const cv::Point2f* midpoint = &facialLandmarks[0];
     const cv::Point2f* lEyei = &facialLandmarks[1]; // inner
     const cv::Point2f* rEyei = &facialLandmarks[2]; // inner
     const cv::Point2f* mouthl = &facialLandmarks[3]; // lhs   
     const cv::Point2f* mouthr = &facialLandmarks[4]; // rhs   
     const cv::Point2f* lEyeo = &facialLandmarks[5]; // outer
     const cv::Point2f* rEyeo = &facialLandmarks[6]; // outer
     const cv::Point2f* nose = &facialLandmarks[7];


     float faceH=0.5*(mouthl->y+mouthr->y)-0.25*(lEyei->y+rEyei->y+lEyeo->y+rEyeo->y);
     float faceW=0.5*(rEyei->x+rEyeo->x-lEyei->x-lEyeo->x);
     
     heights.push_back(faceH);
     widths.push_back(faceW);

     //     cout << "collected face stats for frame" << n << endl;

   }


   int medidx=0.85*widths.size();
   nth_element(widths.begin(), widths.begin()+medidx,
	       widths.end() );
   nth_element(heights.begin(), heights.begin()+medidx,
	       heights.end() );

   float medianW=widths[medidx];
   float medianH=heights[medidx];


   for(int n=first;n<=last;n++){

     //     const std::vector<cv::Point2f>& facialLandmarks = vvi_blackBoard->get<std::vector<cv::Point2f>>(n, FACIAL_LANDMARK_BLACKBOARD_ENTRY);
     
     //     BlackBoardPointer<std::vector<cv::Point2f> > facialLandmarks = 
     //       boost::any_cast<BlackBoardPointer<vector<Point2f> > >(bb[std::pair<int,std::string>(n,"faciallandmarks")]);

//   const std::vector<cv::Point2f>& facialLandmarks = 
//        *boost::any_cast<vector<Point2f> >(&bb[std::pair<int,std::string>(n,"faciallandmarks")]);


//      // The commented-out values seem to go unused
//      // const cv::Point2f* midpoint = &facialLandmarks[0];
//      const cv::Point2f* lEyei = &(*facialLandmarks)[1]; // inner
//      const cv::Point2f* rEyei = &(*facialLandmarks)[2]; // inner
//      const cv::Point2f* mouthl = &(*facialLandmarks)[3]; // lhs   
//      const cv::Point2f* mouthr = &(*facialLandmarks)[4]; // rhs   
//      const cv::Point2f* lEyeo = &(*facialLandmarks)[5]; // outer
//      const cv::Point2f* rEyeo = &(*facialLandmarks)[6]; // outer
//      // const cv::Point2f* nose = &facialLandmarks[7];

 const std::vector<cv::Point2f>& facialLandmarks = 
       *boost::any_cast<vector<Point2f> >(&bb[std::pair<int,std::string>(n,"faciallandmarks")]);

     const cv::Point2f* midpoint = &facialLandmarks[0];
     const cv::Point2f* lEyei = &facialLandmarks[1]; // inner
     const cv::Point2f* rEyei = &facialLandmarks[2]; // inner
     const cv::Point2f* mouthl = &facialLandmarks[3]; // lhs   
     const cv::Point2f* mouthr = &facialLandmarks[4]; // rhs   
     const cv::Point2f* lEyeo = &facialLandmarks[5]; // outer
     const cv::Point2f* rEyeo = &facialLandmarks[6]; // outer
     const cv::Point2f* nose = &facialLandmarks[7];


     float faceH=0.5*(mouthl->y+mouthr->y)-0.25*(lEyei->y+rEyei->y+lEyeo->y+rEyeo->y);
     float faceW=0.5*(rEyei->x+rEyeo->x-lEyei->x-lEyeo->x);

     // this value seems to be unused //Matti
     // bool valid=true;
     if(faceH<0.8*medianH||faceW<0.8*medianW)
       valid_landmarks[n-first]=false;
   }

   // frames w/ invalid landmarks detected
   // now apply simple geometric interpolation

   //cout << "invalid landmarks detected" << endl;

   int lastvalidframe;
   int firstvalidframe;
   
   // special case: copy the first valid landmark locations 
   // for the invalid frames in the beginning of the sequence

   for(firstvalidframe=first;firstvalidframe<=last&&valid_landmarks[firstvalidframe-first]==false;
       firstvalidframe++);
 
   bool interpolation_done=false;

   if(firstvalidframe<=last){
     const std::vector<cv::Point2f>& facialLandmarks = 
       *boost::any_cast<vector<Point2f> >(&bb[std::pair<int,std::string>(firstvalidframe,"faciallandmarks")]);
     //     const std::vector<cv::Point2f>& facialLandmarks = vvi_blackBoard->get<std::vector<cv::Point2f>>(firstvalidframe, FACIAL_LANDMARK_BLACKBOARD_ENTRY);
     
     for(int f=first;f<firstvalidframe;f++)
       bb[std::pair<int,std::string>(f,"faciallandmarks")]=facialLandmarks;
   } else
     interpolation_done=true;

   while(interpolation_done==false){
     for(lastvalidframe=firstvalidframe;lastvalidframe<=last-1&&valid_landmarks[lastvalidframe+1-first]==true;lastvalidframe++);

     //     cout << "valid interval from " << firstvalidframe << " to " << lastvalidframe << " found" << endl;

     // copy the landmarks of the valid interval to localbb
     for(int f=firstvalidframe;f<=lastvalidframe;f++){
       //       const std::vector<cv::Point2f>& facialLandmarks = vvi_blackBoard->get<std::vector<cv::Point2f>>(f, FACIAL_LANDMARK_BLACKBOARD_ENTRY);
       //       bb[std::pair<int,std::string>(f,"faciallandmarks")]=facialLandmarks;
     }

     if(lastvalidframe==last){
       interpolation_done=true;
     } else{
       
       // scan for the beginning of the next valid interval

       for(firstvalidframe=lastvalidframe+1;firstvalidframe<=last&&valid_landmarks[firstvalidframe-first]==false;firstvalidframe++);

       // handle the special case when no more valid intervals were found

       if(firstvalidframe>last){
	 interpolation_done=true;
	 //	 const std::vector<cv::Point2f>& facialLandmarks = vvi_blackBoard->get<std::vector<cv::Point2f>>(lastvalidframe, FACIAL_LANDMARK_BLACKBOARD_ENTRY);	 
     const std::vector<cv::Point2f>& facialLandmarks = 
       *boost::any_cast<vector<Point2f> >(&bb[std::pair<int,std::string>(lastvalidframe,"faciallandmarks")]);

	 for(int f=lastvalidframe+1;f<=last;f++)
	   bb[std::pair<int,std::string>(f,"faciallandmarks")]=facialLandmarks;
       }
       else{

	 // preform the actual interpolation between two valid landmark sets
	 //	 const std::vector<cv::Point2f>& fl1 = vvi_blackBoard->get<std::vector<cv::Point2f>>(lastvalidframe, FACIAL_LANDMARK_BLACKBOARD_ENTRY);	 
	 //	 const std::vector<cv::Point2f>& fl2 = vvi_blackBoard->get<std::vector<cv::Point2f>>(firstvalidframe, FACIAL_LANDMARK_BLACKBOARD_ENTRY);	 
     const std::vector<cv::Point2f>& fl1 = 
       *boost::any_cast<vector<Point2f> >(&bb[std::pair<int,std::string>(lastvalidframe,"faciallandmarks")]);
     const std::vector<cv::Point2f>& fl2 = 
       *boost::any_cast<vector<Point2f> >(&bb[std::pair<int,std::string>(firstvalidframe,"faciallandmarks")]);


	 for(int f=lastvalidframe+1;f<firstvalidframe;f++){

	   float len=firstvalidframe-lastvalidframe;
	   float frac=(f-lastvalidframe)/len;

	   vector<cv::Point2f> facialLandmarks = fl1;
	   for(size_t n=0;n<fl1.size();n++)
	     facialLandmarks[n]=(1-frac)*fl1[n]+frac*fl2[n];
	   
	   bb[std::pair<int,std::string>(f,"faciallandmarks")]=facialLandmarks;
	 }

       }

     }

   }

   // as a check, print out the numbers of the frames for which landmarks are stored

   for(int f=first;f<=last;f++){
     if(bb.count(std::pair<int,std::string>(f,"faciallandmarks"))){
       //        cout << "landmarks stored for frame " << f << endl;
     }
     else
       	cout << "landmarks NOT stored for frame " << f << endl;
   }


   
  }

  void findFaceMaskByLandmarks(int f,const string &cmaskid, localBlackBoardType &bb){

    //    ensureFrameOnBB(f,bb);
    // ensureMasksOnBB(f,bb);


    vector<Point2f> landmarks=*boost::any_cast<vector<Point2f> >(&bb[std::pair<int,std::string>(f,"faciallandmarks")]);

    const cv::Point2f* midpoint = &landmarks[0];
    const cv::Point2f* lEyei = &landmarks[1]; // inner
    const cv::Point2f* rEyei = &landmarks[2]; // inner
    const cv::Point2f* mouthl = &landmarks[3]; // lhs   
    const cv::Point2f* mouthr = &landmarks[4]; // rhs   
    const cv::Point2f* lEyeo = &landmarks[5]; // outer
    const cv::Point2f* rEyeo = &landmarks[6]; // outer
    // Nose seems to go unused //Matti
    // const cv::Point2f* nose = &landmarks[7];
      
    Point2f betweeneyes(0.25*(lEyei->x+lEyeo->x+rEyei->x+rEyeo->x),0.25*(lEyei->y+lEyeo->y+
								   rEyei->y+rEyeo->y));

    float dx=betweeneyes.x-midpoint->x;
    float dy=midpoint->y-betweeneyes.y;



    float alpha=atan(dx/dy);

    //    cout <<"alpha= "<<alpha<<" sin(a)="<<sin(alpha)<<" cos(a)="<<cos(alpha)<<endl;

    // the tilting angle of the face

    float l=sqrt(dx*dx+dy*dy);

    dx=rEyeo->x-lEyeo->x;
    dy=rEyeo->y-lEyeo->y;

    float W=sqrt(dx*dx+dy*dy);

    float L=2.6*l;
    
    Point2f x1=betweeneyes;
    x1.y -= L * cos(alpha);
    x1.x += L * sin(alpha);

    Point2f x1p=x1;
    Point2f x1pp=x1;

    x1p.x += 0.75*W*cos(alpha);
    x1p.y += 0.75*W*sin(alpha);

    x1pp.x -= 0.75*W*cos(alpha);
    x1pp.y -= 0.75*W*sin(alpha);

    Point2f x2=betweeneyes;
    x2.x += 1*W*cos(alpha);
    x2.y += 1*W*sin(alpha);

    Point2f x2p=betweeneyes;
    x2p.x -= 1*W*cos(alpha);
    x2p.y -= 1*W*sin(alpha);

    float l2=(0.5*(mouthl->y+mouthr->y)-midpoint->y)/cos(alpha);

    Point2f mouthlevel=*midpoint;
    mouthlevel.y += l2 * cos(alpha);
    mouthlevel.x -= l2 * sin(alpha);

    Point2f jawlevel=*midpoint;
    jawlevel.y += 2.95*l2 * cos(alpha);
    jawlevel.x -= 2.95*l2 * sin(alpha);

    Point2f x3=mouthlevel,x3p=mouthlevel;
    x3.x += 0.90*W*cos(alpha);
    x3.y += 0.90*W*sin(alpha);

    x3p.x -= 0.90*W*cos(alpha);
    x3p.y -= 0.90*W*sin(alpha);

    Point2f x4=jawlevel,x4p=jawlevel;
    x4.x += 0.40*W*cos(alpha);
    x4.y += 0.40*W*sin(alpha);

    x4p.x -= 0.40*W*cos(alpha);
    x4p.y -= 0.40*W*sin(alpha);


    Mat frame=(*vvi_frameSource)[f];

    //    Mat skin=vvi_blackBoard->get<cv::Mat>(f, SKINDETECTOR_BLACKBOARD_MASK_ENTRY);

	Mat  skin=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"skinmask")]);

//          Mat visframe=frame.clone();

     vector<Scalar> col(8);

     col[0]=CV_RGB(255,255,255);
     col[1]=CV_RGB(255,255,0);
     col[2]=CV_RGB(255,0,255);
     col[3]=CV_RGB(0,255,255);
     col[4]=CV_RGB(255,0,0);
     col[5]=CV_RGB(0,255,0);
     col[6]=CV_RGB(0,0,255);
     col[7]=CV_RGB(0,128,255);

//      if(valid_landmarks[n-first]==false)
//        cout << "landmarks interpolated" << endl;

//      for(int i=0;i<8;i++)
//        cv::circle(visframe, landmarks[i], 4, col[i]);

//      cv::circle(visframe, x1, 4, col[0]);
//      cv::circle(visframe, x1p, 4, col[0]);
//      cv::circle(visframe, x1pp, 4, col[0]);
//      cv::circle(visframe, x2, 4, col[0]);
//      cv::circle(visframe, x2p, 4, col[0]);
//      cv::circle(visframe, x3, 4, col[0]);
//      cv::circle(visframe, x3p, 4, col[0]);
//      cv::circle(visframe, x4, 4, col[0]);
//      cv::circle(visframe, x4p, 4, col[0]);

//      imshow("interpolated facial landmarks",visframe);
//     cv::waitKey(0);

     Mat cmask(frame.rows,frame.cols,CV_8UC1,Scalar::all(0));

     int npts[1];
     npts[0]=9;
     Point *pt=new Point[9];
     pt[0]=x1;
     pt[1]=x1p;
     pt[2]=x2;
     pt[3]=x3;
     pt[4]=x4;
     pt[5]=x4p;
     pt[6]=x3p;
     pt[7]=x2p;
     pt[8]=x1pp;

     const Point *ptc=pt;

     const Point **p=&ptc;

     fillPoly(cmask, p, npts, 1, Scalar::all(255));

     for(int x=0;x<frame.cols;x++)
       for(int y=0;y<frame.rows;y++)
	 if(skin.at<uchar>(y,x)==0) cmask.at<uchar>(y,x)=0;

   //    for(int x=0;x<frame.cols;x++)
//         for(int y=0;y<frame.rows;y++)
//  	 visframe.at<Vec3b>(y,x)[0]=(cmask.at<uchar>(y,x)>0)?255:0;



      //      imshow("constructed head constraint mask",visframe);
      //  cv::waitKey(0);

    bb[std::pair<int,std::string>(f,cmaskid)]=cmask;
    cout << "hedconstraintmask constructed for frame " << f<< endl;
  }


  void deleteNearbyGridletsFromTrackerState(int f,localBlackBoardType &bb){

    vector<vector<Point> > *trackedgridpoints;

    if(bb.count(std::pair<int,std::string>(f,"trackerstate_nonface"))==0)
      return;

    Mat *occl=NULL;
    if(bb.count(std::pair<int,std::string>(f,"unoccludedheadfloatmask")))
      occl=boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"unoccludedheadfloatmask")]);
    
    if(!occl) return;

    Mat m(occl->rows,occl->cols,CV_8UC1,Scalar::all(0));

    for(int x=0;x<m.cols;x++)
      for(int y=0;y<m.rows;y++)
	if(occl->at<float>(y,x)>0.5) m.at<uchar>(y,x)=255;

    int margin=7;

    cv::dilate(m, m, 
	       cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(margin, margin)));

    trackedgridpoints=boost::any_cast<vector<vector<Point> > >(&bb[std::pair<int,std::string>(f,"trackerstate_nonface")]);

    vector<vector<Point> >v=*trackedgridpoints;

    trackedgridpoints->clear();


    for(auto git=v.begin();git!=v.end();git++){
	bool invalid=false;

 	for(auto it=git->begin();it!=git->end();it++)
	  if(m.at<uchar>(it->y,it->x)){
	    invalid=true;
	    break;
	  }

 	if(!invalid)
 	 trackedgridpoints->push_back(*git);
    }
     
  }

  void deleteNonMaskGridletsFromTrackerState(int f, Mat &dmask,localBlackBoardType bb){
    vector<vector<Point> > *trackedgridpoints;

    if(bb.count(std::pair<int,std::string>(f,"trackerstate_nonface"))==0)
      return;

    trackedgridpoints=boost::any_cast<vector<vector<Point> > >(&bb[std::pair<int,std::string>(f,"trackerstate_nonface")]);

    vector<vector<Point> >v=*trackedgridpoints;

    trackedgridpoints->clear();


    for(auto git=v.begin();git!=v.end();git++){
	bool valid=false;

 	for(auto it=git->begin();it!=git->end();it++)
	  if(dmask.at<uchar>(it->y,it->x)){
	    valid=true;
	    break;
	  }

 	if(valid)
 	 trackedgridpoints->push_back(*git);
    }
     

  }


  void expandMaskSequence(int first, int last, const string &srcid, const string &tgtid,
			  int spatialext, int temporalext, localBlackBoardType &bb){

    ensureFrameOnBB(first,bb);

    Mat frame0=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(first,"frame")]);

    // this value does not seem to be used //Matti
    // int n=last-first+1;

    map<int,Mat> tmpseq;

    for(int f=first;f<=last;f++){
      if(bb.count(std::pair<int,std::string>(f,srcid))==0) 
	tmpseq[f]=Mat(frame0.rows,frame0.cols,CV_8UC1,Scalar::all(0));
      else{
	tmpseq[f]=boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,srcid)])->clone();
	cv::dilate(tmpseq[f], tmpseq[f], 
	
	   cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(spatialext, spatialext)));
      }

      // allocate space for the result
      bb[std::pair<int,std::string>(f,tgtid)]=Mat(frame0.rows,frame0.cols,CV_8UC1,Scalar::all(0));

    }

    // spatial expansion done, now temporal

    // interchange the temporal and spatial x dimensions
    // in the sequence for handier data access
    
    int nrows=frame0.rows; // for each row, there's going to be a separate matrix
    int nframes=last-first+1;
    
    int *kval=new int[2*temporalext+1];

    for(int i=0;i<2*temporalext+1;i++)
      kval[i]=1;

    Mat rowmat(nframes,frame0.cols,CV_8UC1,Scalar::all(0));

    cout << "about to construct kernel" << endl;

    IplConvKernel* k=cvCreateStructuringElementEx(1, 2*temporalext+1,0,temporalext,CV_SHAPE_CUSTOM,
						  kval);

    cout << "...done" << endl;

    for(int r=0;r<nrows;r++){

      bool nonzeros=false;

      for(int f=first;f<=last;f++)
	for(int x=0;x<tmpseq[f].cols;x++)
	  {
	    if((rowmat.at<uchar>(f-first,x)=tmpseq[f].at<uchar>(r,x)))
	      nonzeros=true;
	  }

      if(nonzeros){
       cout << "constructed row mat for row " << r << endl;

//       imshow("before filtering",rowmat);
//       cv::waitKey(0);


      CvMat cvimg = rowmat;
 
      // filter rowmat "vertically" (->temporally)
      cvDilate(&cvimg,&cvimg,k);

//       imshow("after filtering",rowmat);
//       cv::waitKey(0);


      // copy filtered results to sequence tgtid on bb

      for(int f=first;f<=last;f++){
	Mat tgtmat=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,tgtid)]);
	for(int x=0;x<tmpseq[f].cols;x++)
	  tgtmat.at<uchar>(r,x)=rowmat.at<uchar>(f-first,x);   
      }
      }

    }



    cvReleaseStructuringElement(&k);
    delete[] kval;


  }

  bool parseCommaSeparatedStringToList(const string &str, vector<string> &l){
    l.clear();
    
    size_t startpos=0;
    size_t len=0;

    for(size_t i=0;i<str.length();i++){
      if(str[i]==','){
	l.push_back(str.substr(startpos,len));
	startpos=i+1;
	len=0;
      } else
	len++;
    }
    l.push_back(str.substr(startpos,len));

    return true;
  }

  void showMotionField(Mat &m){

    Mat v(m.rows,m.cols,CV_8UC3,Scalar::all(0));

    for(int x=0;x<m.cols;x++)
      for(int y=0;y<m.rows;y++){
	v.at<Vec3b>(y,x)[0]=128+m.at<Point2f>(y,x).x;
	v.at<Vec3b>(y,x)[1]=128+m.at<Point2f>(y,x).y;
      }
    
    cv::imshow("motion field",v);


  }

  void trackNeighbourhoods(int first, int last, localBlackBoardType &bb){

    // assume that bb readily contains the following keys for all the frames:

    // "frame"
    // "skinmask"
    // "facerect"
    // "uniqueblobintmask"
    // "blobprogression"
    // "blobinterpretation


    vector<Point> sp;
//       cout << "bbox.width=" << bbox.width <<"bbox.height=" << bbox.height << endl;
      

    BlobProgression bp=*boost::any_cast<BlobProgression>(&bb[std::pair<int,std::string>(0,"blobprogression")]);
    

    map<int,blobinterpretation_type> interp=*boost::any_cast<map<int,blobinterpretation_type> >(&bb[std::pair<int,std::string>(0,"blobinterpretation")]);
    
    // scan the blob interpretation 
    // in order to find first occasion of a blob merging with the head blob

    int firstmerge=-1;

    for(size_t frameidx=fmax(first-bp.firstframe,0); firstmerge<0 && 
	  frameidx<bp.blobs.size();frameidx++){
      for(size_t blobidx=0;blobidx<bp.blobs[frameidx].size();blobidx++){
	BlobInfo *bi=&(bp.blobs[frameidx][blobidx]);
	if(interp[bi->label].face_overlap && 
	   (interp[bi->label].left_hand || interp[bi->label].right_hand)){
	  firstmerge=frameidx+bp.firstframe;
	  break;
	}
      }
    }

    if(firstmerge<=last){

      // show the first merge frame for a check

      //      Mat uniqueblobintmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"uniqueblobintmask")]);
      
      Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(firstmerge,"frame")]);
      cv::imshow("first merge frame",frame);
      cv::waitKey(0);
    }


    for(int f=fmax(first,firstmerge-10);f<firstmerge-1;f++){
       trackFaceGridlet(f,f+1,bb); // these accumulate point clouds on blackboard

       constructFloatMaskFromPointCloud(f+1,f+1,"unoccludedheadpoints","unoccludedheadfloatmask",bb,true);
    }

    
    // from here on, track the left hand

    int lhlbl;

    for(size_t blobidx=0;blobidx<bp.blobs[firstmerge-1-bp.firstframe].size();blobidx++){
      BlobInfo *bi=&(bp.blobs[firstmerge-1-bp.firstframe][blobidx]);
      if(interp[bi->label].left_hand){
	lhlbl=bi->label;
	break;
      }
    }

    vector<Rect> trackedRect;

    int nrrects=22;

    selectRectanglesToTrack(firstmerge-1,lhlbl,nrrects,trackedRect,bb);

    Mat frame0=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(firstmerge-1,"frame")]);

    spiralDeltas(sp,fmin(frame0.cols,frame0.rows));


    Mat dummymask(frame0.rows,frame0.cols,CV_8UC1,Scalar::all(255));

    for(int f=firstmerge;f<=last;f++){

      trackFaceGridlet(f-1,f,bb); // these accumulate point clouds on blackboard

      constructFloatMaskFromPointCloud(f,f,"unoccludedheadpoints","unoccludedheadfloatmask",bb,true);


      Mat tgt_skin=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"skinmask")]);

      Mat *cmask=new Mat;
      *cmask=tgt_skin.clone();

      Mat uniqueblobintmask_ref=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f-1,"uniqueblobintmask")]);
      Mat uniqueblobintmask_tgt=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"uniqueblobintmask")]);
     
      int reflbl=uniqueblobintmask_ref.at<int>(trackedRect[0].y,trackedRect[0].x);


      // prepare a constraintmask for tracking using the 
      // following contraints



      // 1) if label of tracked rectangles continues in the new frame, prohibit 
      //    all other blobs

      if(bp.lbl2blobidx[f].count(reflbl)){
	for(int x=0;x<cmask->cols;x++)
	  for(int y=0;y<cmask->rows;y++)
	    if(uniqueblobintmask_tgt.at<int>(y,x)!=reflbl)
	      cmask->at<uchar>(y,x)=0;
      }

      // 2) if there are two blobs besides the head blob, prohibit the head blob
      // (in the future, this would be also the place for refreshing the
      // rectangle selection)

      int nonheadcount=0;
      int headlbl=-1;

      for(auto it=bp.lbl2blobidx[f].begin(); it!=bp.lbl2blobidx[f].end();it++){
	int lbl=it->first;
	if(interp[lbl].head)
	  headlbl=lbl;
	else
	  nonheadcount++;
      }

      if(headlbl != -1 && nonheadcount>1){
	for(int x=0;x<cmask->cols;x++)
	  for(int y=0;y<cmask->rows;y++)
	    if(uniqueblobintmask_tgt.at<int>(y,x)==headlbl)
	      cmask->at<uchar>(y,x)=0;
      }

      // 3) in the future, prohibit also the blob tracked by the other hand tracker


//       imshow("initial constraint mask",*cmask);
//       cv::waitKey(0);

      // try to shrink the mask by the minumum radius of the rectangles

      int minrad=9999;

      Mat *smask=new Mat;

      for(auto it=trackedRect.begin();it!=trackedRect.end();it++)
	minrad=fmin(minrad,0.5*(it->width-1));

      if(minrad>1){

	cout << "thinning according to minrad="<<minrad<<endl;

	cv::erode(*cmask, *smask,cv::getStructuringElement(cv::MORPH_ELLIPSE, 
							   Size(minrad,minrad)));

	// check for the unlikely case where the thinned mask disappears

	bool nonzero=false;

	for(int x=0;!nonzero && x<smask->cols;x++)
	  for(int y=0;y<smask->rows;y++)
	    if(smask->at<uchar>(y,x)){
	      nonzero=true;
	      break;
	    }

	if(nonzero){
	  delete cmask;
	  cmask=smask;
	}
	else{
	  delete smask;
	}
      }

      Mat occl=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"unoccludedheadfloatmask")]);

      for(int x=0;x<cmask->cols;x++)
	  for(int y=0;y<cmask->rows;y++)
	    if(occl.at<float>(y,x)>0.5)
	      cmask->at<uchar>(y,x)=0;



      // maybe in the future we should add a checking step that replaces the 
      // blob continuity analysis by area overlaps with checks by contents match (tracking)

      

      

      cout << "tracking from frame " << f-1 << " to " << f << endl;

      vector<Rect> origRect=trackedRect;

//       imshow("constraint mask after thinning",*cmask);
//       cv::waitKey(0);

      // initialise tracking by looking for optimal common displacement for all the
      // rectangles

      float res=trackRectangleSetBetweenFrames(trackedRect,f-1,f,bb,cmask);

      // this may fail due to the constraint mask making such displacement inexistent

      if(res<0){
	// in such case
	// initialise track by tracking each rectangle individually

	trackedRect=origRect;

	for(auto it=trackedRect.begin();it!=trackedRect.end();it++)
	  trackRectangleBetweenFrames(*it,f-1,f,bb,cmask);

	// and
	// here we (could) will take the median displacement and 
	// use it to initialise the tracking of all rectangles

	vector<int> dx,dy;
      
	for(int i=0;i<static_cast<int>(trackedRect.size());i++){
	  dx.push_back(trackedRect[i].x-origRect[i].x);
	  dy.push_back(trackedRect[i].y-origRect[i].y);
	}
	
	sort(dx.begin(),dx.end());
	sort(dy.begin(),dy.end());

	int dxMed=dx[0.5*dx.size()];
	int dyMed=dy[0.5*dy.size()];
	
	cout << "median displacement: ("<<dxMed<<","<<dyMed<<")"<<endl;
	
	for(int i=0;i<static_cast<int>(trackedRect.size());i++){
	  trackedRect[i].x = origRect[i].x + dxMed;
	  trackedRect[i].y = origRect[i].y + dyMed;
	}

      }
    
	// then refine tracking by greedy local iteration

      grid_type grid;
      vector<int> radius;

      for(int i=0;i<static_cast<int>(trackedRect.size());i++){
	gridpoint_type p;

	p.refx=origRect[i].x+0.5*(origRect[i].width-1);
	p.refy=origRect[i].y+0.5*(origRect[i].height-1);

	p.tgtx=trackedRect[i].x+0.5*(trackedRect[i].width-1);
	p.tgty=trackedRect[i].y+0.5*(trackedRect[i].height-1);

	grid.push_back(p);
	radius.push_back(0.5*(origRect[i].width-1));
      }


      set_nbr_fullconnectivity(grid);
      set_l0(grid);

    Mat reff=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f-1,"frame")]);
    Mat tgtf=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"frame")]);




//     Mat visref=reff.clone();
//     Mat vistgt=tgtf.clone();

//     showgrid(visref,grid,true,false,true,false);
//     showgrid(vistgt,grid,false,true,false,true);

//     imshow("grid in reference frame",visref);
//     imshow("grid in target frame",vistgt);
//     cv::waitKey(0);

    float alpha=1.0/400;

    int searchr=100;

    vector<bool> needs_update(grid.size(),true);

    int needupdatecount=grid.size();

    ensurePaddedFramesOnBlackboard(f-1,bb);
    ensurePaddedFramesOnBlackboard(f,bb);
 

    Mat reff_zeropad=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f-1,"frame_padded")]);
    Mat tgtf_zeropad=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"frame_padded")]);


    int iterationcount=0;

    int maxiter=600;

    do{

      iterationcount++;

      int r;
      do{
	r=rand(grid.size());
      } while(!needs_update[r]);

      Point oldtgt(grid[r].tgtx,grid[r].tgty);

      cout << "updating grid point " << r << endl; 

//       Mat visb4=tgtf.clone();
//       showgrid(visb4,grid,false,true,false,true);
//       cv::circle(visb4,Point(grid[r].tgtx,grid[r].tgty), 5, CV_RGB(0,255,0));



      updateGridPointGreedyLocal(grid,r,reff_zeropad,tgtf_zeropad,*cmask,2*radius[r]+1,searchr,alpha,1,30,false);

      needs_update[r]=false;
      needupdatecount--;

      if(oldtgt.x!=grid[r].tgtx || oldtgt.y !=grid[r].tgty)
	for(auto it=grid[r].nbr.begin(); it!=grid[r].nbr.end();it++)
	  if(needs_update[*it]==false){
	    needupdatecount++;
	    needs_update[*it]=true;
	  }
//       Mat visref=reff.clone();
//       showgrid(visref,grid,true,false,true,false);
//       cv::circle(visref,Point(grid[r].refx,grid[r].refy), 5, CV_RGB(0,255,0));
//       Mat vistgt=tgtf.clone();
//       showgrid(vistgt,grid,false,true,false,true);
//       cv::circle(vistgt,Point(grid[r].tgtx,grid[r].tgty), 5, CV_RGB(0,255,0));

//       //    imshow("grid in target frame before",visb4);
//       imshow("grid in target frame",vistgt);
//       imshow("grid in reference frame",visref);
//       cv::waitKey(0);

    } while(iterationcount<maxiter && needupdatecount>0);
    

    // transfer the tracking results back from grid to trackedRect
    // prune out points that are tracked to blobs other than the most common blob

    trackedRect.clear();

    Mat uniqueblobintmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"uniqueblobintmask")]);

    map<int,int> labelcounts;

    for(int i=0;i<static_cast<int>(grid.size());i++){
      int lbl=uniqueblobintmask.at<int>(grid[i].tgty,grid[i].tgtx);
      labelcounts[lbl]++;
    }

    int maxcount=-1;
    int maxlbl=-1;

    for(auto it=labelcounts.begin();it!=labelcounts.end();it++)
      if(it->second>maxcount){
	maxcount=it->second;
	maxlbl=it->first;
      }

    cout << "majority lbl is " << maxlbl << " with " << maxcount << " occurrences" << endl;

    for(int i=0;i<static_cast<int>(grid.size());i++){
      int lbl=uniqueblobintmask.at<int>(grid[i].tgty,grid[i].tgtx);
      if(lbl==maxlbl){
	Rect r;

	// here we should shrink radii of rectangles that have drifted towards the 
	// borders of skin region

	for(int ii=0;ii<static_cast<int>(sp.size());ii++){
	  int rr=fmax(fabs(sp[ii].x),fabs(sp[ii].y));
	  if(rr>radius[i]) break;
	  if( grid[i].tgtx+sp[ii].x<0 ||
	      grid[i].tgtx+sp[ii].x>=tgt_skin.cols ||
	      grid[i].tgty+sp[ii].y<0 ||
	      grid[i].tgty+sp[ii].y>=tgt_skin.rows ||
	      tgt_skin.at<uchar>(grid[i].tgty+sp[ii].y,grid[i].tgtx+sp[ii].x)==0){

	    cout << "shrinking radius " << i << " from " << radius[i] << " to ";

	    radius[i]=rr-1;
	    cout << radius[i] << endl;
	    break;
	  }
	}

	r.x=grid[i].tgtx-radius[i];
	r.y=grid[i].tgty-radius[i];
	r.width=2*radius[i]+1;
	r.height=2*radius[i]+1;

	if(radius[i]>2)
	  trackedRect.push_back(r);
      }
    }

    Mat visframe=tgtf.clone();

    for(auto it=trackedRect.begin();it!=trackedRect.end();it++)
      cv::rectangle(visframe,it->tl(),it->br(),CV_RGB(255,0,0));

    // here test for the possibility to refresh the rectangle selection

      nonheadcount=0;

      for(auto it=bp.lbl2blobidx[f].begin(); it!=bp.lbl2blobidx[f].end();it++){
	int lbl=it->first;
	if(!interp[lbl].head)
	  nonheadcount++;
      }

      if(nonheadcount>1){
	selectRectanglesToTrack(f,maxlbl,nrrects,trackedRect,bb);
	for(auto it=trackedRect.begin();it!=trackedRect.end();it++)
	  cv::rectangle(visframe,it->tl(),it->br(),CV_RGB(0,255,0));
      }

    


    imshow("tracked rectangles",visframe);
    cv::waitKey(0);

    delete cmask;


    }

    
  }

  void selectRectanglesToTrack(int frame, int lbl, int nrRects, std::vector<cv::Rect> &result,
			       localBlackBoardType &bb){

    Mat uniqueblobintmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frame,"uniqueblobintmask")]);

//       showDeterministicFalseColour32bit("uniqueblobintmask", uniqueblobintmask);
//       cv::waitKey(0);


//     set<int> ulbl;

//     for(int x=0;x<uniqueblobintmask.cols;x++)
//       for(int y=0;y<uniqueblobintmask.rows;y++)
// 	ulbl.insert(uniqueblobintmask.at<int>(y,x));

//     cout << "unique labels:";

//     for(auto it=ulbl.begin();it!=ulbl.end();it++)
//       cout << " " << *it;
//     cout << endl;

//     cout << "selection label: " << lbl << endl; 
      
//        Mat fr=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frame,"frame")]);

//        Mat visframe=fr.clone();

      vector<Point> center;

      vector<int> radius;

      // idea for selecting rectangles

      // 1. randomly select 3 times the final number of rectangle centers within the blob
      // 2. find out the radius of the maximal within-blob rectangle around each point
      // 3. keep the points with largest radii

      Rect bbox;

      Point br(0,0);

      for(int x=0;x<uniqueblobintmask.cols;x++)
	for(int y=0;y<uniqueblobintmask.cols;y++)
	  if(uniqueblobintmask.at<int>(y,x)==lbl){
	    bbox.x=fmin(bbox.x,x);
	    bbox.y=fmin(bbox.x,x);
	    br.x=fmax(br.x,x);
	    br.y=fmax(br.y,y);
	  }

      bbox.width=br.x-bbox.x+1;
      bbox.height=br.y-bbox.y+1;

      for(int i=0;i<3*nrRects;i++){
	int x=bbox.x+rand(bbox.width);
	int y=bbox.y+rand(bbox.height);
	if(uniqueblobintmask.at<int>(y,x)!=lbl){
	  i--;
	  continue;
	}
	
	center.push_back(Point(x,y));

      }

      // visualise the result

//       for(auto it=center.begin();it!=center.end();it++)
// 	visframe.at<Vec3b>(it->y,it->x)=Vec3b(255,255,255);

//       cv::imshow("center point candidates",visframe);
//       cv::waitKey(0);


      static vector<Point> sp;

//       cout << "bbox.width=" << bbox.width <<"bbox.height=" << bbox.height << endl;
      if(sp.size()==0)
	spiralDeltas(sp,fmin(uniqueblobintmask.cols,uniqueblobintmask.rows));

//       cout << "generated " << sp.size() << " deltas" << endl;

      for(int i=0;i<static_cast<int>(center.size());i++){
	radius.push_back(0);
	for(int ii=0;ii<static_cast<int>(sp.size());ii++)
	  if(center[i].x+sp[ii].x < 0 ||
	     center[i].x+sp[ii].x >= uniqueblobintmask.cols ||
	     center[i].y+sp[ii].y < 0 ||
	     center[i].y+sp[ii].y >= uniqueblobintmask.rows ||
	     uniqueblobintmask.at<int>(center[i].y+sp[ii].y,center[i].x+sp[ii].x)!=lbl){
	    radius[i]=fmin(fabs(sp[ii].x),fabs(sp[ii].y));
	    break;
	  }
      }

      // sort the list of radii

      vector<int> sorted=radius;
      sort(sorted.begin(),sorted.end());

//       cout << "sorted radii:" << endl;
//       for(auto it=sorted.begin();it!=sorted.end();it++)
// 	cout << *it << " ";
//       cout << endl;

      int rtresh=sorted[0.66*sorted.size()];

      result.clear();
      
      for(int i=0;i<static_cast<int>(center.size());i++){
	Rect rect;
	if(radius[i]<rtresh) continue;

	int margin=5;
	radius[i] -= margin;

	if(radius[i]>1){

	  rect.x=center[i].x-radius[i];
	  rect.width=2*radius[i]+1;
	  
	  rect.y=center[i].y-radius[i];
	  rect.height=2*radius[i]+1;
	  
	  result.push_back(rect);
	}
	//	cv::rectangle(visframe,rect.tl(),rect.br(),CV_RGB(255,0,0));

      }

//        cv::imshow("selected rectangles",visframe);
//        cv::waitKey(0);


  }

  float trackRectangleBetweenFrames(Rect &rect,int refframe, int tgtframe,localBlackBoardType &bb, Mat *cmask){

    // the coordinates in pt get replaced by the new ones in the next frame

    Mat reff=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,"frame")]);
    Mat tgtf=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"frame")]);

    ensurePaddedFramesOnBlackboard(refframe,bb);
    ensurePaddedFramesOnBlackboard(tgtframe,bb);
 

    Mat reff_zeropad=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,"frame_padded")]);
    Mat tgtf_zeropad=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"frame_padded")]);

    // initialise tracked rect to the old location of rect

    Rect trackedrect = rect;

    // simple idea: find out the template matching neighbourhood 
    // of trackedpt 

    // this value does not seem to be used //Matti
    // int templater=rect.width/2-1;

    int maxdev=100;

    Rect cutrect;

    cutrect.x=rect.x+intensity_image_zeropad_margin;
    cutrect.y=rect.y+intensity_image_zeropad_margin;

    cutrect.width=rect.width;
    cutrect.height=rect.height;

    Mat templ(reff_zeropad,cutrect);

    cutrect.x=rect.x-maxdev+intensity_image_zeropad_margin;
    cutrect.y=rect.y-maxdev+intensity_image_zeropad_margin;

    cutrect.width=rect.width+2*maxdev;
    cutrect.height=rect.height+2*maxdev;

    Mat tgtmat(tgtf_zeropad,cutrect);

    Mat result;
    
    matchTemplate(tgtmat,templ,result,CV_TM_SQDIFF);

    // invalidate parts of result that are prohibited by the constraintmask

    float largeval=255*255*3*templ.rows*templ.cols;

    if(cmask){
      for(int x=0;x<result.cols;x++){
	  int maskx=x+trackedrect.x-maxdev;
	  if(maskx<0 || maskx>= cmask->cols) continue;

	  for(int y=0;y<result.rows;y++){
	    int masky=y+trackedrect.y-maxdev;
	    if(masky<0 || masky>= cmask->rows) continue;

	    if(cmask->at<uchar>(masky,maskx)==0)
	      result.at<float>(y,x)=largeval;
	}
      }
    }

    double minval;
    Point loc;

    minMaxLoc(result,&minval,NULL,&loc);
   //  floatimagesc("matching result",result);
//     cv::waitKey(0);
	     
    trackedrect.x+=loc.x-maxdev;
    trackedrect.y+=loc.y-maxdev;

//     Mat visframe=reff.clone();
//     Mat visframe2=tgtf.clone();

//     cv::rectangle(visframe,rect.tl(),rect.br(),CV_RGB(255,0,0));
//     cv::rectangle(visframe2,trackedrect.tl(),trackedrect.br(),CV_RGB(255,0,0));

//     imshow("rectangle in reference frame",visframe);
//     imshow("tracked rectangle",visframe2);
//     cv::waitKey(0);

    rect=trackedrect;

    return minval;
    

  }


  float trackRectangleSetBetweenFrames(vector<Rect> &rect,int refframe, int tgtframe,localBlackBoardType &bb, Mat *cmask){

    // the coordinates in rect get replaced by the new ones in the next frame

    Mat reff=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,"frame")]);
    Mat tgtf=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"frame")]);

//     Mat visframe=reff.clone();
//     Mat visframe2=tgtf.clone();

//     for(auto it=rect.begin();it!=rect.end();it++)
//       cv::rectangle(visframe,it->tl(),it->br(),CV_RGB(0,255,0));


    ensurePaddedFramesOnBlackboard(refframe,bb);
    ensurePaddedFramesOnBlackboard(tgtframe,bb);
 

    Mat reff_zeropad=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(refframe,"frame_padded")]);
    Mat tgtf_zeropad=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtframe,"frame_padded")]);

    // initialise tracked rect to the old location of rect

    int minrad=0.5*(rect[0].width-1);

    for(int i=1;i<static_cast<int>(rect.size());i++)
      minrad=fmin(minrad,0.5*(rect[i].width-1));

    Mat cumres;

    int maxdev=100;

    //     cout << "size of reference frame: " << reff.cols << "x"<<reff.rows<<endl;

    for(int i=0;i<static_cast<int>(rect.size());i++){

      Rect ri=rect[i];

      int templater=(ri.width-1)*0.5;

      Rect cutrect;

       // cout << "matching rectangle " << i  
       //  	   <<": ("<< ri.x << ","<<ri.y << ")-("<<ri.br().x<<
       //  	","<<ri.br().y<<")"<<endl;
    

      cutrect.x=ri.x+intensity_image_zeropad_margin;
      cutrect.y=ri.y+intensity_image_zeropad_margin;
      
      cutrect.width=ri.width;
      cutrect.height=ri.height;

        // cout << "template rectangle : ("<< cutrect.x << ","<<cutrect.y << ")
      //-("<<cutrect.br().x<<	","<<cutrect.br().y<<")"<<endl;
      
      Mat templ(reff_zeropad,cutrect);
      
      cutrect.x=ri.x-maxdev+intensity_image_zeropad_margin;
      cutrect.y=ri.y-maxdev+intensity_image_zeropad_margin;
      
      cutrect.width=ri.width+2*maxdev;
      cutrect.height=ri.height+2*maxdev;

      // cout << "target rectangle : ("<< cutrect.x << ","<<cutrect.y << ")-("<<cutrect.br().x<<	","<<cutrect.br().y<<")"<<endl;

    
      Mat tgtmat(tgtf_zeropad,cutrect);

      Mat result;
    
      matchTemplate(tgtmat,templ,result,CV_TM_SQDIFF);

      //result is now of size (2*maxdev+1) x (2*maxdev+1)

      if(i==0)
	cumres=result;
      else
	cumres += result;
    }
  



    // invalidate parts of cumulative result that are prohibited by the constraintmask

    float largeval=255*255*3*rect[0].width*rect[0].width*rect.size()*16;

    if(cmask){
      for(int x=0;x<cumres.cols;x++){
	for(int y=0;y<cumres.rows;y++){
          for(int i=0;i<static_cast<int>(rect.size());i++){
	      Rect ri=rect[i];
	      
	      int radius=(ri.width-1)*0.5;
	      
	      int maskx=x+0.5*ri.width+ri.x-maxdev;
	      int masky=y+0.5*ri.height+ri.y-maxdev;

	      if(maskx<radius-1 || maskx>= cmask->cols-radius+1 || 
		 masky<radius-1 || masky>= cmask->rows-radius+1 ||
		 cmask->at<uchar>(masky,maskx)==0)
		cumres.at<float>(y,x)=largeval;
	    }
	}
      }
    }

    // invalidate also the parts that of result landscape that would take any part of any rectangle
    // outside the image area


    double minval;
    Point loc;
    
    minMaxLoc(cumres,&minval,NULL,&loc);
  //      floatimagesc("matching result",cumres);
//         cv::waitKey(0);
	     

    for(int i=0;i<static_cast<int>(rect.size());i++){
	rect[i].x+=loc.x-maxdev;
	rect[i].y+=loc.y-maxdev;
      }



//     for(auto it=rect.begin();it!=rect.end();it++)
//       cv::rectangle(visframe2,it->tl(),it->br(),CV_RGB(0,255,0));

//     cv::rectangle(visframe,rect.tl(),rect.br(),CV_RGB(255,0,0));
//     cv::rectangle(visframe2,trackedrect.tl(),trackedrect.br(),CV_RGB(255,0,0));

//      imshow("rectangles in reference frame",visframe);
//      imshow("tracked rectangles",visframe2);
//      cv::waitKey(0);

//     rect=trackedrect;
 
     if(minval==largeval) return -1;
     else return minval;
    

  }

  void recordBlobTracks(int first, int last, localBlackBoardType &bb){

    // for each frame (except the first) key "blobtracks" is 
    // stored on bb

    // "blobtracks" is of type map<int,vector<Rect> >, where the int key is 
    // running blob track id, starting from 0

    //       cout << "bbox.width=" << bbox.width <<"bbox.height=" << bbox.height << endl;

    // requires blob progression and blob interpretation to be stored on bb

    bool debugout=false;
      
    Mat frame0=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(first,"frame")]);

    vector<Point> sp;

    spiralDeltas(sp,fmin(frame0.cols,frame0.rows));

    BlobProgression bp=*boost::any_cast<BlobProgression>(&bb[std::pair<int,std::string>(0,"blobprogression")]);
    

    map<int,blobinterpretation_type> interp=*boost::any_cast<map<int,blobinterpretation_type> >(&bb[std::pair<int,std::string>(0,"blobinterpretation")]);

    int nrrects=250;
    
    int nextblobtrackid=0;

    map<int,vector<Rect> > trackedRect;

    int face_references_needed=6;

    bool face_tracking=false;

    int lastocclusionframe=-1;

    // for later filtering, collect an index
    // of (frame,id) -> blob lbl mappings

    map<int,map<int,int>> id2lbl; // semantics: id2lbl[frame][id]=lbl


    for(int f=first;f<=last;f++){
      
      //      if(debugout)
      cout << "frame " << f << endl;

      Mat uniqueblobintmask_tgt=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"uniqueblobintmask")]);


//       showDeterministicFalseColour32bit("uniqueblobintmask", uniqueblobintmask_tgt);
//       cv::waitKey(0);

      Mat tgt_skin=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"skinmask")]);


      // update the face prohibition mask
      if(face_tracking && f>first){
	trackFaceGridlet(f-1,f,bb); // these accumulate point clouds on blackboard

	constructFloatMaskFromPointCloud(f,f,"unoccludedheadpoints","unoccludedheadfloatmask",bb,false);
      }
      

      // track the existing trackerRects

      map<int,int> trackercounts_ref;

      for(auto it=trackedRect.begin(); it!=trackedRect.end();it++){
	if(it->second.size()){
	  Mat uniqueblobintmask_ref=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f-1,"uniqueblobintmask")]);
	  int reflbl=uniqueblobintmask_ref.at<int>(it->second[0].y,it->second[0].x);
	  trackercounts_ref[reflbl]++;
	}
      }

      for(auto it=trackedRect.begin(); it!=trackedRect.end();it++){
	trackRectangleSet(it->second,f-1,f,trackercounts_ref,bb);

//       Mat tgtf=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"frame")]);


// 	Mat visframe=tgtf.clone();

//     for(auto rit=it->second.begin();rit!=it->second.end();rit++)
//       cv::rectangle(visframe,rit->tl(),rit->br(),CV_RGB(255,0,0));

//     imshow("tracked rectangles",visframe);
//     cv::waitKey(0);

      } // for each trackedRects entry

      //  cout << "after tracking existing trackers, the trackers are" << endl;
      // dumpBlobTracks(trackedRect);

      // refresh the rectangle selection if an opportunity arises

      int headlbl=-1;
      map<int,int> trackedcount; // counts the trackers within each blob

      for(auto it=bp.lbl2blobidx[f-bp.firstframe].begin(); it!=bp.lbl2blobidx[f-bp.firstframe].end();it++){
	int lbl=it->first;
	if(interp[lbl].head)
	  headlbl=lbl;
      }

      for(auto it=trackedRect.begin(); it!=trackedRect.end();it++){
	if(it->second.size()){
	  int lbl=uniqueblobintmask_tgt.at<int>(it->second[0].y,it->second[0].x);
	  trackedcount[lbl]++;
	}
      }

      for(auto it=trackedRect.begin(); it!=trackedRect.end();it++){
	if(it->second.size()){
	  int lbl=uniqueblobintmask_tgt.at<int>(it->second[0].y,it->second[0].x);
	  if(lbl!=headlbl && trackedcount[lbl]<2)
	    selectRectanglesToTrack(f,lbl,nrrects,it->second,bb);
	}
      }

      // cout << "after refreshing the trackers are" << endl;
      // dumpBlobTracks(trackedRect);



      // check if there are non-face blobs that have no trackedRects yet
      // if so, insert a trackedRect for these blobs

      map<int,set<int> > lbl2trackerid;

      for(auto it=trackedRect.begin(); it!=trackedRect.end();it++){
	if(it->second.size()){
	  int lbl=uniqueblobintmask_tgt.at<int>(it->second[0].y,it->second[0].x);
	  lbl2trackerid[lbl].insert(it->first);
	}
      }


      set<int> borderblobs;

      for(int x=0;x<uniqueblobintmask_tgt.cols;x++){
	for(int dy=0;dy<20;dy++){
	  borderblobs.insert(uniqueblobintmask_tgt.at<int>(dy,x));
	  borderblobs.insert(uniqueblobintmask_tgt.at<int>(uniqueblobintmask_tgt.rows-1-dy,x));
	}
      }

      for(int y=0;y<uniqueblobintmask_tgt.rows;y++){
	for(int dx=0;dx<20;dx++){
	  borderblobs.insert(uniqueblobintmask_tgt.at<int>(y,dx));
	  borderblobs.insert(uniqueblobintmask_tgt.at<int>(y,uniqueblobintmask_tgt.cols-1-dx));
	}
      }
      if(debugout){
	cout << "border blobs:"<<endl;
	for(auto it=borderblobs.begin();it!=borderblobs.end();it++)
	  cout << " " << *it;
	cout << endl;
      }

      set<int> nontrackedblobs;

      int frameidx=f-bp.firstframe;
      for(size_t blobidx=0;blobidx<bp.blobs[frameidx].size();blobidx++){
	BlobInfo *bi=&(bp.blobs[frameidx][blobidx]);
	if(lbl2trackerid.count(bi->label)==0 && interp[bi->label].face_overlap==false){
	  nontrackedblobs.insert(bi->label);
	}
      }


      for(auto it=nontrackedblobs.begin();it!=nontrackedblobs.end();it++){

	// reject blobs that touch image borders

	if(borderblobs.count(*it)>0) continue;
	if(debugout)
	  cout << "inserting new tracker to blob " << *it << endl;
	vector<Rect> newTracker;
	selectRectanglesToTrack(f,*it,nrrects,newTracker,bb);

	// check if some of the existing trackers should be replaced by the new one

	int lbl2replace=-1;

	for(auto it2=lbl2trackerid.begin();it2!=lbl2trackerid.end();it2++)
	  if(it2->second.size()>1) lbl2replace=it2->first;

	if(lbl2replace>=0){
	  if(debugout)
	    cout << "replacing old tracker in blob " << lbl2replace << endl;
	  
	  set<int> candidates=lbl2trackerid[lbl2replace];

	  if(debugout){
	    cout << "candidate ids are ";
	    for(auto it=candidates.begin();it!=candidates.end();it++)
	      cout << " " << *it;
	    cout << endl;
	  }

	  // predict the location of each existing tracker linearly based on history 
	  // replace the one whose location falls nearest
	  // to the new one (deviation corrected)

	  // in the future some visul similarity based 
	  // track quality measure could be combined with this
	  // purely kinematic criterion

	  // determine the centroid of the new tracker

	  Point2f c(0,0);
	  int ctr=0;

	  for(auto it2=newTracker.begin();it2!=newTracker.end();it2++){
	    c.x += it2->x + it2->br().x;
	    c.y += it2->y + it2->br().y;
	    ctr++;
	  }
	  if(ctr){
	    c.x /= 2*ctr;
	    c.y /= 2*ctr;
	  }

	  int minid=-1;
	  float minerr;

	  for(auto it2=candidates.begin();it2!=candidates.end();it2++){
	    Point2f ppos=c;
	    float dev=1;

	    predictTrackerMotion(*it2, f, ppos,dev,bb);
	    Point2f e=c-ppos;
	    float err=sqrt(e.x*e.x+e.y*e.y)/dev;

	    if(minid<0||err<minerr){
	      minid=*it2;
	      minerr=err;
	    }
	  }

	  trackedRect[minid]=newTracker;

	} else{
	  // otherwise, create new trackerid
	  trackedRect[nextblobtrackid++]=newTracker;
	}
      }

      // remove trackers that

      // a) are in blobs that touch image borders

      // b) have become too small (<5 rectangles w/ radius>3)

      // c) are in the head blob while there are >= two trackers
      //    in the non-head blobs

      int nonheadcount=0;
      for(auto it=trackedRect.begin(); it!=trackedRect.end();it++){
	if(it->second.size()){
	  int lbl=uniqueblobintmask_tgt.at<int>(it->second[0].y,it->second[0].x);
	  if(interp[lbl].head==false)
	    nonheadcount++;
	}
      }

      for(auto it=trackedRect.begin(); it!=trackedRect.end();it++){
	if(it->second.size()){
	int lbl=uniqueblobintmask_tgt.at<int>(it->second[0].y,it->second[0].x);

       if(interp[lbl].head==false){


	 int largecount=0;
	 for(auto rit=it->second.begin();rit!=it->second.end();rit++)
	   if(rit->width>7)
	     largecount++;
	 
	 if(borderblobs.count(lbl)>0 || largecount<5)
	   trackedRect.erase(it);
       } else if(nonheadcount>=2)
	 trackedRect.erase(it);
	}
      }


      // insert the final id-> lbl mapping into the index for
      // later filtering

      for(auto it=trackedRect.begin(); it!=trackedRect.end();it++){
	if(it->second.size()){
	  int lbl=uniqueblobintmask_tgt.at<int>(it->second[0].y,it->second[0].x);
	  id2lbl[f][it->first]=lbl;
	}
      }
      // store the result on bb

      // cout << "after filtering and refreshing, the trackers are" << endl;
      // dumpBlobTracks(trackedRect);

      // before storing the result, remove empty trackers
      for(auto it=trackedRect.begin(); it!=trackedRect.end();it++)
	if(it->second.size()==0) trackedRect.erase(it);

      bb[std::pair<int,std::string>(f,"blobtracks")]=trackedRect;

//       // visualise the result

//        Mat tgtf=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(f,"frame")]);


//        Mat visframe=tgtf.clone();
//        for(auto it=trackedRect.begin(); it!=trackedRect.end();it++){

// 	 cv::Scalar c=CV_RGB(((it->first+1)*87)&255,((it->first+1)*101)&255,
// 			     ((it->first+1)*173)&255);

// 	 for(auto rit=it->second.begin();rit!=it->second.end();rit++)
// 	   cv::rectangle(visframe,rit->tl(),rit->br(),c);

//       }

//      imshow("tracked rectangles",visframe);
//      cv::waitKey(0);

     // test for possible occlusion

     bool possible_occlusion=false;

     for(auto it=trackedRect.begin(); it!=trackedRect.end();it++){
	if(it->second.size()){
	  int lbl=uniqueblobintmask_tgt.at<int>(it->second[0].y,it->second[0].x);
	  if(interp[lbl].head){
	    possible_occlusion=true;
	    break;
	  }
	}
     }
    
     if(possible_occlusion&&face_tracking){
       lastocclusionframe=f;
     }
       

     if(face_tracking){
       // stop face tracking if long enough without occlusion
       if(f-lastocclusionframe>face_references_needed)
	 face_tracking=false;
     } else{

       if(possible_occlusion){
       // initialise face tracking if possible occlusion start
       int startframe=fmax(first,f-face_references_needed-1);
     
       bb.erase(std::pair<int,std::string>(startframe,"trackerstate_face"));       

       for(int ff=startframe;ff<f-1;ff++)
	 trackFaceGridlet(ff,ff+1,bb); 

       face_tracking=true;
       lastocclusionframe=f;
       f--; // backtrack one step and redo tracking with the face mask
       if(debugout)
	 cout << "backtracking one frame due to occlusion start" << endl;
       }
     }
      }
    
    // filter out the cases where a tracker stays 
    // erraneously in the head when a new tracker departs 
    // from the head

    for(int f=first+1;f<=last;f++){

      if(bb.count(std::pair<int,std::string>(f-1,"blobtracks"))==0 ||
	 bb.count(std::pair<int,std::string>(f,"blobtracks"))==0)
	continue;

      // this value seems to go unused //Matti
      // map<int, vector<Rect> > *prevbt=boost::any_cast<map<int,vector<Rect> > >(&bb[std::pair<int,std::string>(f-1,"blobtracks")]);
      map<int, vector<Rect> > *bt=boost::any_cast<map<int,vector<Rect> > >(&bb[std::pair<int,std::string>(f,"blobtracks")]);

      // detect potential erranous splits:
      // head tracker count stays the same
      // while the count of trackers increases

      if(id2lbl[f].size()>id2lbl[f-1].size()){
	int headcount_prev=0;
	int headcount=0;

	int headid_new=-1;
	int nonheadid_new=-1;

	for(auto it=id2lbl[f-1].begin();it!=id2lbl[f-1].end();it++)
	  if(interp[it->second].face_overlap)
	    headcount_prev++;

	for(auto it=id2lbl[f].begin();it!=id2lbl[f].end();it++)
	  if(interp[it->second].face_overlap){
	    headcount++;
	  }

	if(headcount!=headcount_prev) continue;

	// potentially erraneous split detected

	// guess the tracker ids involved as follows

	// take the new non-head id the id closest to the face center
	// among the newly created non-face trackers

	// take the new head id to be the head tracker closest
	// to the new non-head tracker

	// determine the centroids of all the trackers

	map<int,Point2f> centroids;

	for(auto it=bt->begin();it!=bt->end();it++){
	  Point2f c(0,0);
	  int ctr=0;
	  for(auto it2=it->second.begin();it2!=it->second.end();it2++){
	    c.x += it2->x + it2->br().x;
	    c.y += it2->y + it2->br().y;
	    ctr++;
	  }
	  if(ctr){
	    c.x /= 2*ctr;
	    c.y /= 2*ctr;
	  }
	  centroids[it->first]=c;
	}
	
	// centroids determined

	Rect faceLocation=*boost::any_cast<cv::Rect>(&bb[std::pair<int,std::string>(f,"facerect")]);
       
	float mindist=-1;

	for(auto it=id2lbl[f].begin();it!=id2lbl[f].end();it++)
	  if(interp[it->second].face_overlap==false && id2lbl[f-1].count(it->first)==0){
	    float dx=centroids[it->first].x-0.5*(faceLocation.x+faceLocation.br().x);
	    float dy=centroids[it->first].y-0.5*(faceLocation.y+faceLocation.br().y);
	    float dist=dx*dx+dy*dy;

	    if(mindist<0 || dist<mindist){
	      mindist=dist;
	      nonheadid_new=it->first;
	    }
	    
	  }


	mindist=-1;

	for(auto it=id2lbl[f].begin();it!=id2lbl[f].end();it++)
	  if(interp[it->second].face_overlap){
	    float dx=centroids[it->first].x-centroids[nonheadid_new].x;
	    float dy=centroids[it->first].y-centroids[nonheadid_new].y;

	    float dist=dx*dx+dy*dy;

	    if(mindist<0 || dist<mindist){
	      mindist=dist;
	      headid_new=it->first;
	    }
	    
	  }

	if(nonheadid_new<0 || headid_new<0) continue; // could not determine
	// the ids of the trackers involved

	// check if the split -> (headid_new) (nonheadid_new) really is erranous
	// do this by tracking headid_new forwardsin time and detecting 
	// that the tracker either
	// a) disappears b) never leaves the head blob again

	bool erraneous=true;
	for(int ff=f+1;ff<=last;ff++){
	  if(id2lbl[ff].count(headid_new)==0) break;
	  if(interp[id2lbl[ff][headid_new]].face_overlap==false){
	    erraneous=false;
	    break;
	  }
	}

	if(erraneous){
	  if(debugout){
	    cout << "filtering away erraneous split in frame " << f << 
	      " (headid_new="<<headid_new<<" nonheadid_new="<<
	      nonheadid_new<<")"<<endl; 
	  }
	  
	  for(int ff=f;ff<=last;ff++){
	    if(bb.count(std::pair<int,std::string>(ff,"blobtracks"))==0)
	      continue;

	    map<int, vector<Rect> > *bt=boost::any_cast<map<int,vector<Rect> > >(&bb[std::pair<int,std::string>(ff,"blobtracks")]);

	    (*bt)[headid_new]=(*bt)[nonheadid_new];
	    bt->erase(nonheadid_new);

	  }
	}



      }

    }


  }


  void trackRectangleSet(vector<Rect> &r,int reff,int tgtf, 
			 map<int,int> &trackercounts_ref,
			 localBlackBoardType &bb){

    bool debugout=false;

    BlobProgression bp=*boost::any_cast<BlobProgression>(&bb[std::pair<int,std::string>(0,"blobprogression")]);
    map<int,blobinterpretation_type> interp=*boost::any_cast<map<int,blobinterpretation_type> >(&bb[std::pair<int,std::string>(0,"blobinterpretation")]);

    // This value seems to go unused //Matti
    // int nrrects=250;
    Mat uniqueblobintmask_ref=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(reff,"uniqueblobintmask")]);
    Mat uniqueblobintmask_tgt=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtf,"uniqueblobintmask")]);

    static vector<Point> sp;
    if(sp.size()==0)
      spiralDeltas(sp,fmin(uniqueblobintmask_ref.cols,uniqueblobintmask_ref.rows));

    Mat tgt_skin=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtf,"skinmask")]);
    Mat *cmask=new Mat;
    *cmask=tgt_skin.clone();

    int reflbl=uniqueblobintmask_ref.at<int>(r[0].y,r[0].x);

    // prepare a constraintmask for tracking using the 
    // following contraints

    // 1) if label of tracked rectangles continues in the new frame, prohibit 
    //    all other blobs if there was only one tracker in the continuing blob

    if(bp.lbl2blobidx[tgtf-bp.firstframe].count(reflbl) && trackercounts_ref[reflbl]<=1){
      for(int x=0;x<cmask->cols;x++)
	for(int y=0;y<cmask->rows;y++)
	  if(uniqueblobintmask_tgt.at<int>(y,x)!=reflbl)
	    cmask->at<uchar>(y,x)=0;
    }

    //	cout << "other blobs suppressed" << endl;

    // 2) if there are two blobs besides the head blob, prohibit the head blob
    // (in the future, this would be also the place for refreshing the
    // rectangle selection)
    
    int nonheadcount=0;
    int headlbl=-1;
    
    for(auto it=bp.lbl2blobidx[tgtf-bp.firstframe].begin(); it!=bp.lbl2blobidx[tgtf-bp.firstframe].end();it++){
      int lbl=it->first;
      if(interp[lbl].head)
	headlbl=lbl;
      else
	nonheadcount++;
    }

    //	cout << "blobs counted" << endl;

    if(headlbl != -1 && nonheadcount>1){
      for(int x=0;x<cmask->cols;x++)
	for(int y=0;y<cmask->rows;y++)
	  if(uniqueblobintmask_tgt.at<int>(y,x)==headlbl)
	    cmask->at<uchar>(y,x)=0;
    }

    // 3) in the future, prohibit also the blob tracked by the other hand tracker
    // 	       imshow("initial constraint mask",*cmask);
// 	       cv::waitKey(0);
	
    // try to shrink the mask by the minumum radius of the rectangles
	
    int minrad=9999;
	
    Mat *smask=new Mat;
    
    for(auto rit=r.begin();rit!=r.end();rit++)
      minrad=fmin(minrad,0.5*(rit->width-1));
    
    if(minrad>1){
      
      if(debugout) cout << "thinning according to minrad="<<minrad<<endl;
      cv::erode(*cmask, *smask,cv::getStructuringElement(cv::MORPH_ELLIPSE, 
							   Size(minrad,minrad)));

      // check for the unlikely case where the thinned mask disappears

      bool nonzero=false;

      for(int x=0;!nonzero && x<smask->cols;x++)
	for(int y=0;y<smask->rows;y++)
	  if(smask->at<uchar>(y,x)){
	    nonzero=true;
	    break;
	  }
      if(nonzero){
	delete cmask;
	cmask=smask;
      }
      else{
	delete smask;
      }
    }

    // remove regions of the constraint mask that are prohibited by 
    // the unoccluded head mask
    if(bb.count(std::pair<int,std::string>(tgtf,"unoccludedheadfloatmask"))>0){
      Mat occl=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtf,"unoccludedheadfloatmask")]);
      
      for(int x=0;x<cmask->cols;x++)
	for(int y=0;y<cmask->rows;y++)
	  if(occl.at<float>(y,x)>0.5)
	    cmask->at<uchar>(y,x)=0;
    }

    // maybe in the future we should add a checking step that replaces the 
    // blob continuity analysis by area overlaps with checks by contents match (tracking)

//     imshow("final constraint mask",*cmask);
//     cv::waitKey(0);
    if(debugout)
      cout << "tracking from frame " << reff << " to " << tgtf << endl;
    
    vector<Rect> origRect=r;

    //       imshow("constraint mask after thinning",*cmask);
    //       cv::waitKey(0);

    // initialise tracking by looking for optimal common displacement for all the
    // rectangles

    float res=trackRectangleSetBetweenFrames(r,reff,tgtf,bb,cmask);

    if(debugout) cout << "initial common mode tracking tried" << endl;

    // this may fail due to the constraint mask making such displacement inexistent

    if(res<0){
      // in such case
      // initialise track by tracking each rectangle individually

      if(debugout) cout << "tracking rectangles individually" << endl;
      
      r=origRect;
      
      for(auto rit=r.begin();rit!=r.end();rit++)
	trackRectangleBetweenFrames(*rit,reff,tgtf,bb,cmask);

      // and
      // here we (could) will take the median displacement and 
      // use it to initialise the tracking of all rectangles

      vector<int> dx,dy;
      
      for(int i=0;i<static_cast<int>(r.size());i++){
	dx.push_back(r[i].x-origRect[i].x);
	dy.push_back(r[i].y-origRect[i].y);
      }
	
      sort(dx.begin(),dx.end());
      sort(dy.begin(),dy.end());

      int dxMed=dx[0.5*dx.size()];
      int dyMed=dy[0.5*dy.size()];
      
      if(debugout) cout << "median displacement: ("<<dxMed<<","<<dyMed<<")"<<endl;
      
      for(int i=0;i<static_cast<int>(r.size());i++){
	r[i].x = origRect[i].x + dxMed;
	r[i].y = origRect[i].y + dyMed;
      }

    }

    if(debugout) cout << "track initialised" << endl;

//      Mat reff=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(reff,"frame")]);
//       Mat tgtf=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtf,"frame")]);


//       {    
// 	Mat visframe=tgtf.clone();

// 	for(auto rit=r.begin();rit!=r.end();rit++)
// 	  cv::rectangle(visframe,rit->tl(),rit->br(),CV_RGB(255,0,0));

// 	imshow("initial track",visframe);
// 	cv::waitKey(0);

//       }

	// then refine tracking by greedy local iteration

    grid_type grid;
    vector<int> radius;

    for(int i=0;i<static_cast<int>(r.size());i++){
      gridpoint_type p;

      p.refx=origRect[i].x+0.5*(origRect[i].width-1);
      p.refy=origRect[i].y+0.5*(origRect[i].height-1);
      
      p.tgtx=r[i].x+0.5*(r[i].width-1);
      p.tgty=r[i].y+0.5*(r[i].height-1);
      
      grid.push_back(p);
      radius.push_back(0.5*(origRect[i].width-1));
    }
    

    // set_nbr_fullconnectivity(grid);

    // more sparse connectivity could also be ok

    set_nbr_randomconnectivity(grid,4);
    
    set_l0(grid);
    
    //     Mat visref=reff.clone();
    //     Mat vistgt=tgtf.clone();
    
    //     showgrid(visref,grid,true,false,true,false);
    //     showgrid(vistgt,grid,false,true,false,true);
    
    //     imshow("grid in reference frame",visref);
    //     imshow("grid in target frame",vistgt);
    //     cv::waitKey(0);
    
    float alpha=1.0/400;
    
    int searchr=100;
    
    vector<bool> needs_update(grid.size(),true);
    
    int needupdatecount=grid.size();
    
    ensurePaddedFramesOnBlackboard(reff,bb);
    ensurePaddedFramesOnBlackboard(tgtf,bb);
    
    
    Mat reff_zeropad=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(reff,"frame_padded")]);
    Mat tgtf_zeropad=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtf,"frame_padded")]);


    int iterationcount=0;
    int maxiter=200;

    do{

      iterationcount++;
      
      int r;
      do{
	r=rand(grid.size());
      } while(!needs_update[r]);
      
      Point oldtgt(grid[r].tgtx,grid[r].tgty);

      if(debugout) 
	cout << "updating grid point " << r << " (tgtx="<<grid[r].tgtx<<" tgty="<<grid[r].tgty <<
	  " r="<< radius[r] << ")"<<endl; 
      
      //       Mat visb4=tgtf.clone();
      //       showgrid(visb4,grid,false,true,false,true);
      //       cv::circle(visb4,Point(grid[r].tgtx,grid[r].tgty), 5, CV_RGB(0,255,0));

      updateGridPointGreedyLocal(grid,r,reff_zeropad,tgtf_zeropad,*cmask,2*radius[r]+1,searchr,alpha,1,30,false);
      
      needs_update[r]=false;
      needupdatecount--;

      if(oldtgt.x!=grid[r].tgtx || oldtgt.y !=grid[r].tgty)
	for(auto it=grid[r].nbr.begin(); it!=grid[r].nbr.end();it++)
	  if(needs_update[*it]==false){
	    needupdatecount++;
	    needs_update[*it]=true;
	  }
      //       Mat visref=reff.clone();
      //       showgrid(visref,grid,true,false,true,false);
      //       cv::circle(visref,Point(grid[r].refx,grid[r].refy), 5, CV_RGB(0,255,0));
      //       Mat vistgt=tgtf.clone();
      //       showgrid(vistgt,grid,false,true,false,true);
      //       cv::circle(vistgt,Point(grid[r].tgtx,grid[r].tgty), 5, CV_RGB(0,255,0));

      //       //    imshow("grid in target frame before",visb4);
      //       imshow("grid in target frame",vistgt);
      //       imshow("grid in reference frame",visref);
      //       cv::waitKey(0);

    } while(iterationcount<maxiter && needupdatecount>0);
    
    
    // transfer the tracking results back from grid to trackedRect
    // prune out points that are tracked to blobs other than the most common blob
    
    r.clear();
    
    Mat uniqueblobintmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(tgtf,"uniqueblobintmask")]);

    map<int,int> labelcounts;

    for(int i=0;i<static_cast<int>(grid.size());i++){
      int lbl=uniqueblobintmask.at<int>(grid[i].tgty,grid[i].tgtx);
      labelcounts[lbl]++;
    }

    int maxcount=-1;
    int maxlbl=-1;

    for(auto it=labelcounts.begin();it!=labelcounts.end();it++)
      if(it->second>maxcount){
	maxcount=it->second;
	maxlbl=it->first;
      }
    
    if(debugout)
      cout << "majority lbl is " << maxlbl << " with " << maxcount << " occurrences" << endl;

    for(int i=0;i<static_cast<int>(grid.size());i++){

      // ignore points that have drifted outside the image area

      if(grid[i].tgtx-radius[i]+1<0 || grid[i].tgtx+radius[i]>tgt_skin.cols || 
	 grid[i].tgty-radius[i]+1<0 || grid[i].tgty+radius[i]>tgt_skin.rows)
	continue;

      int lbl=uniqueblobintmask.at<int>(grid[i].tgty,grid[i].tgtx);
      if(lbl==maxlbl){
	Rect rect;

	// here we should shrink radii of rectangles that have drifted towards the 
	// borders of skin region

	for(int ii=0;ii<static_cast<int>(sp.size());ii++){
	  int rr=fmax(fabs(sp[ii].x),fabs(sp[ii].y));
	  if(rr>radius[i]) break;
	  if(tgt_skin.at<uchar>(grid[i].tgty+sp[ii].y,grid[i].tgtx+sp[ii].x)==0){
	    if(debugout)
	      cout << "shrinking radius " << i << " from " << radius[i] << " to ";

	    radius[i]=rr-1;
	    if(debugout)
	      cout << radius[i] << endl;
	    break;
	  }
	}

	rect.x=grid[i].tgtx-radius[i];
	rect.y=grid[i].tgty-radius[i];
	rect.width=2*radius[i]+1;
	rect.height=2*radius[i]+1;

	if(radius[i]>2)
	  r.push_back(rect);
      }
    }

    delete cmask;

  }

  void predictTrackerMotion(int id, int tgtframe, Point2f &pos, float &dev,localBlackBoardType &bb){

    // find out the length of history available
    // at the same time, collect the trackers into a vector

    vector<vector<Rect> > trackerpos;

    int maxhistory=6;

    for(int l=1;l<=maxhistory;l++){
      if(bb.count(std::pair<int,std::string>(tgtframe-l,"blobtracks"))==0)
	break;

      map<int,vector<Rect> > *bt=boost::any_cast<map<int,vector<Rect> > >(&bb[std::pair<int,std::string>(tgtframe-l,"blobtracks")]);
      
      if(bt->count(id)==0)
	break;
      
      trackerpos.push_back((*bt)[id]);
    }

    size_t histlen=trackerpos.size();

    if(histlen==0) return; // no motion history, can't predict

    // detemine the centroids of trackers
    
    vector<Point2f> centroids;

    for(auto it=trackerpos.begin();it!=trackerpos.end();it++){
      Point2f c(0,0);
      int ctr=0;
      for(auto it2=it->begin();it2!=it->end();it2++){
	c.x += it2->x + it2->br().x;
	c.y += it2->y + it2->br().y;
	ctr++;
      }
      if(ctr){
	c.x /= 2*ctr;
	c.y /= 2*ctr;
      }

      centroids.push_back(c);

    }

    // handle the special case where there's only one prior observation -> zeroth order 
    // predictor

    float errordiscount=30;

    if(histlen==1){
      pos=centroids[0];
      dev=errordiscount;
    }

    // otherwise predict linearly
    
    pos=2*centroids[0]-centroids[1];

    // estimate the prediction error

    float esum=errordiscount;
    int ectr=1;

    for(int i=0;i<static_cast<int>(histlen-2);i++){
      Point2f err=centroids[i]-(2*centroids[i+1]-centroids[i+2]);
      esum += sqrt(err.x*err.x+err.y*err.y);
      ectr++; 
    }

    dev=esum / ectr;

  }

  void dumpBlobTracks(map<int,vector<Rect> > &bt){
    for(auto it=bt.begin();it!=bt.end();it++){
      cout << "id="<<it->first<<endl;
      for(auto rit=it->second.begin();rit!=it->second.end();rit++)
	cout << "  ("<<rit->x<<","<<rit->y<<")-("<<rit->br().x<<","<<
	  rit->br().y << ")"<<endl;
    }
  }



}


