#include "TsSomCodebook.hpp"
#include "Classifier.hpp"
#include <cstring>

using std::string;
using std::cerr;
using std::endl;
using std::cout;
using std::vector;

namespace slmotion {
  void TsSomCodebook::readSom(const string &fn){

    const int LINELENGTH=10000;

    FILE *cod;

    if((cod=fopen(fn.c_str(),"r"))==NULL){
      cerr << "Failed to open " << fn << " for reading." << endl;
      exit(-1);
    }


    int dim;

    char line[LINELENGTH];

    while(!feof(cod)){
      if(!fgets(line,LINELENGTH,cod)){
	cerr << "error reading cod file" << endl;
	exit(-1);
      }
    
      //    cerr <<"READ LINE:" <<endl << line;

      if(line[0]=='#'){
	while(line[strlen(line)-1] != '\n'){ fgets(line,LINELENGTH,cod);}
	continue;
      }

      char topo[20];
      int w,h,l;

      if(sscanf(line,"%d%s%d%d",&dim,topo,&w,&h) != 4){
	cerr << "couldn't parse level info from 1st line: " << endl;
	cerr << line;
	exit(-1);
      }

      if(strcmp(topo,"rect")){
	cerr << "Map topology != rect." << endl;
	exit(-1);
      }

      cerr << "Read info line " << line <<endl;

      torus= string(line).find("torus") != string::npos;

      l=w*h;

      this->w=w;
      this->h=h;
      this->dim=dim;

      int readdatavec=0;
  
      while(!feof(cod)&&readdatavec<l){
	if(!fgets(line,LINELENGTH,cod)){
	  break;
	  //cerr << "error reading dat file" << endl;
	  //exit(1);
	}

	// cerr <<"READ LINE:" <<endl << line;
      
	if(line[0]=='#'){
	  while(line[strlen(line)-1] != '\n'){ fgets(line,LINELENGTH,cod);}
	  continue;
	}

	vector<float> v(dim);
    
	int offset=0;
	for(int i=0;i<dim;i++){
	  while(line[offset] && isspace(line[offset])) offset++;
	  if(sscanf(line+offset,"%f",&(v[i])) != 1){
	    cerr << "data reading error, line=" << line<<endl;
	    fclose(cod);
	    exit(-1);
	  }
	  while(line[offset] && !isspace(line[offset])) offset++;
	}
    
	units.push_back(v);

	char lblstr[80];
	if(sscanf(line+offset,"%s",lblstr) != 1)
	  labels.push_back("");
	else
	  labels.push_back(lblstr);

	readdatavec++;
	//      cout<<"read "<<readdatavec<<":th codebook vector "<<endl;

      }

      if(readdatavec!=l){
	cerr << "Couldn't read enough data." << endl;
	exit(-1);
      }
      else break;

    }

    fclose(cod);

  }



  void TsSomCodebook::writeSom(const string &fn){
    FILE *cod;

    if((cod=fopen(fn.c_str(),"w"))==NULL){
      cerr << "Failed to open " << fn << " for writing." << endl;
      exit(-1);
    }
    writeSom(cod);
    fclose(cod);

  }

  void TsSomCodebook::writeSom(FILE *cod){
    if(torus)
      fprintf(cod,"%d rect %d %d bubble torus\n",dim,w,h);
    else
      fprintf(cod,"%d rect %d %d bubble\n",dim,w,h);
    for(size_t i=0;i<units.size();i++){
      const vector<float> &v=units[i];
      for(size_t j=0;j<v.size();j++){
	if(j)
	  fprintf(cod," ");
	fprintf(cod,"%f",v[j]);   
      }
      fprintf(cod,"\n");
    }
  }

  void TsSomCodebook::dump(){
    if(labels.size() != units.size()){
      cerr <<"codebook::dump(): labels.size() != units.size()"<<endl;
      exit(-1);
    }

    cout << " dim="<<dim<<" w="<<w<<" h="<<h<<endl;

    for(size_t i=0;i<units.size();i++){
      cout << "["<<i<<"]("<<idx2x(i)<<","<<idx2y(i)<<"):";
      for(size_t j=0;j<units[i].size();j++)
	cout << units[i][j]<<" ";
      cout << labels[i] << endl;      
    }    
  }


  int TsSomCodebook::findclosest(const vector<float> &v){

    float mindist=vectorsqrdist(units[0],v);
    size_t minind=0;
    for(size_t i=0;i<units.size();i++){
      float d=vectorsqrdist(units[i],v,mindist);
      if(d<mindist){
	mindist=d;
	minind=i;
      }
    }
    return minind;
  }

  int TsSomCodebook::findclosest(const vector<float> &v, const Rect& r){

    const float bignumber=999999999;
    float mindist=bignumber;
    int minind=-1;

    for(int y=r.y1;y<=r.y2;y++){
      int yy=y;
      if(torus){
	if(yy<0) yy += h;
	else if(yy>=h) yy-=h;
      }
      int ind=yy*w;
      for(int x=r.x1;x<=r.x2;x++){
	int xx=x;
	if(torus){
	  if(xx<0) xx += w;
	  else if(xx>=w) xx-=w;
	}

	float d=vectorsqrdist(units[ind+xx],v,mindist);
	if(d<mindist){
	  mindist=d;
	  minind=ind+xx;
	}
      }
    }
    return minind;
  }

  int TsSomCodebook::tssom_findbmu(const vector<float> &v){
    int ret;
    if(!upper) ret= findclosest(v);
    else ret= findclosest(v,upper->expanddown(upper->tssom_findbmu(v)));
    //  cout << "tssom_findbmu returning " << ret << endl;
    return ret;

  }

  Rect TsSomCodebook::expanddown(int idx) {
    const int mul=4;

    Rect ret;

    float x=mul*idx2x(idx)+(mul-1)/2.0;
    float y=mul*idx2y(idx)+(mul-1)/2.0;

    // cout << "expanddown ("<<idx<<")->["<<x<<","<<y<<"]"<<endl;

    ret.x1=(int)floor(x-mul);
    ret.x2=(int)ceil(x+mul);
    ret.y1=(int)floor(y-mul);
    ret.y2=(int)ceil(y+mul);

    if(!torus){
      if( ret.x1<0) ret.x1=0;
      if(ret.x2>=mul*w) ret.x2=mul*w-1;
      if(ret.y1<0) ret.y1=0;
      if(ret.y2>=mul*h) ret.y2=mul*h-1;
    }
    return ret;
  }

  float TsSomCodebook::nbf(int i,int j,int d){
    // neighbourhood function of triangular type, width d
    
    if(d<0) d=5;

    int xi=idx2x(i);
    int xj=idx2x(j);

    int yi=idx2y(i);
    int yj=idx2y(j);

    if(xi>xj){int tmp=xi;xi=xj;xj=tmp;}
    if(yi>xj){int tmp=yi;yi=yj;yj=tmp;}

    // ensure {xy}i <= {xy}j

    float dx=xj-xi;
    float dy=yj-yi;

    if(torus){
      int dx2=xi+w-xj;
      int dy2=yi+h-yj;

      if(dx2<dx) dx=dx2;
      if(dy2<dy) dy=dy2;
      
    }

    //cout << "nbf("<<i<<","<<j<<") : (dx,dy)=("<<dx<<","<<dy<<")"<<endl;

    if(dx>d) return 0;
    if(dy>d) return 0;

    return (1-dx/(d+1))*(1-dy/(d+1))/((1+d)*(1+d));

  }

  vector<int> TsSomCodebook::neighbours(int i,int d){

    if(d<0) d=5;

    vector<int> ret;
  
    int x=idx2x(i);
    int y=idx2y(i);

    //cout << "forming neigbourhood for unit "<<i<<" ("<<x<<","<<y<<")"<<endl;

    if(torus){
      for(int xx=x-d;xx<=x+d;xx++)
	for(int yy=y-d;yy<=y+d;yy++){
	  int xxx=xx,yyy=yy;
	  if(xxx<0) xxx+=w;
	  else if(xxx>=w) xxx -= w;

	  if(yyy<0) yyy+=h;
	  else if(yyy>=h) yyy -= h;
	  ret.push_back(xxx+yyy*w);
	}
    } else
      for(int xx=x-d;xx<=x+d;xx++)
	for(int yy=y-d;yy<=y+d;yy++)
	  if(yy>=0&&xx>=0&&yy<h&&xx<w)
	    ret.push_back(xx+yy*w);
  
    return ret;
  }

  void TsSomCodebook::randominit(int w, int h, const DataSet& d){
    dim=d.dim;
    this->w=w;
    this->h=h;
    
    int l=w*h;
    units.clear();

    for(int i=0;i<l;i++)
      units.push_back(d.vec[randomint(l)]);
 
    labels=vector<string>(units.size());

  }

  void TsSomCodebook::initfromabove(){

    const int mul=4;

    if(!upper){
      cerr << "TsSomCodebook::initfromabove(): upper==NULL"<<endl;
      exit(-1);
    }

    const TsSomCodebook &o=*upper;

    w=mul*o.w;
    h=mul*o.h;
    dim=o.dim;

    units.clear();

    for(int oy=0;oy<o.h;oy++) // should use some interpolation
      for(int i=0;i<mul;i++)
	for(int ox=0;ox<o.w;ox++){
	  int oind=ox+oy*o.w;
	  for(int i=0;i<mul;i++)
	    units.push_back(o.units[oind]);
	}

    labels=vector<string>(units.size());
  }

  void TsSomCodebook::trainsom(float a_init,int n_rounds,const DataSet &d,int radius,int dmin){

    const float minradius=dmin;

    int nd=d.vec.size();
    // int nm=units.size();


    for(int r=0;r<n_rounds;r++){
      float d_t = radius*(1-((float)r)/n_rounds);
      if(d_t<minradius) d_t=minradius;
      float a_t=a_init*(1-((float)r)/n_rounds);

      cout << "round "<<r<<": alpha="<<a_t<<" d="<<(int)d_t<<endl;

      for(int i=0;i<nd;i++){
	//    if(i%100==0)
	//	cout << "datavec " << i << endl;
	int bmu=findclosest(d.vec[i]);
	vector<int> nvec=neighbours(bmu,(int)d_t);
	// for(int ind=0;ind<nvec.size();ind++){
	for (size_t ind = 0; ind < nvec.size(); ind++) {
	  int j=nvec[ind];
	  float h=nbf(bmu,j,(int)d_t)*a_t;
	  for(int k=0;k<dim;k++)
	    units[j][k]+=h*(d.vec[i][k]-units[j][k]);
	}
      }
    }
  }

  void TsSomCodebook::regiontrainiteration(TrainingRegion& r, const DataSet &d,
					   float alpha_t, int d_t) {

    //  cout << " current:"<<r.current_data<<" npoints="<<r.data.size()<<" limits=["
    //       << r.limits.x1<<"-"<<r.limits.x2<<","<<r.limits.y1<<"-"
    //       << r.limits.y2 << endl; 
    if(r.data.empty()) return;

    int i=r.data[r.current_data++];
    // if(r.current_data >= r.data.size()) r.current_data=0;
    if(r.current_data >= static_cast<int>(r.data.size()))
      r.current_data = 0;

    int bmu=findclosest(d.vec[i],r.limits);
    vector<int> nvec=neighbours(bmu,(int)d_t);
    // for(int ind=0;ind<nvec.size();ind++){
    for (size_t ind = 0; ind < nvec.size(); ind++) {
      int j=nvec[ind];
      float h=nbf(bmu,j,(int)d_t)*alpha_t;
      for(int k=0;k<dim;k++)
	units[j][k]+=h*(d.vec[i][k]-units[j][k]);
    }
  }
}
