#ifndef SLMOTION_VALIDATOR
#define SLMOTION_VALIDATOR

#include "XmlDocument.hpp"
#include <libxml/parser.h>
#include <libxml/xmlschemas.h>
#include <string>
#include <memory>

namespace slmotion {
  namespace xml {
    /**
     * This is an abstract class can be used to perform XML validation
     *
     * The reason this is made a class lies is because 'resource acquisition
     * is initialisation'
     *
     * @param schemafile The schema filename for annotation templates
     */
    class Validator {
    public:

      /**
       * Constructs a validator
       *
       * @param schemaDoc XML document for the schema.
       */
      explicit Validator(const XmlDocument& schemaDoc) :
        schemaDoc(schemaDoc),
        schemaCtxt((xmlSchemaParserCtxtPtr)NULL, &xmlSchemaFreeParserCtxt),
        schema((xmlSchemaPtr)NULL, &xmlSchemaFree),
        validatorCtxt((xmlSchemaValidCtxtPtr)NULL, &xmlSchemaFreeValidCtxt)
      {
        initialise();
      }




      /**
       * Constructs a validator from a schema file
       *
       * @param filename Filename of the XML schema file
       */
      explicit Validator(const std::string& filename);



      /**
       * Releases any memory allocated for the converter
       */
      // ~Validator();



      /**
       * Validates a document against the schema, and populates it with
       * default values.
       * 
       * @param document The document that needs to be validated
       * The document may be modified.
       *
       * @return True if valid
       */
      bool validate(const XmlDocument& document) const;



      /**
       * Reads the file, and validates the document against the schema.
       *
       * @param file Input filename
       * 
       * @return True if valid
       */
      inline bool validate(const std::string& file) const {
        return xmlSchemaValidateFile(validatorCtxt.get(), file.c_str(), 0) == 0;
      }



      /**
       * Move constructor
       */
      Validator(Validator&& other) :
        schemaDoc(std::move(other.schemaDoc)),
        schemaCtxt(std::move(other.schemaCtxt)),
        schema(std::move(other.schema)),
        validatorCtxt(std::move(other.validatorCtxt)),
        schemaFilename(other.schemaFilename)
      {
        other.schemaFilename = "";
      }


    
      /**
       * Move assignment
       */
      Validator& operator=(Validator&& other); 



      inline std::string getSchema() const {
        return schemaFilename;
      }



      /**
       * Copy constructor. Creates a shallow copy.
       */
      Validator(const Validator&) = default;



      /**
       * Copy assignment. Creates a shallow copy.
       */
      Validator& operator= (const Validator&) = default;



      /**
       * Creates an empty validator that always returns 'valid' for any 
       * document
       */
      Validator() {}



    private:
      /**
       * Initialises the object (called from the constructor, common to all
       * constructors)
       */
      void initialise();


      typedef std::shared_ptr<xmlSchemaParserCtxt> SchemaCtxtPtr;
      typedef std::shared_ptr<xmlSchema> SchemaPtr;
      typedef std::shared_ptr<xmlSchemaValidCtxt> ValidatorCtxtPtr;

      XmlDocument schemaDoc;    
      SchemaCtxtPtr schemaCtxt;
      SchemaPtr schema;
      ValidatorCtxtPtr validatorCtxt;
      std::string schemaFilename;
    };
  }
}
#endif
