#include "math.hpp"
#include "util.hpp"
#include "AsmTracker.hpp"

#ifdef SLMOTION_BOOST_REGEX
#include <boost/regex.hpp>
namespace std {
  using boost::regex;
  namespace regex_constants = boost::regex_constants;
  using boost::smatch;
  using boost::regex_match;
}
#else
#include <regex> // Not properly implemented as of GCC 4.5.1
#endif

#include "Visualiser.hpp"
#include "AsmTracker.hpp"
#include "KLTTracker.hpp"
#include "FeatureCreator.hpp"
#include <boost/algorithm/string.hpp>

#ifdef SLMOTION_ENABLE_LIBFLANDMARK
#include "FaceSuviSegmentator.hpp"
#endif // SLMOTION_ENABLE_LIBFLANDMARK

using cv::Mat;
using cv::Scalar;
using cv::Point;
using cv::Vec3b;
using cv::Size;
using cv::Point2f;
using cv::Rect;
using std::vector;
using std::deque;
using std::list;

namespace slmotion {
  typedef FeatureCreator::tracked_point_map_t tracked_point_map_t;
  typedef FeatureCreator::point_id_to_part_name_map_t point_id_to_part_name_map_t;
  typedef FeatureCreator::velocity_vector_map_t velocity_vector_map_t;
  typedef FeatureCreator::acceleration_vector_map_t acceleration_vector_map_t;



  // these constants are needed when drawing the kinect skeleton
  static size_t nite_linestart[16] = {1, 1, 2, 3, 1, 5, 6, 8, 8, 8, 9,10, 8,12,13, 9};
  static size_t nite_lineend[16]   = {0, 2, 3, 4, 5, 6, 7, 2, 5, 9,10,11,12,13,14,12};



  const struct ColoursStruct COLOURS = {
    cv::Scalar(0, 0, 255),
    cv::Scalar(0, 0, 127),
    cv::Scalar(255, 0, 0),
    cv::Scalar(127, 0, 0),
    cv::Scalar(0, 255, 0),
    cv::Scalar(0, 127, 0),
    cv::Scalar(0, 255, 255),
    cv::Scalar(255, 255, 0),
    cv::Scalar(127, 127, 0),
    cv::Scalar(255, 0, 255),
    cv::Scalar(127, 0, 127),
    cv::Scalar(255, 255, 255),
    cv::Scalar(127, 127, 127)
  };



  void drawDetectedBodyParts(cv::Mat& target, 
                             const std::list<BodyPart>& currentBodyParts) {
    for (std::list<BodyPart>::const_iterator it = currentBodyParts.begin();
         it != currentBodyParts.end(); it++) {
      
      it->getBlob().draw(target, getBodyPartColour(it->getIdentity(),
                                                   it->isAmbiguous()));
      
    }
  }




  static void drawBlobs(Mat& target, const vector<Blob>& blobs) {
    const static Scalar fancyColours[] = {
      Scalar(255, 0, 0),
      Scalar(0, 255, 0),
      Scalar(0, 0, 255),
      Scalar(255, 255, 0),
      Scalar(255, 0, 255),
      Scalar(0, 255, 255)
    };

    int i = 0;
    for (vector<Blob>::const_iterator it = blobs.begin(); it != blobs.end();
         ++it)
      it->draw(target, fancyColours[i++ % 6]);
  }



  static void drawAsms(cv::Mat& target,
                       const vector<Asm::Instance>& asms) {
    for (auto it = asms.cbegin(); it != asms.cend(); ++it)
      it->drawLine(target, 1, cv::Scalar(0,255,0));
  }



  void expandIfNecessary(cv::Mat& matrix, const cv::Point2i& cursor,
                         cv::Point2i& origin, const cv::Size& objectSize) {
    assert(matrix.type() == CV_8UC3);
    cv::Size currentMatrixSize = matrix.size();
    cv::Size newMatrixSize = currentMatrixSize;
    cv::Point2i newOrigin = origin;
    if (origin.x + cursor.x < 0) {
      newMatrixSize.width -= origin.x + cursor.x;
      newOrigin.x -= origin.x + cursor.x;
    }
    if (origin.y + cursor.y < 0) {
      newMatrixSize.height -= origin.y + cursor.y;
      newOrigin.y -= origin.y + cursor.y;
    }
    newMatrixSize.width = std::max(newMatrixSize.width, 
                                   objectSize.width + origin.x + cursor.x);
    newMatrixSize.height = std::max(newMatrixSize.height, 
                                    objectSize.height + origin.y + cursor.y);
    if (newMatrixSize != currentMatrixSize) {
      if (origin != newOrigin) {
        cv::Mat newMatrix(newMatrixSize, matrix.type(), cv::Scalar::all(0));
        for (int i = 0; i < matrix.rows; ++i)
          for (int j = 0; j < matrix.cols; ++j)
            newMatrix.at<cv::Vec3b>(i - origin.y + newOrigin.y,
                                    j - origin.x + newOrigin.x) = 
              matrix.at<cv::Vec3b>(i,j);
        origin = newOrigin;
        matrix = newMatrix;
      }
      else if (!matrix.empty())
        math::expandMat(matrix, newMatrixSize);
      else
        matrix = cv::Mat(newMatrixSize, matrix.type(), cv::Scalar::all(0));
    }
  }



  Mat Visualiser::drawComplexVisualisation(const std::string& visualisationScript,
                                           FrameSource& frameSource,
                                           const BlackBoard& blackBoard,
                                           size_t frameNumber) const {
    const cv::Mat& inFrame = frameSource[frameNumber];
    cv::Point2i origin(0,0); // the origin
    cv::Point2i cursor(0,0); // the cursor i.e. the upper-left corner
                             // of the drawing area
    Mat output(inFrame.size(), CV_8UC3, Scalar::all(0));
    std::vector<std::string> tokens;
    boost::split(tokens, visualisationScript, boost::is_any_of(";"));
    std::smatch m;

    for (size_t i = 0; i < tokens.size(); ++i)
      boost::algorithm::trim(tokens[i]);

    for (auto it = tokens.cbegin(); it != tokens.cend(); ++it) {
      if (regex_match(*it, m, std::regex("showMatrix (.*)"))) {
        BlackBoardPointer<cv::Mat> matrixPtr = blackBoard.get<cv::Mat>(frameNumber, m[1]);
        const Mat& matrix = *matrixPtr;
        expandIfNecessary(output, cursor, origin, matrix.size());
        Mat temp(output,cv::Rect(0, 0, matrix.cols, matrix.rows));
        assert(temp.size() == matrix.size());
        if (matrix.type() == CV_8UC3 || matrix.type() == CV_8UC1)
          math::copyMatrix(temp, matrix);
        else if (matrix.type() == CV_32FC1 || matrix.type() == CV_64FC1)
          math::copyMatrix(temp, matrix * 255.);
        else {
          std::cerr << "Warning: attempting to visualise a matrix \"" << m[1] 
                    << "\" of unsupported type: " 
                    << cvMatTypeToString(matrix.type()) << std::endl;
            
        }
      }
      else if (regex_match(*it, m, std::regex("cropToRect (.*)"))) {
        BlackBoardPointer<cv::Rect> rectPtr = blackBoard.get<cv::Rect>(frameNumber, m[1]);
        const cv::Rect& rect = *rectPtr;
        Mat temp(rect.height, rect.width, output.type());
        Mat temp2(output, rect);
        temp2.copyTo(temp);
        output = temp;
      }
      else if (regex_match(*it, m, std::regex("showAsms"))) {
        vector<BlackBoardPointer<Asm::Instance> > asmPtrs;
        if (blackBoard.has(frameNumber, ASM_TRACKER_LEFT_HAND_ASM_INSTANCE))
          asmPtrs.push_back(blackBoard.get<Asm::Instance>(frameNumber, ASM_TRACKER_LEFT_HAND_ASM_INSTANCE));
        if (blackBoard.has(frameNumber, ASM_TRACKER_LEFT_HAND_ASM_INSTANCE))
          asmPtrs.push_back(blackBoard.get<Asm::Instance>(frameNumber, ASM_TRACKER_RIGHT_HAND_ASM_INSTANCE));
        if (blackBoard.has(frameNumber, ASM_TRACKER_LEFT_HAND_ASM_INSTANCE))
          asmPtrs.push_back(blackBoard.get<Asm::Instance>(frameNumber, ASM_TRACKER_HEAD_ASM_INSTANCE));
        vector<Asm::Instance> asms;
        for (auto it = asmPtrs.cbegin(); it != asmPtrs.cend(); ++it)
          asms.push_back(**it);
        drawAsms(output, asms);
      }
      else if (regex_match(*it, m, std::regex("showBlobs"))) {
        if (blackBoard.has(frameNumber, 
                           BLACKBOARD_BLOBS_ENTRY)) {
          BlackBoardPointer<vector<Blob> > blobs = 
            blackBoard.get<vector<Blob>>(frameNumber,
                                         BLACKBOARD_BLOBS_ENTRY);
          
          drawBlobs(output, *blobs);
        }
      }
      else if (regex_match(*it, m, std::regex("showBodyParts"))) {
        if (blackBoard.has(frameNumber, BODYPARTCOLLECTOR_BLACKBOARD_ENTRY)) {
          BlackBoardPointer<list<BodyPart> > bodyParts = blackBoard.get<list<BodyPart>>(frameNumber, BODYPARTCOLLECTOR_BLACKBOARD_ENTRY);
          drawDetectedBodyParts(output, *bodyParts);
    }
      }
      else if (regex_match(*it, m, std::regex("showPoints (.*)"))) {
        BlackBoardPointer<boost::any> pointsAnyPtr = blackBoard.get(frameNumber, m[1]);
        const boost::any& pointsAny = *pointsAnyPtr;
        assert(pointsAny.type() == typeid(std::vector<cv::Point>) ||
               pointsAny.type() == typeid(std::vector<cv::Point2f>));
        if (pointsAny.type() == typeid(std::vector<cv::Point>)) {
          BlackBoardPointer<std::vector<cv::Point> > pointsPtr = 
            blackBoard.get<std::vector<cv::Point>>(frameNumber, m[1]);
          const std::vector<cv::Point>& points = *pointsPtr;
          for (auto jt = points.cbegin(); jt != points.cend(); ++jt)    
            circle(output, *jt, 3, cv::Scalar::all(255), CV_FILLED);
        }
        if (pointsAny.type() == typeid(std::vector<cv::Point2f>)) {
          BlackBoardPointer<std::vector<cv::Point2f> > pointsPtr = blackBoard.get<std::vector<cv::Point2f>>(frameNumber, m[1]);
          const std::vector<cv::Point2f>& points = *pointsPtr;
            
          for (auto jt = points.cbegin(); jt != points.cend(); ++jt)    
            circle(output, *jt, 3, cv::Scalar::all(255), CV_FILLED);
        }
      }
      else if (regex_match(*it, m, std::regex("showFrame depth"))) {
        for (size_t i = 0; i < frameSource.getNumberOfTracks(); ++i) {
          if (frameSource.getTrack(i).getTrackInfo().type == 
              slmotion::TrackInfo::Type::DEPTH_DATA) {
            cv::Mat depthFrame =
              convertToMarkusFormat(frameSource.getTrack(i)[frameNumber]);
            expandIfNecessary(output, cursor, origin, depthFrame.size());
            Mat temp(output(cv::Rect(cursor.x, cursor.y, depthFrame.cols, 
                                     depthFrame.rows)));
            depthFrame.copyTo(temp);
            break;
          }
          else if (i + 1 == frameSource.getNumberOfTracks())
            throw FrameSourceException("No depth track was found!");
        }
      }
      else if (regex_match(*it, m, std::regex("showKinectSkeleton"))) {
        for (size_t i = 0; i < frameSource.getNumberOfTracks(); ++i) {
          if (frameSource.getTrack(i).getTrackInfo().type == 
              slmotion::TrackInfo::Type::RAW_VECTOR && 
              frameSource.getTrack(i).size() > frameNumber &&
              frameSource.getTrack(i)[frameNumber].cols == 1 && 
              frameSource.getTrack(i)[frameNumber].rows == 30) {
            // here the first raw vector track of correct size is assumed to contain skeleton data
            // this may not be an accurate assumption, however
            const cv::Mat skeleton(frameSource.getTrack(i)[frameNumber]);
            cv::Point2d tl(0,0), br(0,0); // extreme points
            for (int i = 0; i < 16; ++i) {
              tl.x = std::min(skeleton.at<double>(2*i,0), tl.x);
              br.x = std::max(skeleton.at<double>(2*i,0), br.x);
              tl.y = std::min(skeleton.at<double>(2*i+1,0), tl.y);
              br.y = std::max(skeleton.at<double>(2*i+1,0), br.y);
            }
            // expandIfNecessary(output, cursor, origin, cv::Size(br.x-tl.x, br.y-tl.y));
            for (int i = 0; i < 16; ++i) 
              cv::line(output, 
                       cv::Point(cursor.x + origin.x + 
                                 skeleton.at<double>(2*nite_linestart[i], 0), 
                                 cursor.y + origin.y + 
                                 skeleton.at<double>(2*nite_linestart[i]+1, 0)),
                       cv::Point(cursor.x + origin.x + 
                                 skeleton.at<double>(2*nite_lineend[i], 0),
                                 cursor.y + origin.y +
                                 skeleton.at<double>(2*nite_lineend[i]+1,0)), 
                       cv::Scalar(0,255,255),3); 
            break;
          }
          else if (i + 1 == frameSource.getNumberOfTracks())
            throw FrameSourceException("No depth track was found!");
        }
      }
      else if (regex_match(*it, m, std::regex("showFrame ([0-9]+)"))) {
        size_t i = boost::lexical_cast<size_t>(m[1]);
        if (i >= frameSource.getNumberOfTracks())
          throw FrameSourceException(boost::lexical_cast<std::string>(i) + 
                                     " exceeds the number of tracks (" +
                                     boost::lexical_cast<std::string>(frameSource.getNumberOfTracks())
                                     + ") for this frame source!");
        cv::Mat frame = frameSource.getTrack(i).getTrackInfo().type == 
          slmotion::TrackInfo::Type::DEPTH_DATA ?
          convertToMarkusFormat(frameSource.getTrack(i)[frameNumber]) :
          frameSource.getTrack(i)[frameNumber];
        expandIfNecessary(output, cursor, origin, frame.size());
        Mat temp(output(cv::Rect(cursor.x + origin.x, cursor.y + origin.y, 
                                 frame.cols, frame.rows)));
        frame.copyTo(temp);
      }
      else if (regex_match(*it, m, std::regex("showFrame"))) {
        expandIfNecessary(output, cursor, origin, cv::Size(inFrame.cols, inFrame.rows));
        Mat temp(output(cv::Rect(cursor.x + origin.x, cursor.y + origin.y, inFrame.cols, 
                                 inFrame.rows)));
        inFrame.copyTo(temp);
      }
      else if (regex_match(*it, m, std::regex("showRects (.*)"))) {
        BlackBoardPointer<std::vector<cv::Rect> > rectsPtr = blackBoard.get<std::vector<cv::Rect>>(frameNumber, m[1]);
        const std::vector<cv::Rect>& rects = *rectsPtr;
        for (auto jt = rects.begin(); jt != rects.end(); ++jt)
          rectangle(output, jt->tl(), jt->br(), Scalar::all(255));
      }
      else if (regex_match(*it, m, std::regex("showRect (.*)"))) {
        BlackBoardPointer<cv::Rect> rect = blackBoard.get<cv::Rect>(frameNumber, m[1]);
        rectangle(output, rect->tl(), rect->br(), Scalar::all(255));
      }
      else if (regex_match(*it, m, std::regex("locate ([\\-0-9]+|frame-width) "
                                              "([\\-0-9]+|frame-height)"))) {
        cursor = cv::Point2i(m[1] == "frame-width"  ? inFrame.cols : boost::lexical_cast<int>(m[1]), 
                             m[2] == "frame-height" ? inFrame.rows : boost::lexical_cast<int>(m[2]));
        // cv::Size newMinSize(output.size());
        // if (cursor.x > output.cols)
        //   newMinSize.width = cursor.x;
        // if (cursor.x < origin.x)
        //   newMinSize.width += origin.x + cursor.x;
        // if (cursor.y > output.rows)
        //   newMinSize.height = cursor.y;
        // if (cursor.y < origin.y)
        //   newMinSize.height += origin.y + cursor.y;
            
        // expandIfNecessary(output, cursor, origin, newMinSize);
      }
      else if (regex_match(*it, m, std::regex("addAnnotationBar"))) {
        addAnnotationBar(output, frameNumber, frameSource.getLabel(), blackBoard);
      }
#ifdef SLMOTION_ENABLE_LIBFLANDMARK
      else if (regex_match(*it, m, std::regex("showSuviSegments"))) {
        BlackBoardPointer<cv::Mat> segmentsPtr = blackBoard.get<cv::Mat>(frameNumber, FACESUVISEGMENTS_BLACKBOARD_ENTRY);
        const cv::Mat& segments = *segmentsPtr;
        for (int i = 0; i < segments.rows; ++i) {
          for (int j = 0; j < segments.cols; ++j) {
            if (segments.at<uchar>(i,j) > 0) {
              cv::Vec3b& target = output.at<cv::Vec3b>(i,j);
              switch(static_cast<SuviSegment>(segments.at<uchar>(i,j))) {
              case SuviSegment::NIL:
                target = cv::Vec3b(0,0,0);
                break;
              case SuviSegment::CHEEKS:
                target[1] = target[2] = 0;
                break;
              case SuviSegment::NOSE:
                target[0] = target[2] = 0;
                break;
              case SuviSegment::MOUTH:
                target[0] = target[1] = 0;
                break;
              case SuviSegment::NECK:
                target[0] = 0;
                break;
              case SuviSegment::FOREHEAD:
                target[1] = 0;
                break;
              case SuviSegment::OTHER:
                target[0] = target[1] = target[2] =
                  target[0] * 0.299 + target[1] * 0.587 + target[2] * 0.299;
                break;
              }
            }
          }
        }
      }
#endif // SLMOTION_ENABLE_LIBFLANDMARK
      else
        throw SLMotionException("Invalid visualisation script!");
    }
    return output;
  }



  cv::Mat Visualiser::visualise(FrameSource& frameSource, 
                                const BlackBoard& blackBoard, 
                                size_t frameNumber) {
    Mat temp;

    if (complexVisualisation.length() > 0)
      temp = drawComplexVisualisation(complexVisualisation, frameSource, 
                                      blackBoard, frameNumber);
    else
      // inFrame.copyTo(temp);
      frameSource[frameNumber].copyTo(temp);

    if (showSkinMask && blackBoard.has(frameNumber, 
                                       SKINDETECTOR_BLACKBOARD_MASK_ENTRY)) {
      BlackBoardPointer<cv::Mat> skinMask = blackBoard.get<cv::Mat>(frameNumber, SKINDETECTOR_BLACKBOARD_MASK_ENTRY);
      drawSkinMask(temp, *skinMask);
    }
    


    if (blackBoard.has(frameNumber, 
                       BLACKBOARD_BLOBS_ENTRY)) {
      BlackBoardPointer<vector<Blob> > blobs = 
        blackBoard.get<vector<Blob>>(frameNumber,
                                     BLACKBOARD_BLOBS_ENTRY);

      if (showBlobs)
        drawBlobs(temp, *blobs);

      if (showBlobCentroids) 
        drawBlobCentroids(temp, *blobs);
    }



    if (showBodyParts && blackBoard.has(frameNumber, BODYPARTCOLLECTOR_BLACKBOARD_ENTRY)) {
      BlackBoardPointer<list<BodyPart> > bodyParts = blackBoard.get<list<BodyPart>>(frameNumber, BODYPARTCOLLECTOR_BLACKBOARD_ENTRY);
      drawDetectedBodyParts(temp, *bodyParts);
    }

    // note, the Asm Fitting Context won't probably match the one in actual
    // use, so beware
    // if (showGradientMap)
    //   drawGradientMap(temp, temp, 
    //                   asmFittingContext.sobelApertureSize, 
    //                   asmFittingContext.equaliseHistogram, 
    //                   asmFittingContext.gradientMethod, 
    //                   showGradientMap2d);

    // if (showCannyMap)
    //   drawCannyMap(temp, temp, asmFittingContext.cannyThreshold1, 
    //                asmFittingContext.cannyThreshold2, 
    //                asmFittingContext.equaliseHistogram);



    if (showFace && blackBoard.has(frameNumber, 
                                   FACEDETECTOR_BLACKBOARD_ENTRY)) {
      BlackBoardPointer<cv::Rect> faceLocation = blackBoard.get<cv::Rect>(frameNumber, FACEDETECTOR_BLACKBOARD_ENTRY);
      drawFace(temp, *faceLocation);
    }



    if (showCorners && 
        blackBoard.has(frameNumber, 
                       KLTTRACKER_BLACKBOARD_TRACKED_POINTS_ENTRY)) {
      BlackBoardPointer<std::set<TrackedPoint> > kltPoints = 
        blackBoard.get<std::set<TrackedPoint>>(frameNumber, 
                                               KLTTRACKER_BLACKBOARD_TRACKED_POINTS_ENTRY);
      drawCorners(temp, *kltPoints);
    }



    if (showAsms) {
    //   // Create temporary arrays for ASMs and respective anchors
    //   vector<const PDM*> asms { NULL, NULL, NULL };
    // if (blackBoard.has(frameNumber, ASM_TRACKER_HEAD_ASM))
    //   asms[0] = &blackBoard.get<PDM>(frameNumber, ASM_TRACKER_HEAD_ASM);

    // if (blackBoard.has(frameNumber, ASM_TRACKER_LEFT_HAND_ASM))
    //   asms[1] = &blackBoard.get<PDM>(frameNumber, 
    //                                  ASM_TRACKER_LEFT_HAND_ASM);

    // if (blackBoard.has(frameNumber, ASM_TRACKER_RIGHT_HAND_ASM))
    //   asms[2] = &blackBoard.get<PDM>(frameNumber, 
    //                                  ASM_TRACKER_RIGHT_HAND_ASM);
      vector<BlackBoardPointer<Asm::Instance> > asmPtrs;
      if (blackBoard.has(frameNumber, ASM_TRACKER_LEFT_HAND_ASM_INSTANCE))
        asmPtrs.push_back(blackBoard.get<Asm::Instance>(frameNumber, ASM_TRACKER_LEFT_HAND_ASM_INSTANCE));
      if (blackBoard.has(frameNumber, ASM_TRACKER_LEFT_HAND_ASM_INSTANCE))
        asmPtrs.push_back(blackBoard.get<Asm::Instance>(frameNumber, ASM_TRACKER_RIGHT_HAND_ASM_INSTANCE));
      if (blackBoard.has(frameNumber, ASM_TRACKER_LEFT_HAND_ASM_INSTANCE))
        asmPtrs.push_back(blackBoard.get<Asm::Instance>(frameNumber, ASM_TRACKER_HEAD_ASM_INSTANCE));


    // vector<const Point*> anchors { NULL, NULL, NULL };
    // if (blackBoard.has(frameNumber, ASM_TRACKER_HEAD_ANCHOR_BLACKBOARD_ENTRY))
    //   anchors[0] = &blackBoard.get<cv::Point>(frameNumber, ASM_TRACKER_HEAD_ANCHOR_BLACKBOARD_ENTRY);

    // if (blackBoard.has(frameNumber, ASM_TRACKER_LEFT_HAND_ANCHOR_BLACKBOARD_ENTRY))
    //   anchors[1] = &blackBoard.get<cv::Point>(frameNumber, ASM_TRACKER_LEFT_HAND_ANCHOR_BLACKBOARD_ENTRY);

    // if (blackBoard.has(frameNumber, ASM_TRACKER_RIGHT_HAND_ANCHOR_BLACKBOARD_ENTRY))
    //   anchors[2] = &blackBoard.get<cv::Point>(frameNumber, ASM_TRACKER_RIGHT_HAND_ANCHOR_BLACKBOARD_ENTRY);

      // drawAsms(temp, asms, anchors);
      vector<Asm::Instance> asms;
      for (auto it = asmPtrs.cbegin(); it != asmPtrs.cend(); ++it)
        asms.push_back(**it);
      drawAsms(temp, asms);
    }



    if (showEstimatedAnchors) {
    vector<const Point*> anchorGuesses = { NULL, NULL, NULL };
    if (blackBoard.has(frameNumber, ASM_TRACKER_HEAD_ANCHOR_GUESS_BLACKBOARD_ENTRY))
      anchorGuesses[0] = &*blackBoard.get<cv::Point>(frameNumber, ASM_TRACKER_HEAD_ANCHOR_GUESS_BLACKBOARD_ENTRY);

    if (blackBoard.has(frameNumber, ASM_TRACKER_RIGHT_HAND_ANCHOR_GUESS_BLACKBOARD_ENTRY))
      anchorGuesses[1] = &*blackBoard.get<cv::Point>(frameNumber, ASM_TRACKER_RIGHT_HAND_ANCHOR_GUESS_BLACKBOARD_ENTRY);
    
    if (blackBoard.has(frameNumber, ASM_TRACKER_LEFT_HAND_ANCHOR_GUESS_BLACKBOARD_ENTRY))
      anchorGuesses[2] = &*blackBoard.get<cv::Point>(frameNumber, ASM_TRACKER_LEFT_HAND_ANCHOR_GUESS_BLACKBOARD_ENTRY);

      drawEstimatedAnchors(temp, anchorGuesses);
    }


    drawMDFeatures(temp, frameNumber, showMotionVectors, 
                   showAccelerationVectors, showTrackedPoints, 
                   blackBoard);

    return temp;
  }



  void Visualiser::drawSkinMask(cv::Mat& target, const cv::Mat& skinMask) {
    assert(target.type() == CV_8UC3);
    assert(skinMask.type() == CV_8UC1);
    assert(skinMask.cols == target.cols);
    assert(target.rows >= skinMask.rows);

    cv::MatIterator_<Vec3b> it = target.begin<Vec3b>();
    cv::MatConstIterator_<uchar> mit = skinMask.begin<uchar>();
    while(mit != skinMask.end<uchar>()) 
      *it++ = *mit++ ? Vec3b(255,255,255) : Vec3b(0,0,0);    
  }



#if 0
  void Visualiser::drawGradientMap(const Mat& source, Mat& target, 
                                   int apertureSize, bool equaliseHistogram,
                                   PDM::AsmFittingContext::GradientType
                                   gradientType, bool draw2d) {
    assert(target.type() == CV_8UC3);
    assert(target.size() == source.size());

    double minimum, maximum;

    Mat gradientImage, gradx, grady;
    if (draw2d) 
      PDM::computeGradientMap2d(source, gradx, grady, apertureSize, equaliseHistogram, gradientType);

    // gradientImage will always be a map of magnitude
    gradientImage = PDM::computeGradientMap(source, apertureSize, equaliseHistogram, gradientType);
  
    minMaxLoc(gradientImage, &minimum, &maximum);
    Mat temp(target.size(), CV_8UC3);

    for (int i = 0; i < target.size().height; i++) {
      for (int j = 0; j < target.size().width; j++) {
        if (draw2d) {
          Vec3b& elem = temp.at<Vec3b>(i, j);
          elem[1] = 255;
          double d = gradientImage.at<short>(i, j);
          elem[2] = abs(d*255) / std::max(fabs(minimum), maximum);
          d = atan2(-grady.at<short>(i, j), gradx.at<short>(i, j));

          if (d < 0)
            d += 2.0 * M_PI;

          elem[0] = abs((d/(2.0*M_PI)*255.0));
        }
        else {
          Vec3b& elem = target.at<Vec3b>(i, j);
          double d = gradientImage.at<short>(i, j);
          elem[0] = d < 0 ? abs(d*255) / std::max(fabs(minimum), maximum) : 0;
          elem[1] = 0;
          elem[2] = d > 0 ? (d*255) / std::max(fabs(minimum), maximum) : 0;
        }
      }
    }

    if (draw2d) 
      cvtColor(temp, target, CV_HSV2BGR);
  
  }
#endif



  void Visualiser::drawCannyMap(const Mat& source, Mat& target, 
                                double threshold1, double threshold2, 
                                bool equaliseHistogram) {
    assert(source.type() == CV_8UC3);
    assert(target.type() == CV_8UC3);

    Mat gsi;
    cvtColor(source, gsi, CV_BGR2GRAY);

    if (equaliseHistogram)
      equalizeHist(gsi, gsi);

    Mat temp;
    Canny(gsi, temp, threshold1, threshold2, 7, true);
    for (int i = 0; i < target.rows; i++) {
      for (int j = 0; j < target.cols; j++) {
        Vec3b& elem = target.at<Vec3b>(i, j);
        elem[2] = elem[1] = elem[0] = temp.at<uchar>(i,j);
      }
    }
  }



  void Visualiser::drawFace(cv::Mat& target, const cv::Rect& faceLocation) {
    rectangle(target, faceLocation.tl(), faceLocation.br(), Scalar::all(255));
    ellipse(target, Point(faceLocation.x + faceLocation.width / 2, 
			     faceLocation.y + faceLocation.height / 2), 
	    Size(faceLocation.width/2, faceLocation.height/2), 0, 0, 360,
	    Scalar::all(255));
  }



  void Visualiser::drawCorners(Mat& target, 
                               const std::set<TrackedPoint>& kltPoints) {
    for (auto i = kltPoints.cbegin(); i != kltPoints.cend(); ++i) 
      circle(target, static_cast<cv::Point2f>(*i), 3, Scalar::all(255));
  }



  void Visualiser::drawBlobCentroids(Mat& target,
                                     const vector<Blob>& blobs) {
    for (vector<Blob>::const_iterator it = blobs.begin();
         it != blobs.end(); ++it) {
      const Point& p = it->getCentroid();
      
      line(target, Point(p.x-4, p.y-4), Point(p.x+4, p.y+4),
           Scalar(255, 255, 255), 3);
      
      line(target, Point(p.x-4, p.y+4), Point(p.x+4, p.y-4),
           Scalar(255, 255, 255), 3);
    }
  }



  void Visualiser::drawEstimatedAnchors(cv::Mat& target,
                                        const vector<const Point*>& anchors) {
    assert(anchors.size() == 3);
    for (int i = 0; i < 3; i++) {
      const Scalar& col =
        i == 0 ? getBodyPartColour(BodyPart::HEAD) :
        i == 1 ? getBodyPartColour(BodyPart::RIGHT_HAND) :
        getBodyPartColour(BodyPart::LEFT_HAND);
      if (anchors[i])
        circle(target, *anchors[i], 4, col, CV_FILLED);
    }
  }



  void Visualiser::drawMDFeatures(Mat& target, size_t frnumber, 
                                  bool show_mvec,
                                  bool show_avec, 
                                  bool show_tracked_points,
                                  const BlackBoard& blackBoard) {
    if (!(show_tracked_points || show_mvec || show_avec))
      return;

    if (!blackBoard.has(frnumber, FEATURECREATOR_KLT_PRUNED_POINTS_BLACKBOARD_ENTRY))
      return;

    if (!blackBoard.has(frnumber, FEATURECREATOR_KLT_IDENTITIES))
      return;

    if (!blackBoard.has(frnumber, FEATURECREATOR_KLT_VELOCITY_VECTORS_BLACKBOARD_ENTRY))
      return;

    if (!blackBoard.has(frnumber, FEATURECREATOR_KLT_ACCELERATION_VECTORS_BLACKBOARD_ENTRY))
      return;

    BlackBoardPointer<tracked_point_map_t> points =
      blackBoard.get<tracked_point_map_t>(frnumber, FEATURECREATOR_KLT_PRUNED_POINTS_BLACKBOARD_ENTRY);

    BlackBoardPointer<point_id_to_part_name_map_t> bodyParts = 
      blackBoard.get<point_id_to_part_name_map_t>(frnumber,
                                                  FEATURECREATOR_KLT_IDENTITIES);

    BlackBoardPointer<velocity_vector_map_t> velocityVectors =
      blackBoard.get<velocity_vector_map_t>(frnumber,
                                            FEATURECREATOR_KLT_VELOCITY_VECTORS_BLACKBOARD_ENTRY);

    BlackBoardPointer<acceleration_vector_map_t> accelerationVectors =
      blackBoard.get<acceleration_vector_map_t>(frnumber,
                                                FEATURECREATOR_KLT_ACCELERATION_VECTORS_BLACKBOARD_ENTRY);

    for (auto it = points->cbegin(); it != points->cend(); ++it) {
      Point p1, p2;
      const BodyPart::PartName& b = bodyParts->find(it->first)->second;

      p1 = static_cast<TrackedPoint::point_t>(it->second);

      if (show_tracked_points) 
        circle(target, p1, 3, getBodyPartColour(b));

      if (show_mvec) {
        p2 = velocityVectors->find(it->first)->second;
        p2 += p1;
        line(target, p1, p2, getBodyPartColour(b));
      }

      if (show_avec) {
        p2 = accelerationVectors->find(it->first)->second;
        p2 += p1;
        line(target, p1, p2, getBodyPartColour(b));
      }
    }
  }



  void Visualiser::addAnnotationBar(cv::Mat& target, size_t frnumber, 
				    const std::string& label,
				    const BlackBoard& blackBoard) const {
    // // extend the frame so that the annotation bar fits in
    Mat temp;
    copyMakeBorder(target, temp, 0, ANNBARHEIGHT, 0, 0, 
                   cv::BORDER_CONSTANT, Scalar::all(0));
    target = temp;

    std::string frtext = slio->parseAnnFormat(annFormat, frnumber,
					      label, blackBoard);

    std::string text = frtext;
    std::vector<size_t> dotpos;
    while (1) {
      size_t p = text.find("\xc3");
      if (p==std::string::npos)
	break;

      bool err = false;
      if (text[p+1]=='\x84')
	text[p] = 'A';
      else if (text[p+1]=='\x96')
	text[p] = 'O';
      else if (text[p+1]=='\xa4')
	text[p] = 'a';
      else if (text[p+1]=='\xb6')
	text[p] = 'o';
      else {
	text[p] = '?';
	err = true;
      }
      if (!err)
	dotpos.push_back(p);
      text.erase(p+1, 1);
    }

    int fontFace = cv::FONT_HERSHEY_PLAIN;
    double fontScale = 1.0;
    int xoffs = 5, yoffs = 5;
    int baseline = 0;
    cv::Scalar color = Scalar::all(255);

    cv::putText(target, text, Point(xoffs, target.size().height-yoffs),
                fontFace, fontScale, color);
    
    for (size_t i=0; i<dotpos.size(); i++) {
      std::string temptxt = text.substr(0, dotpos[i]+1);
      cv::Size s = cv::getTextSize(temptxt, fontFace, fontScale, 1, &baseline);
      /*
      std::cout << "[" << temptxt << "] " << s.width << "x" << s.height
		<< std::endl;
      */
      int pointy = target.size().height-yoffs-s.height;
      cv::Point pt1a = { xoffs+s.width-7, pointy   };
      cv::Point pt1b = { xoffs+s.width-7, pointy-1 };
      cv::Point pt2a = { xoffs+s.width-4, pointy   };
      cv::Point pt2b = { xoffs+s.width-4, pointy-1 };
      cv::line(target, pt1a, pt1b, color);
      cv::line(target, pt2a, pt2b, color);
    }
  }

#if 0
  static Size computeCombinedFrameSize(const vector<Mat>& visuals,
                                  Visualiser::CombinationStyle cs);

    /**
     * Selects the style automagically, by computing which style creates
     * frames closer to 16:9 aspect ratio
     */
  static Visualiser::CombinationStyle selectCombinationStyle(const vector<Mat>& visuals) {
      Size l2r = computeCombinedFrameSize(visuals, Visualiser::LEFT_TO_RIGHT);
      Size t2b = computeCombinedFrameSize(visuals, Visualiser::TOP_TO_BOTTOM);
      double l2rar = static_cast<double>(l2r.width) / static_cast<double>(l2r.height);
      double t2bar = static_cast<double>(t2b.width) / static_cast<double>(t2b.height);
      if (std::abs(t2bar - 16./9.) < std::abs(l2rar - 16./9.))
        return Visualiser::TOP_TO_BOTTOM;
      return Visualiser::LEFT_TO_RIGHT;
    }

  static Size computeCombinedFrameSize(const vector<Mat>& visuals,
                                  Visualiser::CombinationStyle cs) {
      if (cs == Visualiser::AUTOMAGIC)
        cs = selectCombinationStyle(visuals);

      Size s;

      if (cs == Visualiser::LEFT_TO_RIGHT) {
        for (vector<Mat>::const_iterator it = visuals.begin();
             it != visuals.end(); ++it) 
          s = Size(s.width + it->size().width,
                   std::max(s.height, it->size().height));      
      }
      else if (cs == Visualiser::TOP_TO_BOTTOM) {
        for (vector<Mat>::const_iterator it = visuals.begin();
             it != visuals.end(); ++it) 
          s = Size(std::max(s.width, it->size().width),
                   s.height + it->size().height);
      }
      else
        assert(false && "Unhandled enum");
      return s;
    }
  



  Mat Visualiser::combineVisuals(const vector<Mat>& visuals,
                                 size_t frnumber, const std::string& label,
				 const BlackBoard& blackBoard) const {
    assert(visuals.size() > 0);

    // select the correct style if automagic
    CombinationStyle cs = combinationStyle == AUTOMAGIC ?
      selectCombinationStyle(visuals) : combinationStyle;

    // compute the size for the combined frame
    Size s = computeCombinedFrameSize(visuals, cs);
    Mat temp(disableAnnotationBar ? s.height : ANNBARHEIGHT + s.height, 
             s.width, CV_8UC3, Scalar::all(0));

    // copy images
    Rect r(0,0,0,0);
    // dimension, relevant for shifting
    int& dim = cs == LEFT_TO_RIGHT ? r.width : r.height;
    // position to shift
    int& pos = cs == LEFT_TO_RIGHT ? r.x : r.y; 
    
    for (vector<Mat>::const_iterator it = visuals.begin();
         it != visuals.end(); ++it) {
      r.width = it->cols;
      r.height = it->rows;
      Mat roi(temp, r);
      it->copyTo(roi);
      pos += dim;
    }
    
    // if (!disableAnnotationBar)
    //   addAnnotationBar(temp, frnumber, label, blackBoard);

    return temp;
  }
#endif
}
