#ifndef SLMOTION_VISUALISER
#define SLMOTION_VISUALISER

#include "FrameSource.hpp"
#include "Asm.hpp"
#include <list>
#include <memory>
#include <set>
#include "BodyPart.hpp"
#include "SLIO.hpp"
// #include "TrackedPoint.hpp"
#include "BlackBoard.hpp"



namespace slmotion {
  class TrackedPoint;

    const int ANNBARHEIGHT = 20; ///< annotation bar (below the image frame) height in pixels



  /**
   * Creates an image that represents the minimum values as 0, and maximum
   * values as 255. Everything else in between is scaled linearly.
   */
  template<typename T>
  cv::Mat createMinMaxMap(const cv::Mat& src) {
    assert(src.channels() == 1);
    cv::Mat img(src.size(), CV_8UC1);

    double minimum, maximum;
    cv::minMaxLoc(src, &minimum, &maximum);
    double ud = 255/(maximum - minimum); // unit difference

    cv::MatConstIterator_<T> sit = src.begin<T>();
    cv::MatIterator_<uchar> iit = img.begin<uchar>();
    while (iit != img.end<uchar>()) 
      *iit++ = (*sit++ - minimum) * ud;
    
    return img;
  }



  /**
   * The visualiser class; visualises the input data somehow
   */
  class Visualiser {
  public:
    /**
     * Performs ordinary mid-analysis visualisation, e.g. by showing the 
     * current skin mask or blobs, then returns the output
     */
    cv::Mat visualise(FrameSource& frameSource, const BlackBoard& blackBoard,
                      size_t frameNumber);



    /**
     * Constructs a visualiser object, notably by associating it with an I/O
     * object.
     */
    Visualiser(std::shared_ptr<SLIO> slio) : 
      slio(slio),
      annFormat("Frame: %f"),
      showBlobs(false),
      showSkinMask(false),
      showBodyParts(false),
      showGradientMap(false),
      showGradientMap2d(false),
      showCannyMap(false),
      showFace(false),
      showCorners(false),
      showBlobCentroids(false),
      showAsms(false),
      showEstimatedAnchors(false),
      showTrackedPoints(false),
      showMotionVectors(false),
      showAccelerationVectors(false) //,
      // combinationStyle(NONE)
    { }



    inline void setAnnFormat(const std::string& f) {
      annFormat = f;
    }



    inline void setShowBlobs(bool b) {
      showBlobs = b;
    }



    inline void setShowSkinMask(bool b) {
      showSkinMask = b;
    }



    inline void setComplexVisualisation(const std::string& s) {
      complexVisualisation = s;
    }



    inline void setShowBodyParts(bool b) {
      showBodyParts = b;
    }



    inline void setShowGradientMap(bool b) {
      showGradientMap = b;
    }



    inline void setShowGradientMap2d(bool b) {
      showGradientMap2d = b;
    }



    inline void setShowCannyMap(bool b) {
      showCannyMap = b;
    }



    inline void setShowFace(bool b) {
      showFace = b;
    }



    inline void setShowCorners(bool b) {
      showCorners = b;
    }



    inline void setShowBlobCentroids(bool b) {
      showBlobCentroids = b;
    }



    inline void setShowAsms(bool b) {
      showAsms = b;
    }



    inline void setShowEstimatedAnchors(bool b) {
      showEstimatedAnchors = b;
    }



    inline void setShowTrackedPoints(bool b) {
      showTrackedPoints = b;
    }



    inline void setShowMotionVectors(bool b) {
      showMotionVectors = b;
    }



    inline void setShowAccelerationVectors(bool b) {
      showAccelerationVectors = b;
    }



#if 0
    /**
     * For specifying the way that several input images are combined
     */
    enum CombinationStyle {
      AUTOMAGIC,
      LEFT_TO_RIGHT,
      TOP_TO_BOTTOM,
      NONE // the default, do not combine
    };



    inline void setCombinationStyle(const CombinationStyle& c) {
      combinationStyle = c;
    }



    /**
     * Combines several output frames, usually produced by the visualise 
     * member function, by appending them left-to-right in the order they
     * appear in the vector, then adding an annotation bar below.
     *
     * The resulting combined image is returned.
     */
    cv::Mat combineVisuals(const std::vector<cv::Mat>& visuals,
                           size_t frnumber, const std::string&,
			   const BlackBoard& blackBoard) const;
#endif



    // void setAsmFittingContext(const PDM::AsmFittingContext& context) {
    //   asmFittingContext = context;
    // }



    /**
     * Attaches a line of text at the bottom of the frame according to 
     * annotation format settings. Typically includes the frame number.
     */
    void addAnnotationBar(cv::Mat& target, size_t frnumber, 
			  const std::string& label,
			  const BlackBoard& blackBoard) const;



  private:
    // No copies
    Visualiser(const Visualiser&) = delete;
    Visualiser& operator=(const Visualiser&) = delete;



    /**
     * Draws a complex visualisation contained within the visualisation script
     *
     * The script consists of simple commands, separated by semi-colons (;)
     *
     * The recognised commands are as follows:
     * - showMatrix <name>
     *   reads in the matrix by the name of <name> from the BlackBoard and draws 
     *   it on the screen (or the output matrix)
     * - cropToRect <name>
     *   reads the rectangle by the <name> from the BlackBoard and crops the output to that size
     * - showPoints <name>
     *   given a point vector by the <name> on the BlackBoard, draws a small white circle at every
     *   point
     * - showFrame
     *   draws the default RGB frame
     * - showFrame <int>
     *   draws the frame from the given track
     * - showFrame depth
     *   draws the first depth track frame
     * - locate <int> <int>
     *   relocates the <<cursor>> i.e. the virtual origin (located at the top-left corner of the
     *   frame) to the given co-ordinates. <int> may also be "frame-width" or "frame-height" for
     *   relative placement.
     * - showRects <name>
     *   given a vector of rectangles <name> on the black board, draws each rectangle
     * - showRect <name>
     *   like above but only draw one rectangle
     * - showSuviSegments
     *   draws Suvi segments (according to flandmarks)
     * - showKinectSkeleton
     *   draws the kinect skeleton
     * - addAnnotationBar
     */
    cv::Mat drawComplexVisualisation(const std::string& visualisationScript,
                                     FrameSource& frameSource,
                                     const BlackBoard& blackBoard,
                                     size_t frameNumber) const;



    /**
     * Draws the skin mask onto the target frame with skin pixels coloured 
     * white and non-skin pixels coloured black
     */
    static void drawSkinMask(cv::Mat& target, const cv::Mat& skinMask);



    /**
     * Draws a map of gradients in a frame with the smallest
     * absolute values in blue and largest in red.
     *
     * Can be done in-place.
     *
     * @param source The source image for which the gradients will be
     * calculated 
     * @param target Target image to store the map
     * @param apertureSize Aperture size for the Sobel operation
     * @param equaliseHistogram If true, perform histogram equalisation on 
     * the grayscale image
     * @param gradientType Selects the approximation method for the gradient
     * @param draw2d Draws a two-dimensional representation with the colour
     * representing the direction of the gradient (in HSV), and the value
     * representing the magnitude. Otherwise, only magnitude is used.
     */
    // static void drawGradientMap(const cv::Mat& source, cv::Mat& target, 
    //                             int apertureSize, bool equaliseHistogram, 
    //     			PDM::AsmFittingContext::GradientType 
    //     			gradientType, bool draw2d);



    /**
     * Draws a map of Canny edges on the image
     * @param source Source image
     * @param target Target image
     * @param threshold1 Canny threshold 1
     * @param threshold2 Canny threshold 2
     * @param equaliseHistogram If true, equalise histogram of the grayscale
     * image
     */
    static void drawCannyMap(const cv::Mat& source, cv::Mat& target, 
                             double threshold1, double threshold2, 
                             bool equaliseHistogram);



    /**
     * Draws an ellipse and a rectangle around the face  in the current frame
     */
    static void drawFace(cv::Mat& target, const cv::Rect& faceLocation);



    /**
     * Draws circles around corners that are being tracked by the feature 
     * tracker in the current frame.
     *
     * @param target Target frame
     *
     * @param KLTPoints points tracked
     */
    static void drawCorners(cv::Mat& target, 
                            const std::set<TrackedPoint>& kltPoints);



    /**
     * Draws a cross at blob centroids in the current frame
     */
    static void drawBlobCentroids(cv::Mat& target, 
                                  const std::vector<Blob>& blobs);



    /**
     * Draws estimated body part PDM anchor points in the current frame
     *
     * The anchors should be ordered as follows:
     * head, right hand, left hand
     * No other points should be present.
     */
    static void drawEstimatedAnchors(cv::Mat& target,
                                     const std::vector<const cv::Point*>& anchors);



    /**
     * Draws circles around motion descriptor features, i.e. corners that 
     * have been tracked for at least three consequtive frames
     *
     * @param frnumber Current frame number
     * @param show_mvec Draw motion vectors
     * @param show_avec Draw acceleration vectors
     * @param show_tracked_points Draw tracked points
     */
    static void drawMDFeatures(cv::Mat& target, size_t frnumber, 
                               bool show_mvec,
                               bool show_avec, 
                               bool show_tracked_points,
                               const BlackBoard& blackBoard);



    std::shared_ptr<SLIO> slio;
    std::string annFormat; ///< Sets the annotation string format
    bool showBlobs; ///< If enabled, blobs are drawn on the image in fancy colours
    bool showSkinMask; ///< If enabled, the skin mask is drawn black-and-white
    bool showBodyParts; ///< If enabled, detected body parts are drawn on the image, just like blobs, but the colours match the presumed body part identity
    bool showGradientMap; ///< If enabled, draws a gradient map on the image
    bool showGradientMap2d; ///< If enabled, draws a 2d gradient map on the image
    bool showCannyMap; ///< If enabled, draws a map of edges as detected by the Canny edge detector
    bool showFace; ///< If enabled, a box is drawn around the detected face
    bool showCorners; ///< if enabled, draw circles around corners as detected by the Shi-Tomasi method
    bool showBlobCentroids; ///< If enabled, a cross is drawn at blob centroids
    bool showAsms; ///< If enabled, draws lines around ASMs
    bool showEstimatedAnchors; ///< If set true, draw colour-filled points where anchor points for body part PDMs are estimated to be located

    bool showTrackedPoints; ///< If enabled, draw circles around corners that have been detected for three consecutive frames

    bool showMotionVectors; ///< If enabled, motion vectors are drawn as lines
    bool showAccelerationVectors; ///< If enabled, acceleration vectors are drawn as line
    // CombinationStyle combinationStyle; ///< Specifies how input images are appended when producing a combined output image
    // PDM::AsmFittingContext asmFittingContext; ///< Options for ASM fitting
    std::string complexVisualisation;
  };



  /**
   * Draws detected body parts as follows:
   * Left hand in red
   * Right hand in blue
   * Head in green
   * Combined parts as a sum
   * i.e. combined hands in magenta,
   *  a left hand and a head in yellow
   *  a right hand and a head in cyan
   *  both hands and a head in white
   *
   * Parts from ambiguous frames are drawn in a darker shade
   */
  void drawDetectedBodyParts(cv::Mat& target, 
                             const std::list<BodyPart>& currentBodyParts);



  struct ColoursStruct {
    // Fancy colours
    const cv::Scalar RED;
    const cv::Scalar DARKRED;
    const cv::Scalar BLUE;
    const cv::Scalar DARKBLUE;
    const cv::Scalar GREEN;
    const cv::Scalar DARKGREEN;
    const cv::Scalar YELLOW;
    const cv::Scalar CYAN;
    const cv::Scalar DARKCYAN;
    const cv::Scalar MAGENTA;
    const cv::Scalar DARKMAGENTA;
    const cv::Scalar WHITE;
    const cv::Scalar GREY;
  };
  extern const ColoursStruct COLOURS;
  
  /**
   * Returns a corresponding colour for the body part.
   * 
   * @param b The body part identity
   * @param ambiguous If set, a darker shade is used
   *
   * @return A colour
   */
  const inline cv::Scalar& getBodyPartColour(const BodyPart::PartName& b,
                                             bool ambiguous = false) {
    switch(b) {
    case BodyPart::HEAD:
      return ambiguous ? COLOURS.DARKGREEN : COLOURS.GREEN;
      break;
      
    case BodyPart::LEFT_HAND:
      return ambiguous ? COLOURS.DARKRED : COLOURS.RED;
      break;
      
    case BodyPart::RIGHT_HAND:
      return ambiguous ? COLOURS.DARKBLUE : COLOURS.BLUE;
      break;
      
    case BodyPart::LEFT_HAND_HEAD:
      return COLOURS.YELLOW;
      break;
      
    case BodyPart::RIGHT_HAND_HEAD:
      return COLOURS.CYAN;
      break;
      
    case BodyPart::LEFT_RIGHT_HAND:
      return COLOURS.MAGENTA;
      break;
      
    case BodyPart::LEFT_RIGHT_HAND_HEAD:
      return COLOURS.WHITE;
      break;
      
    case BodyPart::OTHER:
    case BodyPart::UNKNOWN:
      return COLOURS.GREY;
      break;
    }
    return COLOURS.GREY;
  }



  /**
   * Given a matrix, expands it size by filling black to the areas
   * that need to be filled, so that given the origin and the cursor,
   * an object of the given size can be drawn at the given location
   */
  void expandIfNecessary(cv::Mat& matrix, const cv::Point2i& cursor,
                         cv::Point2i& origin, const cv::Size& objectSize);
}

#endif
 
