#ifndef SLMOTION_XML_DOCUMENT
#define SLMOTION_XML_DOCUMENT

#include "XmlNode.hpp"
#include <memory>

namespace slmotion {
  namespace xml {
    /**
     * Represents a very generic XML document format that can be validated
     */
    class XmlDocument {
    public:
      friend class Validator;
      friend void dumpElement(FILE* stream, const XmlDocument& doc, 
                              const XmlNode& element);


      /**
       * An empty constructor with no validator
       */
      XmlDocument() :
        internalDoc(std::unique_ptr<xmlDoc, decltype(&xmlFreeDoc)>(xmlNewDoc(reinterpret_cast<const xmlChar*>("1.0")), &xmlFreeDoc))
      {}

      /**
       * Sets the schema for validator
       */
      // void setSchema(const std::string& schemaFile);


      void setRootElement(XmlNode& node) {
        setRootElement(std::move(node));
      }

      void setRootElement(XmlNode&& node); 

      /**
       * Initialise the document from a file
       */
      explicit XmlDocument(const std::string& xmlDoc);



      /**
       * Releases any memory allocated for the document
       */
      virtual ~XmlDocument() {
        // xmlFreeDoc(internalDoc);
      }


      
      /**
       * Returns the root element
       */
      inline XmlNode getRoot() {
        return rootElement;
      }
      


      /**
       * Returns the root element
       */
      inline const XmlNode getRoot() const {
        return rootElement;
      }



      // inline xmlDocPtr getInternalDoc() const {
      //   return internalDoc.get();
      // }



      /**
       * Saves the document in the specified file.
       */
      virtual void save(const std::string& filename);



      /**
       * Dumps the document into a stream in textual format.
       */
      friend std::ostream& operator<<(std::ostream&, const XmlDocument&);



#if 0 
      /**
       * Validates the document, and fills it with default values
       *
       * @param xmlDocument A pointer to an XML document
       *
       * @return True if valid
       */
      inline bool validate(xmlDocPtr xmlDocument) {
        return validator.validate(xmlDocument);
      }



      /**
       * Validates the document, and fills it with default values
       *
       * @param xmlDocument A pointer to an XML document
       *
       * @return True if valid
       */
      inline bool validate(XmlDocument& xmlDocument) {
        return validator.validate(xmlDocument.internalDoc.get());
      }
#endif
    


      /**
       * Self validate
       */
      // bool validate();



      /**
       * Move constructor
       */
      XmlDocument(XmlDocument&& other) :
        // validator(std::move(other.validator)),
        internalDoc(std::move(other.internalDoc)) {
        other.internalDoc = NULL;
      }



      /**
       * Move assignment
       */
      XmlDocument& operator=(XmlDocument&& other) {
        if (this != &other) {
          std::swap(this->internalDoc, other.internalDoc);
          // std::swap(this->validator, other.validator);
        }
        return *this;
      }



      /**
       * Copy constructor. Creates a deep copy.
       */
      XmlDocument(const XmlDocument& other) :
        // validator(other.validator),
        internalDoc(xmlCopyDoc(other.internalDoc.get(), true), &xmlFreeDoc)
      { }




      /**
       * Copy-assignment operator. Creates a deep copy.
       */
      XmlDocument& operator=(const XmlDocument& other) {
        if (this != &other) {
          XmlDocument temp(other);
          *this = std::move(temp);
        }
        return *this;
      }

    protected:
      /**
       * Internal function used by the output operator
       */
      virtual std::ostream& print(std::ostream&) const;
      


    private:
      /**
       * Reads in the file, fills the internal document and validates it.
       */
      void initialise(const std::string& filename);



      /**
       * Dumps the internal document data in XML format into the target output
       * stream
       *
       * @param os Target output stream
       * @param doc Document reference
       */
      // friend void debugDump(std::ostream& os, const XmlDocument& doc);



      /**
       * Dumps the internal document data in XML format into the target
       * standard C FILE* stream.
       *
       * @param file Target output FILE*
       * @param doc Document reference
       */
      // friend void debugDump(FILE* file, const XmlDocument& doc);

      typedef std::unique_ptr<xmlDoc, decltype(&xmlFreeDoc)> InternalDocPtr;

      // data members
      // std::shared_ptr<Validator> validator;
      InternalDocPtr internalDoc; ///< Internal document
      XmlNode rootElement;
    };



    /**
     * Dumps the document into a stream in textual format.
     */
    std::ostream& operator<<(std::ostream&, const slmotion::xml::XmlDocument&);
  }
}

#endif
