#ifndef SLMOTION_CONFIGURATION
#define SLMOTION_CONFIGURATION

#include "SLMotionConfiguration.hpp"

#ifdef SLMOTION_ENABLE_PICSOM
#include <DataBase.h>
#endif // SLMOTION_ENABLE_PICSOM

#include "Visualiser.hpp"
#include "ColourSpace.hpp"
#include "Filter.hpp"
#include "FaceDetector.hpp"

namespace slmotion {
  // some forward-declarations
  class KLTTracker;
  class Component;
  class Analyser;

  namespace configuration {
    std::string getDefaultConfigurationFile();



    /**
     * Generates the default configuration file, and outputs it to the ostream
     */
    void generateDefaultConfig(std::ostream& os, size_t maxLineWidth);



    /**
     * Creates a configuration file out of the configuration
     */
    void outputConfig(std::ostream& os, size_t maxLineWidth, 
                      const boost::program_options::variables_map& config);



    /**
     * Sets the global working directory
     */
    void setWorkingDirectory(const std::string& dir);

    std::string getWorkingDirectory();

#if 0
    /**
     * A wrapper around the boost options_description class, including a 
     * section description for the configuration file
     */
    struct ConfigurationFileOptionsDescription {
      /**
       * When generating the default configuration file and parsing existing
       * configuration files, this describes the options used by various
       * components.
       */
      boost::program_options::options_description optsDescription;
      /**
       * When generating the default configuration file, this is an overall
       * description of the component section before the individual options
       * are describede in detail.
       */
      std::string sectionDescription;
    };
#endif



    /**
     * Applies the given AnalyserOpts configuration
     */
    void applyConfiguration(const SLMotionConfiguration& config,
                            Visualiser& visualiser);

    void applyConfiguration(const SLMotionConfiguration& config,
                            SLIO& slio);

    void applyConfiguration(const SLMotionConfiguration& config,
                            FrameSource& frameSource);

    std::vector< std::shared_ptr<Component> > createComponents(const SLMotionConfiguration& config, FrameSource* frameSource, BlackBoard* blackBoard);
    
    /**
     * Parses the configuration file, and returns a corresponding 
     * configuration object
     *
     * @param conffile Configuration filename
     */
    SLMotionConfiguration parseConfigurationFile(std::istream& conffileStream);
    
    /**
     * Opens an ifstream for the given file, and calls the function above
     */
    SLMotionConfiguration parseConfigurationFile(const std::string& conffile);
    
    
    
    /**
     * Creates a program_options variables map from the command line string
     * that can then be evaluated using other functions.
     */
    boost::program_options::variables_map parseCommandLine(int argc, char* argv[]);



    boost::program_options::options_description getVisibleOptionsDescription();



    /**
     * Applies the given command line options to the configuration
     *
     * @param argv Args as received by main
     * @param argc Arg count as received by main
     * @param config Output configuration struct
     */
    void applyCommandLineOptions(int argc, char* argv[], SLMotionConfiguration& config);
    
    
    
    const std::set<std::string> STILL_IMAGE_EXTENSIONS = {
      ".jpg", ".jpeg", ".bmp", ".png", ".gif", ".tif", ".tiff"
    };
    
    

    /**
     * Ordinary video file extensions
     */
    const std::set<std::string> VIDEO_EXTENSIONS = {
      ".avi", ".mp4", ".mpg", ".mov", ".mkv", ".ogm"
    };
    
    

    /**
     * Extensions requiring the customised vid file backend (Athitsos
     * et al.)
     */
    const std::set<std::string> VID_EXTENSIONS = {
      ".vid"
    };



    /**
     * Extensions requiring the customised OpenNI ONI file backend
     */
    const std::set<std::string> ONI_EXTENSIONS = {
      ".oni"
    };



    /**
     * Represents the type of the job when creating an analyser. Mainly 
     * conveys information about whether the job is about analysing a video 
     * file, or a still image sequence.
     *
     * Note: VIDEO_FILE = ordinary video file (avi, mpeg etc.)
     * VID_FILE requires the special vid reader library (Athitsos et al.)
     */
    struct Job {
      enum class JobType { 
        DUMMY, ///< A special "no-input" job
        KINECT, ///< Special kinect short-hand
          IMAGE_SEQUENCE,
          VIDEO_FILE
#ifdef SLMOTION_ENABLE_VIDFILE_SUPPORT
          ,VID_FILE
#endif
#ifdef SLMOTION_ENABLE_OPENNI
          ,ONI_FILE
#endif
          } type;
      
      std::string label;
      std::vector<std::string> filenames;
#ifdef SLMOTION_ENABLE_PICSOM
      std::shared_ptr<picsom::PicSOM> picsom;
      picsom::DataBase *picsom_db;
#endif // SLMOTION_ENABLE_PICSOM

      
      Job(JobType type) : type(type)
#ifdef SLMOTION_ENABLE_PICSOM
			,picsom_db(NULL) 
#endif // SLMOTION_ENABLE_PICSOM
      {}

#ifdef SLMOTION_ENABLE_PICSOM
      Job(JobType type, std::shared_ptr<picsom::PicSOM> picsom, 
	  picsom::DataBase* db) : type(type),
				  picsom(picsom),
				  picsom_db(db) 
      {}
#endif // SLMOTION_ENABLE_PICSOM

      Job(JobType type, const std::vector<std::string>& filenames) :
        type(type), filenames(filenames) {}
    };
    
    
    
    /**
     * Given a job, creates the corresponding frame source
     */
    std::shared_ptr<FrameSource> createFrameSourceForJob(const Job& job,
                                                         size_t frameCacheSize,
                                                         SLMotionConfiguration::VideoBackEnd videoBackEnd = SLMotionConfiguration::OPENCV);



    /**
     * Goes through the options, creates all necessary sub components, 
     * associates them with analyser objects, and returns the objects as a 
     * linked list.
     *
     * @param jobs Jobs (filenames)
     * @param conffiles Configuration files. There should be exactly
     * the same number of configuration files as there are jobs.
     * @param analysers Output parameter. Will be used to store the 
     * analysers.
     * @param globalConfiguration Will be used to store the global 
     * configuration (i.e. the one used by the AnalysisController)
     * @param argc Argc as passed to the main function, or zero otherwise
     * @param argv Argv as passed to the main function or NULL
     */
    void createAnalysers(const std::vector<Job>& jobs, 
                         const std::vector<std::string>& conffiles,
                         std::vector<Analyser>& analysers,
                         SLMotionConfiguration& globalConfiguration,
                         int argc, char* argv[]);
    
    
    /**
     * If zero configuration files are supplied, N default configurations will
     * be returned.
     *
     * If one configuration files are supplied, N configurations matching that
     * configuration will be supplied.
     *
     * If the number of configurations matches the number of configuration 
     * files, every configuration will be different.
     *
     * @param conffiles Configuration file names. There should be zero or one
     * entry OR excatly the same number of entries as in 
     * numberOfConfigurations
     *
     * @param numberOfConfigurations Should be a number equal to or greater 
     * than 1; this is the size of the resulting configuration vector. 
     */
    std::vector<SLMotionConfiguration> parseConfigurationFiles(const std::vector<std::string>& conffiles, size_t numberOfConfigurations);
    
    
    
    /**
     * Takes in filenames, and assigns them to jobs.
     */
    std::vector<Job> assignFilenamesToJobs(const std::vector<std::string>&
					   inputFilenames,
					   const std::string& picsom_db_name = "",
                                           const std::string& picsomArgs = "");



    /**
     * The Kinect short-hand for assigning two input video files to a
     * Kinect job.
     */
    std::vector<Job> assignFilenamesToKinectJobs(const std::vector<std::string>& inputFilenames);



    /**
     * Parses the component string using the following rules:
     *  o replace "all" with a list of all components
     *  o likewise, replace "default" with the default set of components
     *  o expand all aliases
     *  o if the component is prefixed with a -, remove all instances
     *  o otherwise, add as-is
     */
    std::vector<std::string> parseComponents(const std::string& componentString,
                                             const std::map<std::string, std::vector<std::string>>& aliases);
  }
}

#endif
